/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include <sys/types.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>

#include "sock.h"
#include "log.h"


/* Used by sock_get_errstr() */

struct _sock_errassoc
{
  long val;
  char *str;
}
_sock_errstr_list[] =
{
  { SOCK_STAT_OK,           "No error." },
  { SOCK_STAT_PARADOX,      "Paradox caused by improper application programming." },
  { SOCK_STAT_INTERRUPT,    "Socket transaction interrupted." },
  { SOCK_STAT_DISCONNECTED, "Disconnected from remote host." },
  { SOCK_STAT_CONNREFUSED,  "Connection refused by remote host." },
  { SOCK_STAT_TIMEOUT,      "Socket transaction timed out." },
  { SOCK_STAT_NONAME,       "Host name lookup or translation failed." },
  { SOCK_STAT_OVERRUN,      "Socket buffers exhausted." },
  { SOCK_STAT_NOMEM,        "Socket subsystem was denied memory." },
  { SOCK_STAT_MEMFAULT,     "Memory violation in socket subsystem." },
  { SOCK_STAT_ERR,          "Unspecified socket error." },
  { -10000,                 "Unknown socket status." }
};

/* --- CODE section --- */

int _sock_interrupted = 0;
int _sock_initialized = 0;


#ifdef OBSOLETE

void _sock_interrupt(int signal)
{
  _sock_interrupted = 1;
  return;
}

#endif


char *sock_status2str(int status)
{
  int i;

  for (i = 0; _sock_errstr_list[i].val != -10000; i++)
    if (_sock_errstr_list[i].val == status)
      return(_sock_errstr_list[i].str);

  return(_sock_errstr_list[i].str);
}

/*
   Returns:
   TRUE   Continue running subsystem (would block, dry stream, ?interrupt?)
   FALSE  Return to caller (connection lost, ?interrupt?)
*/

int _sock_chkerrno(SOCK *s)
{
  int so_errno = 0;
  int so_errno_size = sizeof(so_errno);


  if ((errno == EINTR || errno == ENOTCONN ||  errno == EFAULT) && s->flags & SOCK_LOG)
    log_put_opt(LOG_DEBUG, 0, "[Sock] Error condition from errno: <%s>.",
                strerror(errno));

  switch (errno)
  {
  case EWOULDBLOCK:
    s->status = SOCK_STAT_OK; return(1);
  case EINTR:
    s->status = SOCK_STAT_INTERRUPT; return(1);
  case EPIPE:
    signal(SIGPIPE, SIG_IGN);
  case ENOTCONN:
    sock_disconnect(s, SOCK_INTERNAL | SOCK_IMMEDIATE);
    return(0);
  case EFAULT:
    sock_disconnect(s, SOCK_INTERNAL | SOCK_IMMEDIATE);
    s->status = SOCK_STAT_MEMFAULT; return(0);
  default:
    break;
  }


  if (getsockopt(s->handle, SOL_SOCKET, SO_ERROR, (void *) &so_errno, 
                 &so_errno_size) >= 0)
  {
    if (s->flags & SOCK_LOG)
      log_put_opt(LOG_DEBUG, 0, "[Sock] Error condition from getsockopt: <%s>.",
                  strerror(so_errno));

    switch (so_errno)
    {
    case EWOULDBLOCK:
      s->status = SOCK_STAT_OK; return(1);
    case EINTR:
      s->status = SOCK_STAT_INTERRUPT; return(1);  /* FIXME: Make configurable. */
    case EPIPE:
      signal(SIGPIPE, SIG_IGN);
    case ENOTCONN:
    sock_disconnect(s, SOCK_INTERNAL | SOCK_IMMEDIATE);
      return(0);
    case EFAULT:
    sock_disconnect(s, SOCK_INTERNAL | SOCK_IMMEDIATE);
      s->status = SOCK_STAT_MEMFAULT; return(0);
    default:
      break;
    }
  }

  if (/* !errno && */ !so_errno)
  {
    s->status = SOCK_STAT_OK;
    return(1);
  }
  
  sock_disconnect(s, SOCK_INTERNAL | SOCK_IMMEDIATE);
  return(0);
}


