/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "types.h"
#include "tt.h"

#include <stdio.h>


static void print_indent(TT *tt, FILE *out)
{
  int i;
  
  for (i = tt_depth(tt) - 1; i; i--) fwrite("  ", 2, 1, out);
}


static void print_data(TT *tt, FILE *out)
{
  int quoted = 0;
  unsigned int i;
  byte c;

  if (tt_has_data(tt))
  {
    if (tt_chr(tt, ' ') != -1)
    {
      fwrite("\"", 1, 1, out);
      quoted = 1;
    }
    
    for (i = 0; ; i++)
    {
      if (!tt_data_get_bytes(tt, &c, i, 1)) break;
      
      switch (c)
      {
        case '{':
        case '}':
        case '"':
        case '\'':
        case '\\':
          fwrite("\\", 1, 1, out);
          fwrite(&c, 1, 1, out);
          break;
        
        case '\n':
          fwrite("\\n", 2, 1, out);
          break;
        
        case '\r':
          fwrite("\\r", 2, 1, out);
          break;
        
        case '\t':
          fwrite("\\t", 2, 1, out);
          break;


        default:
          fwrite(&c, 1, 1, out);
          break;
      }
    }

    if (quoted) fwrite("\"", 1, 1, out);
  }
  else fwrite("\"\"", 2, 1, out);
}


static void print_token(TT *tt, FILE *out, TT_PRINT_MODE mode)
{
  if (mode == TT_PRINT_COMPACT)
  {
    if (tt_get_prev(tt)) fwrite(" ", 1, 1, out);
  }
  else
  {
    if (tt_get_prev(tt)) fwrite("\n", 1, 1, out);
    print_indent(tt, out);
  }

  print_data(tt, out);
}


static void print_start_brace(TT *tt, FILE *out, TT_PRINT_MODE mode)
{
  /* Pre-sub-branch space/brace/space */
      
  if (mode == TT_PRINT_COMPACT)
  {
    fwrite(" { ", 3, 1, out);
  }
  else if (mode == TT_PRINT_KNR)
  {
    fwrite(" {\n", 3, 1, out);
  }
  else if (mode == TT_PRINT_ALLMAN)
  {
    fwrite("\n", 1, 1, out);
    print_indent(tt, out);
    fwrite("{\n", 2, 1, out);
  }
}


static void print_end_brace(TT *tt, FILE *out, TT_PRINT_MODE mode)
{
  /* Post-sub-branch brace */

  if (mode == TT_PRINT_COMPACT)
  {
    fwrite(" }", 2, 1, out);
  }
  else if (mode == TT_PRINT_KNR)
  {
    fwrite("\n", 1, 1, out);
    print_indent(tt, out);
    fwrite("}", 1, 1, out);
  }
  else if (mode == TT_PRINT_ALLMAN)
  {
    fwrite("\n", 1, 1, out);
    print_indent(tt, out);
    fwrite("}", 1, 1, out);
  }
}


static void print_branch(TT *tt, FILE *out, TT_PRINT_MODE mode, int honour_meta)
{
  for (; tt; tt = tt_get_next(tt))
  {
    print_token(tt, out, mode);
    
    if (!tt_is_leaf(tt))
    {
      print_start_brace(tt, out, mode);
      print_branch(tt_get_first_child(tt), out, mode, honour_meta);
      print_end_brace(tt, out, mode);
    }
  }
}


void tt_print_to_file(TT *tt, FILE *out, TT_PRINT_MODE mode, int honour_meta)
{
  print_branch(tt_get_first_child(tt), out, mode, honour_meta);
  fwrite("\n", 1, 1, out);
}
