/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: Interface.hh,v 1.3 2001/12/19 12:46:49 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifndef __INTERFACE_HH_FLAG__
#define __INTERFACE_HH_FLAG__

#include <string>

#include <fwbuilder/FWObject.hh>
#include <fwbuilder/FWException.hh>
#include <fwbuilder/IPAddress.hh>

namespace libfwbuilder
{

class Interface : public FWObject 
{
    private:
    
    // This attribute currently not saved in XML
    int bcast_bits;
    // This attribute currently not saved in XML
    bool ostatus;

    /**
     * Type of interface according SNMP
     * (this attribute currently not saved in XML)
     */
    int snmp_type;

    public:
    
    Interface();
    virtual ~Interface();

    Interface(const Interface *i);
    Interface(const Interface &i);

    /**
     * Creates Interface object with given parameters.
     *
     * @param _bcast_bits The value of the least-significant bit in the IP
     *                   broadcast address used for sending datagrams on
     *                   the (logical) interface associated with the IP
     *                   address of this entry.  For example, when the
     *                   Internet standard all-ones broadcast address is
     *                   used, the value will be 1.  This value applies to
     *                   both the subnet and network broadcasts addresses
     *                   used by the entity on this (logical) interface."
     *
     * @param _snmp_type type of interface, as described in RFC-1213 under
     *                   'ifType' variable description.
     * @param _ostatus,  interface operational status. 'true' means up.
     */
    Interface(const std::string& n,const std::string& a,const std::string& n, 
              int bcast_bits=1, int snmp_type=1, bool _ostatus=true,
	      bool ext=false, bool dyn=false, std::string physical_address="");

    Interface(const std::string& n,const std::string& a,const std::string& n,
	      bool ext, const std::string& dyn, std::string physical_address="");

    Interface(const std::string& n,const std::string& a,const std::string& n,
	      int zone, const std::string& dyn, std::string physical_address="");

    IPNetwork getIPNetwork() const throw(FWException);
    
    virtual void fromXML(xmlNodePtr parent);

    virtual bool  validateChild(FWObject *o);

    DECLARE_FWOBJECT_SUBTYPE(Interface);

    /**
     *   each interface must be associated with some security level. Level
     *   is described by interger number between 0 and 100, with 0 being
     *   least secure and 100 most secure levels. By default class Interface
     *   assigns value of 0 to its security level.
     */
    int  getSecurityLevel();
    void setSecurityLevel(int level);

    /**
     *   Convenience methods: interface is considered "external" if its
     *   security level is 0
     */
    void setExt(bool value);
    bool isExt() const ;


    bool isDyn() const ;
    bool isUp () const { return ostatus; }

    const IPAddress  getIPAddress       () const throw(FWException);
    const std::string&    getPhysicalAddress () const;

    int   getSNMPType() const { return snmp_type; };

    virtual Interface& operator=(const Interface &s);

    const std::string &getLabel() const;
    void          setLabel(const std::string& n);
    
};

}

#endif


