/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: GroupContainerManager.java,v 1.9 2006/02/26 00:59:06 pietschy Exp $
 */

package org.pietschy.command;

import org.pietschy.command.log.Logger;

import javax.swing.*;
import java.awt.*;
import java.util.Collection;

/**
 * A  helper class that GroupCommands use to populate and manages its popup/command {@link Container}s.
 * Typcially this class is repsonsible for rebuilding it's container when the GroupCommands members change.
 * GroupCommands can provide their own implementation by providing a custom implementation of {@link GroupMemberFactory}.
 */
public abstract class
GroupContainerManager
{
   protected Logger log;

   private JComponent itemContainer;
   private Object factory;
   private String faceId;

   public GroupContainerManager()
   {
      log = CommandManager.getLogger(getClass());
   }

   /**
    * Configures this instance to manage the specified menu container using the factory and faceid provided.
    *
    * @param container the container to manage.
    * @param factory   the factory to use when creating menu items.
    * @param faceId    the face id to use.
    */
   public void
   initialise(JComponent container, MenuFactory factory, String faceId)
   {
      this.itemContainer = container;
      this.factory = factory;
      this.faceId = faceId;
   }

   /**
    * Configures this instance to manage the specified button container using the factory and faceid provided.
    *
    * @param container the container to manage.
    * @param factory   the factory to use when creating buttons.
    * @param faceId    the face id to use.
    */
   public void
   configure(JComponent container, ButtonFactory factory, String faceId)
   {
      this.itemContainer = container;
      this.factory = factory;
      this.faceId = faceId;
   }

   /**
    * Called to trigger the manager to rebuild its container using the specified collection of
    * {@link GroupMember}s.
    *
    * @param members the current members of the group.
    */
   public abstract void
   rebuildPopupUsing(Collection members);

   /**
    * Gets the container this instance is managing.
    * @return the container this instance is managing.
    */
   protected JComponent
   getItemContainer()
   {
      return itemContainer;
   }

   /**
    * The factory to use when createing new buttons or menus.  It will be an instance of either {@link MenuFactory} or
    * {@link ButtonFactory}.
    * @return the factory to use when createing new buttons or menus.
    */
   protected Object
   getFactory()
   {
      return factory;
   }

   /**
    * The face id to use when creating new buttons or menus.
    * @return the face id to use when creating new buttons or menus.
    */
   protected String
   getFaceId()
   {
      return faceId;
   }

   public String
   toString()
   {
      return getClass().getName() + "[" + itemContainer.getClass().getName() + "]";
   }

}
