/*
    Copyright (C) 2001 Guillaume Morin, Alcve

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: ent.c,v 1.14 2002/01/15 11:27:14 gmorin Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/ent.c,v $
    $Date: 2002/01/15 11:27:14 $
    $Author: gmorin $
*/


#include <stdlib.h>
#include <nss.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <pthread.h>
#include <mysql/mysql.h>

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#ifdef INCLUDE_MY_SYS
	#include <mysql/my_pthread.h>
#endif

#include "lib.h"
#include "parser.h"
#include "passwd.h"
#include "group.h"
#include "nss-shadow.h"
#include "ent.h"

/* This structure is used to share information between
 * all ***..;ent functions. This is a cache of all IDs */

#ifndef ent_t
	#warning ent_t not defined, falling back to uid_t (should not happen)
	#define ent_t uid_t
#endif

typedef struct {
	ent_t * list; /* list of ids */
	ent_t index;  /* index in the list */
	ent_t max; /* end of the list */
	struct passwdoptions options;
	struct groupoptions goptions;
	struct group * gr_list; /* list of pre-computed group entries. */
	struct shadowoptions soptions;
	MYSQL * mysql_auth;
}ENT_S;

static ENT_S * ent[3] = {NULL,NULL,NULL};

/*EOP*/


/* mutexes to make functions reentrant !! */
static pthread_mutex_t mutex[3] = {PTHREAD_MUTEX_INITIALIZER,PTHREAD_MUTEX_INITIALIZER,PTHREAD_MUTEX_INITIALIZER};


/* frees parsing results */
#if USE_SHADOW
#if USE_GROUP
#define free_parse(s) \
_nss_mysql_free_users(&((s)->options)); \
_nss_mysql_free_groups(&((s)->goptions)); \
_nss_mysql_free_shadow(&((s)->soptions)); 
#else 
#define free_parse(s) \
_nss_mysql_free_users(&((s)->options)); \
_nss_mysql_free_shadow(&((s)->soptions)); 
#endif
#else
#if USE_GROUP
#define free_parse(s) \
_nss_mysql_free_users(&((s)->options)); \
_nss_mysql_free_groups(&((s)->goptions));
#else
#define free_parse(s) \
_nss_mysql_free_users(&((s)->options));
#endif
#endif

#define setent_quit(s,t) \
free_parse(s); \
if ((s)->mysql_auth) { \
  if(t == 1) mysql_query((s)->mysql_auth, "unlock tables"); \
	_nss_mysql_db_close(&(s)->mysql_auth); \
} \
if ((s)->list) free((s)->list); \
if ((s)->gr_list) free((s)->gr_list); \
free(s); \
(s) = NULL; \
pthread_mutex_unlock(&mutex[t]); \
if (result) mysql_free_result(result); \
if (ent_count) free(ent_count)

#define setent_check_mem(x,s,t) if ( (x) == NULL ) { \
	_nss_mysql_log(LOG_ERR,"setent: not enough memory"); \
	if (s) { \
		setent_quit(s,t); \
	} \
	return NSS_STATUS_TRYAGAIN; \
}

#if USE_GROUP
#if USE_SHADOW
#define check_type(t,f) \
switch(t) { \
	case 0: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for passwd(0)"); break; \
	case 1: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for group(1)"); break; \
	case 2: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for shadow(2)"); break; \
	default: _nss_mysql_log(LOG_ERR,f " called for unknow type %d",t); \
			 return NSS_STATUS_UNAVAIL; \
};
#else
#define check_type(t,f) \
switch(t) { \
	case 0: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for passwd(0)"); break; \
	case 1: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for group(1)"); break; \
	default: _nss_mysql_log(LOG_ERR,f " called for unknow type %d",t); \
			 return NSS_STATUS_UNAVAIL; \
};
#endif
#else
#if USE_SHADOW
#define check_type(t,f) \
switch(t) { \
	case 0: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for passwd(0)"); break; \
	case 2: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for shadow(2)"); break; \
	default: _nss_mysql_log(LOG_ERR,f " called for unknow type %d",t); \
			 return NSS_STATUS_UNAVAIL; \
};
#else
#define check_type(t,f) \
switch(t) { \
	case 0: if (DEBUG) _nss_mysql_log(LOG_ERR,f " called for passwd(0)"); break; \
	default: _nss_mysql_log(LOG_ERR,f " called for unknow type %d",t); \
			 return NSS_STATUS_UNAVAIL; \
};
#endif
#endif

/* setent
 * Initializes data for gr**ent functions
 * NOTE this function does _NOT_ use errno
 */
enum nss_status _nss_mysql_setent (int type) {
	char * sql = NULL;
	MYSQL_RES *result = NULL;
	MYSQL_ROW sql_row;
	register unsigned i;
	int * ent_count = 0;
	ent_t max;
	ENT_S ** s;
	pthread_mutex_t * m;
	int ret = 0;
	
	
	check_type(type,"setent");
	s = &ent[type];
	m = &mutex[type];
	
	pthread_mutex_lock(m);
	if ((*s) != NULL) {
		/* ent[type] must be completely reinitialized */
		if (DEBUG) _nss_mysql_log(LOG_ERR,"Warning: setent(%d): ent"
				" was != NULL, it has been deleted", type);
		if ((*s)->list != NULL) 
			free((*s)->list);
		if ((*s)->mysql_auth) {
			my_thread_init();
			_nss_mysql_db_close(&(*s)->mysql_auth);
			my_thread_end();
		}
		free_parse(*s);
		free(*s);
		*s = NULL;
	}
	/* new ent */
	(*s) = malloc(sizeof(ENT_S));
	setent_check_mem(*s,*s,type);
	memset((*s),0,sizeof(ENT_S));

	/* we parse conf files */
	switch(type) {
		case 0: ret = _nss_mysql_read_conf_file("users",&(*s)->options,NULL,NULL); break;
		case 1: ret = _nss_mysql_read_conf_file("groups",&(*s)->options,&(*s)->goptions,NULL); break;
		case 2: ret = _nss_mysql_read_conf_file("shadow",NULL,NULL,&(*s)->soptions); break;
	}
	
	if (! ret) {
		_nss_mysql_log(LOG_ERR,"setent(%d): conf file parsing failed",type);
		setent_quit(*s,type);
		return NSS_STATUS_UNAVAIL;
	}

	/* DB connection */
	switch(type) {
		case 0:
		case 1: ret = _nss_mysql_db_connect(&(*s)->mysql_auth,(*s)->options.host,(*s)->options.dbuser,
				(*s)->options.dbpasswd,(*s)->options.database); break;
		case 2: ret = _nss_mysql_db_connect(&(*s)->mysql_auth,(*s)->soptions.host,(*s)->soptions.dbuser,
				(*s)->soptions.dbpasswd,(*s)->soptions.database); break;
	}
	if (!ret) {
		setent_quit(*s,type);
		return NSS_STATUS_UNAVAIL;
	}

	/* we look for all elements in the DB */
	switch(type) {
		case 0:
			sql = _nss_mysql_sqlprintf("select %s from %s where %s order by %s",
				(*s)->options.uidcolumn,
				(*s)->options.table,
				(*s)->options.where[0] ? (*s)->options.where : "1=1",
				(*s)->options.uidcolumn
			);
			break;
		case 1:
			/*
			 * Count the number of members in each group. Will be used for bulk
			 * retrieval at a later time.
			 */
			sql = _nss_mysql_sqlprintf("select %s,count(*) from %s,%s,%s where %s = %s and %s = %s and %s and %s group by %s order by %s",
				(*s)->goptions.gidcolumn,
				/* from */
				(*s)->goptions.groupinfotable,
				(*s)->goptions.memberstable,
				(*s)->options.table,
				/* where */
				(*s)->goptions.mgroupidcolumn,
				(*s)->goptions.groupidcolumn,
				(*s)->options.useridcolumn,
				(*s)->goptions.museridcolumn,
				(*s)->goptions.where[0] ? (*s)->goptions.where : "1=1",
				(*s)->options.where[0] ? (*s)->options.where : "1=1",
				/* order & group */
			  (*s)->goptions.gidcolumn,
			  (*s)->goptions.gidcolumn
				);
			break;
		case 2:
			sql = _nss_mysql_sqlprintf("select %s from %s where %s order by %s",
				(*s)->soptions.useridcolumn,
				(*s)->soptions.table,
				(*s)->soptions.where[0] ? (*s)->soptions.where : "1=1",
				(*s)->soptions.useridcolumn
			);
	}		

	setent_check_mem(sql,*s,type);
	if (DEBUG) _nss_mysql_log(LOG_ERR,"setent(%d): SQL statement: %s",type,sql);

#if USE_GROUP
	/*
	 * We are going to rely on the count returned by the first SQL statement
	 * in order to parse the second one. Therefore we must make sure the
	 * tables are not changed in the meantime.
	 */
	if(type == 1) {
		char * sql_lock = NULL;
		sql_lock = _nss_mysql_sqlprintf("lock tables %s read, %s read, %s read",
																		(*s)->options.table,
																		(*s)->goptions.groupinfotable,
																		(*s)->goptions.memberstable
																		);
		setent_check_mem(sql_lock,*s,type);
		if (DEBUG) _nss_mysql_log(LOG_ERR,"setent(%d): SQL statement: %s",type,sql_lock);
		if (mysql_query((*s)->mysql_auth, sql_lock)) {
			_nss_mysql_log(LOG_ERR,"setent(%d): mysql_query failed: %s",type,mysql_error((*s)->mysql_auth));
			setent_quit(*s,type);
			my_thread_end();
			return NSS_STATUS_UNAVAIL;
		}
		free(sql_lock);
	}
#endif
	
	if (mysql_query((*s)->mysql_auth, sql)) {
		_nss_mysql_log(LOG_ERR,"setent(%d): mysql_query failed: %s",type,mysql_error((*s)->mysql_auth));
		setent_quit(*s,type);
		my_thread_end();
		return NSS_STATUS_UNAVAIL;
	}
		
	result = mysql_store_result((*s)->mysql_auth);
	if (! result)	{
		_nss_mysql_log(LOG_ERR,"setent(%d): mysql_store_result failed: %s",
				type,mysql_error((*s)->mysql_auth));
		setent_quit(*s,type);
		my_thread_end();
		return NSS_STATUS_UNAVAIL;
	}
	
	max = mysql_num_rows(result);

	(*s)->list = malloc(sizeof(ent_t) * max);
	setent_check_mem((*s)->list,*s,type);
	ent_count = malloc(sizeof(int) * max);
	setent_check_mem(ent_count,*s,type);
	/* Every id is copied to list, so we have a cache now */
	for (i = 0 ; i < max ; ++i) {
		int error;
		sql_row = mysql_fetch_row(result);
		if (! sql_row) {
			_nss_mysql_log(LOG_ERR,"setent(%d): mysql_fetch_row failed: %s.",
					       type,mysql_error((*s)->mysql_auth));
			setent_quit(*s,type);
			my_thread_end();
			return NSS_STATUS_UNAVAIL;
		}
		(*s)->list[i] = _nss_mysql_strtol(sql_row[0],-1,&error);
		if (error) {
			_nss_mysql_log(LOG_ERR,"seten(%d)t: Can't convert [UG]ID %s into integer. "
					                   "Reverting to -1. This won't work as "
									   "expected. Fix your database.",
									   type,sql_row[0]);
		}
		ent_count[i] = _nss_mysql_strtol(sql_row[1],1,&error);
		if (error) {
			_nss_mysql_log(LOG_ERR,"seten(%d)t: Can't convert count %s into integer. "
					                   "Reverting to 1. Check %s statement.",
									   type,sql_row[1],sql);
		}
	}
	
	(*s)->max = max;
	(*s)->index = 0; /* index of list */

	free(sql);
	mysql_free_result(result);
	result = NULL;

#if USE_GROUP
	/*
	 * Cache all results before walking groups because requests to 
	 * repeatidly get the group members are a CPU hog, no matter how
	 * optimized the indexes are.
	 */
	if (type == 1) {
		(*s)->gr_list = malloc(sizeof(struct group) * max);
		setent_check_mem((*s)->gr_list,*s,type);
		memset((*s)->gr_list,0,sizeof(struct group) * max);

		sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s from %s,%s,%s where %s = %s and %s = %s and %s and %s order by %s",
															 (*s)->goptions.groupnamecolumn,
															 (*s)->goptions.gidcolumn,
															 (*s)->goptions.passwordcolumn,
															 (*s)->options.usercolumn,
															 /* from */
															 (*s)->goptions.groupinfotable,
															 (*s)->options.table,
															 (*s)->goptions.memberstable,
															 /* where */
															 (*s)->goptions.groupidcolumn,
															 (*s)->goptions.mgroupidcolumn,
															 (*s)->options.useridcolumn,
															 (*s)->goptions.museridcolumn,
															 (*s)->goptions.where[0] ? (*s)->goptions.where : "1=1",
															 (*s)->options.where[0] ? (*s)->options.where : "1=1",
															 (*s)->goptions.gidcolumn);
		setent_check_mem(sql,*s,type);
		if (DEBUG) _nss_mysql_log(LOG_ERR,"setent(%d): get all groups %s",type,sql);
		if (mysql_query((*s)->mysql_auth, sql)) {
			_nss_mysql_log(LOG_ERR,"setent(%d): mysql_query failed: %s",type,mysql_error((*s)->mysql_auth));
			setent_quit(*s,type);
			my_thread_end();
			return NSS_STATUS_UNAVAIL;
		}

		result = mysql_store_result((*s)->mysql_auth);
		if (! result)	{
			_nss_mysql_log(LOG_ERR,"setent(%d): mysql_store_result failed: %s",
										 type,mysql_error((*s)->mysql_auth));
			setent_quit(*s,type);
			my_thread_end();
			return NSS_STATUS_UNAVAIL;
		}

		for (i = 0 ; i < max ; ++i) {
			enum nss_status status;
			int error;
			if((status = _nss_mysql_group_result_to_struct(&((*s)->gr_list[i]), &error, (*s)->mysql_auth, result, ent_count[i])) != NSS_STATUS_SUCCESS) {
				setent_quit(*s,type);
				my_thread_end();
				return NSS_STATUS_UNAVAIL;
			}
		}

		mysql_free_result(result);
		free(sql);

		if (DEBUG) _nss_mysql_log(LOG_ERR,"setent(%d): get all groups end",type);

		if (mysql_query((*s)->mysql_auth, "unlock tables")) {
			_nss_mysql_log(LOG_ERR,"setent(%d): mysql_query failed: %s",type,mysql_error((*s)->mysql_auth));
			setent_quit(*s,type);
			my_thread_end();
			return NSS_STATUS_UNAVAIL;
		}
	}
#endif

	if (DEBUG) _nss_mysql_log(LOG_ERR,"setent(%d): finished, max=%d",type,(*s)->max);
	my_thread_end();
	pthread_mutex_unlock(m);
	return NSS_STATUS_SUCCESS;
}

/* endent
 * Kills all data for [gs]et..ent functions
 * NOTE this function does _NOT_ use errno
 */
enum nss_status _nss_mysql_endent (int type) {
	ENT_S ** s;
	pthread_mutex_t * m;
	
	check_type(type,"endent");
	s = &ent[type];
	m = &mutex[type];
	
	pthread_mutex_lock(m);
	my_thread_init();
	if ((*s) != NULL ) {
		if ((*s)->list != NULL) {
			/* normal operation */
			free((*s)->list);
			(*s)->list = NULL;
		} else {
			/* should not happen, who knows :-) */
			if (DEBUG) _nss_mysql_log(LOG_ERR, "endent(%d): ent != NULL but ent->list is",type);
		}
#if USE_GROUP
		if (type == 1) {
			if ((*s)->gr_list != NULL) {
				struct group * list = (*s)->gr_list;
				int i;
				int count = 0;
				for (i = 0 ; i < (*s)->max ; ++i) {
					if(list[i].gr_name) {
						int j;
						count++;
						free(list[i].gr_name);
						free(list[i].gr_passwd);
						for (j = 0 ; list[i].gr_mem[j] ; j++) {
							free(list[i].gr_mem[j]);
						}
					}
				}
				if (DEBUG) _nss_mysql_log(LOG_ERR, "endent(%d): flushed %d cache entries",type,count);
				(*s)->gr_list = NULL;
			} else {
				/* should not happen, who knows :-) */
				if (DEBUG) _nss_mysql_log(LOG_ERR, "endent(%d): ent != NULL but ent->gr_list is",type);
			}
		}
#endif
		if ((*s)->mysql_auth != NULL) {
			_nss_mysql_db_close(&(*s)->mysql_auth);
			(*s)->mysql_auth = NULL;
		} else
			if (DEBUG) _nss_mysql_log(LOG_ERR, "endent(%d): ent != NULL but ent->mysql_auth is",type);
		free_parse(*s);
		free((*s));
		(*s) = NULL;
	} else {
		if (DEBUG) _nss_mysql_log(LOG_ERR,"endent(%d): ent was NULL",type);
	}
	
	if (DEBUG) _nss_mysql_log(LOG_ERR,"endent(%d) finished",type);
	my_thread_end();
	pthread_mutex_unlock(m);
	return NSS_STATUS_SUCCESS;
}


/* getgrent
 * Gets info for every group
 * Arguments:
 * gr: array to fill
 * buffer: buffer, unused
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

enum nss_status _nss_mysql_getent_r (int type, void * to_fill,
                char * buffer, size_t buflen,int * errnop) {
	enum nss_status status = NSS_STATUS_NOTFOUND; /* to silence a warning */
	ENT_S ** s;
	pthread_mutex_t * m;
	
	check_type(type,"getent");
	s = &ent[type];
	m = &mutex[type];
	
	pthread_mutex_lock(m);
	*errnop = ENOENT;
	
	/* sanity checks */
	if ((*s) == NULL) {
		if ( DEBUG) _nss_mysql_log(LOG_ERR,"getent(%d): called with ent == NULL",type);
		pthread_mutex_unlock(m);
		return NSS_STATUS_UNAVAIL;
	}
	if ((*s)->list == NULL) {
		if (DEBUG) _nss_mysql_log(LOG_ERR,"getent(%d): called with ent->list == NULL",type);
		pthread_mutex_unlock(m);
		return NSS_STATUS_UNAVAIL;
	}
	if ((*s)->mysql_auth == NULL) {
		int ret = 0;
		if (DEBUG) _nss_mysql_log(LOG_ERR,"getent(%d): called with ent->mysql_auth == NULL",type);
		/* we try to reconnect */
		switch(type) {
			case 0:
			case 1: ret = _nss_mysql_db_connect(&(*s)->mysql_auth,(*s)->options.host,(*s)->options.dbuser,
					(*s)->options.dbpasswd,(*s)->options.database); break;
			case 2: ret = _nss_mysql_db_connect(&(*s)->mysql_auth,(*s)->soptions.host,(*s)->soptions.dbuser,
					(*s)->soptions.dbpasswd,(*s)->soptions.database); break;
		}	
		if (! ret) {
		
			pthread_mutex_unlock(m);
			return NSS_STATUS_UNAVAIL;
		}
	} else {
		my_thread_init();
	}
	if (DEBUG) _nss_mysql_log(LOG_ERR,"getent(%d): entered for i=%u (max=%u)",type,(*s)->index,(*s)->max);
	
	/*
	 * Only bother to ping if not using cached data for groups.
	 */
	if (type != 1) {
		/* Is the server still alive ? */
		if (mysql_ping((*s)->mysql_auth)) {
			_nss_mysql_log(LOG_ERR,"getent(%d): can't sustain connection : %s",
										 type,mysql_error((*s)->mysql_auth));
			(*s)->mysql_auth = NULL;
			pthread_mutex_unlock(m);
			my_thread_end();
			return NSS_STATUS_UNAVAIL;
		}
	}
		
	while ((*s)->index < (*s)->max)	{
		/* we get the info on this group */
		switch(type) {
			case 0:
					status = _nss_mysql_passwd_fill_struct(NULL,NULL,(*s)->list[(*s)->index],(struct passwd *) to_fill,errnop,
						(*s)->mysql_auth,&(*s)->options,NULL);break;
#if USE_GROUP
			case 1: 
					if (DEBUG) _nss_mysql_log(LOG_ERR,"getent(%d): get %s from cache",type, (*s)->gr_list[(*s)->index].gr_name);
					memcpy(to_fill,&((*s)->gr_list[(*s)->index]),sizeof(struct group));
					/*
					 * It is the responsibility of the caller to free the content of
					 * the structure, just forget about it.
					 */
					memset(&((*s)->gr_list[(*s)->index]),0,sizeof(struct group));
					status = NSS_STATUS_SUCCESS;
					break;
#endif
#if USE_SHADOW
			case 2:

					status = _nss_mysql_shadow_fill_struct(NULL,NULL,(*s)->list[(*s)->index],(struct spwd *) to_fill,errnop,
						(*s)->mysql_auth,&(*s)->soptions,NULL);
					break;
#endif
		}
		if (status == NSS_STATUS_UNAVAIL || status == NSS_STATUS_SUCCESS) {
			break;
		}
		/* cache is not uptodate */
		if (DEBUG) _nss_mysql_log(LOG_ERR,"getent(%d): not found occured in g_fill_struct",type);
		++((*s)->index);
	}

	/* the lookup is finished */
	if ((*s)->index == (*s)->max) {
		pthread_mutex_unlock(m);
		my_thread_end();
		return NSS_STATUS_NOTFOUND;
	}

	if (status == NSS_STATUS_UNAVAIL)	{
		_nss_mysql_log(LOG_ERR,"getent(%d): fill_struct returned UNAVAIL",type);
		pthread_mutex_unlock(m);
		my_thread_end();
		return NSS_STATUS_UNAVAIL;
	}
	
	if (DEBUG) _nss_mysql_log(LOG_ERR,"getent(%d):successfully exiting",type);
	/* next entry man ! */
	++(*s)->index;
	pthread_mutex_unlock(m);
	*errnop = 0;
	my_thread_end();
	return NSS_STATUS_SUCCESS;
}

