/*
    Copyright (C) 2000 Steve Brown
	Copyright (C) 2000-2001 Guillaume Morin, Alcve

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: passwd.c,v 1.67 2002/01/15 11:27:14 gmorin Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/passwd.c,v $
    $Date: 2002/01/15 11:27:14 $
    $Author: gmorin $
*/


#include <stdlib.h>
#include <nss.h>
#include <mysql/mysql.h>
#include <pwd.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <pthread.h>

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#ifdef INCLUDE_MY_SYS
	#include <mysql/my_pthread.h>
#endif

#include "lib.h"
#include "parser.h"
#include "passwd.h"
#include "ent.h"


#define FALLBACK_GID 65534 /* if the gid column can't be read, fall back to this GID. should be nogroup */
#define FALLBACK_UID 65534

#define ENT_TYPE 0

enum nss_status _nss_mysql_getpwuid_r(uid_t,struct passwd *,char *, size_t,int *);
enum nss_status _nss_mysql_setpwent (void);
enum nss_status _nss_mysql_getpwent_r (struct passwd *pw, char * buffer, size_t buflen,int * errnop); 
enum nss_status _nss_mysql_endpwent (void);
enum nss_status _nss_mysql_getpwnam_r(const char *,struct passwd *,char *,size_t,int *);
static int check_connection(MYSQL ** mysql_auth,struct passwdoptions * options,pthread_mutex_t * mutex);


/* generic quit macro */
#define clean_quit() \
if (sql) free(sql); \
if (result) mysql_free_result(result); \
*errnop = ENOENT /* most common situation */


/* check if the arg is NULL */
#define check_mem(x) if ( (x) == NULL ) { \
	/*if (DEBUG) */ _nss_mysql_log(LOG_ERR,"passwd.c: not enough memory"); \
	clean_quit(); \
	*errnop = EAGAIN; \
	return NSS_STATUS_TRYAGAIN; \
}

/* _nss_mysql_passwd_fill_struct
 * Get info in the DB by name or by uid
 * Arguments:
 * if uname not NULL 
 *    uname is the name of the user we look for
 *    name is a buffer to escape uname
 * if uname is NULL
 *    uid is the uid of the user we look for
 *
 * pw: the struc we'll fill
 * errnop: ptr to the application errno
 */

#define lockm() \
if (m) pthread_mutex_lock(m);

#define unlockm() \
if (m) pthread_mutex_unlock(m);

enum nss_status _nss_mysql_passwd_fill_struct(const char *uname, char * name, int uid, struct passwd *pw, int *errnop, MYSQL * mysql_auth,struct passwdoptions * options,pthread_mutex_t * m) {
	char  * sql = NULL;
	MYSQL_RES *result = NULL;
	MYSQL_ROW sql_row;
	int i;

	*errnop = ENOENT;

	if (DEBUG) {
		if (uname) {
			/* looking by name */
			_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct called for user %s",uname);
		} else {
			/* looking by uid */
			_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct called for uid %d",uid);
		}
	}
	
	/* escaping name */
	if (uname)	{
#ifndef HAVE_MYSQL_REAL_ESCAPE_STRING
		mysql_escape_string(name,uname,strlen(uname));
#else
		mysql_real_escape_string(mysql_auth,name,uname,strlen(uname));
#endif
	}

	if (name) {
		sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s,%s,%s,%s from %s "
				                   "where %s='%s' and %s is not null and %s",
			options->usercolumn,
			options->uidcolumn,
#if ! USE_SHADOW
			options->passwdcolumn,
#else
			NULL,
#endif
			options->realnamecolumn,
			options->shellcolumn,
			options->homedircolumn,
			options->gidcolumn,
			options->table,
			options->usercolumn,
			name,
			options->uidcolumn,
			options->where[0] ? options->where : "1=1"
			);

	} else {
            /* "select usercolumn,uidcolumn,(passwdcolumn|uidcolumn),realnamecolumn,
			shellcolumn,(homedircolumn|uidcolumn),(gidcolumn|uidcolumn)
			from table where uidcolumn=uid" */
		sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s,%s,%s,%s from %s where %s=%d and %s",
			options->usercolumn,
			options->uidcolumn,
#if ! USE_SHADOW
			options->passwdcolumn,
#else
			NULL,
#endif
			options->realnamecolumn,
			options->shellcolumn,
			options->homedircolumn,
			options->gidcolumn,
			options->table,
			options->uidcolumn,
			uid,
			options->where[0] ? options->where : "1=1"
			);
	}
	check_mem(sql);
	if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: SQL statement: %s",sql);

	lockm();
	if (mysql_query(mysql_auth, sql)) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: mysql_query failed: %s",mysql_error(mysql_auth));
		clean_quit();
		unlockm();
		return NSS_STATUS_UNAVAIL;
	}
	
	result = mysql_store_result(mysql_auth);
	if (! result) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: mysql_store_result failed: %s",mysql_error(mysql_auth));
		clean_quit();
		unlockm();
		return NSS_STATUS_UNAVAIL;
	}

	unlockm();
	if ((i=mysql_num_rows(result)) == 1) {
		
		/* we've found one entry */
		
		sql_row = mysql_fetch_row(result);
		if (! sql_row) {
			_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: mysql_fetch_row has failed: %s",
					       mysql_error(mysql_auth));
			clean_quit();
			return NSS_STATUS_UNAVAIL;
		}
					
		/* sanity checks */
		if (_nss_mysql_isempty(sql_row[0])) {
			_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: NULL or empty username. Fix your database");
			clean_quit();
			return NSS_STATUS_UNAVAIL;
		}
		
#if ! USE_SHADOW
		if (_nss_mysql_isempty(sql_row[2])) {
			_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: NULL or empty password for %s. "
					               "Fix your database",sql_row[0]);
			clean_quit();
			return NSS_STATUS_UNAVAIL;
		}
#endif
		
		/* name is stored */
		pw->pw_name = strdup(sql_row[0]);
		check_mem(pw->pw_name);
		if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: username == %s", sql_row[0]);
		
		/* storing password */
#if ! USE_SHADOW
		pw->pw_passwd = strdup(sql_row[2]);
		check_mem(pw->pw_passwd);
		if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: password == %s", sql_row[2]);
#else
		pw->pw_passwd = strdup("x");
		check_mem(pw->pw_passwd);
#endif
		
		/* storing uid */
		pw->pw_uid = _nss_mysql_strtol(sql_row[1],FALLBACK_UID,&i);
		if (i) {
			_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: User %s has invalid uid(%s). "
					               "Reverted to %d. Fix your database.", 
								   pw->pw_name, sql_row[1]||"NULL",pw->pw_uid);
		}
		if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: uid == %d", pw->pw_uid);

		/* check if a GID was returned */
		pw->pw_gid = _nss_mysql_strtol(sql_row[6],FALLBACK_GID,&i);
		if (i) {
			_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct: User %s has invalid gid(%s). "
					               "Reverted to %d. Fix your database.",
								   pw->pw_name, sql_row[6],FALLBACK_GID);
		}
		if (DEBUG) _nss_mysql_log(LOG_ERR, "_nss_mysql_passwd_fill_struct: gid == %d", pw->pw_gid);
		
		/* Realname, empty realname is valid, so no warning */
		pw->pw_gecos = sql_row[3] ? strdup(sql_row[3]) : strdup("");
		check_mem(pw->pw_gecos);

		/* home */
		if (_nss_mysql_isempty(sql_row[5]))	{
			_nss_mysql_log(LOG_ERR,"Empty or NULL home column for user %s(%d). "
					       "Falling back to /tmp. Fix your database.",
					       pw->pw_name,pw->pw_uid);
			pw->pw_dir = strdup("/tmp");
		} else {
			pw->pw_dir = strdup(sql_row[5]);
		}
		check_mem(pw->pw_dir);

		/* shell */
		if (_nss_mysql_isempty(sql_row[4]))	{
			_nss_mysql_log(LOG_ERR,"Empty or NULL shell column for user %s(%d). "
					       "Falling back to /bin/sh. Fix your database.",
					       pw->pw_name,pw->pw_uid);
			pw->pw_shell = strdup("/bin/sh");
		} else {
			pw->pw_shell = strdup(sql_row[4]);
		}
		
		check_mem(pw->pw_shell);
		clean_quit();
		*errnop = 0;
		if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct finished sucessfully");
		return NSS_STATUS_SUCCESS;
	}
	/* i should be 0 here no more */
	if (i) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_fill_struct found %d matching rows. Check the database",i);
	} else {
		if (DEBUG) _nss_mysql_log(LOG_ERR,"User not found...");
	}
	clean_quit();
	return NSS_STATUS_NOTFOUND;
}


/* setpwent
 * Initializes data for ...pwent functions
 * NOTE this function does _NOT_ use errno
 */

enum nss_status _nss_mysql_setpwent (void) {
	return _nss_mysql_setent(ENT_TYPE);
}


/* endpwent
 * Kills all data for ...pwent functions
 * NOTE this function does _NOT_ use errno
 */
enum nss_status _nss_mysql_endpwent (void) {
	return _nss_mysql_endent(ENT_TYPE);
}

/* getpwent
 * Gets info for every group
 * Arguments:
 * pw: array to fill
 * buffer: buffer, unused 
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

enum nss_status _nss_mysql_getpwent_r (struct passwd *pw,
                char * buffer, size_t buflen,int * errnop) {
	return _nss_mysql_getent_r(ENT_TYPE,(void *) pw,buffer,buflen,errnop);
}

/* check_connection 
 * checks if a connection has been opened and if it is still alive
 * tries to open a connection if not
 */

int check_connection(MYSQL ** mysql_auth,struct passwdoptions * options,pthread_mutex_t * mutex) {
	/* Is the server still alive ? */
	pthread_mutex_lock(mutex);
	if (*mysql_auth != NULL) {
		my_thread_init();
		if (mysql_ping(*mysql_auth)) {
			if (DEBUG) _nss_mysql_log(LOG_ERR,"check_connection: can't sustain connection : %s",
					mysql_error(*mysql_auth));
			*mysql_auth = NULL;
		}
	}
	
	/* DB connection */
	if (*mysql_auth == NULL) {
		if  (! _nss_mysql_db_connect(mysql_auth,options->host,options->dbuser,options->dbpasswd,options->database)) {
			pthread_mutex_unlock(mutex);
			my_thread_end();
			return 0;
		}
	} 
	pthread_mutex_unlock(mutex);
	return 1;
}


#define getpw_quit() \
_nss_mysql_free_users(&options); 

/* getpwnam
 * looks for an user by its name
 * Arguments:
 * uid: user's uid
 * result: struct we'll fill
 * buf: buffer (not used)
 * buflen: sizeof(buffer)
 * errnop: ptr on the application errno
 */


static MYSQL * pwnam_auth = NULL;
static pthread_mutex_t pwnam_mutex = PTHREAD_MUTEX_INITIALIZER;

enum nss_status _nss_mysql_getpwnam_r (const char *name, struct passwd *result,
		char *buf, size_t buflen, int *errnop) {
	enum nss_status toreturn = NSS_STATUS_UNAVAIL;
	char * buffer;	
	struct passwdoptions options;
	
	memset(&options,0,sizeof(struct passwdoptions));

	if (DEBUG) _nss_mysql_log(LOG_ERR,"getpwnam called for %s\n", name);
	*errnop = ENOENT;		

	if (!name) return NSS_STATUS_NOTFOUND;

	/* conf files parsing */
	if (! _nss_mysql_read_conf_file("users",&options,NULL,NULL)) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_getpwnam: conf file parsing failed");
		getpw_quit();
		return NSS_STATUS_UNAVAIL;
	}

	if (! check_connection(&pwnam_auth,&options,&pwnam_mutex)) {
		getpw_quit();
		return NSS_STATUS_UNAVAIL;
	}

	if (result)	{
		buffer = malloc(strlen(name) * 2 + 1);
		if (buffer == NULL) {
			*errnop = EAGAIN;
			getpw_quit();
			my_thread_end();
			return NSS_STATUS_TRYAGAIN;
		}
		toreturn = _nss_mysql_passwd_fill_struct(name, buffer,-1, result,errnop,pwnam_auth,&options,&pwnam_mutex);
		free(buffer);
	}
	my_thread_end();
	getpw_quit();
	return toreturn;
}

/* getpwuid
 * looks for an user by uid
 * Arguments:
 * uid: user's uid
 * result: struct we'll fill
 * buf: buffer (not used)
 * buflen: sizeof(buffer)
 * errnop: ptr on the application errno
 */


static pthread_mutex_t pwuid_mutex = PTHREAD_MUTEX_INITIALIZER;
static MYSQL * pwuid_auth = NULL;

enum nss_status _nss_mysql_getpwuid_r (uid_t uid, struct passwd *result, 
	char *buf, size_t buflen, int *errnop) {
	enum nss_status status = NSS_STATUS_UNAVAIL;
	struct passwdoptions options;

	memset(&options,0,sizeof(struct passwdoptions));
	
	if (DEBUG) _nss_mysql_log(LOG_ERR,"getpwuid called for %d",uid);

	/* conf files parsing */
	if (! _nss_mysql_read_conf_file("users",&options,NULL,NULL)) {
		_nss_mysql_log(LOG_ERR,"_nss_mysql_getpwuid: conf file parsing failed");
		getpw_quit();
		return NSS_STATUS_UNAVAIL;
	}

	if (! check_connection(&pwuid_auth,&options,&pwuid_mutex)) {
		getpw_quit();
		return NSS_STATUS_UNAVAIL;
	}
	
	status = _nss_mysql_passwd_fill_struct(NULL, NULL, uid, result,errnop,pwuid_auth,&options,&pwuid_mutex);
	getpw_quit();
	my_thread_end();
	return status;
}
