#include "funcprotos.h"
#include <quicktime/quicktime.h>
#define LQT_LIBQUICKTIME
#include <quicktime/lqt_codecapi.h>

#include "twos.h"

/* We decode this many samples at a time */

#define SAMPLES_PER_BLOCK 1024



/* =================================== private for twos */


typedef struct
{
	uint8_t *work_buffer;
	long buffer_size;
        int le; /* For 8 bits means unsigned operation, for more bits, little endian */
        int encode_initialized;

/* Decode specific stuff */
        int decode_buffer_size;
        int decode_buffer_alloc;
        uint8_t * decode_buffer;
        uint8_t * decode_buffer_ptr;

        int16_t ** decode_sample_buffer_i;
        float ** decode_sample_buffer_f;
        int decode_sample_buffer_size;
        int decode_sample_buffer_pos;
        int decode_initialized;
        int decode_block_align;

  } quicktime_twos_codec_t;

static int byte_order(void)
{                /* 1 if little endian */
	int16_t byteordertest;
	int byteorder;

	byteordertest = 0x0001;
	byteorder = *((unsigned char *)&byteordertest);
	return byteorder;
}

static int get_work_buffer(quicktime_t *file, int track, long bytes)
{
	quicktime_twos_codec_t *codec = ((quicktime_codec_t*)file->atracks[track].codec)->priv;

        if(codec->buffer_size < bytes)
          {
          codec->buffer_size = bytes + 128;
          codec->work_buffer = realloc(codec->work_buffer, codec->buffer_size);
          }
        return 0;
}

/* =================================== public for twos */

static int delete_codec(quicktime_audio_map_t *atrack)
{
	quicktime_twos_codec_t *codec = ((quicktime_codec_t*)atrack->codec)->priv;

	if(codec->work_buffer) free(codec->work_buffer);
	codec->work_buffer = 0;
	codec->buffer_size = 0;

        if(codec->decode_sample_buffer_i)
          {
          free(codec->decode_sample_buffer_i[0]);
          free(codec->decode_sample_buffer_i);
          }
        if(codec->decode_sample_buffer_f)
          {
          free(codec->decode_sample_buffer_f[0]);
          free(codec->decode_sample_buffer_f);
          }
        if(codec->decode_buffer)
          free(codec->decode_buffer);


        free(codec);
	return 0;
}

static int swap_bytes(uint8_t *buffer, long samples, int bits)
{
	long i = 0;
	char byte1;
	uint8_t *buffer1, *buffer2;

	switch(bits)
	{
                case 8:
                  /* Swap sign */
                  for(i = 0; i < samples; i++)
                    buffer[i] ^= 0x80;
                  break;

		case 16:
                        if(!byte_order()) return 0;
			buffer1 = buffer;
			buffer2 = buffer + 1;
			while(i < samples * 2)
			{
				byte1 = buffer2[i];
				buffer2[i] = buffer1[i];
				buffer1[i] = byte1;
				i += 2;
			}
			break;

		case 24:
                        if(!byte_order()) return 0;
			buffer1 = buffer;
			buffer2 = buffer + 2;
			while(i < samples * 3)
                          {
                                byte1 = buffer2[i];
				buffer2[i] = buffer1[i];
				buffer1[i] = byte1;
				i += 3;
			}
			break;

		default:
			break;
	}
	return 0;
}


static int decode(quicktime_t *file, 
					int16_t **output_i, 
					float **output_f, 
					long samples, 
					int track) 
{
        int j;
        quicktime_audio_map_t *track_map = &(file->atracks[track]);
	quicktime_twos_codec_t *codec = ((quicktime_codec_t*)track_map->codec)->priv;

	int64_t chunk, chunk_sample;
	int64_t i;

        int samples_decoded, samples_copied;

        int64_t samples_to_skip = 0;
        int bits;
        
        bits = quicktime_audio_bits(file, track);

        //        fprintf(stderr, "Bits: %d\n", bits);
        
        if(!codec->decode_initialized)
          {
          codec->decode_initialized = 1;

          if(bits <= 16)
            {
            codec->decode_sample_buffer_i    = malloc(sizeof(*(codec->decode_sample_buffer_i)) * file->atracks[track].channels);
            codec->decode_sample_buffer_i[0] = malloc(sizeof(*(codec->decode_sample_buffer_i[0])) * file->atracks[track].channels *
                                                      SAMPLES_PER_BLOCK);
            for(i = 1; i < file->atracks[track].channels; i++)
              codec->decode_sample_buffer_i[i] = codec->decode_sample_buffer_i[0] + i * SAMPLES_PER_BLOCK;
            
            }
          else
            {
            codec->decode_sample_buffer_f    = malloc(sizeof(*(codec->decode_sample_buffer_f)) * file->atracks[track].channels);
            codec->decode_sample_buffer_f[0] = malloc(sizeof(*(codec->decode_sample_buffer_f[0])) * file->atracks[track].channels *
                                                      SAMPLES_PER_BLOCK);

            for(i = 1; i < file->atracks[track].channels; i++)
              codec->decode_sample_buffer_f[i] = codec->decode_sample_buffer_f[0] + i * SAMPLES_PER_BLOCK;
            }

          codec->decode_block_align = track_map->channels * bits / 8;

          /* Read first chunk */

          codec->decode_buffer_size = lqt_read_audio_chunk(file,
                                                           track, file->atracks[track].current_chunk,
                                                           &(codec->decode_buffer),
                                                           &(codec->decode_buffer_alloc));
          if(codec->decode_buffer_size <= 0)
            return 0;
          /* Handle AVI byte order */
          if(codec->le)
            swap_bytes(codec->decode_buffer, 
                       (codec->decode_buffer_size * track_map->channels) / codec->decode_block_align,
                       quicktime_audio_bits(file, track));
          
          codec->decode_buffer_ptr = codec->decode_buffer;
          }
        
        if(file->atracks[track].current_position != file->atracks[track].last_position)
          {
          /* Seeking happened */
          quicktime_chunk_of_sample(&chunk_sample, &chunk,
                                    file->atracks[track].track,
                                    file->atracks[track].current_position);

          /* Read the chunk */

          if(file->atracks[track].current_chunk != chunk)
            {
            file->atracks[track].current_chunk = chunk;
            codec->decode_buffer_size = lqt_read_audio_chunk(file,
                                                             track, file->atracks[track].current_chunk,
                                                             &(codec->decode_buffer),
                                                             &(codec->decode_buffer_alloc));
            if(codec->decode_buffer_size <= 0)
              return 0;
            
            codec->decode_buffer_ptr = codec->decode_buffer;

            if(codec->le)
              swap_bytes(codec->decode_buffer, 
                         (codec->decode_buffer_size * track_map->channels) / codec->decode_block_align,
                         quicktime_audio_bits(file, track));
            }
          else
            {
            codec->decode_buffer_size += (int)(codec->decode_buffer_ptr - codec->decode_buffer);
            codec->decode_buffer_ptr = codec->decode_buffer;
            }

          /* Skip frames */
          
          samples_to_skip = file->atracks[track].current_position - chunk_sample;
          if(samples_to_skip < 0)
            {
            fprintf(stderr, "twos: Cannot skip backwards\n");
            samples_to_skip = 0;
            }
          
          codec->decode_buffer_ptr = codec->decode_buffer + samples_to_skip * codec->decode_block_align;
          codec->decode_buffer_size -= samples_to_skip * codec->decode_block_align;
          codec->decode_sample_buffer_size = 0;
          codec->decode_sample_buffer_pos = 0;
          }

        /* Decode until we are done */

        samples_decoded = 0;

        while(samples_decoded < samples)
          {
          /* Decode new frame if necessary */
          if(!codec->decode_sample_buffer_size)
            {
            /* Get new chunk if necessary */
            
            if(!codec->decode_buffer_size)
              {
              file->atracks[track].current_chunk++;
              codec->decode_buffer_size = lqt_read_audio_chunk(file,
                                                               track, file->atracks[track].current_chunk,
                                                               &(codec->decode_buffer),
                                                               &(codec->decode_buffer_alloc));
              if(codec->decode_buffer_size <= 0)
                break;
              /* Handle AVI byte order */
              if(codec->le)
		swap_bytes(codec->decode_buffer, 
                           (codec->decode_buffer_size * track_map->channels) / codec->decode_block_align,
                           quicktime_audio_bits(file, track));
              
              codec->decode_buffer_ptr = codec->decode_buffer;
              }
            
            codec->decode_sample_buffer_size = codec->decode_buffer_size / codec->decode_block_align;
            if(codec->decode_sample_buffer_size > SAMPLES_PER_BLOCK)
              {
              codec->decode_sample_buffer_size = SAMPLES_PER_BLOCK;
              }

            codec->decode_sample_buffer_pos = 0;
            
            /* Decode the stuff */
            
            switch(bits)
              {
              case 8:
                for(i = 0; i < codec->decode_sample_buffer_size; i++)
                  {
                  for(j = 0; j < track_map->channels; j++)
                    {
                    codec->decode_sample_buffer_i[j][i] = ((int16_t)(*codec->decode_buffer_ptr)) << 8;
                    codec->decode_buffer_ptr++;
                    codec->decode_buffer_size--;
                    }
                  }
                break;
              case 16:
                for(i = 0; i < codec->decode_sample_buffer_size; i++)
                  {
                  for(j = 0; j < track_map->channels; j++)
                    {
                    codec->decode_sample_buffer_i[j][i] = ((int16_t)codec->decode_buffer_ptr[0]) << 8 |
                      ((unsigned char)codec->decode_buffer_ptr[1]);
                    codec->decode_buffer_ptr+= 2;
                    codec->decode_buffer_size-=2;
                    }
                  }
                break;
              case 24:
                for(i = 0; i < codec->decode_sample_buffer_size; i++)
                  {
                  for(j = 0; j < track_map->channels; j++)
                    {
                    codec->decode_sample_buffer_i[j][i] = (float)((((int)codec->decode_buffer_ptr[0]) << 16) | 
                                                           (((unsigned char)codec->decode_buffer_ptr[1]) << 8) |
                                                           ((unsigned char)codec->decode_buffer_ptr[2])) / 0x7fffff;
                    codec->decode_buffer_ptr+= 3;
                    codec->decode_buffer_size-=3;
                    }
                  }
                break;
              default:
                break;
              }
            }
          
          /* Copy samples */
          //          fprintf(stderr, "Copy samples: %d\n", codec->decode_buffer_size);
          samples_copied = lqt_copy_audio(output_i,                                     // int16_t ** dst_i
                                          output_f,                                     // float ** dst_f
                                          codec->decode_sample_buffer_i,                  // int16_t ** src_i
                                          codec->decode_sample_buffer_f,                  // float ** src_f
                                          samples_decoded,                              // int dst_pos
                                          codec->decode_sample_buffer_pos,              // int src_pos
                                          samples - samples_decoded,                    // int dst_size
                                          codec->decode_sample_buffer_size,                    // int src_size
                                          file->atracks[track].channels);               // int num_channels
          
          samples_decoded += samples_copied;
          codec->decode_sample_buffer_size -= samples_copied;
          codec->decode_sample_buffer_pos += samples_copied;
          }
        
        file->atracks[track].last_position = file->atracks[track].current_position + samples_decoded;
        return samples_decoded;

        
#if 0 /* Old (hopefully obsolete version) */
        
        int step = track_map->channels * quicktime_audio_bits(file, track) / 8;

	get_work_buffer(file, track, samples * step);
/*
 * printf("decode 1 %d\n", quicktime_audio_bits(file, track));
 * sleep(1);
 */
	result = !quicktime_read_audio(file, codec->work_buffer, samples, track);


/* Undo increment since this is done in codecs.c */
	track_map->current_position -= samples;

/* Handle AVI byte order */
	if(codec->le)
		swap_bytes(codec->work_buffer, 
			samples, 
			track_map->channels, 
			quicktime_audio_bits(file, track));

	switch(quicktime_audio_bits(file, track))
	{
		case 8:
			if(output_i && !result)
			{
				for(i = 0, j = channel; i < samples; i++)
				{
					output_i[i] = ((int16_t)codec->work_buffer[j]) << 8;
					j += step;
				}
			}
			else
			if(output_f && !result)
			{
				for(i = 0, j = channel; i < samples; i++)
				{
					output_f[i] = ((float)codec->work_buffer[j]) / 0x7f;
					j += step;
				}
			}
			break;
		
		case 16:
			if(output_i && !result)
			{
				for(i = 0, j = channel * 2; i < samples; i++)
				{
					output_i[i] = ((int16_t)codec->work_buffer[j]) << 8 |
									((unsigned char)codec->work_buffer[j + 1]);
					j += step;
				}
			}
			else
			if(output_f && !result)
			{
				for(i = 0, j = channel * 2; i < samples; i++)
				{
					output_f[i] = (float)((((int16_t)codec->work_buffer[j]) << 8) |
									((unsigned char)codec->work_buffer[j + 1])) / 0x7fff;
					j += step;
				}
			}
			break;
		
		case 24:
			if(output_i && !result)
			{
				for(i = 0, j = channel * 3; i < samples; i++)
				{
					output_i[i] = (((int16_t)codec->work_buffer[j]) << 8) | 
									((unsigned char)codec->work_buffer[j + 1]);
					j += step;
				}
			}
			else
			if(output_f && !result)
			{
				for(i = 0, j = channel * 3; i < samples; i++)
				{
					output_f[i] = (float)((((int)codec->work_buffer[j]) << 16) | 
									(((unsigned char)codec->work_buffer[j + 1]) << 8) |
									((unsigned char)codec->work_buffer[j + 2])) / 0x7fffff;
					j += step;
				}
			}
			break;
		
		default:
			break;
	}


	return result;
#endif
}

#define CLAMP(x, y, z) ((x) = ((x) <  (y) ? (y) : ((x) > (z) ? (z) : (x))))

static int encode(quicktime_t *file, 
							int16_t **input_i, 
							float **input_f, 
							int track, 
							long samples)
{
	int result = 0;
	long i, j;
        quicktime_trak_t *trak;
	quicktime_audio_map_t *track_map = &(file->atracks[track]);
	quicktime_twos_codec_t *codec = ((quicktime_codec_t*)track_map->codec)->priv;
	int step = track_map->channels * quicktime_audio_bits(file, track) / 8;
	int sample;
	float sample_f;
        int bits = quicktime_audio_bits(file, track);
        
        if(!codec->encode_initialized)
          {
          trak = track_map->track;
          if(trak->strl)
            {
            /* strh stuff */
            trak->strl->dwRate = (trak->mdia.minf.stbl.stsd.table[0].sample_rate * track_map->channels * bits) / 8;
            trak->strl->dwScale = (track_map->channels * bits) / 8;
            trak->strl->dwSampleSize = bits/8;
            
            /* WAVEFORMATEX stuff */
            
            trak->strl->nBlockAlign = trak->strl->dwScale;
            trak->strl->nAvgBytesPerSec =  trak->strl->dwRate;
            trak->strl->wBitsPerSample = bits;
            }
          codec->encode_initialized = 1;
          }

        get_work_buffer(file, track, samples * step);

	if(input_i)
	{
		for(i = 0; i < track_map->channels; i++)
		{
			switch(bits)
			{
				case 8:
					for(j = 0; j < samples; j++)
					{
						sample = input_i[i][j] >> 8;
						codec->work_buffer[j * step + i] = sample;
					}
					break;
				case 16:
					for(j = 0; j < samples; j++)
					{
						sample = input_i[i][j];
						codec->work_buffer[j * step + i * 2] = ((unsigned int)sample & 0xff00) >> 8;
						codec->work_buffer[j * step + i * 2 + 1] = ((unsigned int)sample) & 0xff;
					}
					break;
				case 24:
					for(j = 0; j < samples; j++)
					{
						sample = input_i[i][j];
						codec->work_buffer[j * step + i * 3] = ((unsigned int)sample & 0xff00) >> 8;
						codec->work_buffer[j * step + i * 3 + 1] = ((unsigned int)sample & 0xff);
						codec->work_buffer[j * step + i * 3 + 2] = 0;
					}
					break;
			}
		}
	}
	else
	{
		for(i = 0; i < track_map->channels; i++)
		{
			switch(quicktime_audio_bits(file, track))
			{
				case 8:
					for(j = 0; j < samples; j++)
					{
						sample_f = input_f[i][j];
						if(sample_f < 0)
							sample = (int)(sample_f * 0x7f - 0.5);
						else
							sample = (int)(sample_f * 0x7f + 0.5);
						CLAMP(sample, -0x7f, 0x7f);
						codec->work_buffer[j * step + i] = sample;
					}
					break;
				case 16:
					for(j = 0; j < samples; j++)
					{
						sample_f = input_f[i][j];
						if(sample_f < 0)
							sample = (int)(sample_f * 0x7fff - 0.5);
						else
							sample = (int)(sample_f * 0x7fff + 0.5);
						CLAMP(sample, -0x7fff, 0x7fff);
						codec->work_buffer[j * step + i * 2] = ((unsigned int)sample & 0xff00) >> 8;
						codec->work_buffer[j * step + i * 2 + 1] = ((unsigned int)sample) & 0xff;
					}
					break;
				case 24:
					for(j = 0; j < samples; j++)
					{
						sample_f = input_f[i][j];
						if(sample_f < 0)
							sample = (int)(sample_f * 0x7fffff - 0.5);
						else
							sample = (int)(sample_f * 0x7fffff + 0.5);
						CLAMP(sample, -0x7fffff, 0x7fffff);
						codec->work_buffer[j * step + i * 3] = ((unsigned int)sample & 0xff0000) >> 16;
						codec->work_buffer[j * step + i * 3 + 1] = ((unsigned int)sample & 0xff00) >> 8;
						codec->work_buffer[j * step + i * 3 + 2] = ((unsigned int)sample) & 0xff;
					}
					break;
			}
		}
	}

/* Handle AVI byte order */
	if(codec->le)
		swap_bytes(codec->work_buffer, 
                           samples * 
                           track_map->channels, 
			quicktime_audio_bits(file, track));

	result = quicktime_write_audio(file, codec->work_buffer, samples, track);

	return result;
}

void quicktime_init_codec_twos(quicktime_audio_map_t *atrack)
{
	quicktime_twos_codec_t *codec;
	quicktime_codec_t *codec_base = (quicktime_codec_t*)atrack->codec;

/* Init public items */
	codec_base->delete_acodec = delete_codec;
	codec_base->decode_audio = decode;
	codec_base->encode_audio = encode;
	codec_base->fourcc = QUICKTIME_TWOS;
	codec_base->title = "Twos complement";
	codec_base->desc = "Twos complement";

/* Init private items */
	codec = codec_base->priv = calloc(1, sizeof(quicktime_twos_codec_t));
	codec->work_buffer = 0;
	codec->buffer_size = 0;
}

void quicktime_init_codec_sowt(quicktime_audio_map_t *atrack)
{
	quicktime_twos_codec_t *codec;
	quicktime_codec_t *codec_base = (quicktime_codec_t*)atrack->codec;

/* Init public items */
	codec_base->delete_acodec = delete_codec;
	codec_base->decode_audio = decode;
	codec_base->encode_audio = encode;
	codec_base->fourcc = "SOWT";
	codec_base->title = "Twos complement (Little endian)";
	codec_base->desc = "Twos complement (Little endian)";
	codec_base->wav_id = 0x01;

/* Init private items */
	codec = codec_base->priv = calloc(1, sizeof(quicktime_twos_codec_t));
	codec->work_buffer = 0;
	codec->buffer_size = 0;
        codec->le = 1;
}
