/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/* 
	This file shows an example of how to use the database interface to do some
	spatial queries involving objects with points, lines and polygon geometries.

	Authors: 
		Karine Reis Ferreira  
		Lubia Vinhas
*/

#include <TeMySQL.h>
#include <TeImportExport.h>



int main()
{
 	// Datatabase server parameters
	string host = "localhost";
	string dbname = "TerraTeste";
	string user = "root";
	string password = "";

	// Open a connection to the TerraTeste MySQL database 
	TeDatabase* db = new TeMySQL();
	if (!db->connect(host, user, password, dbname))
	{
		cout << "Error: " << db->errorMessage() << endl;
		cout << endl << "Press Enter\n";
		getchar();
		return 1;
	}
	cout << "Connection successful to the database \"" << dbname << "\" on MySQL server \"" << host << "\" !\n";;

	// Check whether there is a layer of polygons called "states".
	// In negative case, import it from the states.shp file
	string layerName = "states";

	TeDatum wgs84 = TeDatumFactory::make("WGS84");
	TeLatLong* proj = new TeLatLong(wgs84);

	if (db->layerExist(layerName) == false)
	{
		// Create a new layer in the database
		TeLayer* layer = new TeLayer(layerName, db, proj);
		string filename = "../data/states.shp";	// Shapefile path
		string tablename = "States";	// Name of the attribute table

		if (TeImportShape(layer, filename, tablename))
			cout << "The shapefile \"states.shp\" was imported successfully into the TerraLib database!\n" << endl;
		else
			cout << "Error: Fail to import the shapefile \"rivers.shp\"!\n" << endl;
	}	

	// Check whether there is a layer of lines called "rivers".
	// In negative case, import it from the states.shp file
	layerName = "rivers";

	if (db->layerExist(layerName) == false)
	{
		// Create a new layer in the database
		TeLayer* layer = new TeLayer(layerName, db, proj);
		string filename = "../data/rivers.shp";	// Shapefile path
		string tablename = "Rivers";	// Name of the attribute table

		if (TeImportShape(layer, filename, tablename))
			cout << "The shapefile \"rivers.shp\" was imported successfully into the TerraLib database!\n" << endl;
		else
			cout << "Error: Fail to import the shapefile \"rivers.shp\"!\n" << endl;
	}

	// Check whether there is a layer of points called "cities".
	// In negative case, import it from the states.shp file
	layerName = "cities";

	if (db->layerExist(layerName) == false)
	{
		// Create a new layer in the database
		TeLayer* layer = new TeLayer(layerName, db, proj);
		string filename = "../data/cities.shp";	// Shapefile path
		string tablename = "Cities";	// Name of the attribute table

		if (TeImportShape(layer, filename, tablename))
			cout << "The shapefile \"cities.shp\" was imported successfully into the TerraLib database!\n" << endl;
		else
			cout << "Error: Fail to import the shapefile \"cities.shp\"!\n" << endl;
	}

	// Retrieve from the database a layer of polygons, a layer of lines and a layer of points
    TeLayer* states = new TeLayer("states");		// states have polygons
    if (!db->loadLayer(states))                
    {                                                                                               
        cout << "Fail to load layer \"states\": " << db->errorMessage() << endl;
		cout << endl << "Press Enter\n";
		getchar();
		return 1;
    }

    TeLayer* rivers = new TeLayer("rivers");	// rivers have lines
    if (!db->loadLayer(rivers))                
    {                                                                                               
        cout << "Fail to load layer \"rivers\": " << db->errorMessage() << endl;
		cout << endl << "Press Enter\n";
		getchar();
		return 1;
    }

	TeLayer* cities = new TeLayer("cities");	// cities have points
    if (!db->loadLayer(cities))                
    {                                                                                               
        cout << "Fail to load layer \"cities\": " << db->errorMessage() << endl;
		cout << endl << "Press Enter\n";
		getchar();
		return 1;
    }

	vector<string> objsOut;	// holds the identification of the resulting objects
	
	vector<string> objsIn;  // objects to be queried		
	objsIn.push_back("23");

	// Retrieve the states that touches state 23
	bool res = db->spatialRelation(states->tableName(TePOLYGONS), TePOLYGONS, objsIn,
		                      objsOut, TeTOUCHES);
	if (res)
	{
		cout << "States that touch state \"23\": \n";
		unsigned int i;
		for (i=0; i<objsOut.size(); i++)
			cout << "State: " << objsOut[i] << endl;

	}

	// Retrieve the rivers that crosses state 23
	res = db->spatialRelation(states->tableName(TePOLYGONS), TePOLYGONS, objsIn,
		                            rivers->tableName(TeLINES), TeLINES, objsOut, TeCROSSES);
	if (res)
	{
		cout << "\nRivers that cross state \"23\": \n";
		unsigned int i;
		for (i=0; i<objsOut.size(); i++)
			cout << "River: " << objsOut[i] << endl;
	}

	// Retrieve the cities that are within state 23
	res = db->spatialRelation(states->tableName(TePOLYGONS), TePOLYGONS, objsIn,
		                      cities->tableName(TePOINTS), TePOINTS, objsOut, TeWITHIN);
	if (res)
	{
		cout << "\nCities that are within state \"23\": \n";
		unsigned int i;
		for (i=0; i<objsOut.size(); i++)
			cout << "City: " << objsOut[i] << endl;

	}

	db->close();
	cout << endl << "Press Enter\n";
	cout.flush();
	getchar();
	return 0;
}

