/************************************************************************************
TerraView - visualization and exploration of geographical databases
using TerraLib.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.
This file is part of TerraView. TerraView is free software; you can redistribute it 
and/or modify it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

You should have received a copy of the GNU General Public License
along with TerraView.
The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The software provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use of
this program and its documentation.
*************************************************************************************/

#include <TeQtCanvas.h>
#include <stdio.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qcursor.h>
#include <qpaintdevicemetrics.h> 

#define MIN(a,b) a<b?a:b
#define MAX(a,b) a>b?a:b

//TeQtCanvas::TeQtCanvas(QWidget *parent, const char *name ) : QScrollView(parent,name,WNorthWestGravity)
TeQtCanvas::TeQtCanvas(QWidget *parent, const char *name ) : TeQtBasicCanvas(parent,name)
{
	setVScrollBarMode(QScrollView::Auto);
	setHScrollBarMode(QScrollView::Auto);
    viewport()->setMouseTracking( TRUE );
    viewport()->setFocusPolicy( StrongFocus );
    viewport()->setBackgroundColor( backgroundColor() );
	setMargin (0);
	width_ = viewport()->width();
	height_ = viewport()->height();
	popupCanvas_ = new QPopupMenu(this);
}

TeQtCanvas::~TeQtCanvas()
{
	if (backRaster_)
		delete backRaster_;
	backRaster_ = 0;

	QPaintDevice* dev = painter_.device();
	if(dev)
		painter_.end();

	if(pixmap0_)
		delete pixmap0_;
	pixmap0_ = 0;

	if(pixmap1_)
		delete pixmap1_;
	pixmap1_ = 0;

	if(pixmap2_)
		delete pixmap2_;
	pixmap2_ = 0;

	if(pixmap3_)
		delete pixmap3_;
	pixmap3_ = 0;
}

void TeQtCanvas::setMode (CursorMode m)
{
	cursorMode_ = m;
	down_ = false;
}

void TeQtCanvas::clear()
{
	int ww, hh;
	ww = viewport()->width();
	hh = viewport()->height();
	resizeContents (ww,hh);

// Clear the window
	QPaintDevice* dev = painter_.device();
	if(dev)
		painter_.end();
	painter_.begin(viewport());
	painter_.eraseRect(0, 0, ww, hh);
	painter_.end();
	if(dev)
	painter_.begin(dev);

	emit windowClear();
}

void TeQtCanvas::clear(TeBox box)
{
	int ww, hh;
	ww = viewport()->width();
	hh = viewport()->height();
	resizeContents (ww,hh);

	mapCanvasWorldToViewport(box);
// Clear the window
	QPaintDevice* dev = painter_.device();
	plotOnWindow();
	painter_.eraseRect((int)box.x1(), (int)box.y1(), (int)box.width(), (int)box.height());
	painter_.end();
	if(dev)
		painter_.begin(dev);
	emit windowClear();
}

void TeQtCanvas::clearAll()
{
	// Clear the window
	clear();

	// Clear the pixmap
	if(pixmap0_)
		pixmap0_->fill(paletteBackgroundColor());
	if(pixmap1_)
		pixmap1_->fill(paletteBackgroundColor());
	if(pixmap2_)
		pixmap2_->fill(paletteBackgroundColor());
	if(pixmap3_)
		pixmap3_->fill(paletteBackgroundColor());

	if (backRaster_)
		delete backRaster_;
	backRaster_ = 0;

}

void TeQtCanvas::clearAll(TeBox box)
{
	// Clear the window
	clear(box);

	// Clear the pixmap
	mapCanvasWorldToViewport(box);

	QPaintDevice* dev = painter_.device();
	if(dev)
		painter_.end();

	if(pixmap0_)
	{
		painter_.begin(pixmap0_);
		painter_.fillRect((int)box.x1(), (int)box.y1(), (int)box.width(), (int)box.height(), backgroundColor());
		painter_.end();
	}
	if(pixmap1_)
	{
		painter_.begin(pixmap1_);
		painter_.fillRect((int)box.x1(), (int)box.y1(), (int)box.width(), (int)box.height(), backgroundColor());
		painter_.end();
	}
	if(pixmap0_)
	{
		painter_.begin(pixmap2_);
		painter_.fillRect((int)box.x1(), (int)box.y1(), (int)box.width(), (int)box.height(), backgroundColor());
		painter_.end();
	}

	if(dev)
		painter_.begin(dev);
}

void TeQtCanvas::setView(int w, int h, QPaintDevice *pd)
{
    int ww = w, 
           hh = h;

    int dpix, dpiy, ncolors, wmm, hmm, depth, pdepth;

	if (pd == 0)
	{
		if (ww == 0)
			ww = viewport()->width();
		if (hh == 0)
			hh = viewport()->height();
		resizeContents (ww,hh);
		x0_ = 0;
		y0_ = 0;
	}
	else
	{
		QPaintDeviceMetrics pdm( pd );
		if (ww == 0)
			ww = pdm.width();
		if (hh == 0)
			hh = pdm.height ();
//		x0_ = pdm.width()*.1;
//		y0_ = pdm.height()*.1;
		x0_ = 0;
		y0_ = 0;
		dpix = pdm.logicalDpiX();
		dpiy = pdm.logicalDpiY();
		ncolors = pdm.numColors ();
		wmm = pdm.widthMM ();
		hmm = pdm.heightMM ();
		depth = pdm.depth ();
		QPaintDevice* dev = painter_.device();
		if(dev)
			painter_.end();
		painter_.begin(pd);
	    painter_.setClipRect( x0_, y0_, ww, hh );
	}

// Build new pixmap if window has been resized
	if (pixmap0_ == 0 || width_ != ww || height_ != hh)
	{
		QPaintDevice* dev = painter_.device();
		if(dev)
			painter_.end();

		if (pixmap0_)
			delete pixmap0_;
		pixmap0_ = 0;

		if (pixmap1_)
			delete pixmap1_;
		pixmap1_ = 0;

		if (pixmap2_)
			delete pixmap2_;
		pixmap2_ = 0;

		if (pixmap3_)
			delete pixmap3_;
		pixmap3_ = 0;

// Build a new pixmap
		pixmap0_ = new QPixmap (ww,hh);
		pixmap1_ = new QPixmap (ww,hh);
		pixmap2_ = new QPixmap (ww,hh);
		pixmap3_ = new QPixmap (ww,hh);

		pdepth = pixmap0_->depth ();
	}

	width_ = ww;
	height_ = hh;
	params_.ncols_ = width_;
	params_.nlines_ = height_;

	if (pd == 0)
	{
		QPaintDevice* dev = painter_.device();
		if(dev)
			painter_.end();
		painter_.begin(pixmap0_);
	}

	down_ = false;
	xul_ = xmin_;
	yul_ = ymax_;
	xlr_ = xmax_;
	ylr_ = ymin_;
	clearAll();
}

void TeQtCanvas::correctScrolling (QPoint &p)
{
	if(painter_.device() == viewport())
	{
		QPoint o (contentsX (), contentsY ());
		p -= o;
	}
}

void TeQtCanvas::initCursorArea(QPoint p)
{
	if (!pixmap0_)
		return;

	down_ = true;
	TeCoord2D pw = mapVtoW(p);
	xul_ = pw.x();
	yul_ = pw.y();
	ixul_ = ixlr_ = p.x();
	iyul_ = iylr_ = p.y();
}

void TeQtCanvas::contentsMousePressEvent( QMouseEvent* e)
{
	if (!pixmap0_)
		return;

	QPoint m = e->pos();
	TeCoord2D p = mapVtoW(m);

	if(e->button() == LeftButton)
	{
		if (cursorMode_ == Area || cursorMode_ == Edit)
		{
			plotOnWindow();
			down_ = true;
			xul_ = p.x();
			yul_ = p.y();
			ixul_ = ixlr_ = e->pos().x();
			iyul_ = iylr_ = e->pos().y();
			emit mouseMoved (p, e->state(), m);
		}
		else
			emit mousePressed (p, e->state(), m);
	}
	else if(e->button() == RightButton)
		emit mouseRightButtonPressed (p, e->state(), m);
}

void TeQtCanvas::contentsMouseMoveEvent ( QMouseEvent* e)
{
	if (!pixmap0_)
		return;
	QPoint m = e->pos();
	TeCoord2D p = mapVtoW(m);
	if (down_)
	{
		QPoint o = offset();
		painter_.setPen (green);
		painter_.setRasterOp (Qt::XorROP);
		painter_.drawLine(ixul_-o.x(),iyul_-o.y(),ixlr_-o.x()-1,iyul_-o.y());            
		painter_.drawLine(ixlr_-o.x(),iyul_-o.y(),ixlr_-o.x(),iylr_-o.y() -1);            
		painter_.drawLine(ixlr_-o.x(),iylr_-o.y(),ixul_-o.x()-1,iylr_-o.y() );            
		painter_.drawLine(ixul_-o.x(),iylr_-o.y(),ixul_-o.x(),iyul_-o.y()-1 );            
		ixlr_ = e->pos().x();
		iylr_ = e->pos().y();
		painter_.drawLine(ixul_-o.x(),iyul_-o.y(),ixlr_-o.x()-1,iyul_-o.y());            
		painter_.drawLine(ixlr_-o.x(),iyul_-o.y(),ixlr_-o.x(),iylr_ -o.y()-1);            
		painter_.drawLine(ixlr_-o.x(),iylr_-o.y(),ixul_-o.x()-1,iylr_-o.y() );            
		painter_.drawLine(ixul_-o.x(),iylr_-o.y(),ixul_-o.x(),iyul_-o.y()-1 );            
		painter_.setRasterOp (Qt::CopyROP);
	}
	emit mouseMoved (p, e->state(), m);
}

void TeQtCanvas::contentsMouseReleaseEvent( QMouseEvent* e)
{
	if (!pixmap0_)
		return;
	QPoint m = e->pos();
	TeCoord2D p = mapVtoW(m);
	if (down_)
	{
		ixlr_ = e->pos().x();
		iylr_ = e->pos().y();
		xlr_ = p.x();
		ylr_ = p.y();
		if(xul_ > xlr_)
		{
			double a = xlr_;
			xlr_ = xul_;
			xul_ = a;
		}
		if(ylr_ > yul_)
		{
			double a = ylr_;
			ylr_ = yul_;
			yul_ = a;
		}
		down_ = false;
	}
	emit mouseReleased (p, e->state(), m);
}

void TeQtCanvas::contentsMouseDoubleClickEvent ( QMouseEvent* e)
{
	if (!pixmap0_)
		return;

	QPoint m = e->pos();
	TeCoord2D p = mapVtoW(m);

	if(e->button() == LeftButton)
		emit mouseDoublePressed (p, e->state(), m);
}


void TeQtCanvas::contentsContextMenuEvent( QContextMenuEvent* e)
{
//	popupCanvas_->exec(QCursor::pos());
	QMouseEvent *m = (QMouseEvent *)e;
	emit popupCanvasSignal(m);
}


void TeQtCanvas::leaveEvent ( QEvent * )
{
	emit mouseLeave();
}

void TeQtCanvas::keyPressEvent (QKeyEvent* e)
{
	emit keyPressed(e);
}

void TeQtCanvas::keyReleaseEvent (QKeyEvent* e)
{
	emit keyReleased(e);
}

void TeQtCanvas::viewportPaintEvent (QPaintEvent*)
{
	copyPixmapToWindow ();
	emit paintEvent ();
}

void TeQtCanvas::resizeEvent ( QResizeEvent * e)
{
//	viewport()->resize(e->size());
	QScrollView::resizeEvent(e);
}

void TeQtCanvas::copyPixmapToWindow()
{
	if (pixmap0_)
	{
		int cx = contentsX();
		int cy = contentsY();
		bitBlt (viewport(),0,0,pixmap0_,cx,cy,width_-cx,height_-cy,CopyROP,true);
	}
}

double TeQtCanvas::mapVtoData (int pixels)
{
	TeBox dbox = getDataWorld();
	double dwidth = dbox.width();
	int	width = viewport()->width();

	double d = (double)pixels * dwidth / (double)width;
	return d;
}

int TeQtCanvas::mapDatatoV (double a)
{
	TeBox dbox = getDataWorld();
	double dwidth = dbox.width();
	int	pwidth = viewport()->width();

	int d = (int)((double)pwidth * a / (double)dwidth);
	return d;
}

TeBox 
TeQtCanvas::getPieBarBox(TeThemeApplication* theme)
{
	TeBox	boxOut;

	if(theme->chartAttributes_.size() == 0)
		return boxOut;

	string objectId = theme->chartSelected_;

	string	TS = theme->collectionTable();
	string	sel = "SELECT * FROM " + TS + " WHERE ";

/*	if(theme->chartObjects() == TeSelectedByPointing)
		sel += TS + ".sel_by_pointing = 1 AND ";
	else if(theme->chartObjects() == TeNotSelectedByPointing)
		sel += TS + ".sel_by_pointing = 0 AND ";
	else if(theme->chartObjects() == TeSelectedByQuery)
		sel += TS + ".sel_by_query = 1 AND ";
	else if(theme->chartObjects() == TeNotSelectedByQuery)
		sel += TS + ".sel_by_query = 0 AND ";
	else if(theme->chartObjects() == TeGrouped)
		sel += TS + ".c_legend_id <> 0 AND ";
	else if(theme->chartObjects() == TeNotGrouped)
		sel += TS + ".c_legend_id = 0 AND ";
*/
	sel += TS + ".c_object_id = '" + objectId + "'";

	TeDatabasePortal *portal = theme->layer()->database()->getPortal();
	if (!portal)
		return false;

	if (!portal->query(sel) || !portal->fetchRow())
	{
		delete portal;
		return boxOut;
	}

	setDataProjection(theme->layer()->projection());

	double	width = theme->barWidth();
	double	maxh = theme->barMaxHeight();
	double	minh = theme->barMinHeight();
	double	maxd = theme->pieMaxDiameter();
	double	mind = theme->pieMinDiameter();
	double	diameter = theme->pieDiameter();


	if(theme->keepDimension() != 0)
	{
		width = mapVtoData((int)width);
		maxh = mapVtoData((int)maxh);
		minh = mapVtoData((int)minh);
		maxd = mapVtoData((int)maxd);
		mind = mapVtoData((int)mind);
		diameter = mapVtoData((int)diameter);
	}

	unsigned int i;

	TeChartType chartType = (TeChartType)theme->chartType();
	if(chartType == TePieChart)
	{
		double delta = diameter / 2.;
		if(!(theme->pieDimAttribute() == "NONE"))
			delta = maxd / 2.;

		string attrs;
		for(i=0; i<theme->chartAttributes_.size(); i++)
			attrs += theme->chartAttributes_[i] + ",";
		attrs += "label_x,label_y";
		if(!(theme->pieDimAttribute() == "NONE"))
			attrs += "," + theme->pieDimAttribute();

		string q = "SELECT " + attrs + theme->sqlFrom() + " WHERE " + TS + ".c_object_id = '" + objectId + "'";
		portal->freeResult();
		if(portal->query(q))
		{
			if(portal->fetchRow())
			{
				double tot = 0.;
				for(i=0; i<theme->chartAttributes_.size(); i++)
				{
					string v = portal->getData(i);
					if(v.empty())
					{
						delete portal;
						return boxOut;
					}
					tot += atof(v.c_str());
				}
				if(tot == 0.)
				{
					delete portal;
					return boxOut;
				}

				if(!(theme->pieDimAttribute() == "NONE"))
				{
					if(theme->chartMaxVal() - theme->chartMinVal() == 0)
						diameter = 0.;
					else
					{
						double	adim = portal->getDouble(theme->chartAttributes_.size()+2);
						double a = (adim - theme->chartMinVal()) / (theme->chartMaxVal() - theme->chartMinVal());
						diameter = a * (maxd - mind) + mind;
					}
				}
				double	x = theme->chartPoint_.x();
				double	y = theme->chartPoint_.y();
				boxOut.x1_ = x - diameter / 2.;
				boxOut.y1_ = y - diameter / 2.;
				boxOut.x2_ = x + diameter / 2.;
				boxOut.y2_ = y + diameter / 2.;
			}
		}
	}
	else
	{
		double	n = theme->chartAttributes_.size();
		double	maxv = theme->chartMaxVal();
		double	minv = theme->chartMinVal();

		string attrs;
		for(i=0; i<theme->chartAttributes_.size(); i++)
			attrs += theme->chartAttributes_[i] + ",";
		attrs += "label_x,label_y";

		string q = "SELECT " + attrs + theme->sqlFrom() + " WHERE " + TS + ".c_object_id = '" + objectId + "'";

		vector<double> dvec;
		portal->freeResult();
		if(portal->query(q))
		{
			if(portal->fetchRow())
			{
				double	x = theme->chartPoint_.x();
				double	xmin = x - n * width / 2.;
				double	xmax = x + n * width / 2.;
				double	y = theme->chartPoint_.y();

				dvec.clear();
				for(i=0; i<theme->chartAttributes_.size(); i++)
				{
					string val = portal->getData(i);
					if(val.empty())
					{
						delete portal;
						return boxOut;
					}
					dvec.push_back(atof(val.c_str()));
				}

				double mheight = -10;
				for(i=0; i<dvec.size(); i++)
				{
					double height;
					double v = dvec[i];
					if(maxv - minv == 0)
						height = 0.;
					else
						height = (v - minv) * (maxh - minh) / (maxv - minv) + minh;
					mheight = MAX(height, mheight);
				}
				boxOut.x1_ = xmin;
				boxOut.y1_ = y;
				boxOut.x2_ = xmax;
				boxOut.y2_ = y + mheight;
			}
		}
	}
	delete portal;
	return boxOut;
}


