#include "TeMrSID.h"
#include "MrSIDNavigator.h" 
#include "MrSIDImageFile.h" 
#include "lt_xTrans.h" 
#include "lt_exception.h"

LT_USE_NAMESPACE(LizardTech)

// MrSID not initialized yet
bool TeMrSIDReader::mrsid_initialized = false; 

// Constructor
TeMrSIDReader::TeMrSIDReader(const char* fname) :
	sidNav_(0),
    sidImageFile_(0),
	imgBuffer_(0),
	mrsidFileName_(fname)
{
  if(!mrsid_initialized)
  {
    setlocale(LC_ALL,"English");
    XTrans::initialize();
    mrsid_initialized = true;
  }

  FileSpecification filespec(fname);
  try
  {
	  sidImageFile_ = new MrSIDImageFile(filespec);
	  sidNav_ = new MrSIDNavigator(*sidImageFile_);
  }
  catch(...)
  {
	  int j = 0;
  }

	return;
};

TeMrSIDReader::~TeMrSIDReader()
{
  if(sidNav_) delete sidNav_;
 // if(sidImageFile_) delete sidImageFile_;		// ????!!!
  clear();
}

void TeMrSIDReader::clear()
{
  if(imgBuffer_ != NULL) 
  {
    free(imgBuffer_);
    imgBuffer_ = NULL;
  }
}

int TeMrSIDReader::nBands() const
{
  return sidNav_->nband();
}

unsigned int TeMrSIDReader::bitsPerPixel() const
{
  return sidImageFile_->colorSpace().samplesPerPixel();
}

TeMrSIDReader::ColorModel TeMrSIDReader::getColorModel() const
{
	ColorSpace::Scheme scheme = sidImageFile_->colorSpace().scheme();
	if (scheme == ColorSpace::RGB)
		return ColorModelRGB;
	else if (scheme == ColorSpace::GRAYSCALE)
		return ColorModelGray;
	else
		return ColorModelUnknown;
}

unsigned int TeMrSIDReader::getWidth() const 
{
  return sidNav_->imageWidth();
}

unsigned int TeMrSIDReader::getHeight() const
{
  return sidNav_->imageHeight();
}

bool TeMrSIDReader::hasWorldInfo()
{
  return sidNav_->hasWorldInfo();
}

double TeMrSIDReader::originX()
{
  double x;

  if(!sidImageFile_->xu(x))
    return 0;

  x-= resX() * 0.5;

  return x;
}

double TeMrSIDReader::originY()
{
  double y;

  if(!sidImageFile_->yu(y))
    return 0;

  y-= resY() * 0.5;

  return y;
}

double TeMrSIDReader::resX()
{
  double res;

  MetadataReader myReader = sidImageFile_->metadata();
  if (myReader.empty()) return 1;
  if (myReader.keyExists("IMAGE::X_RESOLUTION"))
  {
    MetadataElement element = myReader.getValue("IMAGE::X_RESOLUTION");
    if (!(element.type() == MetadataValue::DOUBLE)) return 1;
    res = (double) element.getMetadataValue();
    return res;
  }
  else return 1;
}

double TeMrSIDReader::resY()
{
  double res;

  MetadataReader myReader = sidImageFile_->metadata();
  if (myReader.empty()) return -1;
  if (myReader.keyExists("IMAGE::Y_RESOLUTION"))
  {
    MetadataElement element = myReader.getValue("IMAGE::Y_RESOLUTION");
    if (!(element.type() == MetadataValue::DOUBLE)) return -1;
    res = (double) element.getMetadataValue();
    return -res;
  }
  else return -1;
}

void TeMrSIDReader::getWorld(double& x0, double& y0, double& x1, double& y1)
{
  double sx, sy;
  unsigned int w, h;

  getOrigin(x0, y0);
  getResolution(sx, sy);
  getDimensions(w, h);

  x1 = x0 + w * sx;
  y1 = y0 + h * sy;
}

void TeMrSIDReader::getCurrentBoundingBox(double& xmin, double& ymin, double& xmax, double& ymax)
{
  double sx, sy;
  unsigned int w, h;
  double x0, y0, x1, y1;

  sidNav_->xres(sx);
  sidNav_->yres(sy);

  sidNav_->panTo(0,0,IntRect::TOP_LEFT);
  if(!sidNav_->xu(x0))
    x0 = 0.0;
  else
    x0-= sx * 0.5;

  if(!sidNav_->yu(y0))
    y0 = 0.0;
  else
    y0-= sy * 0.5;

  sy = -1*sy;
  int l = sidNav_->level();

  LizardTech::IntDimension dim = sidImageFile_->getDimensionsAtLevel(l);
  w = dim.width;
  h = dim.height;

  x1 = x0 + w * sx;
  y1 = y0 + h * sy;

  if(x0 < x1)
  {
    xmin = x0;
    xmax = x1;
  }
  else
  {
    xmin = x1;
    xmax = x0;
  }

  if(y0 < y1)
  {
    ymin = y0;
    ymax = y1;
  }
  else
  {
    ymin = y1;
    ymax = y0;
  }
}

void TeMrSIDReader::getBoundingBox(double& xmin, double& ymin, double& xmax, double& ymax)
{
  double sx, sy;
  unsigned int w, h;
  double x0, y0, x1, y1;

  sx = resX();
  sy = resY();

  if(!sidImageFile_->xu(x0))
    x0 = 0.0;
  else
    x0-= sx * 0.5;

  if(!sidImageFile_->yu(y0))
    y0 = 0.0;
  else
    y0-= sy * 0.5;

  w = sidNav_->imageWidth();
  h = sidNav_->imageHeight();

  x1 = x0 + w * sx;
  y1 = y0 + h * sy;

  if(x0 < x1)
  {
    xmin = x0;
    xmax = x1;
  }
  else
  {
    xmin = x1;
    xmax = x0;
  }

  if(y0 < y1)
  {
    ymin = y0;
    ymax = y1;
  }
  else
  {
    ymin = y1;
    ymax = y0;
  }
}

ImageBuffer* TeMrSIDReader::getImageBuffer(unsigned int size, unsigned int nbands,
                                           const unsigned int* bands, void* data)
{
	unsigned int i;
	ImageBufferInfo properties;
	int* sample_map;

	if(data == NULL)
		throw OutOfMemoryException();

	// compute each band offset
	sample_map = (int*)calloc(nbands, sizeof(int));
	if (sample_map == NULL)
		throw OutOfMemoryException();

	for(i = 0;i < nbands;i++)
		sample_map[i] = bands[i] * size;

	/* 
	Set image buffer properties
	Attention! The order in which the ImageBufferInfo properties are set matters.
	So, care must be taken during modification of code part.
	*/
	if(nBands() == 1)
		properties.setColorSpace(ColorSpace::GRAYSCALE);
	else
		properties.setColorSpace(ColorSpace::RGB);

	properties.setSampleMap((const int*)sample_map);
	properties.setBandDistribution(ImageBufferInfo::BSQ);
	properties.setSampleType(ImageBufferInfo::UINT8);
	properties.setAlignment(ImageBufferInfo::NO_ALIGN);
	properties.setStride(1);
	free(sample_map);
	return new ImageBuffer(properties, data);
}

ImageBuffer* TeMrSIDReader::getImageBuffer2(unsigned int size, unsigned int nbands,
                                const unsigned int* bands, void* data[])
{
  unsigned int i;
  ImageBufferInfo properties;
  int* sample_map;

  // allocate data for the image buffer
  if(imgBuffer_ != NULL) 
	  free(imgBuffer_);
  imgBuffer_ = (unsigned char*)calloc(nbands, size);

  if(imgBuffer_ == NULL)
    throw OutOfMemoryException();

  // compute each band offset
  sample_map = (int*)calloc(nbands, sizeof(int));
  if(sample_map == NULL)
    throw OutOfMemoryException();

  for(i = 0;i < nbands;i++)
  {
    sample_map[i] = bands[i] * size;
    data[i] = imgBuffer_ + sample_map[i];
  }

  /* 
    Set image buffer properties
    Attention! The order in which the ImageBufferInfo properties are set matters.
    So, care must be taken during modification of code part.
  */
  if(nBands() == 1)
    properties.setColorSpace(ColorSpace::GRAYSCALE);
  else
    properties.setColorSpace(ColorSpace::RGB);

  properties.setSampleMap((const int*)sample_map);
  properties.setBandDistribution(ImageBufferInfo::BSQ);
  properties.setSampleType(ImageBufferInfo::UINT8);
  properties.setAlignment(ImageBufferInfo::NO_ALIGN);
  properties.setStride(1);

  free(sample_map);

  return new ImageBuffer(properties, imgBuffer_);
}

int TeMrSIDReader::getMaxZoomLevel()
{ 
	if (sidNav_)
		return sidNav_->getMaxZoomLevel();
	return -1;
}

int TeMrSIDReader::nlev()
{
	if (sidNav_)
		return sidNav_->nlev();
	return 0;
}

int TeMrSIDReader::getCurrentLevel()
{
	if (sidNav_)
		return sidNav_->level();
	return 0;
}

void TeMrSIDReader::zoomTo(int level)
{
	if (sidNav_)
		sidNav_->zoomTo(level);
}

void TeMrSIDReader::getCurrentLevelResolution(double& resx, double& resy)
{
	if (!sidNav_)
		return;
	sidNav_->yres(resy);
	sidNav_->xres(resx);
}

void TeMrSIDReader::getCurrentRectangleDimension(int& w, int& h)
{
	if (!sidNav_)
		return;
	w = sidNav_->width();
	h = sidNav_->height();
}

void TeMrSIDReader::getDimensionAtLevel(int level, int& w, int& h)
{
	if (!sidImageFile_)
		return;
	LizardTech::IntDimension dim;
	dim = sidImageFile_->getDimensionsAtLevel(level);
	w = dim.width;
	h = dim.height;
}

/* Pans the specified rectangle navigator to a new point and redefines it's dimension*/
bool 
TeMrSIDReader::selectArea(int leftColumn, int topRow, int width, int height)
{
	if(!sidImageFile_)
		return false;

	if(!sidNav_)
		return false;
	try{
		sidNav_->panTo(leftColumn, topRow, IntRect::TOP_LEFT);
	    sidNav_->resize(width, height, IntRect::TOP_LEFT);
	}
	catch(...)
	{
		return false;
	}
	return true;
}

/* Loads the image selected by the navigator rectangle into memory.
   Expects that the pointer to unsiged char was already allocated */
bool 
TeMrSIDReader::getSelectedArea(unsigned char* data)
{
	if(!sidImageFile_)
		return false;

	if(!sidNav_)
		return false;

	try{
		ImageBuffer* sidBuffer = getImageBuffer(sidNav_->width() * sidNav_->height(), 
			                                    nBands(), data);
		sidNav_->loadImage(*sidBuffer);
		delete sidBuffer;
	}
	catch(...)
	{
		return false;
	}
	return true;
}


/* Allocates an image buffer as expected from MrSID */
ImageBuffer* 
TeMrSIDReader::getImageBuffer(unsigned int size, unsigned int nbands, void* data)
{
	ImageBufferInfo properties;
	if (nbands == 1)
		properties.setColorSpace(ColorSpace::GRAYSCALE);
	else
		properties.setColorSpace(ColorSpace::RGB);
    properties.setBandDistribution(ImageBufferInfo::BIP);
    properties.setSampleType(ImageBufferInfo::UINT8);
    properties.setAlignment(ImageBufferInfo::NO_ALIGN);
	if (nbands == 3)
	{
	    properties.setStride(3);
		int sampleMap[] = {0, 1, 2,};
		properties.setSampleMap(sampleMap);
	}
	else
	    properties.setStride(1);
	return new ImageBuffer(properties, data);
}
