/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/
/*! \file TeDecoder.h
    This file deals with decoding of raster structures 
*/
#ifndef  __TERRALIB_INTERNAL_DECODER_H
#define  __TERRALIB_INTERNAL_DECODER_H

#ifdef WIN32
#pragma warning ( disable: 4786 )
#endif

#include "TeProjection.h"
#include "TeBox.h"
#include "TeRasterParams.h"
#include "TeFactory.h"

using namespace std;
#include <string>
#include <list>
#include <map>

class TeRaster;
class TeDecoderMemory;

//! An abstract class to decode raster data 
/* 
	A decoder keeps its own copy of the  parameters associated
	to the raster being decoded.
	
	\sa TeRaster, TeRasterParams
*/
class TeDecoder {

public:
	//! Empty constructor
	TeDecoder() {}

	//! Constructor from  raster parameters
	/*!
		\param par raster parameters
	*/
	TeDecoder(TeRasterParams& par) : params_(par) {}

	//! Virtual destructor (derived classe might have their own)
	virtual ~TeDecoder() {}	

	//! Returns the raster parameters
	TeRasterParams& params()
		{ return params_; }

	//! Updates the raster parameters
	void updateParams(TeRasterParams& par)
	{	params_ = par; }

	//! Sets the value of a specific raster pixel 
	/*!
		\param col pixel column identifier
		\param lin pixel line identifier
		\param val pixel value being inserted
		\param band pixel band identifier
	*/
	virtual bool setElement(int col, int lin, double val, int band=0 ) = 0;

	//! Gets an specific element (col, lin, band) of a raster data
	/*!
		\param col pixel column identifier
		\param lin pixel line identifier
		\param val pixel value being retrieved
		\param band pixel band identifier
	*/
	virtual bool getElement(int col, int lin, double& val, int band=0 ) = 0;

	//! Transform a geographical coordinate to an index (lin, col) coordinate
	virtual TeCoord2D coord2Index (TeCoord2D &pt)
	{	return params_.coord2Index(pt);	}

	//! Transform an index (lin, col) coordinate to geographical coordinate
	virtual TeCoord2D index2Coord (TeCoord2D &pt)
	{	return params_.index2Coord(pt);	}

	//! Initializes the internal structures of the decoder from an external raster parameters structure
	/*  
	    Copy the raster parameters to the internal parameters of the raster
		calls the init() method
	*/
	virtual void init( TeRasterParams& par ) 
	{ clear(); params_= par; init();}

	//! Initializes the internal structures of the decoder, from its raster parameters structure.
	virtual void init() = 0;

	//! Clears its internal structures
	virtual bool clear() = 0;

	//! Returns a default object - for compatibility reasons with TeDecoderFactory
	static TeDecoder* DefaultObject( const TeRasterParams& /* par */) 
	{ return 0; }

/** @name Selection
  These routines are used to implement a selection of blocks or tiles 
  that intercept a given bounding box, and provide them sequentially.
  Decoders that decodify rasters in formats that support tiling should
  reimplement theses functions in an optimized way.
*/
//@{ 
	
	//! Returns TRUE if it there are raster blocks that intersect a given box
	virtual bool selectBlocks(TeBox& /* bb */, int /* resFac */, TeRasterParams& /*parBlock*/)  
	{ return false; }

	//! Gets the the current raster block selected as a raster in memory
	virtual bool getSelectedRasterBlock(TeDecoderMemory* /* memDec */) 
	{ return false; }

	//! Clear a block selection previously done
	virtual void clearBlockSelection() {}

	//! Returns the raster best resolution level available to fill a given bounding box with a given number of lines and columns in a given projection
	virtual int bestResolution(TeBox& /*bb*/, int /*ncols*/, int /*nlines*/, TeProjection* /*proj*/)
	{	return 1; }

	//! Returns the number of blocks selected in the last query
	virtual int numberOfSelectedBlocks()
	{	return 0; }

//@}

protected:

//! The parameters that describes a raster data
	TeRasterParams  params_;	
};

//! A decoder abstract factory
/*! 
	Implements an abstract factory that builds appropriate decoders
	according to a string identifier.
*/
class TeDecoderFactory: public TeFactory<TeDecoder,TeRasterParams> 
{
public:

	//! Maps synonyms to decoders identifiers (example "TIFF" or "tif" to  decoderTIFF
	typedef map<string,string> TeNames2Decoders;

	//! Unique instance instance of map from synonyms to decoder identifiers
	static TeNames2Decoders& instanceName2Dec ()
	{ 
		static TeNames2Decoders names2dec_;
		return names2dec_;
	}

	//! Builds an appropriate decoder from a identifier
	TeDecoderFactory(const string& name) : TeFactory<TeDecoder,TeRasterParams>(name) { }

	//! Destructor
	virtual ~TeDecoderFactory() {}
};

#endif
