/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#ifdef WIN32
#pragma warning ( disable: 4786 )
#endif

#include "TeIntersector.h"
#include <map>

using namespace std;

//---------------- TeIntersectionsBO ----------------//
bool TeINTERSECTOR2::TeIntersects(const TeLine2D& redLine, const TeLine2D& blueLine)
{
	if(redLine.size() < 2 || blueLine.size() < 2) 
		return false; 

	if(TeDisjoint(redLine.box(), blueLine.box()))
		return false;

	unsigned int i = 0;
	unsigned int j = 0;

	TeBox interBox;
	
	// Creates a intersection box from lines boxes.
	::TeIntersection(redLine.box(), blueLine.box(), interBox);

	unsigned int nstep_redLine  = redLine.size() - 1;
	unsigned int nstep_blueLine = blueLine.size() - 1;

	TeSegmentIntersectionType t = TeImproperIntersection;

	for(i = 0; i < nstep_redLine; ++i)
	{
		// Creates the segment box.
		TeBox red_box = makeBox(redLine[i].x(), redLine[i].y(), redLine[i+1].x(), redLine[i+1].y());

		// See if red segment box intersects with the intersection box.
		if(TeDisjoint(interBox, red_box))
			continue;	// If it doesn't intersect, go to another segment => skip comparasion beteween other "m" segments.

		for(j = 0; j < nstep_blueLine; ++j)
		{
			// Check intersection.
			if(TeIntersects(redLine[i], redLine[i+1], blueLine[j], blueLine[j+1], t))
				return true;
		}
	}	

	return false;
}



bool TeINTERSECTOR2::TeSafeIntersections(const TeLine2D& redLine, const TeLine2D& blueLine, TeVectorBoundaryIP& report, const unsigned int& redObjId, const unsigned int& blueObjId)
{
	if(redLine.size() < 2 || blueLine.size() < 2) 
		return false; 

	if(TeDisjoint(redLine.box(), blueLine.box()))
		return false;

	unsigned int i = 0;
	unsigned int j = 0;

	TeBox interBox;
	
	bool hasIntersections = false;

	// Creates a intersection box from lines boxes.
	::TeIntersection(redLine.box(), blueLine.box(), interBox);

	unsigned int nstep_redLine  = redLine.size() - 1;
	unsigned int nstep_blueLine = blueLine.size() - 1;

	TeSegmentIntersectionType t = TeImproperIntersection;

	for(i = 0; i < nstep_redLine; ++i)
	{
		// Creates the segment box.
		TeBox red_box = makeBox(redLine[i].x_, redLine[i].y_, redLine[i+1].x_, redLine[i+1].y_);

		// See if red segment box intersects with the intersection box.
		if(TeDisjoint(interBox, red_box))
			continue;	// If it doesn't intersect, go to another segment => skip comparasion beteween other "m" segments.

		for(j = 0; j < nstep_blueLine; ++j)
		{
			// Check intersection.
			TeBoundaryIP ip;

			if(TeIntersection(redLine[i], redLine[i+1], blueLine[j], blueLine[j+1], ip, t))
			{
				hasIntersections = true;

				ip.redPartNum_ = redObjId;
				ip.redSegNum_ = i;
				ip.bluePartNum_ = blueObjId;
				ip.blueSegNum_ = j;

				if(ip.coords_.size() == 2)	//overlap
				{
					// Verificar se os pontos esto em ordem crescente
					if(ip.coords_[0].x_ < ip.coords_[1].x_)
					{
						report.push_back(ip);
					}
					else if(ip.coords_[0].x_ > ip.coords_[1].x_)
					{
						swap(ip.coords_[0], ip.coords_[1]);
						report.push_back(ip);
					}
					else if(ip.coords_[0].y_ < ip.coords_[1].y_)
					{
						report.push_back(ip);						
					}
					else
					{
						swap(ip.coords_[0], ip.coords_[1]);
						
						report.push_back(ip);						
					}

				}
				else
				{
					report.push_back(ip);
				}
			}
		}
	}	

	return hasIntersections;
}


bool TeINTERSECTOR2::TeSafeIntersections(const TePolygonSet& redPols, const TePolygonSet& bluePols, TeVectorBoundaryIP& report)
{
	unsigned int redPart = 0;
	unsigned int redPolsSize = redPols.size();

	// Loops through red polygons
	for(unsigned int i = 0; i < redPolsSize; ++i)
	{
		TePolygon redPol = redPols[i];
		unsigned int redPolSize = redPol.size();

		// Loops through red polygons rings
		for(register unsigned int j = 0; j < redPolSize; ++j)
		{
			TeLinearRing redRing = redPol[j];			

			// Loops through blue polygons
			unsigned int bluePart = 0;
			unsigned int bluePolsSize =  bluePols.size();

			for(unsigned int k = 0; k < bluePolsSize; ++k)
			{
				// Loops through blue polygons rings
				TePolygon bluePol = bluePols[k];

				unsigned int bluePolSize = bluePol.size();

				for(register unsigned int l = 0; l < bluePolSize; ++l)
				{
					TeLinearRing blueRing = bluePol[l];

					TeSafeIntersections(redRing, blueRing, report, redPart, bluePart);

					++bluePart;
				}
			}

			++redPart;
		}
	}

	return !report.empty();
}

