/*  gtktiemu - a TI89/92/92+ emulator
 *  (c) Copyright 2000, Romain Lievin and Thomas Corvazier
 *  (c) Copyright 2001, Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __LIB68K__
#define __LIB68K__

#include <time.h>

#ifdef __cplusplus
extern "C"{
#endif

#include "callbacks.h"
#include "keydef.h"
#include "tilibs.h"
  
  /*************/
  /* Constants */
  /*************/  

#define MAXCHARS 256
  
  // Constants
#define TI92 1
#define TI89 2
#define MODULEPLUS 4
#define TI92p (TI92 | MODULEPLUS)
  
#define INTERNAL 1
#define FLASH_ROM 2
  
  extern void link_progress(int type, char *name, int size);

  // Bkpts types (access)
#define BK_BYTE 1
#define BK_WORD 2
#define BK_LONG 4
#define BK_READ 16
#define BK_WRITE 32

#define BK_READ_BYTE (BK_READ | BK_BYTE)
#define BK_READ_WORD (BK_READ | BK_WORD)
#define BK_READ_LONG (BK_READ | BK_LONG)
#define BK_READ_LONG_WORD BK_READ_LONG
#define BK_WRITE_BYTE (BK_WRITE | BK_BYTE)
#define BK_WRITE_WORD (BK_WRITE | BK_WORD)
#define BK_WRITE_LONG (BK_WRITE | BK_LONG)
#define BK_WRITE_LONG_WORD BK_WRITE_LONG

  // Constants for setBreakpointVector()
#define BK_NONE                -1
#define BK_BUS_ERROR           2
#define BK_ADDRESS_ERROR       3
#define BK_ILLEGAL_INSTRUCTION 4
#define BK_ZERO_DIVIDE         5
#define BK_CHK_INSTRUCTION     6
#define BK_TRAPV_INSTRUCTION   7
#define BK_PRIVILEGE_VIOLATION 8
#define BK_TRACE               9
#define BK_LINE_1010           10
#define BK_LINE_1111           11
#define BK_NONINIT_INTERRUPT   15

  // Constants for setBreakpointAutoint()
#define BK_SPURIOUS  0
#define BK_AUTOINT_1 1
#define BK_AUTOINT_2 2
#define BK_AUTOINT_3 3
#define BK_AUTOINT_4 4
#define BK_AUTOINT_5 5
#define BK_AUTOINT_6 6
#define BK_AUTOINT_7 7

  // Constants for setBreakpointTrap()
#define BK_TRAP_NONE -1
#define BK_TRAP_0 0
#define BK_TRAP_1 1
#define BK_TRAP_2 2
#define BK_TRAP_3 3
#define BK_TRAP_4 4
#define BK_TRAP_5 5
#define BK_TRAP_6 6
#define BK_TRAP_7 7
#define BK_TRAP_8 8
#define BK_TRAP_9 9
#define BK_TRAP_A 10
#define BK_TRAP_B 11
#define BK_TRAP_C 12
#define BK_TRAP_D 13
#define BK_TRAP_E 14
#define BK_TRAP_F 15

  // Breakpoints cause (get with getBkptCause())
#define BK_CAUSE_ACCESS       1
#define BK_CAUSE_ACCESS_RANGE 2
#define BK_CAUSE_ADDRESS      3
#define BK_CAUSE_VECTOR       4
#define BK_CAUSE_TRAP         5
#define BK_CAUSE_AUTOINT      6

  /*********/
  /* Types */
  /*********/

  /*
    Store informations about TIB files or ROM images
  */
  struct ti68k_rom_infos_
  {
    int loaded;
    int calc_type;
    char version[20];
    int flash;
    int internal;
    int size;
    int tib;
  };
  typedef struct ti68k_rom_infos_ ROM_INFO;
  typedef struct ti68k_rom_infos_ Ti68kRomInfo;
  typedef struct ti68k_rom_infos_ TIB_INFO;
  typedef struct ti68k_rom_infos_ Ti68kTibInfo;

  /* 
     Refresh/progress functions
     This structure allows to implement a kind of callbacks mechanism (which
     allow libti68k to interact with user without being dependant of a GUI).
  */
  struct ti68k_info_update
  {
    /* Variables to update */
    int cancel;                // Abort the current transfer
    char label_text[MAXCHARS]; // A text to display (varname, ...)
    float percentage;          // Percentage of the current operation
    float main_percentage;     // Percentage of all operations
    float prev_percentage;     // Previous percentage of current operation
    float prev_main_percentage;// Previous percentage of all operations
    int count;                 // Number of bytes exchanged
    int total;                 // Number of bytes to exchange
    clock_t start_time;        // Time when transfer has begun
    clock_t current_time;      // Current time
    
    /* Functions for updating */
    void (*start)   (void);                   // Init internal variables
    void (*stop)    (void);                   // Release internal variables
    void (*refresh) (void);                   // Pass control to GUI for refresh
    void (*msg_box) (const char *t, char *s); // Display a message box
    void (*pbar)    (void);                   // Refresh the progress bar
    void (*label)   (void);                   // Refresh the label
    int  (*choose)  (char *cur_name, 
		     char *new_name);         // Display choice box
  };
  typedef struct ti68k_info_update Ti68kInfoUpdate;

  
  /*************/
  /* Functions */
  /*************/

  // Main functions
  extern void ti68k_set_update(Ti68kInfoUpdate *iu,
			       void (*start)    (void),
			       void (*stop)     (void),
			       void (*resfresh) (void),
			       void (*msg_box)  (const char *t, char *s),
			       void (*pbar)     (void),
			       void (*label)    (void),
			       int  (*choose)   (char *cur_name, 
						 char *new_name));

  extern int ti68k_setGuiCallbacks(callback_iv_t initSpecific, 
				   callback_iv_t exitSpecific,
				   callback_iv_t updateScreen,
				   callback_iv_t updateKeys,
				   callback_iv_t screenOn,
				   callback_iv_t screenOff,
				   callback_vi_t setScreenPtr,
				   callback_ii_t setContrast);

  extern int ti68k_loadDefaultConfig(void);

  extern int ti68k_initLib68k(void);
  extern int ti68k_resetLib68k(void);  
  extern int ti68k_closeLib68k(void);

  // Error handling functions
  extern int ti68k_getError(int err_num, char *error_msg);
#define ti68k_get_error ti68k_getError //preferred
  
  // State functions
  extern int ti68k_loadState(char *filename);
  extern int ti68k_saveState(char *filename);

  // ROM related functions
  extern int ti68k_loadRom(char *filename);
  extern int ti68k_loadTib(char *filename);
  extern int ti68k_convertTibToRom(char *f_in, char *f_out);
  extern void* ti68k_getRomPtr(void);
  extern int ti68k_getRomSize(void);
  extern const char *ti68k_getRomVersion(void);
  extern int ti68k_isRomOk(void);
  extern int ti68k_getRomType(void);
  extern int ti68k_getRomImageInfo(const char *filename, ROM_INFO *ri);
  //extern int ti68k_getUpdateInfo(const char *filename, ROM_INFO *ri);
  //extern int ti68k_getRomInfo(ROM_INFO *ri);
  extern int ti68k_dumpRom(char *filename);
  extern int ti68k_scanRomImages(const char *dirname, const char *filename);

  // RAM related functions
  extern void* ti68k_getRamPtr(void);
  extern int ti68k_getRamSize(void);
  extern int ti68k_dumpRam(char *filename);

  // Debugger functions
  extern void ti68k_setInstructionsDisplay(int bVal);
  extern int ti68k_disasm(int addr,char *output);
  extern callback_iv_t ti68k_defineDebugger(callback_iv_t new_debugger);
  extern int ti68k_launchDebugger(void);
  extern void ti68k_getBreakpointCause(int *type, int *id, int *mode);
  extern int ti68k_doSingleStep(void);
  extern int ti68k_doInstructions(int n);

  // Breakpoints handling functions  
  extern int ti68k_setBreakpointAddress(int address);
  extern int ti68k_setBreakpointAccess(int address, int mode);
  extern int ti68k_setBreakpointAccessRange(int addressMin,int addressMax,
					    int mode);
  extern int ti68k_setBreakpointVector(int vector);
  extern int ti68k_setBreakpointAutoint(int autoint);
  extern int ti68k_setBreakpointTrap(int trap);
  extern void ti68k_delBreakpointAddress(int i);
  extern void ti68k_delBreakpointAccess(int i, int mode);
  extern void ti68k_delBreakpointAccessRange(int i, int mode);
  extern void ti68k_delBreakpointVector(int i);
  extern void ti68k_delBreakpointAutoint(int i);
  extern void ti68k_delBreakpointTrap(int i);

  extern void ti68k_setDataRegister(int n, int val);
  extern void ti68k_setAddressRegister(int n, int val);
  extern void ti68k_setSpRegister(int val);
  extern void ti68k_setPcRegister(int val);
  extern void ti68k_setSrRegister(int val);
  extern void ti68k_setFlagRegister(unsigned char flag);
  extern int ti68k_getDataRegister(int n);
  extern int ti68k_getAddressRegister(int n);
  extern int ti68k_getSpRegister();
  extern int ti68k_getPcRegister();
  extern int ti68k_getSrRegister();
  extern char *ti68k_getFlagRegister(void);

  extern struct intlist* listBreakpointAddress;
  extern int nBreakpointAddress;

  // Keyboard functions
  extern void ti68k_setActiveKey(int key, int active);
  extern int  ti68k_isKeyPressed(int key);
#define isKeyPressed ti68k_isKeyPressed

  // LCD functions
  extern void* ti68k_getLcdPtr(void);
  extern char ti68k_getContrast(void);

  // Linkport functions
  void ti68k_open_linkport();
  void ti68k_close_linkport();
  int ti68k_setInternalLinkTimeout(int value);
  int ti68k_getInternalLinkTimeout(int value);
  int ti68k_sendFile(char *filename);
  int ti68k_setLinkCable(LinkCable *lc);
  LinkCable* ti68k_getLinkCable();

  // Miscellaneous
  extern int ti68k_getCalcType(void);
#define getCalcType ti68k_getCalcType

#ifdef __cplusplus
}
#endif

#endif
