#   MailScanner - SMTP E-Mail Virus Scanner
#   Copyright (C) 2001  Julian Field
#
#   $Id: sweep.pl,v 1.50 2002/03/05 14:09:33 jkf Exp $
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#   The author, Julian Field, can be contacted by email at
#      Jules@JulianField.net
#   or by paper mail at
#      Julian Field
#      Dept of Electronics & Computer Science
#      University of Southampton
#      Southampton
#      SO17 1BJ
#      United Kingdom
#

# Run Sophos sweep over the directory hierarchy I'm given.
# Parse the output and find all the viruses.
# Passed in a directory..
# Return a hash. Keys are "messageid,messagepart", values are reports
# and only exist when there was a virus in the messagepart.

use strict;
use POSIX qw(setsid);
package Sweep;

my($LOCK_SH) = 1;
my($LOCK_EX) = 2;
my($LOCK_NB) = 4;
my($LOCK_UN) = 8;

my (
    $S_NONE,			# Not present
    $S_UNSUPPORTED,		# Present but you're on your own
    $S_ALPHA,			# Present but not tested -- we hope it works!
    $S_BETA,			# Present and tested to some degree -- we think it works!
    $S_SUPPORTED,		# People use this; it'd better work!
   ) = (0,1,2,3,4);

my %Scanners = (
  sophos => {
    Lock		=> 'SophosBusy.lock',
    # In next line, '-ss' makes it work nice and quietly
    CommonOptions	=> '-sc -f -all -rec -ss -archive -loopback --no-follow-symlinks --no-reset-atime -TNEF',
    DisinfectOptions	=> '-di',
    ScanOptions		=> '',
    InitParser		=> \&InitSophosParser,
    ProcessOutput	=> \&ProcessSophosOutput,
    SupportScanning	=> $S_SUPPORTED,
    SupportDisinfect	=> $S_SUPPORTED,
  },
  mcafee		=> {
    Lock		=> 'McAfeeBusy.lock',
    CommonOptions	=> '--recursive --ignore-links --analyze --secure --noboot',
    DisinfectOptions	=> '--clean',
    ScanOptions		=> '',
    InitParser		=> \&InitMcAfeeParser,
    ProcessOutput	=> \&ProcessMcAfeeOutput,
    SupportScanning	=> $S_SUPPORTED,
    SupportDisinfect	=> $S_SUPPORTED,
  },
  command		=> {
    Lock		=> 'CommandBusy.lock',
    CommonOptions	=> '-packed -archive',
    DisinfectOptions	=> '-disinf',
    ScanOptions		=> '',
    InitParser		=> \&InitCommandParser,
    ProcessOutput	=> \&ProcessCommandOutput,
    SupportScanning	=> $S_SUPPORTED,
    SupportDisinfect	=> $S_SUPPORTED,
  },
  inoculate	=> {
    Lock		=> 'InoculateBusy.lock',
    CommonOptions	=> '-nex -arc -mod reviewer -spm h ',
    DisinfectOptions	=> '-act cure -sca mf',
    ScanOptions		=> '',
    InitParser		=> \&InitInoculateParser,
    ProcessOutput	=> \&ProcessInoculateOutput,
    SupportScanning	=> $S_SUPPORTED,
    SupportDisinfect	=> $S_SUPPORTED,
  },
  inoculan	=> {
    Lock		=> 'InoculanBusy.lock',
    CommonOptions	=> '-nex -rev ',
    DisinfectOptions	=> '-nex -cur',
    ScanOptions		=> '',
    InitParser		=> \&InitInoculanParser,
    ProcessOutput	=> \&ProcessInoculanOutput,
    SupportScanning	=> $S_SUPPORTED,
    SupportDisinfect	=> $S_SUPPORTED,
  },
  kaspersky	=> {
    Lock		=> 'KasperskyBusy.lock',
    CommonOptions	=> '',
    DisinfectOptions	=> '-- -I2',
    ScanOptions		=> '-I0',
    InitParser		=> \&InitKasperskyParser,
    ProcessOutput	=> \&ProcessKasperskyOutput,
    SupportScanning	=> $S_BETA,
    SupportDisinfect	=> $S_ALPHA,
  },
  "f-secure"	=> {
    Lock		=> 'FSecureBusy.lock',
    CommonOptions	=> '--dumb --archive',
    DisinfectOptions	=> '--auto --disinf',
    ScanOptions		=> '',
    InitParser		=> \&InitFSecureParser,
    ProcessOutput	=> \&ProcessFSecureOutput,
    SupportScanning	=> $S_BETA,
    SupportDisinfect	=> $S_BETA,
  },
  "f-prot"	=> {
    Lock		=> 'FProtBusy.lock',
    CommonOptions	=> '-old -archive -dumb',
    DisinfectOptions	=> '-disinf -auto',
    ScanOptions		=> '',
    InitParser		=> \&InitFProtParser,
    ProcessOutput	=> \&ProcessFProtOutput,
    SupportScanning	=> $S_SUPPORTED,
    SupportDisinfect	=> $S_SUPPORTED,
  },
  nod32		=> {
    Lock		=> 'Nod32Busy.lock',
    CommonOptions	=> '-log-',
    DisinfectOptions	=> '-clean -delete',
    ScanOptions		=> '',
    InitParser		=> \&InitNOD32Parser,
    ProcessOutput	=> \&ProcessNOD32Output,
    SupportScanning	=> $S_SUPPORTED,
    SupportDisinfect	=> $S_SUPPORTED,
  },

  "none"		=> {
    Lock		=> 'NoneBusy.lock',
    CommonOptions	=> '',
    DisinfectOptions	=> '',
    ScanOptions		=> '',
    InitParser		=> \&NeverHappens,
    ProcessOutput	=> \&NeverHappens,
    SupportScanning	=> $S_NONE,
    SupportDisinfect	=> $S_NONE,
  },
);

# Put in a SIGCHLD handler to reap zombie processes
#$SIG{CHLD} = sub { 1 while wait() != -1 }; # Commented out in 2.53

sub PrintSummary {
  my($Reports, $Clean, $Dirty) = @_;
  my($id,$part,$value,$text);

  while(($id,$value) = each %$Reports) {
    while(($part, $text) = each %$value) {
      print "Virus found in message $id part $part\n   $text\n";
    }
  }
  print "Clean Messages are " . join(', ', @$Clean) . "\n";
  print "Dirty Messages are " . join(', ', @$Dirty) . "\n";
}

# Write whole-message infection reports for every message which we
# could not parse for some reason.
sub LogUnparsable {
  my($Reports, $InfTypes, $CantParse) = @_;
  my($id);

  foreach $id (@$CantParse) {
    $Reports->{"$id"}{""} = "Could not parse message $id";
    $InfTypes->{"$id"}{""} .= "e";
  }
}

# Write attachment infection reports for every message which has a
# TNEF attachment we could not parse.
sub LogBadTNEF {
  my($Reports, $InfTypes, $CantParse, $BaseDir) = @_;
  local(*DIR);
  my($id, $path, @tneffiles, $tnef);

  foreach $id (@$CantParse) {
    $path = "$BaseDir/$id";
    opendir(DIR, $path);
    @tneffiles = map { /(winmail\d*\.dat\d*)/i } readdir DIR;
    closedir DIR;
    foreach $tnef (@tneffiles) {
      $Reports->{"$id"}{"$tnef"} = "Could not parse Outlook Rich Text attachment";
      $InfTypes->{"$id"}{"$tnef"} = "e";
    }
  }
}

# Construct lists of the clean and infected messages.
# Passed in a hash of reports and a ref to list of message ids.
# Pass out a list of clean message ids and a list of infected message ids.
sub CleanAndDirty {
  my($Reports, $IdList, $Clean, $Dirty) = @_;
  my(%clean, %dirty, $key, $id, $part);

  map { $clean{$_} = 1 } @$IdList;
  foreach $id (keys %$Reports) {
    delete $clean{$id};
    $dirty{$id} = 1;
  }
  @$Clean = keys %clean;
  @$Dirty = keys %dirty;
}

sub VirusScan {
  my($BaseDir, $InfectionTypes) = @_;

  my(%infections, $NumInfections);

  $NumInfections = CallCommercialChecking($BaseDir, \%infections, $InfectionTypes)
    if $Config::VirusScanning;
  $NumInfections += CallOwnChecking($BaseDir, \%infections, $InfectionTypes);

  Log::InfoLog("Found $NumInfections viruses in messages " .
          join(',', keys %infections)) if $NumInfections;

  return \%infections;
}

sub CallCommercialChecking {
  my($BaseDir, $infections, $inftypes) = @_;

  local(*BASEDIR);
  my($NumInfections, $success, $id);

  chdir $BaseDir or die "Cannot chdir to $BaseDir, $!";

  $success = TryCommercial('.', $infections, $inftypes, \$NumInfections, $BaseDir);
  unless ($success) {
    # Virus checking the whole batch of messages timed out, so now check them
    # one at a time to find the one with the DoS attack in it.
    Log::WarnLog("Denial Of Service attack detected!");
    opendir BASEDIR, '.'
      or Log::DieLog("Can't open $BaseDir in CallCommercialChecking, $!");
    while($id = readdir BASEDIR) {
      next unless -d "$id";   # Only check directories
      next if $id =~ /^\.+$/; # Don't check myself or my parent
      # The "./" is important as it gets the path right for Sophos parser code
      $success = TryCommercial("./$id", $infections, $inftypes, \$NumInfections, $BaseDir);
      unless ($success) {
        # We have found the DoS attack message
        $infections->{"$id"}{""} = "Denial Of Service attack in message!";
        $inftypes->{"$id"}{""} .= "d";
        Log::WarnLog("Denial Of Service attack is in message $id");
      }
    }
    closedir BASEDIR;
  }

  # Return value is the number of infections we found
  return $NumInfections;
}

# Try all the installed commercial virus scanners
sub TryCommercial {
  my($subdir, $infections, $inftypes, $rCounter, $BaseDir) = @_;

  my($number, @scanners, @commands, $result);

  $Config::VirusScanner =~ s/^[\s,]*//; # Remove any leading or trailing
  $Config::VirusScanner =~ s/[\s,]*$//; # list separators
  $Config::Sweep        =~ s/^[\s,]*//; # Remove any leading or trailing
  $Config::Sweep        =~ s/[\s,]*$//; # list separators

  @scanners = split(/[\s,]+/, $Config::VirusScanner);
  @commands = split(/[\s,]+/, $Config::Sweep);
  $result = 0;

  foreach $number (0..$#scanners) {
    $result += TryOneCommercial($scanners[$number], $commands[$number], $subdir,
                                $infections, $inftypes, $rCounter, $BaseDir);
  }

  return $result;
}

# Try one of the commercial virus scanners
sub TryOneCommercial {
  my($scanner, $sweepcommand, $subdir, $infections, $inftypes, $rCounter, $BaseDir) = @_;

  local(*CHECKER, *LOCK, *KID);
  my ($rScanner, $VirusLock, $voptions);
  my($Counter, $TimedOut, $PipeReturn, $pid);

  Log::DieLog("Never heard of scanner '$scanner'!")
    unless exists $Scanners{$scanner};

  $rScanner = $Scanners{$scanner};

  if ($rScanner->{"SupportScanning"} == $S_NONE){
      Log::DebugLog("Scanning using scanner \"$scanner\" not supported; not scanning");
      return 1;
  }

  CheckCodeStatus($rScanner->{"SupportScanning"})
    or Log::DieLog("Bad return code from CheckCodeStatus - should it have quit?");

  $VirusLock = $Config::LockDir . "/" . $rScanner->{"Lock"}; # lock file
  $voptions  = $rScanner->{"CommonOptions"}; # Set common command line options
  $voptions .= " " . $rScanner->{"ScanOptions"}; # Add command line options to "scan only"
  &{$$rScanner{"InitParser"}}(); # Initialise scanner-specific parser


  # Check that the virus checker files aren't currently being updated,
  # and wait if they are.
  open(LOCK, ">$VirusLock")
    or Log::DieLog("Cannot create $VirusLock, $!");
  flock(LOCK, $LOCK_SH);
  print LOCK "Virus checker locked for scanning by $scanner $$\n";

  Log::DebugLog("Commencing scanning by $scanner...");

  $TimedOut = 0;
  eval {
    die "Can't fork: $!" unless defined($pid = open(KID, "-|"));
    if ($pid) {
      # In the parent
      local $SIG{ALRM} = sub { $TimedOut = 1; die "Command Timed Out" };
      alarm $Config::SweepTimeout;
      while(<KID>) {
	$Counter += &{$$rScanner{"ProcessOutput"}}($_, $infections, $inftypes, $BaseDir);
      }
      close KID;
      $PipeReturn = $?;
      $pid = 0; # 2.54
      alarm 0;
    } else {
      # In the child
      POSIX::setsid(); 
      exec "$sweepcommand $voptions $subdir"
        or die "Can't run commercial checker $scanner (\"$sweepcommand\"): $!";
    }
  };
  alarm 0; # 2.53

  # Note to self: I only close the KID in the parent, not in the child.
  Log::DebugLog("Completed scanning by $scanner");

  # Catch failures other than the alarm
  Log::DieLog("Commercial virus checker failed with real error: $@")
    if $@ and $@ !~ /Command Timed Out/;

  #print STDERR "pid = $pid and \@ = $@\n";

  # In which case any failures must be the alarm
  if ($@ or $pid>0) {
    # Kill the running child process
    my($i);
    kill -15, $pid;
    # Wait for up to 10 seconds for it to die
    for ($i=0; $i<10; $i++) {
      sleep 1;
      ($pid=0),last unless kill(0, $pid);
      kill -15, $pid;
    }
    # And if it didn't respond to 11 nice kills, we kill -9 it
    kill -9, $pid if $pid;
    wait; # 2.53
  }

  flock(LOCK, $LOCK_UN);
  close LOCK;
  $$rCounter = $Counter; # Set up output value

  # Return failure if the command timed out, otherwise return success
  Log::WarnLog("Commercial scanner $scanner timed out!") if $TimedOut;
  return 0 if $TimedOut;
  return 1;
}

# Initialise any state variables the Sophos output parser uses
sub InitSophosParser {
  ;
}

# Initialise any state variables the McAfee output parser uses
my($currentline);
sub InitMcAfeeParser {
  $currentline = '';
}

# Initialise any state variables the Command (CSAV) output parser uses
sub InitCommandParser {
  ;
}

# Initialise any state variables the Inoculate-IT output parser uses
sub InitInoculateParser {
  ;
}

# Initialise any state variables the Inoculan 4.x output parser uses
sub InitInoculanParser {
  ;
}

# Initialise any state variables the Kaspersky output parser uses
my ($kaspersky_CurrentObject);
sub InitKasperskyParser {
  $kaspersky_CurrentObject = "";
}

# Initialise any state variables the F-Secure output parser uses
my ($fsecure_InCruft);
sub InitFSecureParser {
  $fsecure_InCruft=(-1);
}

# Initialise any state variables the F-Prot output parser uses
my ($fprot_InCruft);
sub InitFProtParser {
  $fprot_InCruft=(-3);
}

# Initialise any state variables the Nod32 output parser uses
sub InitNod32Parser {
  ;
}

# These functions must be called with, in order:
# * The line of output from the scanner
# * A reference to the hash containing problem details
# * A reference to the hash containing types of problem
# * The base directory in which we are working.
#
# The base directory must contain subdirectories named
# per message ID, and must have no trailing slash.
#
#
# These functions must return with:
# * return code 0 if no problem, 1 if problem.
# * type of problem (currently only "v" for virus)
#   appended to $types{messageid}{messagepartname}
# * problem report from scanner appended to
#   $infections{messageid}{messagepartname} -- don't
#   forget the terminating newline.
#
# If the scanner may refer to the same file multiple times,
# you should consider appending to the $infections rather
# than just setting it, I guess.
#
sub ProcessSophosOutput {
  my($line, $infections, $types, $BaseDir) = @_;
  my($report, $infected, $dot, $id, $part, @rest);

  #print "$line";
  chomp $line;
  Log::WarnLog($line) if $line =~ /error/i;
  return 0 unless $line =~ /virus.*found/i;
  Log::InfoLog($line);
  $report = $line;
  $infected = $line;
  $infected =~ s/^.*found\s*in\s*file\s*//i;
  # JKF 10/08/2000 Used to split into max 3 parts, but this doesn't handle
  # viruses in zip files in attachments. Now pull out first 3 parts instead.
  ($dot, $id, $part, @rest) = split(/\//, $infected);
  $infections->{"$id"}{"$part"} .= $report . "\n";
  $types->{"$id"}{"$part"} .= "v"; # it's a real virus
  return 1;
}
  
sub ProcessMcAfeeOutput {
  my($line, $infections, $types, $BaseDir) = @_;

  my($lastline, $report, $dot, $id, $part, @rest);

  chomp $line;
  $lastline = $currentline;
  $currentline = $line;

  # SEP: need to add code to log warnings
  return 0 unless $line =~ /Found/;

  # McAfee prints the whole path as opposed to
  # ./messages/part so make it the same
  $lastline =~ s/$BaseDir//;

  # make an equivalent report line from the last 2
  $report = "$lastline$currentline";
  # note: '$dot' does not become '.'
  ($dot, $id, $part, @rest) = split(/\//, $lastline);
  $infections->{"$id"}{"$part"} .= $report . "\n";
  $types->{"$id"}{"$part"} .= "v";
  return 1;
}

# This next function originally contributed in its entirety by 
# "Richard Brookhuis" <brookhuis@busschers.nl>
sub ProcessCommandOutput {
  my($line, $infections, $types, $BaseDir) = @_;
  #my($line) = @_;

  my($report, $infected, $dot, $id, $part, @rest);

  #print "$line";
  chomp $line;
  Log::WarnLog($line) if $line =~ /error/i;
  return 0 unless $line =~ /Infection:/i;
  Log::InfoLog($line);
  $report = $line;
  $infected = $line;
  $infected =~ s/\s+Infection:.*$//i;
  # JKF 10/08/2000 Used to split into max 3 parts, but this doesn't handle
  # viruses in zip files in attachments. Now pull out first 3 parts instead.
  $line =~ s/-\>/\//; # JKF Handle archives rather better
  ($dot, $id, $part, @rest) = split(/\//, $infected);
  $infections->{"$id"}{"$part"} .= $report . "\n";
  $types->{"$id"}{"$part"} .= "v"; # it's a real virus
  #print "ID: $id  PART: $part  REPORT: $report\n";
  return 1;
}

# This next function contributed in its entirety by 
# sfarrell@icconsulting.com.au
sub ProcessInoculateOutput {
  my($line, $infections, $types, $BaseDir) = @_;
  my($report, $infected, $dot, $id, $part, @rest);

  #print "$line";

  chomp $line;
  Log::WarnLog($line) if $line =~ /error/i;
  Log::WarnLog($line) if $line =~ /Error/i;
  return 0 unless $line =~ /is infected by virus:/i;
  Log::InfoLog($line);

  # Ino prints the whole path as opposed to
  # ./messages/part so make it the same
  # Scott Farrell's system definitely requires the extra /
  # Output looks like this:
  # File: /var/spool/MailScanner/incoming/./message-id/filename
  $line =~ s/$BaseDir\///;

  # ino uses <file.ext> instead of /files.ext/ in archives
  $line =~ s/</\//;
  $line =~ s/>/\//;

  $report = $line;
  $infected = $line;
#  $infected =~ s/^.*found\s*in\s*file\s*//i;
  # JKF 10/08/2000 Used to split into max 3 parts, but this doesn't handle
  # viruses in zip files in attachments. Now pull out first 3 parts instead.
  ($dot, $id, $part, @rest) = split(/\//, $infected);
  $infections->{"$id"}{"$part"} .= $report . "\n";
  $types->{"$id"}{"$part"} .= "v"; # so we know what to tell sender
  return 1;
}

# Inoculan 4.x parser, contributed in its entirety by Gabor.Funk@hunetkft.hu
#
# This next function is the modified version of sfarrell@icconsulting.com.au's
# inoculateit 6.0 section by gabor.funk@hunetkft.hu - 2002.03.01 - v1.0
# It works with Inoculan 4.x inocucmd which is a beta/test/unsupported version
# Can be downloaded from: ftp://ftp.ca.com/getbbs/linux.eng/inoctar.LINUX.Z
# This package is rarely modified but you can download virsig.dat from other
# 4.x package such as the NetWare package (smallest and non MS compressed)
# It can be found at: ftp://ftp.ca.com/pub/InocuLAN/il0156.zip
# wget it; unzip il0156.zip VIRSIG.DAT; mv VIRSIG.DAT virsig.dat
# and since the last engine was "corrected" not to accept newer signature
# files, you have to patch the major version number to the same or below as
# the one which come with the inoctar.LINUX.Z (currently 32). In virsig.dat
# the major version number is located at address 10h for v33.17 this would
# be 33h. You simply have to change it to 32h and it should work. Note:
# using a higher version virsig.dat with a lower version engine is highly
# discouraged by CA and can result not to recognize newer viruses.
# inocucmd needs libstdc++-libc6.1-1.so.2 so you need to link it to your
# actual one (it was libstdc++-3-libc6.2-2-2.10.0.so on my debian testing).
# location: inocucmd and virsig.dat (the two required files) should be at
# /opt/CA, /usr/local/bin or other location specified in $CAIGLBL0000
# test: inocucmd .  (inocucmd without argument can report bogus virsig.dat
# version number but it's ok if it scans the file with no error)
# I like inocucmd because it needs 2 file alltogether, requires no building
# and/or "installation" so is very ideal for testing. 

sub ProcessInoculanOutput {
  my($line, $infections, $types, $BaseDir) = @_;
  my($report, $infected, $dot, $id, $part, @rest);
 
  chomp $line;
  Log::WarnLog($line) if $line =~ /error/i;
  Log::WarnLog($line) if $line =~ /Error/i;
  return 0 unless $line =~ /was infected by virus/i;
  Log::InfoLog($line);

  # Sample outputs for an unpacked and a packed virus
  # "[././cih-sfl.exe] was infected by virus [Win95/CIH.1003]"
  # "[././w95.arj:SLIDER10.EXE] was infected by virus [Win95/Slider 1.0.Trojan]"
 
  $report   = $line;
  $infected = $line;
  $infected =~ s/^\[\.\///i;
  $infected =~ s/([:\]]).*//i;
   
  ($dot, $id, $part, @rest) = split(/\//, $infected);
  $infections->{"$id"}{"$part"} .= $report . "\n";
  $types->{"$id"}{"$part"} .= "v"; # so we know what to tell sender
  return 1;
}

# If you use Kaspersky, look at this code carefully
# and then be very grateful you didn't have to write it.
#
sub ProcessKasperskyOutput {
  my($line, $infections, $types, $BaseDir) = @_;
  #my($line) = @_;
  
  my($report, $infected, $dot, $id, $part, @rest);
  
  # Don't know what kaspersky means by "object" yet...

  # Lose trailing cruft
  return 0 unless defined $kaspersky_CurrentObject;

  if ($line =~ /^Current\sobject:\s(.*)$/) {
    $kaspersky_CurrentObject = $1;
  }
  elsif ($kaspersky_CurrentObject eq "") {
    # Lose leading cruft
    return 0;
  }
  else {
    chomp $line;
    $line =~ s/^\r//;
    # We can rely on BaseDir not having trailing slash.
    # Prefer s/// to m// as less likely to do unpredictable things.
    if ($line =~ / infected: /) {
      $report = $line;
      $line =~ s/^$BaseDir//;
      $line =~ s/(.*) infected:.*/$1/;
      ($dot,$id,$part,@rest) = split(/\//, $line);
      $infections->{"$id"}{"$part"} .= $report . "\n";
      $types->{"$id"}{"$part"} .= "v"; # so we know what to tell sender
      return 1;
    }
    # see commented code below if you think this regexp looks fishy
    if ($line =~ /^([\r ]*)Scan\sprocess\scompleted\.\s*$/) {
      undef $kaspersky_CurrentObject;
      # uncomment this to see just one reason why I hate kaspersky AVP -- nwp
      # foreach(split //, $1) {
      #	  print ord($_) . "\n";
      # }
    }
  }
  return 0;
}

sub ProcessFSecureOutput {
  my($line, $infections, $types, $BaseDir) = @_;
  #my($line) = @_;
  
  my($report, $infected, $dot, $id, $part, @rest);

  chomp $line;

  # Lose cruft
  return 0 if $fsecure_InCruft > 0;
  if ($line eq "") {
    $fsecure_InCruft += 1;
    return 0;
  }
  $fsecure_InCruft == 0 or return 0;

  # Prefer s/// to m// as less likely to do unpredictable things.
  # We hope.
  if ($line =~ /\tinfection:\s/) {
    $report = $line;
    # Get to relevant filename in a reasonably but not
    # totally robust manner (*impossible* to be totally robust
    # if we have square brackets and spaces in filenames)
    # Strip archive bits if present
    $line =~ s/^\[(.*?)\] .+(\tinfection:.*)/$1$2/;
    # Get to the meat or die trying...
    $line =~ s/\tinfection:[^:]*$//
      or Log::DieLog("Dodgy things going on in F-Secure output:\n$report\n");
    ($dot,$id,$part,@rest) = split(/\//, $line);
    $infections->{"$id"}{"$part"} .= $report . "\n";
    $types->{"$id"}{"$part"} .= "v"; # so we know what to tell sender
    return 1;
  }

  Log::DieLog("Either you've found a bug in MailScanner's F-Secure\noutput parser, or F-Secure's output format has changed!\nPlease mail the author of MailScanner!\n");
}

sub ProcessFProtOutput {
  my($line, $infections, $types, $BaseDir) = @_;
  #my($line) = @_;
  
  my($report, $infected, $dot, $id, $part, @rest);

#  print STDERR $line;

  chomp $line;

  # Lose cruft
  return 0 if $fprot_InCruft > 0;
  if ($line eq "") {
    $fprot_InCruft += 1;
    return 0;
  }
  $fprot_InCruft == 0 or return 0;

  # Prefer s/// to m// as less likely to do unpredictable things.
  # We hope.
  # JKF 5+11/1/2002 Make "security risk" and "joke program" lines look like
  #                 virus infections for easier parsing.
  # JKF 25/02/2002  Add all sorts of patterns gleaned from a coredump of F-Prot
  $report = $line;
  if ($line =~ /(is|could be) a (security risk|virus construction)/) {
    # Reparse the rest of the line to turn it into an infection report
    $line =~ s/(is|could be) a (security risk|virus construction).*$/Infection: /;
  }
  if ($line =~ /(is|could be) a( boot sector)? virus dropper/) {
    # Reparse the rest of the line to turn it into an infection report
    $line =~ s/(is|could be) a( boot sector)? virus dropper.*$/Infection: /;
  }
  if ($line =~ /(is|could be) a corrupted or intended/) {
    # Reparse the rest of the line to turn it into an infection report
    $line =~ s/(is|could be) a corrupted or intended.*$/Infection: /;
  }
  if ($line =~ /(is|could be) a (joke|destructive) program/) {
    # Reparse the rest of the line to turn it into an infection report
    $line =~ s/(is|could be) a (joke|destructive) program.*$/Infection: /;
  }
  if ($line =~ /\s\sInfection:\s/) {
    # Get to relevant filename in a reasonably but not
    # totally robust manner (*impossible* to be totally robust
    # if we have slashes, spaces and "->" in filenames)
    $line =~ s/^(.*?)->.+(\s\sInfection:.*)/$1$2/;	# strip archive bits if present
    $line =~ s/^.*(\/.*\/.*)\s\sInfection:[^:]*$/$1/		# get to the meat or die trying
      or Log::DieLog("Dodgy things going on in F-Prot output:\n$report\n");
#    print STDERR "**$line\n";
    ($dot,$id,$part,@rest) = split(/\//, $line);
    $infections->{"$id"}{"$part"} .= $report . "\n";
    $types->{"$id"}{"$part"} .= "v"; # so we know what to tell sender
    return 1;
  }

  # Ignore files we couldn't scan as they were encrypted
  if ($line =~ /\s\sNot scanned \(encrypted\)/) {
    return 0;
  }

  Log::WarnLog("Either you've found a bug in MailScanner's F-Prot output parser, or F-Prot's output format has changed! F-Prot said this \"$line\". Please mail the author of MailScanner");
  return 0;
}

# This function provided in its entirety by Ing. Juraj Hantk <hantak@wg.sk>
sub ProcessNOD32Output {
  my($line, $infections, $types, $BaseDir) = @_;
  my($report, $infected, $dot, $id, $part, @rest);
    
  chomp $line;
  Log::WarnLog($line) if $line =~ /error/i;
  return 0 unless $line =~ /\s-\s/i;
 
  my ($part1,$part2,$part3,$part4,@ostatne)=split(/\.\//,$line);
  $line="./".$part4;
  Log::InfoLog($line);
  $report = $line;
  $infected = $line;
  $infected =~ s/^.*\s*-\s*//i;
 
  # JKF 10/08/2000 Used to split into max 3 parts, but this doesn't handle
  # viruses in zip files in attachments. Now pull out first 3 parts instead.
  ($dot, $id, $part, @rest) = split(/[\/,-]/, $report);
  $part =~ s/\s$//g;
  $infections->{"$id"}{"$part"} .= $report . "\n";
  $types->{"$id"}{"$part"} .= "v"; # it's a real virus
 
  return 1;
}


sub CallOwnChecking {
  my($BaseDir, $infections, $inftypes) = @_;

  # Insert your own checking here.
  # In $BaseDir, you will find a directory for each message, which has the
  # same name as the message id. Also there is a messageid.header file
  # containing all the headers for the message.
  # Add entries into %$infections, where they are referenced as
  # $infections->{"message id"}{"filename"} but please don't over-write ones
  # that are already there.
  # If the danger was detected in a header or applies to the whole message
  # then append the error report (and a newline) to
  # $infections->{"message id"}{""}.
  # Return the number of infections/problems you found.

  chdir $BaseDir or Log::DieLog("Could not chdir $BaseDir, %s", $!);

  my($id, $attach, $DirEntry);
  local(*BASE, *MESSAGE, *HEADER);
  my $counter = 0;

  opendir BASE, "." or Log::DieLog("Could not opendir $BaseDir, %s", $!);
  while ($DirEntry = readdir BASE) {
    next if $DirEntry =~ /^\./;

    # Test for presence of dangerous headers, such as "X-Spanska:"
    if (-f $DirEntry && $DirEntry =~ /\.header$/) {
      open(HEADER, $DirEntry) or next;
      $id = $DirEntry;
      $id =~ s/\.header$//;
      my @headers = <HEADER>;
      if (grep /^X-Spanska:/i, @headers) {
        Log::InfoLog("Found Happy virus in $id");
        $infections->{$id}{""} .= "\"Happy\" virus\n";
        $inftypes->{$id}{""} .= "v";
        $counter++;
      }
      # Comment out next if statement completely from production code
      #if (grep /Jules Private Virus Trigger/i, @headers) {
      #  Log::InfoLog("Found Jules Private Virus Trigger in $id");
      #  $infections->{$id}{""} .= "\"JKF Header Test\" virus\n";
      #  $counter++;
      #}

      close HEADER;
    }
    
    # Test for dangerous attachment filenames, such as *.jpeg.vbs,
    # *.lnk, *.vbs and "pretty park.exe"
    if (-d $DirEntry) {
      $id = $DirEntry;
      opendir MESSAGE, $id or next;
      while ($attach = readdir MESSAGE) {
        next unless -f "$id/$attach";

        # Work through the attachment filename rules, using the first
        # rule that matches.
        my $i;
        my $MatchFound = 0;
        for ($i=0; $i<@Config::NameAllow && !$MatchFound; $i++) {
          my $regexp = $Config::NameRE[$i];
          next unless $attach =~ /$regexp/i; # Next rule if this rule doesn't match
          $MatchFound = 1;
          if ($Config::NameAllow[$i] eq 'deny') {
            # It's a rejection rule, so log the error.
            Log::InfoLog($Config::NameLog[$i] . " in $attach");
            $infections->{$id}{$attach} .= $Config::NameUser[$i] .
                                           " in $attach\n";
            $inftypes->{$id}{$attach} .= "f";
            $counter++;
          }
        }
      }
      closedir MESSAGE;
    }
  }
  closedir BASE;
  return $counter;
}

# Call the commercial scanners, but to disinfect this time.
# Ignore all output from them.
# Re-check the files later to see if the disinfection was successful.
sub CallDisinfector {
  my($BaseDir) = @_;

  my($number, @scanners, @commands);

  # No need to do this here, will have already been done.
  #$Config::VirusScanner =~ s/^[\s,]*//; # Remove any leading or trailing
  #$Config::VirusScanner =~ s/[\s,]*$//; # list separators
  #$Config::Sweep        =~ s/^[\s,]*//; # Remove any leading or trailing
  #$Config::Sweep        =~ s/[\s,]*$//; # list separators

  @scanners = split(/[\s,]+/, $Config::VirusScanner);
  @commands = split(/[\s,]+/, $Config::Sweep);

  foreach $number (0..$#scanners) {
    CallOneDisinfector($scanners[$number], $commands[$number], $BaseDir);
  }
}

# Call one of the commercial scanners, but to disinfect.
# Ignore all output from it this time.
sub CallOneDisinfector {
  my($scanner, $sweepcommand, $BaseDir) = @_;

  local(*SWEEP, *LOCK);
  my($rScanner, $VirusLock, $voptions, $TimedOut, $pid, $PipeReturn);

  Log::DieLog("Never heard of scanner '$scanner'!")
    unless exists $Scanners{$scanner};

  $rScanner = $Scanners{$scanner};

  if ($rScanner->{"SupportDisinfect"} == $S_NONE){
      Log::DebugLog("Disinfection using scanner \"$scanner\" not supported; not disinfecting");
      return 1;
  }

  CheckCodeStatus($rScanner->{"SupportDisinfect"})
    or Log::DieLog("Bad return code from CheckCodeStatus - should it have quit?");

  $VirusLock = $Config::LockDir . "/" . $rScanner->{"Lock"}; # lock file
  $voptions  = $rScanner->{"CommonOptions"}; # Set common command line options
  $voptions .= " " . $rScanner->{"DisinfectOptions"}; # Add command line options to disinfect
  
  chdir $BaseDir or die "Cannot chdir to $BaseDir, $!";

  # Check that Sophos IDE files aren't currently being updated,
  # and wait if they are.
  open(LOCK, ">$VirusLock") or Log::DieLog("Cannot create $VirusLock, $!");
  flock(LOCK, $LOCK_SH);
  print LOCK "Sophos locked for scanning by $scanner $$\n";

  $TimedOut = 0;
  eval {
    die "Can't fork: $!" unless defined($pid = open(KID, "|-"));
    if ($pid) {
      # In the parent
      local $SIG{ALRM} = sub { $TimedOut = 1; die "Command Timed Out" };
      alarm $Config::SweepTimeout;
      print KID "A\n" if $scanner eq 'sophos'; # Tell sweep to disinfect all
      close KID;
      $PipeReturn = $?;
      $pid = 0;
      alarm 0;
    } else {
      # In the child
      POSIX::setsid();
      exec "$sweepcommand $voptions ."
        or die "Can't run commercial disinfector $scanner (\"$sweepcommand\"): $!";
    }
  };
  alarm 0; # 2.53

  # Note to self: I only close the KID in the parent, not in the child.

  # Catch failures other than the alarm
  Log::DieLog("Commercial virus checker failed with real error: $@")
    if $@ and $@ !~ /Command Timed Out/;

  # In which case any failures must be the alarm
  if ($@ or $pid>0) {
    # Kill the running child process
    my($i);
    kill -15, $pid;
    # Wait for up to 10 seconds for it to die
    for ($i=0; $i<10; $i++) {
      sleep 1;
      ($pid=0),last unless kill(0, $pid);
      kill -15, $pid;
    }
    # And if it didn't respond to 11 nice kills, we kill -9 it
    kill -9, $pid if $pid;
    wait; # 2.53
  }

  # Don't care about return code in this case, we ignore it anyway
  Log::InfoLog("Commercial disinfector $scanner returned $PipeReturn") if $PipeReturn;
  flock(LOCK, $LOCK_UN);
  close LOCK;
}

sub NeverHappens {
  Log::DieLog("THIS SHOULD NEVER HAPPEN.\nPlease report this as a bug to the authors.");
}

# Should be called when we're about to try to run some code to
# scan or disinfect (after checking that code is present)
sub CheckCodeStatus {
  my ($codestatus) = @_;

  my $allowedlevel = $S_SUPPORTED;

  $Config::CodeStatus =~ /^beta/i and $allowedlevel = $S_BETA;
  $Config::CodeStatus =~ /^alpha/i and $allowedlevel = $S_ALPHA;
  $Config::CodeStatus =~ /^unsup/i and $allowedlevel = $S_UNSUPPORTED;
  $Config::CodeStatus =~ /^none/i and $allowedlevel = $S_NONE;

  $codestatus >= $allowedlevel and return 1;

  Log::WarnLog("Looks like a problem... dumping status information");
  Log::WarnLog("Minimum acceptable stability = $allowedlevel ($Config::CodeStatus)");
  Log::WarnLog("Using Scanner \"$Config::VirusScanner\"");
  foreach (keys %Scanners) {
    my $statusinfo = "Scanner \"$_\": scanning code status ";
    $statusinfo .= $Scanners{$_}{"SupportScanning"};
    $statusinfo .= " - disinfect code status ";
    $statusinfo .= $Scanners{$_}{"SupportDisinfect"};
    Log::WarnLog($statusinfo);
  }
  Log::WarnLog("FATAL: Encountered code that does not meet configured acceptable stability"); 
  Log::DieLog("FATAL: *Please go and READ* http://www.sng.ecs.soton.ac.uk/mailscanner/install/codestatus.shtml"); 
}

1;
