/*
 * Copyright (c) 2001-2003 Shiman Associates Inc. All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */
/*
 * $Id: mas_endian_device.c,v 1.2 2003/03/10 22:28:44 rocko Exp $
 *
 * Copyright (c) 2000, 2001 by Shiman Associates Inc. and Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions: The above
 * copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the names of the authors or
 * copyright holders shall not be used in advertising or otherwise to
 * promote the sale, use or other dealings in this Software without
 * prior written authorization from the authors or copyright holders,
 * as applicable.
 *
 * All trademarks and registered trademarks mentioned herein are the
 * property of their respective owners. No right, title or interest in
 * or to any trademark, service mark, logo or trade name of the
 * authors or copyright holders or their licensors is granted.
 *
 */

/* 2 OCT 2002 - rocko - verified reentrant
 * 2 OCT 2002 - rocko - verified timestamp clean
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "mas/mas_dpi.h"
#include "profile.h"

/* what's the actual work we're going to do? */
#define MAS_END_NONE   0
#define MAS_END_SWAP   1

/************************************************************************
 * endian_state
 *
 * State memory structure for this device instance.
 *
 ************************************************************************/
struct endian_state
{
    int32 reaction;
    int32 sink;
    int32 source;
    int32 from_bytes;
    int32 from_endian;
    int32 to_endian;
    int32 to_bytes;
    int32 work;
};

/*************************************************************************
 * ACTIONS
 *************************************************************************/


/* standard actions ****************************************************/
int32 mas_dev_init_instance( int32 , void* );
int32 mas_dev_show_state( int32 device_instance, void* predicate );

/* device specific actions *********************************************/
int32 mas_endian_convert( int32 , void* );

int32
mas_dev_init_instance( int32 device_instance, void* predicate )
{
    struct endian_state*  state;
    
    /* Allocate state holder and cast it so we can work on it */
    state       = MAS_NEW( state );
    if ( state == 0 )
	return mas_error(MERR_MEMORY);

    masd_set_state(device_instance, state); /* set device state */
    
    masd_get_port_by_name( device_instance, "sink", &state->sink );
    masd_get_port_by_name( device_instance, "source", &state->source );
    masd_get_port_by_name( device_instance, "reaction", &state->reaction );

    state->from_endian = state->to_endian = MAS_NO_ENDIAN_FMT;
    
    return 0;
}

int32
mas_dev_configure_port( int32 device_instance, void* predicate )
{
    struct endian_state*  state;
    struct mas_data_characteristic* dc;
    int32 portnum = *(int32*)predicate;
    int32* dataflow_port_dependency;
    uint8  format, resolution, channels, endian;
    uint32 srate;
    int32  err;

    masd_get_state(device_instance, (void**)&state);

    err = masd_get_data_characteristic( portnum, &dc );
    if ( err < 0 )
        return mas_error(MERR_INVALID);
    
    err = masc_scan_audio_basic_dc( dc, &format, &srate, &resolution, &channels, &endian );
    if ( err < 0 )
        return mas_error(MERR_INVALID);
    
    if ( portnum == state->sink )
    {
        state->from_endian = endian;
        state->from_bytes = masc_get_audio_basic_bpstc( resolution, channels ) / channels;
    }
    else if ( portnum == state->source )
    {
        state->to_endian = endian;
        state->to_bytes = masc_get_audio_basic_bpstc( resolution, channels ) / channels;
    }
    else
    {
        return mas_error( MERR_NOTDEF );
    }

    /* if both ports are configured */
    if ( ( state->from_endian != MAS_NO_ENDIAN_FMT ) && ( state->to_endian != MAS_NO_ENDIAN_FMT ) )
    {
        /* Figure out if we're actually doing work. */

        state->work = MAS_END_NONE; /* the default */
        switch ( state->from_endian )
        {
        case MAS_BIG_ENDIAN_FMT:
            if ( state->to_endian == MAS_LITTLE_ENDIAN_FMT )
                state->work = MAS_END_SWAP;
#if (defined MAS_LITTLE_ENDIAN)
            if ( state->to_endian == MAS_HOST_ENDIAN_FMT )
                state->work = MAS_END_SWAP;
#endif
            break;
        case MAS_LITTLE_ENDIAN_FMT:
            if ( state->to_endian == MAS_BIG_ENDIAN_FMT )
                state->work = MAS_END_SWAP;
#if (defined MAS_BIG_ENDIAN)
            if ( state->to_endian == MAS_HOST_ENDIAN_FMT )
                state->work = MAS_END_SWAP;
#endif
            break;
        case MAS_HOST_ENDIAN_FMT:
#if (defined MAS_LITTLE_ENDIAN)
            if ( state->to_endian == MAS_BIG_ENDIAN_FMT )
                state->work = MAS_END_SWAP;
#endif
#if (defined MAS_BIG_ENDIAN)
            if ( state->to_endian == MAS_LITTLE_ENDIAN_FMT )
                state->work = MAS_END_SWAP;
#endif
            break;
        default: break;
            
        }
        
        /* schedule our dataflow dependency on data_sink */
        dataflow_port_dependency = masc_rtalloc( sizeof (int32) );
        *dataflow_port_dependency = state->sink;
        err = masd_reaction_queue_action(state->reaction, device_instance, 
                                         "mas_endian_convert", 0, 0, 0, 0, 0,
                                         MAS_PRIORITY_DATAFLOW, 1, 1, 
                                         dataflow_port_dependency);
        if ( err < 0 ) return err;
    }
    

    return 0;
}

int32
mas_dev_disconnect_port( int32 device_instance, void* predicate )
{
    struct endian_state*  state;
    int32 portnum = *(int32*)predicate;

    MASD_GET_STATE(device_instance, state);

    if ( portnum == state->sink )
    {
        state->from_endian = MAS_NO_ENDIAN_FMT;
    }
    else if ( portnum == state->source )
    {
        state->to_endian = MAS_NO_ENDIAN_FMT;
    }

    return 0;
}

int32
mas_dev_exit_instance( int32 device_instance, void* predicate )
{
    struct endian_state*  state;
    
    MASD_GET_STATE(device_instance, state);
    masc_rtfree( state );
    
    return 0;
}

int32
mas_dev_terminate( int32 device_instance, void* predicate )
{
    return 0;
}

int32
mas_dev_show_state( int32 device_instance, void* predicate )
{
    struct endian_state*  state;
    
    masd_get_state(device_instance, (void**)&state);

    if ( state->work )
        masc_log_message( 0, "endian %d is byte-swapping %d-bit words", device_instance, state->from_bytes*8 );
    
    if ( state->work )
        masc_log_message( 0, "endian %d isn't byte-swapping", device_instance );
    
    return 0;
}

int32
mas_endian_convert( int32 device_instance, void* predicate )
{
    struct endian_state* state;
    struct mas_data*     data;
    int16*               words;
    int32*               dwords;
    int                  i;
    
    masd_get_state(device_instance, (void**)&state);

    masd_get_data( state->sink, &data );

    if ( state->from_bytes > 2 )
    {
        /* Treat 20, 24 bit samples as 32-bit ints. */
        if ( state->work )
        {
            dwords = (int32*)data->segment;
            for (i=0; i< (data->length/4); i++)
                dwords[i] = mas_swap32( dwords[i] );
        }
        
    }
    else if ( state->from_bytes == 2 )
    {
        if ( state->work )
        {
            words = (int16*)data->segment;
            for (i=0; i< (data->length/2); i++)
                words[i] = mas_swap16( words[i] );
        }
    }

    /* don't convert if 8-bit */
        
    masd_post_data( state->source, data );

    return 0;
}

