/* miniapm - A tiny battery monitor

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/


#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h>
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <apm.h>

#define TIME_LEFT  0
#define PERCENTAGE 1

#define WIN_WIDTH  8
#define WIN_HEIGHT 14

#define CLOCK_DISP 1
#define BATT_DISP  0

int read_apm(int *values)
{
  /* add stat function here ? */
 
  apm_info info;
  apm_read(&info);

  values[TIME_LEFT] = info.battery_time;
  values[PERCENTAGE] = info.battery_percentage;

  return 0;
}


GC
create_gc(Display* display, Window win, int reverse_video)
{
  GC gc;			
  unsigned long valuemask = 0;	
				
  XGCValues values;		
  unsigned int line_width = 1;	
  int line_style = LineSolid;	
  int cap_style = CapButt;	
  int join_style = JoinMiter;	
  int screen_num = DefaultScreen(display);

  XFontStruct* font_info;

  /* try to load the given font. */
  char* font_name = "6x13";

  gc = XCreateGC(display, win, valuemask, &values);
  if (gc < 0) {
	fprintf(stderr, "XCreateGC: \n");
  }

  /* allocate foreground and background colors for this GC. */
  if (reverse_video) {
    XSetForeground(display, gc, WhitePixel(display, screen_num));
    XSetBackground(display, gc, BlackPixel(display, screen_num));
  } else {
    XSetForeground(display, gc, BlackPixel(display, screen_num));
    XSetBackground(display, gc, WhitePixel(display, screen_num));
  }

  /* define the style of lines that will be drawn using this GC. */
  XSetLineAttributes(display, gc,line_width, line_style, cap_style, join_style );

  /* define the fill style for the GC. to be 'solid filling'. */
  XSetFillStyle(display, gc, FillSolid);

  font_info = XLoadQueryFont(display, font_name);
  if (!font_info) {
    fprintf(stderr, "XLoadQueryFont: failed loading font '%s'\n", font_name);
  }
  XSetFont(display, gc, font_info->fid);

  return gc;
}

int update(Display* display, Window win, GC gc, GC rev_gc, 
	   int mode, Pixmap mask, GC mask_gc)
{
  int apm_vals[2];
  int rect_width;
  int screen_num = DefaultScreen(display);

  Colormap cmp;

  /* move this outside of the func !!!!!! */
  XColor Red, Orange, Green, Exact;  

  cmp = DefaultColormap(display, DefaultScreen(display));
  XAllocNamedColor(display, cmp, "red", &Red, &Exact);
  XAllocNamedColor(display, cmp, "green", &Green, &Exact);
  XAllocNamedColor(display, cmp, "orange", &Orange, &Exact);

  read_apm(apm_vals);

  XClearWindow(display,win);

  XSetForeground(display, mask_gc,
		 XWhitePixel(display, DefaultScreen(display)));
  XSetForeground(display, mask_gc,
		 XBlackPixel(display, DefaultScreen(display)));
  XFillRectangle(display, mask, mask_gc, 0, 0, WIN_WIDTH, WIN_HEIGHT);

  XSetForeground(display, mask_gc,
		 XBlackPixel(display, DefaultScreen(display)));
  XSetForeground(display, mask_gc,
		 XWhitePixel(display, DefaultScreen(display)));


  XShapeCombineMask (display, win, ShapeBounding, 0, 0, mask, ShapeSet);

  if (apm_vals[PERCENTAGE] <= 0 || apm_vals[PERCENTAGE] > 99)
     XSetForeground(display, gc, WhitePixel(display, screen_num));
  else if (apm_vals[PERCENTAGE] < 25)
     XSetForeground(display, gc, Red.pixel);
  else if (apm_vals[PERCENTAGE] < 75)
     XSetForeground(display, gc, Orange.pixel);
  else if (apm_vals[PERCENTAGE] <= 100)
     XSetForeground(display, gc, Green.pixel);

  XDrawRectangle(display, win, gc, 2, 0, 4, 2);  /* battery top */
  XDrawRectangle(display, win, gc, 1, 2, 6, 11); /* battery body */
  
  XDrawRectangle(display, mask, mask_gc, 2, 0, 4, 2);
  XDrawRectangle(display, mask, mask_gc, 1, 2, 6, 11);
  
  rect_width = apm_vals[PERCENTAGE]/10;
  
  XFillRectangle(display, win, gc, 1, 12-rect_width, 6, rect_width+1);
  XFillRectangle(display, mask, mask_gc, 1, 12-rect_width, 6, rect_width+1);

  XSetForeground(display, gc, BlackPixel(display, screen_num));
  XShapeCombineMask (display, win, ShapeBounding, 0, 0, mask, ShapeSet);

  XFlush(display);

  return 0;
}


void printversion(void)
{
  printf("Version: 0.2 \n");
}

void usage(void)
{
  printf("Usage: miniapm <options>\n");
  printf("Options:\n");
  printf("  -display  <display>\n");
  printf("  -v        version\n");
}



int main(int argc, char* argv[])
{
  Display* display;	
  int screen_num;	
  Window win;
  GC gc, rev_gc, mask_gc;
  Pixmap mask;
  
  char *geometry     = NULL;
  char *display_name = (char *)getenv("DISPLAY");  
  int depth, i;
  int mode = BATT_DISP;

  time_t last_time = 0;
  time_t curTime;

  int update_req = 0;
  int screenWidth;
  int screenHeight;

  /* pass command line */
  for (i=1; argv[i]; i++) {
    char *arg = argv[i];
    if (*arg=='-') {
      switch (arg[1]) {
	case 'd' : /* display */
	  display_name = argv[i+1];
	  i++;
	  break;
	case 'g' : /* geometry */
	  geometry = argv[i+1];
	  i++;
	  break;
	case 'v' :
	  printversion();
	  exit(0);
	  break;
	default:
	  usage();
	  exit(0);
	  break;
      }
    }
  }

  display = XOpenDisplay(display_name);
  if (display != NULL) {

     Atom window_type_atom = XInternAtom(display,
					 "_NET_WM_WINDOW_TYPE" , False);
     Atom window_type_dock_atom = XInternAtom(display,
					      "_NET_WM_WINDOW_TYPE_DOCK",False);

  screen_num = DefaultScreen(display);
  win = XCreateSimpleWindow(display,
			    RootWindow(display, screen_num),
			    0, 0,
			    WIN_WIDTH, WIN_HEIGHT,
			    0, BlackPixel(display, screen_num),
			    WhitePixel(display, screen_num));

  gc = create_gc(display, win, 0);
  rev_gc = create_gc(display, win, 1);
  depth = DefaultDepth(display, DefaultScreen(display));

  mask = XCreatePixmap(display, win, WIN_WIDTH, WIN_HEIGHT, 1);
  mask_gc = create_gc(display, mask, 0);
  XFillRectangle(display, mask, mask_gc, 0, 0, WIN_WIDTH, WIN_HEIGHT );

  XShapeCombineMask (display, win, ShapeBounding, 0, 0, mask, ShapeSet);

  XChangeProperty(display, win, window_type_atom, XA_ATOM, 32, 
		  PropModeReplace, (unsigned char *) &window_type_dock_atom, 1);

  XMapWindow(display, win);

  update(display,win,gc,rev_gc,mode,mask, mask_gc);

  XSync(display, False);
  XFlush(display);

  screenWidth = DisplayWidth(display,screen_num);
  screenHeight = DisplayHeight(display,screen_num);


  XSelectInput(display, win, ExposureMask | StructureNotifyMask |
	       SubstructureNotifyMask ); 
  {
    int done = 0;
    int isButtonPressed = 0;
    XEvent an_event;

    while (!done)
      {
	curTime = time(0);
	if ( curTime != last_time )
	  {
	    last_time = curTime;
	    update_req = 1;
	  }
	while (XPending(display)) {
	  XNextEvent(display, &an_event);
	  switch (an_event.type) {
	    case Expose:
	      update_req = 1;
	      break;
	    case DestroyNotify:	      
	      done = 1;
	      break;
	    default:
	      break;
	  } 
	}
	if (update_req)
	  {
	    update(display,win,gc,rev_gc,mode,mask, mask_gc);
	    update_req = 0;
	  } 
	if (!isButtonPressed) /* dont sleep while being dragged */
	    usleep(300000L);
      } 
    
  }


  XFreePixmap(display,mask);


  XCloseDisplay(display);
  } else {
    fprintf(stderr, "%s: cannot connect to X server '%s'\n",
            argv[0], display_name);
    exit(1);
  }

  
  return 0;
}








