/*
 *  minisys - tiny sys monitor
 *
 *  cpu reading code based on wmbubblemon
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

struct {

   /* cpu data  */
   int loadIndex;
   int samples;
   u_int64_t *load, *total;

   /* memory data  */
   u_int64_t mem_used;
   u_int64_t mem_max;
   u_int64_t swap_used;
   u_int64_t swap_max;
   unsigned int swap_percent;  /* swap used, in percent */
   unsigned int mem_percent;   /* memory used, in percent */
   
   /* X bits */
   Display* dpy;
   Window win;

} msd;   


/* returns current CPU load in percent, 0 to 100 */
int system_cpu(void)
{
    unsigned int cpuload;
    u_int64_t load, total, oload, ototal;
    u_int64_t ab, ac, ad, ae;
    int i;
    FILE *stat;

    stat = fopen("/proc/stat", "r");
    fscanf(stat, "%*s %Ld %Ld %Ld %Ld", &ab, &ac, &ad, &ae);
    fclose(stat);

    /* Find out the CPU load */
    /* user + sys = load
     * total = total */
    load = ab + ac + ad;	/* cpu.user + cpu.sys; */
    total = ab + ac + ad + ae;	/* cpu.total; */

    /* "i" is an index into a load history */
    i = msd.loadIndex;
    oload = msd.load[i];
    ototal = msd.total[i];

    msd.load[i] = load;
    msd.total[i] = total;
    msd.loadIndex = (i + 1) % msd.samples;

    if (ototal == 0)	
	cpuload = 0;
    else
	cpuload = (100 * (load - oload)) / (total - ototal);

    return cpuload;
}

int system_memory(void)
{
    u_int64_t my_mem_used, my_mem_max;
    u_int64_t my_swap_used, my_swap_max;

    static int mem_delay = 0;
    FILE *mem;
    static u_int64_t aa, ab, ac, ad, ae, af, ag, ah;
    /* put this in permanent storage instead of stack */
    static char not_needed[2048];

    if (mem_delay-- <= 0) {
	mem = fopen("/proc/meminfo", "r");
	fgets(not_needed, 2048, mem);
	
	fscanf(mem, "%*s %Ld %Ld %Ld %Ld %Ld %Ld", &aa, &ab, &ac,
	       &ad, &ae, &af);
	fscanf(mem, "%*s %Ld %Ld", &ag, &ah);
	fclose(mem);
	mem_delay = 25;

	/* calculate it */
	my_mem_max = aa;	/* memory.total; */
	my_swap_max = ag;	/* swap.total; */

	my_mem_used = ah + ab - af - ae;

	/* No swap on ipaq 
	if (my_mem_used > my_mem_max) {
	   my_swap_used = my_mem_used - my_mem_max;
	    my_mem_used = my_mem_max;
	} else {
	   my_swap_used = 0;
	}
	*/
	
	msd.mem_used = my_mem_used;
	msd.mem_max = my_mem_max;
	//msd.swap_used = my_swap_used;
	//msd.swap_max = my_swap_max;

	msd.mem_percent = (100 * msd.mem_used) / msd.mem_max;
	//msd.swap_percent = (100 * msd.swap_used) / msd.swap_max;
	/* memory info changed - update things */
	return 1;
	}
    /* nothing new */
    return 0;
}

void usage(void)
{
   printf("Usage: minisys [options...]\n");
   printf("Options:\n");
   printf("  -display  <display>\n");
   printf("  -geometry <geometry>\n");
   printf("  -m <color>  set memory bar line color\n");
   printf("  -c <color>  set cpu bar line color\n");
   printf("  -t  set window type to toolbar\n");
   printf("  -n  set window type to normal\n");
   printf("  -h  this help\n\n");
   exit(1);
}

#define WIN_DOCK    1
#define WIN_TOOLBAR 2
#define WIN_NORMAL  3


int main(int argc, char *argv[])
{
   int i;
   u_int64_t load = 0, total = 0;

   int winw=32, winh=16, winx=0, winy=0, cpusize, memsize;
   int scr;
   Atom type_atom;
   Atom type_dock_atom;
   Atom type_toolbar_atom;
   GC bg_gc, mem_gc, cpu_gc; 
   XGCValues gv;
   XColor bg_col, mem_col, cpu_col, exact_col;
   XWMHints *wm_hints;
   Pixmap backing;
   XEvent an_event;

   char *display_name = (char *)getenv("DISPLAY");
   char *geometry = NULL;
   int wintype = WIN_DOCK;

   char *memcol_str = "blue";
   char *cpucol_str = "green";
   
   
   /* init load data */
   msd.samples = 16;
   
   if (msd.load) {
      load = msd.load[msd.loadIndex];
      free(msd.load);
   }
   
   if (msd.total) {
      total = msd.total[msd.loadIndex];
      free(msd.total);
   }
   
   msd.loadIndex = 0;
   msd.load = malloc(msd.samples * sizeof(u_int64_t));
   msd.total = malloc(msd.samples * sizeof(u_int64_t));
   for (i = 0; i < msd.samples; i++) {
      msd.load[i] = load;
      msd.total[i] = total;
   }

   for (i=1; argv[i]; i++) {
      char *arg = argv[i];
      if (*arg=='-') {
	 switch (arg[1]) {
	    case 'd' :
	       display_name = argv[i+1];
	       i++;
	       break;
	    case 'g' :
	       geometry = argv[i+1];
	       i++;
	       break; 
	    case 't':
	       wintype = WIN_TOOLBAR;
	       winh = 32;
	       break;
	    case 'n':
	       winw = 320; winh = 100;
	       wintype = WIN_NORMAL;
	       break;
	    case 'm' :
	       i++;
	       if (argv[i] != NULL) memcol_str = argv[i];
	       break;
	    case 'c':
	       i++;
	       if (argv[i] != NULL) cpucol_str = argv[i];
	       break;
	    default:
	       usage();
	       exit(0);
	       break;
	 }
      }	
   }

   if (geometry != NULL)
      XParseGeometry(geometry, &winx, &winy, &winw, &winh );


   /* init X stuff */

   if ((msd.dpy = XOpenDisplay(display_name)) == NULL)
   {
      fprintf(stderr, "%s: failed to open display", argv[0]);
   }
   scr = DefaultScreen(msd.dpy);

   XAllocNamedColor(msd.dpy, DefaultColormap(msd.dpy, scr),
		    "black", &bg_col, &exact_col);
   XAllocNamedColor(msd.dpy, DefaultColormap(msd.dpy, scr),
		    memcol_str, &mem_col, &exact_col);
   XAllocNamedColor(msd.dpy, DefaultColormap(msd.dpy, scr),
		    cpucol_str,  &cpu_col, &exact_col);
   
   gv.graphics_exposures = False;
   gv.function   = GXcopy;
   gv.foreground = bg_col.pixel;

   bg_gc = XCreateGC(msd.dpy, RootWindow(msd.dpy, scr),
		     GCGraphicsExposures|GCFunction|GCForeground, &gv);

   gv.foreground = mem_col.pixel;
   mem_gc = XCreateGC(msd.dpy, RootWindow(msd.dpy, scr),
		     GCGraphicsExposures|GCFunction|GCForeground, &gv);

   gv.foreground = cpu_col.pixel;
   gv.function   = GXxor;
   cpu_gc = XCreateGC(msd.dpy, RootWindow(msd.dpy, scr),
		     GCGraphicsExposures|GCFunction|GCForeground, &gv);

   backing = XCreatePixmap(msd.dpy, RootWindow(msd.dpy, scr), winw, winh,
			   DefaultDepth(msd.dpy,scr));

   msd.win = XCreateSimpleWindow(
				 msd.dpy, RootWindow(msd.dpy, scr),
				 0, 0, winw, winh, 0,
				 WhitePixel(msd.dpy, scr),
				 BlackPixel(msd.dpy, scr)
				 );
   
  type_atom = XInternAtom(msd.dpy, "_NET_WM_WINDOW_TYPE", False);
  type_dock_atom = XInternAtom(msd.dpy, "_NET_WM_WINDOW_TYPE_DOCK",False);
  type_toolbar_atom =XInternAtom(msd.dpy, "_NET_WM_WINDOW_TYPE_TOOLBAR",False);

  switch (wintype)
  {
     case WIN_DOCK:
	XChangeProperty(msd.dpy, msd.win, type_atom, XA_ATOM, 32, 
			PropModeReplace,
			(unsigned char *) &type_dock_atom, 1);
	break;
     case WIN_TOOLBAR:

	XChangeProperty(msd.dpy, msd.win, type_atom, XA_ATOM, 32, 
			PropModeReplace,
			(unsigned char *) &type_toolbar_atom, 1);
	wm_hints = XAllocWMHints();
	wm_hints->input = False;
	wm_hints->flags = InputHint;
	XSetWMHints(msd.dpy, msd.win, wm_hints );

	break;
     default:
	break;
  }
  
  XStoreName(msd.dpy, msd.win, "minisys");
  
  XMapWindow(msd.dpy, msd.win);
  
  XFillRectangle(msd.dpy, msd.win, bg_gc, 0, 0, winw, winh);
  XFillRectangle(msd.dpy, backing, bg_gc, 0, 0, winw, winh);
  
  XSelectInput(msd.dpy, msd.win, ExposureMask | StructureNotifyMask);

  while(1)
  {
      
     cpusize = (int)(( system_cpu() / 100.00 ) * winh);
     memsize = (int)(( msd.mem_percent / 100.00 ) * winh);
     
     XCopyArea(msd.dpy, backing, backing, bg_gc, 1, 0, winw-1, winh, 0, 0);
     XDrawLine(msd.dpy, backing, bg_gc, winw-1, winh, winw-1, 0);
     XDrawLine(msd.dpy, backing, mem_gc, winw-1, winh, winw-1, winh-memsize);
     XDrawLine(msd.dpy, backing, cpu_gc, winw-1, winh, winw-1, winh-cpusize);
     
     XCopyArea(msd.dpy, backing, msd.win, bg_gc, 0, 0, winw, winh, 0, 0);
     
     while ( XPending(msd.dpy) ) 
     {
	XNextEvent(msd.dpy, &an_event);
	if (an_event.type == ConfigureNotify)
	{
	   if ( an_event.xconfigure.width != winw
		|| an_event.xconfigure.height != winh)
	   {
	      winw = an_event.xconfigure.width;
	      winh = an_event.xconfigure.height;
	      XFreePixmap(msd.dpy, backing);
	      backing = XCreatePixmap(msd.dpy, RootWindow(msd.dpy, scr),
				      winw, winh,
				      DefaultDepth(msd.dpy,scr));
	      XFillRectangle(msd.dpy, msd.win, bg_gc, 0, 0, winw, winh);
	      XFillRectangle(msd.dpy, backing, bg_gc, 0, 0, winw, winh);
	   }
	}
     }
     system_memory();
     usleep(50000);
  }
}
