""" The repository for all loaded plugin definitions. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.traits.api import Dict, HasTraits, Instance, List, Property, Str

# Local imports.
from plugin_definition import PluginDefinition


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class PluginDefinitionRegistry(HasTraits):
    """ The repository for all loaded plugin definitions. """

    #### 'PluginDefinitionRegistry' interface #################################

    # The application that the registry is part of.
    application = Instance('enthought.envisage.core.application.Application')

    # All of the plugin definitions in the registry.
    definitions = Property(List(PluginDefinition))

    #### Private interface ####################################################

    # All of the plugin definitions in the registry.
    _definitions = Dict(Str, Instance(PluginDefinition))

    ###########################################################################
    # 'PluginDefinitionRegistry' interface.
    ###########################################################################

    #### Properties ###########################################################

    def _get_definitions(self):
        """ Returns all the plugin definitions in the registry (as a list). """

        return self._definitions.values()

    #### Methods ##############################################################

    def add_definition(self, definition):
        """ Adds a plugin definition to the registry. """

        # Make sure that we don't import the same definition more than once.
        if not self._definitions.has_key(definition.id):
            self._definitions[definition.id] = definition
            if definition.enabled:
                # Add all extension points and extensions found in the plugin
                # definition to the extension registry.
                self._register_extension_points(definition)
                self._register_extensions(definition)

        else:
            logger.warn('duplicate plugin %s' % definition.id)

        return

    def get_definition(self, id):
        """ Returns the plugin definition with the specified *id*.

        Returns '''None''' if no such plugin definition exists.

        """

        return self._definitions.get(id)

    def has_definition(self, id):
        """ Does the registry contain a definition with the specified *id*? """

        return self._definitions.has_key(id)

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _register_extension_points(self, definition):
        """ Adds a plugin definition's extension points to the extension
        registry.
        """

        extension_registry = self.application.extension_registry

        for extension_point in definition.extension_points:
            extension_registry.add_extension_point(extension_point)

        return

    def _register_extensions(self, definition):
        """ Adds a plugin definition's extensions to the extension registry. """

        extension_registry = self.application.extension_registry

        for extension in definition.extensions:
            # Tag the extension with the plugin definition that contributed it.
            extension._definition_ = definition

            # Add the extension.
            extension_registry.add_extension(extension)

        return

#### EOF ######################################################################
