""" The base class for all views. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.debug.api import log_called_from
from enthought.envisage.api import Application
from enthought.traits.api import Any, Bool, Enum, Float, HasTraits, Instance
from enthought.traits.api import List, Property, Str


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class View(HasTraits):
    """ The base class for all views. """

    #### 'View' interface #####################################################

    # The application that the view is part of. This is a convenience
    # property and is equivalent to 'self.window.application'.
    application = Property(Instance(Application))

    # Is the view busy? (i.e., should the busy cursor (often an hourglass) be
    # displayed?).
    busy = Bool(False)

    # The toolkit-specific control that represents the view.
    #
    # The framework sets this to the value returned by 'create_control'.
    control = Any

    # Does the view currently have the focus?
    has_focus = Bool(False)

    # The view's globally unique identifier.
    id = Str

    # The view's name.
    name = Str

    # The current selection within the view.
    selection = List

    # The workbench window that the view is in.
    #
    # The framework sets this when the view is added to a window.
    window = Instance('enthought.envisage.workbench.Window')

    # Whether the view is visible or not.
    visible = Bool(False)

    # The following traits specify the *default* position of the view. This
    # information is only used if the view is added to a perspective that
    # it is not explicitly part of (i.e. it does not appear in the
    # perspective's 'contents'.
    #
    # This often happens because:-
    #
    # a) The application doesn't define any perspectives
    # b) The user can add/remove any view to/from any perspective
    #
    # fixme: These traits are idential to those in 'PerspectiveItem'. How can
    # we unify them?

    # The default position of the view relative to the item specified in the
    # 'relative_to' trait.
    #
    # 'top'    puts the view above the 'relative_to' item.
    # 'bottom' puts the view below the 'relative_to' item.
    # 'left'   puts the view to the left of  the 'relative_to' item.
    # 'right'  puts the view to the right of the 'relative_to' item.
    # 'with'   puts the view in the same region as the 'relative_to' item.
    #
    # If the position is specified as 'with' you must specify a 'relative_to'
    # item other than the editor area (i.e., you cannot position a view 'with'
    # the editor area).
    position = Enum('left', 'top', 'bottom', 'right', 'with')

    # The Id of the view to position this view relative to. If this is not
    # specified (or if no view exists with this Id) then the view will be
    # placed relative to the editor area.
    relative_to = Str

    # The default width of the view (as a fraction of the window width).
    #
    # e.g. 0.5 == half the window width.
    #
    # Note that this is treated as a suggestion, and it may not be possible for
    # the workbench to allocate space requested.
    width = Float(-1)

    # The default height of the view (as a fraction of the window height).
    #
    # e.g. 0.5 == half the window height.
    #
    # Note that this is treated as a suggestion, and it may not be possible for
    # the workbench to allocate space requested.
    height = Float(-1)

    ###########################################################################
    # 'View' interface.
    ###########################################################################

    #### Properties ###########################################################

    def _get_application(self):
        """ Property getter. """

        return self.window.application

    #### Methods ##############################################################

    def create_control(self, parent):
        """ Creates the toolkit-specific control that represents the view.

        'parent' is the toolkit-specific control that is the view's parent.

        View implementors should override this!

        """

        # fixme: We only call the 'protected' method '_create_contents' here
        # to make views backward compatible with the views in the old UI
        # plugin! View implementors should really override *this* method.
        logger.warn('VIEW [%s] FAILED TO OVERRIDE "create_control"', self)
        control = self._create_contents(parent)

        return control

    def destroy_control(self):
        """ Destroys the toolkit-specific control that represents the view.

        """

        if self.control is not None:
            logger.debug('destroying control for view [%s]', self)
            self.control.Destroy()

        else:
            logger.debug('no control to destroy for view [%s]', self)

        return

    def set_focus(self):
        """ Sets the focus to the appropriate control in the view.

        By default we set the focus to be the view's top-level control. If
        you need to give focus to some child control then override.

        """

        if self.control is not None:
            self.control.SetFocus()

        return

    # Convenience methods for common operations on the service registry.
    def get_service(self, interface, query=None):
        """ Returns a service that implements the specified interface.

        Raises a **SystemError** if no such service is found.

        """

        logger.warn('DEPRECATED: Use self.application.get_service')
        log_called_from()

        return self.application.get_service(interface, query)

    def get_services(self, interface, query=None):
        """ Returns all services that match the specified query.

        If no services match the query, then an empty list is returned.

        """

        logger.warn('DEPRECATED: Use self.application.get_services')
        log_called_from()

        return self.application.get_services(interface, query)

    def register_service(self, interface, obj, properties=None):
        """ Registers a service that implements the specified interface.

        Returns a service ID (a unique ID for the service within the
        application).

        """

        logger.warn('DEPRECATED: Use self.application.register_service')
        log_called_from()

        return self.application.register_service(interface, obj, properties)

    def unregister_service(self, service_id):
        """ Unregisters a service. """

        logger.warn('DEPRECATED: Use self.application.unregister_service')
        log_called_from()

        self.application.unregister_service(service_id)

        return

    ###########################################################################
    # DEPRECATED Protected 'View' interface.
    ###########################################################################

    def _create_contents(self, parent):
        """ Create the toolkit-specific control that represents the view.

        'parent' is the toolkit-specific control that is the view's parent.

        DEPRECATED: Override 'create_control' instead.

        """

        logger.warn('VIEW [%s] FAILED TO OVERRIDE "_create_contents"', self)

        # By default we create a red panel!
        import wx

        panel = wx.Panel(parent, -1)
        panel.SetBackgroundColour("red")
        panel.SetSize((100, 200))

        return panel

    ###########################################################################
    # Private interface.
    ###########################################################################

    #### Trait change handlers ################################################

    def _busy_changed(self):
        """ Static trait change handler. """

        self.window.application.gui.busy = self.busy

        return

#### EOF ######################################################################
