/* sel-graphs.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "sel-graphs.h"
#include <libgnomedb/libgnomedb.h>
#include "graph/mg-graph.h"

/*
 *
 * Module for all graphs managed by MgConf
 *
 */
static void         module_graphs_fill_model (Module *module);
static void         module_graphs_free (Module *module);
static const gchar *module_graphs_col_name (Module *module, guint colno);
static void         module_graphs_model_store_data (Module *module, GtkTreeIter *iter);
static const gchar *module_graphs_render_graph_type (MgGraph *graph);

Module *
sel_module_graphs_new (MgSelector *mgsel, gboolean insert_header, 
		       GtkTreeIter *iter, gpointer data)
{
	Module *module;

	module = g_new0 (Module, 1);
	module->selector = mgsel;
	module->fill_model = module_graphs_fill_model;
	module->free = module_graphs_free;
	module->col_name = module_graphs_col_name;
	module->obj_manager = NULL;
	module->model_store_data = module_graphs_model_store_data;
	module->mod_data = NULL;
	module->iter = NULL;
	module->parent_module = NULL;
	module->sub_modules = NULL;

	if (insert_header) {
		GdkPixbuf *pixbuf = NULL;
		GtkTreeModel *model = mgsel->priv->model;

		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_RELATIONS);
		module->iter = g_new0 (GtkTreeIter, 1);
		gtk_tree_store_append (GTK_TREE_STORE (model), module->iter, iter);
		gtk_tree_store_set (GTK_TREE_STORE (model), module->iter, NAME_COLUMN, 
				    _("Graphs"), PIXBUF_COLUMN, pixbuf, 
				    CONTENTS_COLUMN, CONTENTS_TOP_CATEGORY, 
				    SUB_MODULE_COLUMN, NULL, -1);		
	}
	else {
		if (iter)
			module->iter = gtk_tree_iter_copy (iter);
	}

	return module;
}


static GSList *module_graphs_get_objects_list (Module *module);
static void
module_graphs_fill_model (Module *module)
{
	GtkTreeModel *model;
	GdkPixbuf *pixbuf_graph = NULL;

	pixbuf_graph = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-relations_16x16.png");

	/* Module's private data */
	module->mod_data = g_new0 (ModFlatData, 1);
	FLAT_DATA (module)->manager = NULL;
	FLAT_DATA (module)->manager_weak_refed = FALSE;
	FLAT_DATA (module)->fallback_obj_pixbuf = pixbuf_graph;
	FLAT_DATA (module)->get_objects_list = module_graphs_get_objects_list;

	/* Initial model filling */
	model = module->selector->priv->model;
	flat_init_model_fill (module, model);

	/* Signals handlers */
	g_signal_connect (G_OBJECT (module->selector->priv->conf), "graph_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (G_OBJECT (module->selector->priv->conf), "graph_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (G_OBJECT (module->selector->priv->conf), "graph_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);
}

static GSList *
module_graphs_get_objects_list (Module *module)
{
	return mg_conf_get_graphs (module->selector->priv->conf);
}

static void
module_graphs_free (Module *module)
{
	GSList *list = module->sub_modules;

	/* Free the sub modules */
	while (list) {
		(MODULE (list->data)->free) (MODULE (list->data));
		g_free (list->data);
		list = g_slist_next (list);
	}
	if (module->sub_modules) {
		g_slist_free (module->sub_modules);
		module->sub_modules = NULL;
	}

	/* free this module */
	g_signal_handlers_disconnect_by_func (G_OBJECT (module->selector->priv->conf),
					      G_CALLBACK (flat_obj_added_cb), module);
	g_signal_handlers_disconnect_by_func (G_OBJECT (module->selector->priv->conf),
					      G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_handlers_disconnect_by_func (G_OBJECT (module->selector->priv->conf),
					      G_CALLBACK (flat_obj_updated_cb), module);

	if (module->iter)
		gtk_tree_iter_free (module->iter);

	flat_free_mod_data (module);
	g_free (module->mod_data);
	module->mod_data = NULL;
}


static const gchar *
module_graphs_col_name (Module *module, guint colno)
{
	switch (colno) {
	case 0:
		return _("Graph");
		break;
	case EXTRA1_COLUMN:
		return _("Type");
		break;
	default:
		return NULL;
		break;
	}
}

static void
module_graphs_model_store_data (Module *module, GtkTreeIter *iter)
{
	GObject *obj;
	GtkTreeModel *model;

	model = module->selector->priv->model;
	gtk_tree_model_get (model, iter, OBJ_COLUMN, &obj, -1);

	if (obj && IS_MG_GRAPH (obj)) {
		const gchar *str1;
		gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    EXTRA1_COLUMN, module_graphs_render_graph_type (MG_GRAPH (obj)),
				    -1);
		str1 = mg_base_get_name (MG_BASE (obj));
		if (!str1 || !(*str1)) {
			gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    NAME_COLUMN, _("Graph <no name>"),
				    -1);
		}
	}
}

static const gchar *
module_graphs_render_graph_type (MgGraph *graph)
{
	switch (mg_graph_get_graph_type (graph)) {
	case MG_GRAPH_DB_RELATIONS:
		return _("Database relations");
	case MG_GRAPH_QUERY_JOINS:
		return _("Query joins");
	case MG_GRAPH_MODELLING:
		return _("Model");
	default:
		g_assert_not_reached ();
	}

	return NULL;
}

