/***************************** LICENSE START ***********************************

 Copyright 2014 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvQFolderWatcher.h"

#include <QDebug>
#include <QTimer>

#include "Folder.h"
#include "FolderPresenter.h"
 
QList<MvQFolderWatcher*> MvQFolderWatcher::items_;	
QMap<Folder*,MvQTimeoutInfo> MvQFolderWatcher::timeoutLog_;

//This is the timeou period between two rescans
const int MvQFolderWatcher::defaultTimeout_=8000;

MvQFolderWatcher::MvQFolderWatcher(FolderPresenter *fp) :
 presenter_(fp),
 timeout_(defaultTimeout_)
{
  	timer_= new QTimer(this);
	timer_->setInterval(timeout_);
	
	connect(timer_,SIGNAL(timeout()),
		this,SLOT(slotTimeout()));
		
	//Scan immediately
	slotTimeout();
	
	//Start timeout
	timer_->start();
	
	items_ << this;
}

Folder* MvQFolderWatcher::folder() 
{
	return presenter_->currentFolder();
}	

bool MvQFolderWatcher::doTimeout(bool forced)
{
  	Folder *f=folder();
  	if(f && (forced || isTimeoutDue()))
	{
	  	//qDebug() << "rescan -->" << QString::fromStdString(f->fullName());
		//qDebug() << "    elapsed:" << timeoutLog_[f].last.elapsed() << "ms";
	  	timeoutLog_[f].last.start();
		f->scan();
		timeoutLog_[f].lenght=timeoutLog_[f].last.elapsed();
		//qDebug() << "    lenght:" << timeoutLog_[f].lenght << "ms";
		timeoutLog_[f].last.start(); 
		
		return true;
	}	
	
	return false;
}  
//Check if the timeout action should really be performed. If there is not long enough time elapsed 
//since the last timeout it will return false
bool MvQFolderWatcher::isTimeoutDue()
{
  	QMap<Folder*,MvQTimeoutInfo>::iterator it=timeoutLog_.find(folder());
	
	//If the folder is already in the log
	if(it != timeoutLog_.end())
	{  
		//if more than the 80% of the timeout period passed!!
	  	return it.value().last.elapsed() > +8*timeout_/10;
	}
	//If the folder is not in the log (no timeout has happened so for it) we
	//add it to the log
	else
	{	
		timeoutLog_[folder()].last.start();  	
	}
	
	return true;
}

void MvQFolderWatcher::cleanLog()
{
	QList<Folder*> lst;
  	foreach(MvQFolderWatcher* item,items_)
	{  	
		lst << item->folder();
	}
		
	QList<Folder*> keys=timeoutLog_.keys();
	foreach(Folder* f,keys)
	{
		if(!lst.contains(f))  
			timeoutLog_.remove(f);
	}
}
	  
void MvQFolderWatcher::slotTimeout()
{
  	//Try to perform the timeout action. If it was succesfull we
	//restart the timer
	if(doTimeout(false))
		timer_->start();  
}  

void MvQFolderWatcher::reload()
{
	timer_->stop();
	doTimeout(true);	
	timer_->start();
	
	//print();
	
}  

MvQFolderWatcher* MvQFolderWatcher::add(FolderPresenter* fp)
{
	MvQFolderWatcher *item=find(fp);
	if(!item)
	{
	  	cleanLog();	
		item=new MvQFolderWatcher(fp);
	}
	
	return item;
}	
	
void MvQFolderWatcher::remove(FolderPresenter* fp)
{
  	if(MvQFolderWatcher *item=find(fp))
	{  
		items_.removeAll(item);
		delete item;
		cleanLog();
	}
}	

void MvQFolderWatcher::reload(FolderPresenter* fp)
{
	MvQFolderWatcher *item=find(fp);
	if(item)
	{  
	  	item->reload();
	}
	//safety measure
	else
	{
		item=MvQFolderWatcher::add(fp);
		item->reload();
	}	
}

MvQFolderWatcher* MvQFolderWatcher::find(FolderPresenter* fp)
{
 	foreach(MvQFolderWatcher* item,items_)
	{
	  	if(item->folderPresenter() == fp) 
		  	return item;
	}
	return NULL;
}  

void MvQFolderWatcher::print()
{
  	qDebug() << "Watched folders: \n----------------";
  	
  	foreach(MvQFolderWatcher* item,items_)
	{	
		qDebug() << QString::fromStdString(item->folder()->fullName()) << item->timeout()/1000 << "s";
	}	
}		
		