;;; mew-pop.el

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Jun 28, 2000

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; POP info
;;;

(defvar mew-pop-info-list
  '("case"
    "user" "server" "port"
    "process" "ssh-process"
    "directive" "flush" "skip-list" "save-uidl"
    "status" "cnt" "left" "input" "keep-input" "diag"
    "auth" "key" "passwd" "try"
    "rtrs" "dels" "refs" "rmvs" "uidl"
    "size" "body-lines" "truncated" "delete"
    "fld" "msgdb"))

(mew-info-defun "mew-pop-" mew-pop-info-list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Process name
;;;

(defconst mew-pop-info-prefix "mew-pop-info-")

(defsubst mew-pop-info-name (case)
  (let ((server (mew-pop-server case))
	(port (mew-pop-port case))
	(user (mew-pop-user case))
	(sshsrv (mew-pop-ssh-server case))
	(name mew-pop-info-prefix))
    (setq name (concat name user "@" server))
    (unless (string= port mew-pop-port)
      (setq name (concat name ":" port)))
    (if sshsrv
	(concat name "%" sshsrv)
      name)))

(defsubst mew-pop-buffer-name (pnm)
  (concat mew-buffer-prefix pnm))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; FSM
;;;

(defvar mew-pop-fsm
  '(("grtg"         nil ("+OK" . "capa"))
    ("capa"         t   ("+OK" . "auth") ("-ERR" . "pswd"))
    ("cram-md5"     nil ("+OK" . "pwd-cram-md5") ("-ERR" . "wpwd"))
    ("pwd-cram-md5" nil ("+OK" . "list") ("-ERR" . "wpwd"))
    ("apop"         nil ("+OK" . "list") ("-ERR" . "wpwd"))
    ("user"         nil ("+OK" . "pass") ("-ERR" . "wpwd2"))
    ("pass"         nil ("+OK" . "list") ("-ERR" . "wpwd"))
    ("list"         t   ("+OK" . "uidl"))
    ("uidl"         t   ("+OK" . "umsg") ("-ERR" . "pre-retr"))
    ("dels"	    nil ("+OK" . "dels"))
    ("retr"         t   ("+OK" . "dele"))
    ("dele"         nil ("+OK" . "retr"))
    ("quit"         nil ("+OK" . "noop"))))

(defsubst mew-pop-fsm-by-status (status)
  (assoc status mew-pop-fsm))

(defsubst mew-pop-fsm-next (status code)
  (cdr (mew-assoc-match code (nthcdr 2 (mew-pop-fsm-by-status status)) 0)))

(defsubst mew-pop-fsm-reply (status)
  (nth 1 (mew-pop-fsm-by-status status)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Commands
;;;

(defsubst mew-pop-message (pnm &rest args)
  (if (mew-pop-get-diag pnm) (apply (function message) args)))

(defun mew-pop-retrieve (case directive &rest args)
  ;; in +inbox
  (let* ((server (mew-pop-server case))
	 (port (mew-pop-port case))
	 (sshsrv (mew-pop-ssh-server case))
	 (pnm (mew-pop-info-name case))
	 (ret t)
	 (buf (get-buffer-create (mew-pop-buffer-name pnm)))
	 (diag (not (eq directive 'biff)))
	 process sshpro sshname lport)
    (if (mew-pop-get-process pnm)
	(message "Another POP process is running. Try later")
      (if (null sshsrv)
	  (setq process (mew-pop-open pnm server port diag))
	(setq sshpro (mew-open-ssh-stream server port sshsrv))
	(when sshpro
	  (setq sshname (process-name sshpro))
	  (setq lport (mew-ssh-pnm-to-lport sshname))
	  (if lport (setq process (mew-pop-open pnm "localhost" lport diag)))))
      (cond
       (process
	(mew-pop-message pnm "Communicating with the POP server...")
	(mew-summary-lock process "POPing")
	(mew-info-clean-up pnm)
	(mew-pop-set-process pnm process)
	(mew-pop-set-port pnm port)
	(mew-pop-set-ssh-process pnm sshpro)
	(mew-pop-set-diag pnm diag)
	(mew-pop-set-status pnm "grtg")
	(mew-pop-set-case pnm case)
	(mew-pop-set-directive pnm directive)
	(mew-pop-set-cnt pnm 0)
	(mew-pop-set-try pnm 0)
	(mew-pop-set-keep-input pnm t)
	(mew-pop-set-server pnm server)
	(mew-pop-set-auth pnm (mew-pop-auth case))
	(mew-pop-set-user pnm (mew-pop-user case))
	(mew-pop-set-fld pnm (mew-inbox-folder case))
	(mew-pop-set-size pnm (mew-pop-size case))
	(mew-pop-set-body-lines pnm (mew-pop-body-lines case))
	(cond
	 ((eq directive 'biff)
	  (mew-pop-set-passwd pnm (nth 0 args))) ;; password
	 ((eq directive 'inc)
	  (mew-pop-set-flush pnm (nth 0 args)) ;; no-flush
	  (mew-pop-set-save-uidl pnm t)
	  (mew-pop-set-delete pnm (mew-pop-delete case)))
	 ((eq directive 'get)
	  (mew-pop-set-fld pnm (nth 0 args))
	  (mew-pop-set-refs pnm (nth 1 args))
	  (mew-pop-set-skip-list pnm t)
	  (mew-pop-set-delete pnm (mew-pop-delete case)))
	 ((eq directive 'scan)
	  (mew-pop-set-fld pnm (nth 0 args))
	  (mew-sinfo-set-mdrop-case case))
	 ((eq directive 'mdrop)
	  (mew-pop-set-fld pnm (nth 0 args))
	  (mew-pop-set-refs pnm (nth 1 args))
	  (mew-pop-set-rmvs pnm (nth 2 args))
	  (mew-pop-set-skip-list pnm t)))
	;;
	(set-process-sentinel process 'mew-pop-sentinel)
	(set-process-filter process 'mew-pop-filter)
	(set-process-buffer process buf)
	(mew-sinfo-set-scan-id nil)))
      ret)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Biff
;;;

(defvar mew-biff-string nil)
(defvar mew-pop-biff-timer-id nil)

(defun mew-pop-biff ()
  (let* ((case mew-case-input)
	 (inbox (mew-inbox-folder case))
	 (tag (mew-pop-passtag2 case))
	 passwd)
    (save-excursion
      (set-buffer inbox)
      (when (and (mew-summary-exclusive-p)
		 (and mew-use-cached-passwd
		      (setq passwd (mew-passwd-get-passwd tag))))
	(mew-sinfo-set-scan-form (mew-summary-scan-form inbox))
	(mew-pop-retrieve case 'biff passwd)))))

(defun mew-pop-check ()
  (interactive)
  (let* ((case mew-case-input)
	 (inbox (mew-inbox-folder case)))
    (save-excursion
      (set-buffer inbox)
      (when (mew-summary-exclusive-p)
	(mew-sinfo-set-scan-form (mew-summary-scan-form inbox))
	(mew-pop-retrieve case 'biff)))))

(defun mew-pop-biff-setup ()
  (if (not mew-use-biff)
      (mew-pop-biff-clean-up)
    (if mew-pop-biff-timer-id (cancel-timer mew-pop-biff-timer-id))
    (setq mew-pop-biff-timer-id
	  (mew-timer (* 60 mew-pop-biff-interval) (function mew-pop-biff))))
  (let ((ent '(mew-biff-string mew-biff-string)))
    (unless (member ent global-mode-string)
      (if global-mode-string
	  (setq global-mode-string
		(append global-mode-string (list " " ent)))
	(setq global-mode-string (list ent))))))

(defun mew-pop-biff-clean-up ()
  (if mew-pop-biff-timer-id (cancel-timer mew-pop-biff-timer-id))
  (setq mew-pop-biff-timer-id nil))

(defun mew-pop-biff-bark (n)
  (if (= n 0)
      (setq mew-biff-string nil)
    (if (and mew-use-biff-bell (eq mew-biff-string nil))
	(beep))
    (setq mew-biff-string (format "Mail(%d)" n))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Opening POP
;;;

(defun mew-pop-open (pnm server port diag)
  (let ((sprt (mew-port-sanity-check port))
	pro tm)
    (condition-case emsg
	(progn
	  (setq tm (mew-timer mew-pop-timeout-time 'mew-pop-timeout))
	  (if diag (message "Connecting to the POP server..."))
	  (setq pro (open-network-stream pnm nil server sprt))
	  (process-kill-without-query pro)
	  (mew-set-process-cs pro mew-cs-text-for-net mew-cs-text-for-net)
	  (if diag (message "Connecting to the POP server...done")))
      (quit
       (if diag (message "Can't connect to the POP server due to time out"))
       (setq pro nil))
      (error
       (if diag (message "%s, %s" (nth 1 emsg) (nth 2 emsg)))
       (setq pro nil)))
    (if tm (cancel-timer tm))
    pro))

(defun mew-pop-timeout ()
  (signal 'quit nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Closing POP
;;;

(defun mew-pop-close (pnm)
  (interactive)
  (let* ((process (get-process pnm))
	 (buf (process-buffer process)))
    (when (and (processp process) (memq (process-status process) '(open)))
      (set-process-buffer process nil)
      (mew-remove-buffer buf)
      (set-process-filter process nil)
      (process-send-string process (format "QUIT%s" mew-cs-eol))))
  (mew-pop-tear-down pnm))

(defun mew-pop-tear-down (pnm)
  (let ((process (get-process pnm))
	(sshpro (mew-pop-get-ssh-process pnm))
	(fld (mew-pop-get-fld pnm)))
    ;; (mew-pop-set-process pnm nil)
    (mew-info-clean-up pnm)
    (if (processp process) (delete-process process))
    (if (processp sshpro) (delete-process sshpro))
    (when (and fld (get-buffer fld))
      (save-excursion
	(set-buffer fld)
	(mew-summary-unlock)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Filter and sentinel
;;;

(defun mew-pop-debug (label string)
  (when (mew-debug 'net)
    (save-excursion
      (set-buffer (get-buffer-create mew-buffer-debug))
      (goto-char (point-max))
      (insert (format "\n<%s>\n%s\n" label string)))))

(defun mew-pop-filter (process string)
  (let* ((pnm (process-name process))
	 (status (mew-pop-get-status pnm))
	 (cnt (mew-pop-get-cnt pnm))
	 (msgs (mew-pop-get-rtrs pnm))
	 (msg (car msgs))
	 (siz (nth 2 msg))
	 (fld (mew-pop-get-fld pnm))
	 (mulrep (mew-pop-fsm-reply status))
	 stay next func progress total)
    (mew-pop-debug (upcase status) string)
    (mew-filter
     ;; This code may create a long string. So, take care.
     (if (mew-pop-get-keep-input pnm)
	 (mew-pop-set-input pnm (concat (mew-pop-get-input pnm) string)))
     (cond
      ((string= status "grtg")
       (mew-summary-set-process-status fld "Auth'ing"))
      ((string= status "list")
       (mew-summary-set-process-status fld "Checking"))
      ((string= status "dels")
       (mew-summary-set-process-status fld "Deleting"))
      ((string= status "retr")
       (setq total (+ (length msgs) cnt))
       (if (>= cnt total)
 	   (setq progress nil)
  	 (setq siz (string-to-int siz))
	 (if (= siz 0) (setq siz 1)) ;; xxx
 	 (setq progress (format "%d/%d:%02d%%" (1+ cnt) total
  				(if (< 10000 siz)
  				    (/ (buffer-size) (/ siz 100))
  				  (/ (* (buffer-size) 100) siz)))))
       (mew-summary-set-process-status fld "Retrieving" progress)))
     ;; xxx
     ;; Process's buffer
     (goto-char (point-max))
     (mew-set-buffer-multibyte nil)
     (insert string)
     (cond
      ((and (and (goto-char (point-min)) (looking-at "-ERR"))
	    (and (goto-char (1- (point-max))) (looking-at "\n$")))
       (setq next (mew-pop-fsm-next status "-ERR")))
      ((and (and (goto-char (point-min)) (looking-at "\\+")) ;; +OK
	    (or (and mulrep
		     (goto-char (point-max))
		     (forward-line -1)
		     (looking-at "^\\.\r?$"))
		(and (not mulrep)
		     (goto-char (1- (point-max)))
		     (looking-at "\n$"))))
       (setq next (mew-pop-fsm-next status "+OK")))
      (t
       (setq stay t)))
     (unless stay
       (unless next (setq next "quit"))
       (mew-pop-set-status pnm next)
       (setq func (intern-soft (concat "mew-pop-command-" next)))
       (goto-char (point-min))
       (if (fboundp func)
	   (and func (funcall func process pnm))
	 (error "No function called %s" (symbol-name func)))
       (mew-pop-set-input pnm nil)
       (mew-erase-buffer)))))

(defun mew-pop-message2 (pnm msg left)
  (let (msg2)
    (cond
     ((or (null left) (= left 0))
      )
     ((= left 1)
      (setq msg2 " (1 message left)"))
     (t
      (setq msg2 (format " (%d messages left)" left))))
    (if msg2 (setq msg (concat msg msg2)))
    (mew-pop-message pnm msg)))

(defun mew-pop-sentinel (process event)
  (let* ((pnm (process-name process))
	 (cnt (mew-pop-get-cnt pnm))
	 (left (mew-pop-get-left pnm))
	 (fld (mew-pop-get-fld pnm))
	 (directive (mew-pop-get-directive pnm))
	 (flush (mew-pop-get-flush pnm))
	 (n (length (mew-pop-get-rtrs pnm)))
	 savep)
    (mew-pop-debug "POP SENTINEL" event)
    (mew-filter
     (if (eq directive 'biff)
	 (funcall mew-pop-biff-function n)
       (setq mew-biff-string nil)) ;; received
     (cond
      ((eq cnt nil)
       ())
      ((= cnt 0)
       (if (eq directive 'mdrop)
	   (mew-pop-message2 pnm "Deleting...done" left)
	 (mew-pop-message2 pnm "No new messages" left)))
      ((= cnt 1)
       (mew-pop-message2 pnm "1 message retrieved" left)
       (setq savep t))
      (t
       (mew-pop-message2
	pnm (format "%d messages retrieved" cnt) left)
       (setq savep t)))
     (set-buffer fld)
     (if savep (mew-summary-folder-cache-save))
     (set-buffer-modified-p nil)
     (when (eq directive 'mdrop)
       (let ((msgdb (mew-pop-get-msgdb pnm)))
	 (while msgdb
	   (mew-touch-folder (car (car msgdb)))
	   (setq msgdb (cdr msgdb)))))
     (mew-pop-tear-down pnm)
     (unless (eq directive 'biff)
       (run-hooks 'mew-pop-sentinel-non-biff-hook))
     (run-hooks 'mew-pop-sentinel-hook)
     (when (and mew-auto-flush-queue flush)
       (mew-smtp-flush-queue
	(mew-queue-folder mew-case-output) mew-case-output)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; POP AUTH
;;;

(defun mew-pop-passtag (pnm)
  (let ((server (mew-pop-get-server pnm))
	(port (mew-pop-get-port pnm))
	(user (mew-pop-get-user pnm)))
    (concat user "@" server ":" port)))

(defun mew-pop-passtag2 (case)
  (let ((server (mew-pop-server case))
	(port (mew-pop-port case))
	(user (mew-pop-user case)))
    (concat user "@" server ":" port)))

(defun mew-pop-input-passwd (prompt pnm)
  (let ((tag (mew-pop-passtag pnm))
	(directive (mew-pop-get-directive pnm)))
    (if (eq directive 'biff)
	(or (mew-pop-get-passwd pnm)       ;; mew-pop-biff
	    (mew-input-passwd prompt tag)) ;; mew-pop-check
      (mew-input-passwd prompt tag))))

(defvar mew-pop-auth-alist
  '(("CRAM-MD5" mew-pop-command-auth-cram-md5)))

(defsubst mew-pop-auth-get-func (auth)
  (nth 1 (mew-assoc-case-equal auth mew-pop-auth-alist 0)))

(defun mew-pop-command-auth-cram-md5 (pro pnm)
  (process-send-string
   pro (format "AUTH CRAM-MD5%s" mew-cs-eol))
  (mew-pop-set-status pnm "cram-md5"))

(defun mew-pop-command-pwd-cram-md5 (pro pnm)
  (let ((str (mew-pop-get-input pnm))
	(user (mew-pop-get-user pnm))
	challenge passwd cram-md5)
    (if (string-match " \\([A-Za-z0-9+/]+=*\\)" str) ;; xxx
	(setq challenge (mew-match 1 str)))
    (setq passwd (mew-pop-input-passwd "CRAM-MD5: " pnm))
    (setq cram-md5 (mew-cram-md5 user passwd challenge))
    (process-send-string pro (format "%s%s" cram-md5 mew-cs-eol))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; UIDL
;;;

(defvar mew-pop-uidl-db nil)

(defsubst mew-pop-uidl-db-get (pnm)
  (cdr (assoc (mew-pop-passtag pnm) mew-pop-uidl-db)))

(defsubst mew-pop-uidl-db-set (pnm uidl)
  (let* ((tag (mew-pop-passtag pnm))
	 (ent (assoc tag mew-pop-uidl-db)))
    (if ent
	(setcdr ent uidl)
      (setq mew-pop-uidl-db (cons (cons tag uidl) mew-pop-uidl-db)))))

(defvar mew-pop-uidl-file ".mew-uidl")

(defsubst mew-time-diff (t1 t2)
  (/ (+ (* (- (nth 0 t2) (nth 0 t1)) 65536)
	(- (nth 1 t2) (nth 1 t1)))
     86400.0)) ;; one day (* 60 60 24)

(defsubst mew-pop-expired-p (time keep)
  (cond
   ((and (consp keep)
	 (stringp (nth 0 keep)) (file-exists-p (nth 0 keep))
	 (numberp (nth 1 keep)))
    (if (>= (mew-time-diff time (mew-file-get-time (nth 0 keep))) (nth 1 keep))
	t))
   ((numberp keep)
    (if (>= (mew-time-diff time (current-time)) keep) t))
   ;; ((eq keep t) t)
   ;; This case MUST not be included because messages marked with 'T'
   ;; will be deleted.
   (t nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Set up and tear down
;;;

(defun mew-pop-setup ()
  (setq mew-pop-uidl-db (mew-lisp-load mew-pop-uidl-file))
  (add-hook 'kill-emacs-hook (function mew-pop-clean-up)))

(defun mew-pop-clean-up ()
  (remove-hook 'kill-emacs-hook (function mew-pop-clean-up))
  (mew-lisp-save mew-pop-uidl-file mew-pop-uidl-db))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Filters
;;;

(defun mew-pop-command-capa (pro pnm)
  (if (re-search-forward "<[^>]+>" nil t)
      (mew-pop-set-key pnm (mew-match 0)))
  (process-send-string pro (format "capa%s" mew-cs-eol)))

(defun mew-pop-command-auth (pro pnm)
  (cond
   ((eq (mew-pop-get-auth pnm) t) ;; t means SASL
    (let ((str (mew-pop-get-input pnm)) auth func)
      (if (and (string-match "SASL \\([^\n\r]+\\)\r?\n" str)
	       (setq auth (mew-auth-select
			   (mew-match 1 str) mew-pop-auth-list))
	       (setq func (mew-pop-auth-get-func auth))
	       (fboundp func))
	  (progn
	    (mew-pop-set-auth pnm auth)
	    (funcall func pro pnm))
	(mew-pop-debug "<AUTH>" "No preferred POP AUTH.\n"))))
   (t
    (mew-pop-command-pswd pro pnm))))

(defun mew-pop-command-pswd (pro pnm)
  (let ((auth (mew-pop-get-auth pnm)))
    (cond
     ((or (eq auth 'pass) (eq auth 'user))
      (mew-pop-set-status pnm "user")
      (mew-pop-command-user pro pnm))
     (t
      (mew-pop-set-status pnm "apop")
      (mew-pop-command-apop pro pnm)))))

(defun mew-pop-command-user (pro pnm)
  (mew-pop-set-try pnm 0)
  (process-send-string
   pro (format "user %s%s" (mew-pop-get-user pnm) mew-cs-eol)))

(defun mew-pop-command-pass (pro pnm)
  (let (passwd)
    (setq passwd (mew-pop-input-passwd "POP password: " pnm))
    (mew-pop-message pnm "Sending your POP password to the POP server...")
    (process-send-string
     pro (format "pass %s%s" passwd mew-cs-eol))))

(defun mew-pop-command-apop (pro pnm)
  (let ((try (mew-pop-get-try pnm))
	passwd key kmd5)
    (catch 'passwd-try
      (cond
       ((= try 0)
	(setq key (mew-pop-get-key pnm)))
       ((< try 3)
	(mew-pop-message pnm "APOP password is wrong!")
	(mew-passwd-set-passwd (mew-pop-passtag pnm) nil)
	(setq key (mew-pop-get-key pnm)))
       (t
	(mew-passwd-set-passwd (mew-pop-passtag pnm) nil)
	(mew-pop-set-status pnm "quit")
	(mew-pop-command-quit pro pnm)
	(throw 'passwd-try nil)))
      (setq passwd (mew-pop-input-passwd "APOP password: " pnm))
      (setq kmd5 (mew-keyed-md5 key passwd))
      (mew-pop-set-try pnm (1+ try))
      (mew-pop-message pnm "Sending your APOP password to the POP server...")
      (process-send-string
       pro (format "apop %s %s%s" (mew-pop-get-user pnm) kmd5 mew-cs-eol)))))

(defun mew-pop-command-wpwd (pro pnm)
  (let ((auth (mew-pop-get-auth pnm))
	(str (mew-pop-get-input pnm))
	(clear-pass t))
    (cond
     ((and (stringp str) (string-match " lock" str)) ;; very ad hoc
      (mew-pop-message pnm "The mailbox is locked!")
      (setq clear-pass nil))
     ((or (eq auth 'pass) (eq auth 'user))
      (mew-pop-message pnm "POP password is wrong!"))
     ((eq auth 'apop)
      (mew-pop-message pnm "APOP password is wrong!"))
     ((stringp auth)
      (mew-pop-message pnm "%s password is wrong!" (upcase auth)))
     (t
      ;; pnm may be cleared already
      (mew-pop-message pnm "Password is wrong!")))
    (if clear-pass (mew-passwd-set-passwd (mew-pop-passtag pnm) nil))
    (mew-pop-tear-down pnm))) ;; Typical servers disconnect, so no quit

(defun mew-pop-command-wpwd2 (pro pnm)
  (mew-pop-message pnm "Stronger password scheme should be used!")
  (mew-passwd-set-passwd (mew-pop-passtag pnm) nil)
  (mew-pop-tear-down pnm)) ;; Typical servers disconnect, so no quit

(defun mew-pop-command-list (pro pnm)
  (mew-pop-message pnm "Communicating with the POP server...")
  (mew-pop-set-keep-input pnm nil)
  (cond
   ((mew-pop-get-skip-list pnm)
    (mew-pop-set-status pnm "uidl")
    (process-send-string pro (format "uidl%s" mew-cs-eol)))
   (t
    (process-send-string pro (format "list%s" mew-cs-eol)))))

(defun mew-pop-command-uidl (pro pnm)
  (let (msgs num siz)
    (while (re-search-forward "^\\([0-9]+\\) +\\([0-9]+\\)" nil t)
      (setq num (mew-match 1))
      (setq siz (mew-match 2))
      (setq msgs (cons (list num nil siz) msgs)))
    (if msgs
	(progn
	  (setq msgs (nreverse msgs))
	  (mew-pop-set-rtrs pnm msgs)
	  (mew-pop-set-left pnm (length msgs))
	  (process-send-string pro (format "uidl%s" mew-cs-eol)))
      (mew-pop-set-status pnm "quit")
      (mew-pop-command-quit pro pnm))))

(defun mew-pop-command-umsg (pro pnm)
  (let* ((ctime (current-time))
	 (list-msgs (mew-pop-get-rtrs pnm)) ;; (num nil siz)
	 (old-uidl (mew-pop-uidl-db-get pnm))
	 (refs (mew-pop-get-refs pnm)) ;; ((uid siz del (+fld msg)) ...)
	 (rmvs (mew-pop-get-rmvs pnm))
	 (directive (mew-pop-get-directive pnm))
	 (keep (mew-pop-get-delete pnm))
	 (left 0)
	 rtr rtrs dels
	 num uid uidl uid-time ent)
    (while (re-search-forward "^\\([0-9]+\\) +\\([!-~]*\\)" nil t)
      (setq left (1+ left))
      (setq num (mew-match 1))
      (setq uid (mew-match 2))
      (setq uid-time nil)
      ;; A broken POP server may return a null UID.
      (if (string= uid "") (setq uid nil))
      (cond
       ((or (eq directive 'get) (eq directive 'mdrop))
	(cond
	 ((setq rtr (assoc uid refs))
	  (setq rtrs (cons (cons num rtr) rtrs)))
	 ((member uid rmvs)
	  (setq dels (cons num dels)))))
       ((eq directive 'biff)
	(when (and uid (not (assoc uid old-uidl)))
	  (setq rtrs (cons (list num) rtrs))))
       ((eq directive 'scan)
	(when uid
	  (setq ent (assoc num list-msgs))
	  (setcar (cdr ent) uid)
	  (setq rtrs (cons ent rtrs))))
       ((eq directive 'inc)
	(if uid (setq uid-time (cdr (assoc uid old-uidl))))
	(cond
	 (uid-time
	  (setq uidl (cons (cons uid uid-time) uidl))
	  (if (mew-pop-expired-p uid-time keep)
	      (setq dels (cons num dels))))
	 (t
	  (setq uidl (cons (cons uid ctime) uidl))
	  (setq ent (assoc num list-msgs))
	  (setcar (cdr ent) uid)
	  (setq rtrs (cons ent rtrs)))))))
    (mew-pop-set-uidl pnm uidl)
    (setq rtrs (nreverse rtrs))
    (setq dels (nreverse dels))
    (mew-pop-set-rtrs pnm rtrs)
    (mew-pop-set-dels pnm dels)
    (mew-pop-set-left pnm left)
    (if dels
	(mew-pop-command-pre-dels pro pnm)
      (mew-pop-command-pre-retr pro pnm))))

(defun mew-pop-command-pre-dels (pro pnm)
  (let* ((dels (mew-pop-get-dels pnm))
	 (directive (mew-pop-get-directive pnm))
	 (n (length dels)))
    (cond
     ((= n 0) ;; should not occur
      (mew-pop-command-pre-retr pro pnm))
     ((= n 1)
      (if (eq directive 'mdrop)
	  (mew-pop-message pnm "Deleting 1 message in background..."))
      (mew-pop-set-status pnm "dels")
      (mew-pop-command-dels pro pnm))
     (t
      (if (eq directive 'mdrop)
	  (mew-pop-message pnm "Deleting %d messages in background..." n))
      (mew-pop-set-status pnm "dels")
      (mew-pop-command-dels pro pnm)))))

(defun mew-pop-command-dels (pro pnm)
  (let ((dels (mew-pop-get-dels pnm))
	(left (mew-pop-get-left pnm))
	num)
    (if (null dels)
	(mew-pop-command-pre-retr pro pnm)
      (setq num (car dels))
      (mew-pop-set-dels pnm (cdr dels))
      (mew-pop-set-left pnm (1- left))
      (process-send-string pro (format "dele %s%s" num mew-cs-eol)))))

(defun mew-pop-command-pre-retr (pro pnm)
  (let* ((rtrs (mew-pop-get-rtrs pnm))
	 (n (length rtrs))
	 (directive (mew-pop-get-directive pnm)))
    (cond
     ((= n 0)
      (unless (eq directive 'mdrop)
	(mew-pop-message pnm "No new messages"))
      (mew-pop-set-status pnm "quit")
      (mew-pop-command-quit pro pnm))
     ((= n 1)
      (mew-pop-message pnm "Retrieving 1 message in background...")
      (mew-pop-set-status pnm "retr")
      (mew-pop-command-retr pro pnm))
     (t
      (mew-pop-message pnm "Retrieving %d messages in background..." n)
      (mew-pop-set-status pnm "retr")
      (mew-pop-command-retr pro pnm)))))

(defun mew-pop-command-retr (pro pnm)
  (let* ((directive (mew-pop-get-directive pnm))
	 (rtrs (mew-pop-get-rtrs pnm))
	 (rtr  (car rtrs))
	 (num  (nth 0 rtr))
	 (siz  (nth 2 rtr))
	 (lim (mew-pop-get-size pnm))
	 (lines (mew-pop-get-body-lines pnm)))
    (cond
     ((or (null rtr) (eq directive 'biff))
      (mew-pop-set-truncated pnm nil)
      (mew-pop-set-status pnm "quit")
      (mew-pop-command-quit pro pnm))
     ((or (eq directive 'get) (eq directive 'mdrop))
      (mew-pop-set-truncated pnm nil)
      (process-send-string pro (format "retr %s%s" num mew-cs-eol)))
     ((eq directive 'scan)
      (mew-pop-set-truncated pnm t)
      (process-send-string
       pro (format "top %s 0%s" num mew-cs-eol)))
     ((or (= lim 0) (< (string-to-int siz) lim))
      (mew-pop-set-truncated pnm nil)
      (process-send-string pro (format "retr %s%s" num mew-cs-eol)))
     (t
      (mew-pop-set-truncated pnm t)
      (process-send-string
       pro (format "top %s %d%s" num lines mew-cs-eol))))))

(defun mew-pop-get-new-message (pnm fld msg)
  (if msg
      (cons msg (mew-expand-folder fld msg)) ;; 'get
    (let ((msgdb (mew-pop-get-msgdb pnm))
	  fld-msg file)
      (cond
       ((and msgdb (setq fld-msg (assoc fld msgdb)))
	(setq msg (cdr fld-msg))
	(setcdr fld-msg (int-to-string (1+ (string-to-int msg))))
	(setq file (mew-expand-folder fld msg)))
       (t
	(setq file (mew-folder-new-message fld))
	(setq msg (file-name-nondirectory file))
	(mew-pop-set-msgdb
	 pnm
	 (cons (cons fld (int-to-string (1+ (string-to-int msg)))) msgdb))))
      (cons msg file))))

(defun mew-pop-command-dele (pro pnm)
  (let* ((case (mew-pop-get-case pnm))
	 (width (1- (mew-scan-width)))
	 (left (mew-pop-get-left pnm))
	 (rtrs (mew-pop-get-rtrs pnm))
	 (rtr  (car rtrs))
	 (num  (nth 0 rtr))
	 (uid  (nth 1 rtr))
	 (siz  (nth 2 rtr))
	 (del  (or (eq (nth 3 rtr) t) ;; delete may be a number
		   (eq (mew-pop-get-delete pnm) t)))
	 (fld-msg (nth 4 rtr))
	 (fld  (or (nth 0 fld-msg) (mew-pop-get-fld pnm)))
	 (msg  (nth 1 fld-msg))
	 (truncated (mew-pop-get-truncated pnm))
	 (xmu mew-x-mew-uidl:)
	 vec file msg-file)
    ;; deleting +OK
    (goto-char (point-min))
    (forward-line)
    (delete-region (point-min) (point))
    ;; line delimiters
    (when (string= mew-cs-eol "\r\n")
      (goto-char (point-min))
      (while (search-forward "\r\n" nil t) (replace-match "\n" nil t)))
    ;; deleting .\n
    (goto-char (point-max))
    (forward-line -1)
    (delete-region (point) (point-max))
    ;; unescape ^.
    (goto-char (point-min))
    (while (re-search-forward "^\\." nil t)
      (delete-char -1)
      (forward-line))
    (setq msg-file (mew-pop-get-new-message pnm fld msg))
    (setq msg (car msg-file) file (cdr msg-file))
    (when truncated
      (goto-char (point-min))
      (when uid
	(if case
	    (mew-header-insert xmu (concat uid " " siz " " case) 'no-fold)
	  (mew-header-insert xmu (concat uid " " siz) 'no-fold))))
    (catch 'write-error
      (condition-case nil
	  (mew-frwlet 
	   mew-cs-dummy mew-cs-text-for-write
	   (write-region (point-min) (point-max) file nil 'no-msg))
	(error
	 (mew-pop-set-status pnm "quit")
	 (mew-pop-command-quit pro pnm)
	 (throw 'write-error nil)))
      (set-file-modes file mew-file-mode)
      (mew-pop-set-cnt pnm (1+ (mew-pop-get-cnt pnm)))
      ;;
      (let ((links (nthcdr 5 rtr))
	    fld msg alias)
	(while links
	  (setq fld (nth 0 (car links)))
	  (setq msg (nth 1 (car links)))
	  (setq links (cdr links))
	  (setq alias (cdr (mew-pop-get-new-message pnm fld msg)))
	  (mew-link file alias)))
      ;;
      (mew-set-buffer-multibyte t)
      (setq vec (mew-pop-scan-header))
      (mew-scan-set-folder vec fld)
      (mew-scan-set-message vec msg)
      (mew-set-buffer-multibyte nil)
      (mew-scan-insert-line fld vec width msg nil)
      (mew-pop-set-rtrs pnm (cdr rtrs))
      (if (and del (not truncated))
	  (progn
	    (mew-pop-set-left pnm (1- left))
	    (process-send-string
	     pro (format "dele %s%s" num mew-cs-eol)))
	(mew-pop-set-status pnm "retr")
	(mew-pop-command-retr pro pnm)))))

(defun mew-pop-command-quit (pro pnm)
  (when (mew-pop-get-save-uidl pnm)
    (mew-pop-uidl-db-set pnm (mew-pop-get-uidl pnm))
    (mew-lisp-save mew-pop-uidl-file mew-pop-uidl-db))
  (process-send-string
   pro (format "quit%s" mew-cs-eol)))

(defun mew-pop-command-noop (pro pnm)
  ())

(defun mew-pop-scan-header ()
  (goto-char (point-min))
  (unless (re-search-forward mew-eoh nil t)
    (goto-char (point-max)))
  (mew-scan-header))

(provide 'mew-pop)

;;; Copyright Notice:

;; Copyright (C) 1999-2002 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-pop.el ends here
