/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: parse.c,v 1.46 2002/01/01 11:43:26 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "config.h"

#ifdef HAVE_LIBADNS
#include <adns.h>
#endif

#include "mlocale.h"
#include "mplugins.h"
#include "mrecord.h"
#include "mdatatypes.h"
#include "datatypes/query/datatype.h"
#include "datatypes/record/datatype.h"
#include "misc.h"

#include "plugin_config.h"

const char *short_month[] = {	"Jan", "Feb", "Mar", "Apr", "May", "Jun", 
			"Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL};

int find_os (mconfig *ext_conf, char *str) {
	config_input *conf = ext_conf->plugin_conf;
	mlist *l = conf->match_os;
	if (!str || !l) return 0;
	
	while (*str == ' ') str++;
	
	while (l) {
		mdata *data = l->data;
		
		if (data && strmatch(data->data.match.match, NULL, str)) {
			return 1;
		}
		
		l = l->next;
	}
	
	return 0;
}

int find_ua (mconfig *ext_conf, char *str) {
	config_input *conf = ext_conf->plugin_conf;
	mlist *l = conf->match_ua;
	if (!str || !l) return 0;
	
	while (*str == ' ') str++;
	
	while (l) {
		mdata *data = l->data;
		
		if (data && strmatch(data->data.match.match, NULL, str)) {
			return 1;
		}
		
		l = l->next;
	}
	
	return 0;
}

int parse_timestamp(mconfig *ext_conf, const char *str, mlogrec *record) {
#define N 20 + 1
	int ovector[3 * N], n, i;
	char buf[10];
	struct tm tm;
	config_input *conf = ext_conf->plugin_conf;
	
	if ((n = pcre_exec(conf->match_timestamp, conf->match_timestamp_extra, str, strlen(str), 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "string doesn't match: %s\n", str);
			return M_RECORD_CORRUPT;
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "execution error while matching: %d\n", n);
			return M_RECORD_HARD_ERROR;
		}
	}
	
	/* everything has matched, take the different pieces and be happy :) */
	pcre_copy_substring(str, ovector, n, 1, buf, sizeof(buf));
	tm.tm_mday = strtol(buf, NULL, 10);
	
	pcre_copy_substring(str, ovector, n, 2, buf, sizeof(buf));
	for (i = 0; short_month[i];i++) {
		if (!strcmp(buf, short_month[i])) {
			tm.tm_mon = i;
		}
	}
	
	pcre_copy_substring(str, ovector, n, 3, buf, sizeof(buf));
	tm.tm_year = strtol(buf, NULL, 10)-1900;
	
	pcre_copy_substring(str, ovector, n, 4, buf, sizeof(buf));
	tm.tm_hour = strtol(buf, NULL, 10);
	pcre_copy_substring(str, ovector, n, 5, buf, sizeof(buf));
	tm.tm_min = strtol(buf, NULL, 10);
	pcre_copy_substring(str, ovector, n, 6, buf, sizeof(buf));
	tm.tm_sec = strtol(buf, NULL, 10);
	
	record->timestamp = mktime (&tm);
	
	return M_RECORD_NO_ERROR;
#undef  N
}

int parse_useragent(mconfig *ext_conf,const char *str, mlogrec_web_extclf *record) {
/* get user agent */
	char *pc1, *pc3, *pc2, *buf_copy;
	
	buf_copy = malloc(strlen(str)+1);
	strcpy(buf_copy, str);
	
	pc1 = pc2 = buf_copy;

	if ((pc3 = strchr(pc1, '(') )) {
		if (strstr(pc3, "compatible;")) {
			int finished = 0;
			char *unknown = NULL;
			
			pc1 = pc2 = (pc3+1) + /* strlen("compatible;") */ 11;
			
			/* pc3: point to the ( before compatible */
			
			while (!finished && !(record->req_useragent && record->req_useros)) {
				while (*pc2 && !(*pc2 == ';' || *pc2 == ')')) pc2++;
				
				if (!*pc2 || *pc2 == ')') {
					finished = 1;
				}
				
				/* trim spaces */
				while (*pc1 == ' ') pc1++;
				
				if (pc1 != pc2) {
					*pc2 = '\0';
					
					/* a datafield to parse */
					
					if (!record->req_useragent && find_ua(ext_conf, pc1)) {
						record->req_useragent = malloc(pc2-pc1+1);
						strcpy(record->req_useragent, pc1);
					} else if (!record->req_useros && find_os(ext_conf, pc1)) {
						record->req_useros = malloc(pc2-pc1+1);
						strcpy(record->req_useros, pc1);
					} else {
						if (unknown == NULL) {
							unknown = malloc(pc2-pc1+1);
							strcpy(unknown, pc1);
						} else {
							unknown = realloc(unknown, strlen(unknown) + (pc2-pc1) + 1);
							strcat(unknown, pc1);
						}
					}
				}
				pc1 = ++pc2;
			}
			
			if (unknown) {
				if (!(record->req_useragent)) {
					record->req_useragent = unknown;
				} else {
					free(unknown);
				}
			}

		} else {
			int finished = 0;
			
			pc2 = pc3;

			*pc2 = '\0';
			
			record->req_useragent = malloc(pc2-pc1+1);
			strcpy(record->req_useragent, pc1);
			
			pc1 = pc2 = (pc3+1);

			while (!finished) {
				while (*pc2 && !(*pc2 == ';' || *pc2 == ')')) pc2++;
				if (!*pc2 || *pc2 == ')') {
					finished = 1;
				}

				while (*pc1 == ' ') pc1++;
				
				*pc2 = '\0';
				
				if (!record->req_useros && find_os(ext_conf, pc1)) {
					record->req_useros = malloc(strlen(pc1)+1);
					strcpy(record->req_useros, pc1);
				}
				pc1 = ++pc2;
			}
		}
	} else {
		record->req_useragent = malloc(strlen(str)+1);
		strcpy(record->req_useragent, str);
	}
	
	free(buf_copy);
	
	return 0;
}

int parse_url(mconfig *ext_conf,const char *str, mlogrec_web *record) {
#define N 20 + 1
	int ovector[3 * N], n;
#if 0
#define DEBUG_INPUT
#endif
#ifdef DEBUG_INPUT
	int i;
#endif
	config_input *conf = ext_conf->plugin_conf;
	const char **list;
	
	if (strcmp("-", str) == 0) {
/* if the url is '-' we don't have to cry about it. 
 * if someone knows what a url == '-' is good for, tell me please.
 * doing it this should suppress the warning.
 */
		return M_RECORD_IGNORED;
	}
	
	if ((n = pcre_exec(conf->match_url, conf->match_url_extra, str, strlen(str), 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "string doesn't match: %s\n", str);
			return M_RECORD_CORRUPT;
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "execution error while matching: %d\n", n);
			return M_RECORD_HARD_ERROR;
		}
	}
	
	if (n > 2) {
		/* everything has matched, take the different pieces and be happy :) */
		pcre_get_substring_list(str, ovector, n, &list);
	
		record->req_method = malloc(strlen((char *)list[1])+1);
		strcpy(record->req_method, (char *)list[1]);
		
		if (strcmp("-", (char *)list[2]) == 0) {
			return M_RECORD_IGNORED;
		}
		
		record->req_url = malloc(strlen((char *)list[2])+1);
		strcpy(record->req_url, (char *)list[2]);
		
		if (n > 3) {
			if (strlen((char *)list[3])) {
				record->req_getvars = malloc(strlen((char *)list[3])+1);
				strcpy(record->req_getvars, (char *)list[3]);
			}
		}
			
		if (n > 4) {
			record->req_protocol = malloc(strlen((char *)list[4])+1);
			strcpy(record->req_protocol, (char *)list[4]);
		}

		free(list);
	} else {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "Matched fields below minimum: %d\n", n);
		return M_RECORD_HARD_ERROR;
	}
	
	return M_RECORD_NO_ERROR;
#undef  N
}

int parse_referrer(mconfig *ext_conf,const char *str, mlogrec_web_extclf *record) {
#define N 20 + 1
	int ovector[3 * N], n;
	config_input *conf = ext_conf->plugin_conf;
	const char **list;
	
	if ((n = pcre_exec(conf->match_referrer, conf->match_referrer_extra, str, strlen(str), 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "string doesn't match: %s\n", str);
			return M_RECORD_CORRUPT;
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "execution error while matching: %d\n", n);
			return M_RECORD_HARD_ERROR;
		}
	}
	
	if (n >= 2) {
		/* everything has matched, take the different pieces and be happy :) */
		pcre_get_substring_list(str, ovector, n, &list);
	
		record->ref_url = malloc(strlen((char *)list[1])+1);
		strcpy(record->ref_url, (char *)list[1]);
		
		if (n > 3) {
			record->ref_getvars = malloc(strlen((char *)list[3])+1);
			strcpy(record->ref_getvars, (char *)list[3]);
		}
#ifdef DEBUG_INPUT		
		fprintf(stderr, "%s.%d: %s, %s\n", __FILE__, __LINE__, record->ref_url, record->ref_getvars);
#endif
		free(list);
	} else {
		fprintf(stderr, "%s.%d: Matched fields below minimum: %d\n", __FILE__, __LINE__, n);
		return -1;
	}
	
	
	
	return 0;
#undef  N
}

int parse_record_pcre(mconfig *ext_conf, mlogrec *record, char *_buffer) {
#define N 20 + 1
	const char **list;
	int ovector[3 * N], n;
	config_input *conf = ext_conf->plugin_conf;
	mlogrec_web *recweb = NULL; 
	int ret;
	
	record->ext_type = M_RECORD_TYPE_WEB;
	record->ext = mrecord_init_web();
	
	recweb = record->ext;
	
	if (recweb == NULL) return M_RECORD_HARD_ERROR;
	
	if (strncmp("format=", _buffer, 7) == 0) {
		fprintf(stderr, "%s.%d: detected a NetScape Server Log - perhaps it goes wrong\n", __FILE__, __LINE__);
		fprintf(stderr, "%s.%d: use the netscape plugin instead\n", __FILE__, __LINE__);
		return M_RECORD_HARD_ERROR;
	}
	
/* parse a CLF record */	
	if ((n = pcre_exec(conf->match_clf, conf->match_clf_extra, _buffer, strlen(_buffer), 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "string doesn't match: %s\n", _buffer);
			return M_RECORD_CORRUPT;
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "execution error while matching: %d\n", n);
			return M_RECORD_HARD_ERROR;
		}
	}
	
	if (n >= 7) {
		int i;
		pcre_get_substring_list(_buffer, ovector, n, &list);
		
		recweb->req_host = malloc(strlen((char *)list[1])+1);
		strcpy(recweb->req_host, (char *)list[1]);
	
		recweb->req_user = malloc(strlen((char *)list[3])+1);
		strcpy(recweb->req_user, (char *)list[3]);
	
		switch ((ret = parse_timestamp(ext_conf, list[4], record))) {
		case M_RECORD_NO_ERROR:
			break;
		case M_RECORD_HARD_ERROR:
			fprintf(stderr, "%s.%d: parse_timestamp died on %s\n", __FILE__, __LINE__, _buffer);
			free(list);
			return M_RECORD_HARD_ERROR;
		case M_RECORD_IGNORED:
			free(list);
			return M_RECORD_IGNORED;
		case M_RECORD_CORRUPT:
			free(list);
			return M_RECORD_CORRUPT;
		default:
			fprintf(stderr, "%s.%d: parse_timestamp return a unknown ret-value on %d\n", 
				__FILE__, __LINE__, 
				ret);
			free(list);
			return M_RECORD_HARD_ERROR;
		}
		
		switch ( (ret = parse_url(ext_conf, list[5], recweb)) ) {
		case M_RECORD_NO_ERROR:
			break;
		case M_RECORD_HARD_ERROR:
			fprintf(stderr, "%s.%d: parse_url died on %s\n", __FILE__, __LINE__, _buffer);
			free(list);
			return M_RECORD_HARD_ERROR;
		case M_RECORD_IGNORED:
			free(list);
			return M_RECORD_IGNORED;
		case M_RECORD_CORRUPT:
			free(list);
			return M_RECORD_CORRUPT;
		default:
			fprintf(stderr, "%s.%d: parse_url return a unknown ret-value: %d\n", 
				__FILE__, __LINE__, 
				ret);
			free(list);
			return M_RECORD_CORRUPT;
		}
	
		recweb->req_status = strtol(list[6], NULL,10);
		recweb->xfersize = strtol(list[7], NULL,10);
		
		/* referrer & useragent */
		if (n == 11) {
			mlogrec_web_extclf *recext;
		
			recext = mrecord_init_web_extclf();
			
			if (recext != NULL) {
				if (parse_referrer(ext_conf, list[9], recext) == -1) {
					mrecord_free_web_extclf(recext);
					fprintf(stderr, "%s.%d: parse_referrer died on %s\n", __FILE__, __LINE__, _buffer);
					free(list);
					return M_RECORD_CORRUPT;
				}

				if (parse_useragent(ext_conf, list[10], recext) == -1) {
					mrecord_free_web_extclf(recext);
					fprintf(stderr, "%s.%d: parse_useragent died on %s\n", __FILE__, __LINE__, _buffer);
					free(list);
					return M_RECORD_CORRUPT;
				}
				recweb->ext_type = M_RECORD_TYPE_WEB_EXTCLF;
				recweb->ext = recext;
			}
		} else if (n == 10) {
			/* squid */
			mlogrec_web_squid *recext;
		
			recweb->ext_type = M_RECORD_TYPE_WEB_SQUID;
			recweb->ext = mrecord_init_web_extclf();
		
			recext = recweb->ext;
		
			if (recext != NULL) {
			}
		}
		free(list);
	} else {
		fprintf(stderr, "%s.%d: Matched fields below minimum: %d\n", __FILE__, __LINE__, n);
		return M_RECORD_HARD_ERROR;
	}
	
	return M_RECORD_NO_ERROR;
#undef  N
}

int parse_record_dynamic(mconfig *ext_conf, mlogrec *record, char *_buffer) {
#define N 20 + 1
	const char **list;
	int ovector[3 * N], n, i;
	config_input *conf = ext_conf->plugin_conf;
	mlogrec_web *recweb = NULL; 
	mlogrec_web_extclf *recext = NULL; 
	
	/* remove the carriage return */
	if (_buffer[strlen(_buffer)-1] == '\r') {
		_buffer[strlen(_buffer)-1] = '\0';
	}
	
	if (conf->match_clf == NULL) return M_RECORD_HARD_ERROR;
	
	recweb = mrecord_init_web();
	
	record->ext_type = M_RECORD_TYPE_WEB;
	record->ext = recweb;
	
	recext = mrecord_init_web_extclf();
	
	recweb->ext_type = M_RECORD_TYPE_WEB_EXTCLF;
	recweb->ext = recext;
	
	if (recweb == NULL) return M_RECORD_HARD_ERROR;

/* parse a CLF record */	
	if ((n = pcre_exec(conf->match_clf, conf->match_clf_extra, _buffer, strlen(_buffer), 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			fprintf(stderr, "%s.%d: string doesn't match: %s\n", __FILE__, __LINE__, _buffer);
			return M_RECORD_CORRUPT;
		} else {
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
			return M_RECORD_HARD_ERROR;
		}
	}
	
	pcre_get_substring_list(_buffer, ovector, n, &list);
	
	for (i = 0; i < n-1; i++) {
		switch (def[conf->trans_fields[i]].id) {
		case M_CLF_FIELD_TIMESTAMP:
			switch ( parse_timestamp(ext_conf, (char *)list[i+1], record) ) {
			case M_RECORD_NO_ERROR:
				break;
			case M_RECORD_HARD_ERROR:
				fprintf(stderr, "%s.%d: parse_timestamp died on %s\n", __FILE__, __LINE__, _buffer);
				free(list);
				return M_RECORD_HARD_ERROR;
			case M_RECORD_IGNORED:
				free(list);
				return M_RECORD_IGNORED;
			case M_RECORD_CORRUPT:
				free(list);
				return M_RECORD_CORRUPT;
			default:
				fprintf(stderr, "%s.%d: *args* on %s\n", __FILE__, __LINE__, _buffer);
				free(list);
				return M_RECORD_HARD_ERROR;
			}
			
			break;
		case M_CLF_FIELD_REQ_HOST:
			recweb->req_host = malloc(strlen((char *)list[i+1])+1);
			strcpy(recweb->req_host, (char *)list[i+1]);
			break;
		case M_CLF_FIELD_USERNAME:
			recweb->req_user = malloc(strlen((char *)list[i+1])+1);
			strcpy(recweb->req_user, (char *)list[i+1]);
			break;
		case M_CLF_FIELD_STATUS:
			recweb->req_status = strtol(list[i+1], NULL,10);
			break;
		case M_CLF_FIELD_BYTES_SEND:
			recweb->xfersize = strtod(list[i+1], NULL);
			break;
		case M_CLF_FIELD_SERVER_PORT:
			recext->srv_port = malloc(strlen((char *)list[i+1])+1);
			strcpy(recext->srv_port, (char *)list[i+1]);
			break;
		case M_CLF_FIELD_SERVER_IP:
			recext->srv_host = malloc(strlen((char *)list[i+1])+1);
			strcpy(recext->srv_host, (char *)list[i+1]);
			break;
		case M_CLF_FIELD_REQUEST:
			switch ( parse_url(ext_conf, list[i+1], recweb) ) {
			case M_RECORD_NO_ERROR:
				break;
			case M_RECORD_HARD_ERROR:
				fprintf(stderr, "%s.%d: parse_url died on %s\n", __FILE__, __LINE__, _buffer);
				free(list);
				return M_RECORD_HARD_ERROR;
			case M_RECORD_IGNORED:
				free(list);
				return M_RECORD_IGNORED;
			case M_RECORD_CORRUPT:
				free(list);
				return M_RECORD_CORRUPT;
			default:
				fprintf(stderr, "%s.%d: *args* on %s\n", __FILE__, __LINE__, _buffer);
				free(list);
				return M_RECORD_HARD_ERROR;
			}
			break;
		case M_CLF_FIELD_USER_AGENT:
			switch (parse_useragent(ext_conf, list[i+1], recext)) {
			case M_RECORD_NO_ERROR:
				break;
			case M_RECORD_HARD_ERROR:
				fprintf(stderr, "%s.%d: parse_useragent died on %s\n", __FILE__, __LINE__, _buffer);
				free(list);
				return M_RECORD_HARD_ERROR;
			case M_RECORD_IGNORED:
				free(list);
				return M_RECORD_IGNORED;
			case M_RECORD_CORRUPT:
				free(list);
				return M_RECORD_CORRUPT;
			default:
				fprintf(stderr, "%s.%d: *args* on %s\n", __FILE__, __LINE__, _buffer);
				free(list);
				return M_RECORD_HARD_ERROR;
			}
			break;
		case M_CLF_FIELD_REFERRER:
			switch (parse_referrer(ext_conf, list[i+1], recext)) {
			case M_RECORD_NO_ERROR:
				break;
			case M_RECORD_HARD_ERROR:
				fprintf(stderr, "%s.%d: parse_referrer died on %s\n", __FILE__, __LINE__, _buffer);
				free(list);
				return M_RECORD_HARD_ERROR;
			case M_RECORD_IGNORED:
				free(list);
				return M_RECORD_IGNORED;
			case M_RECORD_CORRUPT:
				free(list);
				return M_RECORD_CORRUPT;
			default:
				fprintf(stderr, "%s.%d: *args* on %s\n", __FILE__, __LINE__, _buffer);
				free(list);
				return M_RECORD_HARD_ERROR;
			}
			break;
		/* no mapping */
		case M_CLF_FIELD_AUTH_USERNAME:
		case M_CLF_FIELD_DURATION:
			M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
				"the field '%s' (%d) is known, but not supported yet.\n",def[conf->trans_fields[i]].field, def[conf->trans_fields[i]].id);
			break;
		/* no mapping, silent */
		case M_CLF_FIELD_REMOTE_IP:
			break;

		default:
			fprintf(stderr, "the field '%s' (%d) is unknown\n", def[conf->trans_fields[i]].field, def[conf->trans_fields[i]].id);
			break;
		}
	}

	free(list);
	
	return M_RECORD_NO_ERROR;
#undef  N
}

int get_line (mconfig *ext_conf) {
	config_input *conf = ext_conf->plugin_conf;
	int newline = 1;
	
	/* TODO: handle !\n as a expected linewrap 
	 * 
	 * ...bla!\n
	 * saber\n
	 * 
	 * becomes
	 * 
	 * ...blasaber\n
	 * 
	 */
	
	if (!fgets(conf->buffer, conf->buf_len-1,conf->inputfile)) {
		newline = 0;
	}

	while (newline && conf->buffer[strlen(conf->buffer)-1] != '\n') {
		conf->buffer = realloc(conf->buffer, (conf->buf_len+conf->buf_inc+1) * sizeof(char));
		
		if (!fgets(conf->buffer+strlen(conf->buffer), conf->buf_inc-1,conf->inputfile)) {
			newline = 0;
		}
		
		conf->buf_len += conf->buf_inc;
	}
	
	return newline;
}

#ifdef HAVE_LIBADNS
const char* reverse_ip(const char *ip) {
	static char ip_buf[30];
	pcre *m;
#define N 20 + 1
	const char **list;
	int ovector[3 * N], n;
	const char *errptr;
	int erroffset = 0;
	
	if (!ip) return NULL;
	
	if ((m = pcre_compile(
		"^([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})$", 
		0, &errptr, &erroffset, NULL)) == NULL) {
		
		fprintf(stderr, "%s.%d: rexexp compilation error at %s\n", __FILE__, __LINE__, errptr);
		return NULL;
	} 
	
	if ((n = pcre_exec(m, NULL, ip, strlen(ip), 0, 0, ovector, 3 * N)) < 0) {
		if (n != PCRE_ERROR_NOMATCH) {
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
		}
		return NULL;
	}
	
	pcre_get_substring_list(ip, ovector, n, &list);
	
	sprintf(ip_buf, "%s.%s.%s.%s.in-addr.arpa.",
		list[4],
		list[3],
		list[2],
		list[1]
		);
	
	pcre_free(list);
	free(m);
	
	return ip_buf;
}
#endif

int mplugins_input_clf_get_next_record(mconfig *ext_conf, mlogrec *record) {
	int ret = 0;
	config_input *conf = ext_conf->plugin_conf;
	int newline = 1;
	mlist *l;
	
	if (record == NULL) return M_RECORD_HARD_ERROR;
	
	/* inserting the records into to list */
	while (newline && conf->read_ahead_count < conf->read_ahead_limit) {
		newline = get_line(ext_conf);
		
		if (newline) {
			mlogrec *new_record = mrecord_init();
			mdata *data = NULL;
			char string[255];
#ifdef HAVE_LIBADNS
			const char *conv_ip;
			adns_query *query = NULL;
			mlogrec_web *web = NULL;
#endif
			ret = (conf->format)
				? parse_record_dynamic(ext_conf, new_record, conf->buffer)
				: parse_record_pcre(ext_conf, new_record, conf->buffer);
			
			switch(ret) {
			case M_RECORD_NO_ERROR:
				break;
			case M_RECORD_IGNORED:
				mrecord_free(new_record);
				return M_RECORD_IGNORED;
			case M_RECORD_HARD_ERROR:
				mrecord_free(new_record);
				return M_RECORD_HARD_ERROR;
			case M_RECORD_CORRUPT:
				mrecord_free(new_record);
				return M_RECORD_CORRUPT;
			default:
				mrecord_free(new_record);
				fprintf(stderr, "%s.%d: unknown return value -> %d\n", __FILE__, __LINE__, ret);
				return M_RECORD_HARD_ERROR;
			}
			
#ifdef HAVE_LIBADNS
			if (ext_conf->enable_resolver) {
				web = new_record->ext;
				conv_ip = reverse_ip(web->req_host);
			
				if (conv_ip) {
					mdata *data = NULL;
					
					if (!mhash_in_hash(ext_conf->query_hash, web->req_host)) {
						query = malloc(sizeof(adns_query));
	
						adns_submit(*(ext_conf->adns), 
							conv_ip,
							adns_r_ptr,
							adns_qf_quoteok_cname|adns_qf_cname_loose,
							NULL,
							query
						);
				
			/* put ip and query into a hash */
					
						data = mdata_Query_create(web->req_host, query);
						mhash_insert_sorted(ext_conf->query_hash, data);
					}
			/* don't free query !! it will be removed by mhash_free */
				}
			}
#endif
			
			
			sprintf(string, "%12ld", new_record->timestamp);
			
			data = mdata_Record_create(string, new_record);
			mlist_insert_sorted(conf->record_list, data);
			/* don't free record !! it will be removed by mhash_free */
			
			conf->read_ahead_count++;
		}
	}
	
	/* take them out of the list */
	l = conf->record_list;
	
	if (l && l->data) {
		mrecord_copy(record, l->data->data.record.record);
		
		mdata_free(l->data);
		l->data = NULL;
		
		if (l->next) {
			conf->record_list = l->next;
			l->next->prev = NULL;
			l->prev = NULL;
			l->next = NULL;
			mlist_free(l);
		}
		
		conf->read_ahead_count--;
		
		return M_RECORD_NO_ERROR;
	}

	return M_RECORD_EOF;
}
