/*
 * Copyright (C) 2001  CodeFactory AB
 * Copyright (C) 2001  Richard Hult
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult <rhult@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <gtk/gtk.h>
#include <libgnomeui/gnome-canvas.h>
#include <libgnomeui/gnome-canvas-util.h>
#include <libgnomeui/gnome-canvas-line.h>

#include "util/type-utils.h"
#include "arrow-item.h"

static void arrow_item_init	    (ArrowItem      *arrow_item);
static void arrow_item_class_init   (ArrowItemClass *klass);

GNOME_CLASS_BOILERPLATE (ArrowItem,
			 arrow_item,
			 GnomeCanvasLine,
			 gnome_canvas_line);


static void
arrow_item_class_init (ArrowItemClass *klass)
{
}

static void
arrow_item_init (ArrowItem *item)
{
}

static void
set_arrow_end (ArrowItem *arrow)
{
	GnomeCanvasPoints *points;
	gdouble            x1, y1, x2, y2;

	x1 = GNOME_CANVAS_ITEM (arrow->item)->x1;
	y1 = GNOME_CANVAS_ITEM (arrow->item)->y1;
	x2 = GNOME_CANVAS_ITEM (arrow->item)->x2;
	y2 = GNOME_CANVAS_ITEM (arrow->item)->y2;

	gtk_object_get (GTK_OBJECT (arrow), "points", &points, NULL);

	points->coords[2] = x1;
	points->coords[3] = y1 + (y2 - y1) / 2;

	gnome_canvas_item_raise_to_top (GNOME_CANVAS_ITEM (arrow));

	gnome_canvas_item_set (GNOME_CANVAS_ITEM (arrow),
			       "points", points,
			       NULL);
}

static void
set_arrow_start (ArrowItem *arrow)
{
	GnomeCanvasPoints *points;
	gdouble            x1, y1, x2, y2;

	x1 = GNOME_CANVAS_ITEM (arrow->predecessor)->x1;
	y1 = GNOME_CANVAS_ITEM (arrow->predecessor)->y1;
	x2 = GNOME_CANVAS_ITEM (arrow->predecessor)->x2;
	y2 = GNOME_CANVAS_ITEM (arrow->predecessor)->y2;

	gtk_object_get (GTK_OBJECT (arrow), "points", &points, NULL);
	
	points->coords[0] = x2;
	points->coords[1] = y1 + (y2 - y1) / 2;

	gnome_canvas_item_raise_to_top (GNOME_CANVAS_ITEM (arrow));
	
	gnome_canvas_item_set (GNOME_CANVAS_ITEM (arrow),
			       "points", points,
			       NULL);
}

static void
item_moved (NetworkItem *item, ArrowItem *arrow)
{
	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_NETWORK_ITEM (item));
	g_return_if_fail (arrow != NULL);
	g_return_if_fail (IS_ARROW_ITEM (arrow));

	set_arrow_end (arrow);
}

static void
predecessor_moved (NetworkItem *item, ArrowItem *arrow)
{
	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_NETWORK_ITEM (item));
	g_return_if_fail (arrow != NULL);
	g_return_if_fail (IS_ARROW_ITEM (arrow));

	set_arrow_start (arrow);
}

static void
item_destroyed (NetworkItem *item, ArrowItem *arrow)
{
	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_NETWORK_ITEM (item));
	g_return_if_fail (arrow != NULL);
	g_return_if_fail (IS_ARROW_ITEM (arrow));

	gtk_object_destroy (GTK_OBJECT (arrow));
}

ArrowItem *
arrow_item_new (NetworkItem *item, NetworkItem *predecessor)
{
	ArrowItem         *arrow;
	GnomeCanvasGroup  *root;
	GnomeCanvasPoints *points;
	
	g_return_val_if_fail (item != NULL, NULL);
	g_return_val_if_fail (IS_NETWORK_ITEM (item), NULL);
	g_return_val_if_fail (predecessor != NULL, NULL);
	g_return_val_if_fail (IS_NETWORK_ITEM (predecessor), NULL);
	
	root = gnome_canvas_root (GNOME_CANVAS_ITEM (item)->canvas);
	points = gnome_canvas_points_new (2);

	arrow = ARROW_ITEM (gnome_canvas_item_new (root,
						   TYPE_ARROW_ITEM,
						   "points", points,
						   "last_arrowhead", TRUE,
						   "arrow_shape_a", 6.0,
						   "arrow_shape_b", 6.0,
						   "arrow_shape_c", 3.0,
						   "fill_color", "black",
						   NULL));
	
	arrow->item = item;
	arrow->predecessor = predecessor;

	gnome_canvas_points_unref (points);

	set_arrow_start (arrow);
	set_arrow_end (arrow);
	
	gtk_signal_connect_while_alive (GTK_OBJECT (item),
					"moved",
					item_moved,
					arrow,
					GTK_OBJECT (arrow));

	gtk_signal_connect_while_alive (GTK_OBJECT (item),
					"destroy",
					item_destroyed,
					arrow,
					GTK_OBJECT (arrow));

	gtk_signal_connect_while_alive (GTK_OBJECT (predecessor),
					"moved",
					predecessor_moved,
					arrow,
					GTK_OBJECT (arrow));

	gtk_signal_connect_while_alive (GTK_OBJECT (predecessor),
					"destroy",
					item_destroyed,
					arrow,
					GTK_OBJECT (arrow));

	return arrow;
}
