/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001      Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

/* TODO:
 *
 * Ability to hide the shortcutbar and show notebook tabs instead.
 * UTF-8 issues for the shortcutbar labels (e.g. swedish translation of network).
 * Ability to add a component after a project is loaded.
 *
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <bonobo/bonobo-ui-util.h>
#include <bonobo/bonobo-win.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <gal/widgets/e-unicode.h>
#include <gal/shortcut-bar/e-shortcut-bar.h>
#include <gal/e-paned/e-hpaned.h>
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "util/file-sel.h"
#include "client/widgets/mr-hpaned.h"
#include "client/widgets/mr-message-box.h"
#include "preferences-dialog.h"
#include "project-window.h"
#include "project-properties-dialog.h"
#include "template-dialog.h"
#include "shell.h"

#include "util/debug.h"

static void      project_window_init        (GtkObject         *object);
static void      project_window_class_init  (GtkObjectClass    *object_class);
static gchar    *window_title_from_name     (const gchar       *str);
static gchar    *get_project_name           (ProjectWindow     *window);
static void      get_project_range          (ProjectWindow     *window);
static void      add_shell_listener         (ProjectWindow     *window);
static void      remove_shell_listener      (ProjectWindow     *window);
static gboolean  project_window_delete_event(ProjectWindow     *window, 
					     gpointer           user_data);
static void 	 project_window_exit        (ProjectWindow     *window);
static gboolean  project_window_really_exit (ProjectWindow     *window);
static void      setup_progress_bar         (ProjectWindow     *window);
static void      set_pixmap                 (BonoboUIComponent *uic,
					     const char        *xml_path,
					     const char        *icon);

GNOME_CLASS_BOILERPLATE (ProjectWindow, project_window, 
			 BonoboWindow, bonobo_window);


static GList *window_list = NULL;
extern GSList *open_projects;

struct _ProjectWindowPriv {
	BonoboUIComponent *ui_component;
	Shell             *shell;
	gint               listener_id;
	GtkWidget         *notebook;
	GtkWidget         *shortcut_bar;
	GtkWidget         *paned;
	GtkWidget         *progress_bar;
	GtkWidget         *status;
};

static void
project_window_destroy (GtkObject *object)
{
	ProjectWindow      *window;
	GM_Project          project;

	window = PROJECT_WINDOW (object);

	d(puts (__FUNCTION__));
	
	remove_shell_listener (window);
	
	project = shell_get_project (window->priv->shell);
	open_projects = g_slist_remove (open_projects, project);

	shell_remove_from_project (window->priv->shell);
	bonobo_object_unref (BONOBO_OBJECT (window->priv->shell));

	g_free (window->priv);
	window->priv = NULL;

	window_list = g_list_remove (window_list, window);
	if (g_list_length (window_list) == 0) {
		gtk_main_quit ();
	}
}

static void
new_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	project_window_new (NULL);
}

static void
new_details_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GtkWidget *dialog, *window;
	gint       ret;
	gchar     *filename;

	/* FIXME: when we have templates, add this again. */
	new_cb (component, data, cname);
	return;
	
	dialog = template_dialog_new ();
	ret =  template_dialog_run (TEMPLATE_DIALOG (dialog), &filename);
	gtk_widget_destroy (dialog);

	if (ret != GNOME_OK) {
		return;
	}

	window = NULL;
	
	if (filename) {
		GM_Project         project_co;
		CORBA_Environment  ev;
		gchar             *uri;

		uri = gnome_vfs_get_uri_from_local_path (filename);
		
		window = project_window_new (uri);
	
		/* Reset the filename so we don't try to save over it. */
		project_co = shell_get_project (PROJECT_WINDOW (window)->priv->shell);
		CORBA_exception_init (&ev);

		GNOME_MrProject_Project_setURI (project_co, "", &ev);
		CORBA_exception_free (&ev);

		g_free (uri);
	}
	else {
		window = project_window_new (NULL);
	}
		
	gtk_widget_show_all (window);
	g_free (filename);
}

static void
open_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ProjectWindow *window, *new_window;
	gchar         *filename, *uri;

	window = PROJECT_WINDOW (data);

	filename = file_sel_run ("Open Project");
	if (filename) {
		GM_Project         project_co;
		CORBA_Environment  ev;
		GM_ProjectState    state;
		gchar             *name;

		project_co = shell_get_project (window->priv->shell);

		CORBA_exception_init (&ev);

		state = GNOME_MrProject_Project_getState (project_co, &ev);
		if (BONOBO_EX (&ev)) {
			g_log_exception (&ev, "");

			CORBA_exception_free (&ev);
			g_free (filename);
			return;
		}

		uri = gnome_vfs_get_uri_from_local_path (filename);
		g_free (filename);
		
		if (state == GNOME_MrProject_PROJECT_STATE_EMPTY) {
			/* The project we already have is empty,
			 * use that to load the new project into.
			 */
			GNOME_MrProject_Project_load (project_co, uri, &ev);

			if (BONOBO_EX (&ev)) {
				g_log_exception (&ev, _("Could not open project."));

				CORBA_exception_free (&ev);
				g_free (uri);
				return;
			}

			open_projects = g_slist_prepend (open_projects, project_co);
		} else {
			/* Ok, we already have a project that is being used.
			 * Open a new window for our new project.
			 */
			new_window = (ProjectWindow *) project_window_new (uri);
			if (new_window) {
				gtk_widget_show_all (GTK_WIDGET (new_window));
			
				name = get_project_name (new_window);
				gtk_window_set_title (GTK_WINDOW (new_window), name);
			}
		}
		CORBA_exception_free (&ev);
		g_free (uri);
	}
}

static void
save_as_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ProjectWindow *window;
	gchar         *filename, *uri;

	window = PROJECT_WINDOW (data);
	
	filename = file_sel_run (_("Save Project as"));
	if (filename) {
		GM_Project         project_co;
		CORBA_Environment  ev;
		gchar             *tmp;

		project_co = shell_get_project (window->priv->shell);

		if (strstr (filename, ".mrproject") == NULL) {
			tmp = g_strconcat (filename, ".mrproject", NULL);
			g_free (filename);
			filename = tmp;
		}

		uri = gnome_vfs_get_uri_from_local_path (filename);
		g_free (filename);
		
		CORBA_exception_init (&ev);
		GNOME_MrProject_Project_save (project_co, uri, &ev);
		if (BONOBO_EX (&ev)) {
			g_log_exception (&ev, "");
		}

		g_free (uri);
		CORBA_exception_free (&ev);
	}
}

static void
save_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ProjectWindow     *window;
	GM_Project         project_co;
	CORBA_Environment  ev;
	CORBA_char        *uri;

	window = PROJECT_WINDOW (data);
	project_co = shell_get_project (window->priv->shell);
	
	CORBA_exception_init (&ev);

	uri = GNOME_MrProject_Project_getURI (project_co, &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not get URI (%s).", 
			   bonobo_exception_get_text (&ev));
		CORBA_exception_free (&ev);
		return;
	}

	if (uri == CORBA_OBJECT_NIL || strlen (uri) == 0) {
		/* Dispatch to save as. */
		save_as_cb (component, data, cname); /* FIXME: Ugly. */
	} else {
		GNOME_MrProject_Project_save (project_co, uri, &ev);
		if (BONOBO_EX (&ev)) {
			g_warning ("Could not save (%s).",
				   bonobo_exception_get_text (&ev));
		}
	}

	CORBA_free (uri);
	CORBA_exception_free (&ev);
}

static void
print_setup_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
}

static void
print_preview_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
}

static void
revert_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ProjectWindow     *window;
	GM_Project         project_co;
	CORBA_char        *uri = NULL;
	CORBA_Environment  ev;

	window = PROJECT_WINDOW (data);
	
	CORBA_exception_init (&ev);

	project_co = shell_get_project (window->priv->shell);

	/* Get the uri of the current project. */
	uri = GNOME_MrProject_Project_getURI (project_co, &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not get URI (%s).", 
			   bonobo_exception_get_text (&ev));

		goto revert_error;
	}

	/* Make sure that that the name is semi-valid and load it. */
	if (uri != CORBA_OBJECT_NIL && strlen (uri) > 0) {
		shell_open (window->priv->shell, uri, &ev);

		if (BONOBO_EX (&ev)) {
			g_log_exception (&ev, "Reverting file");
			goto revert_error;
		}
	}

 revert_error:	
	CORBA_exception_free (&ev);
	if (uri) {
		CORBA_free(uri);
	}
}

static void
properties_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ProjectWindow *window;
	GM_Shell       shell_co;

	window = PROJECT_WINDOW (data);
	shell_co = BONOBO_OBJREF (window->priv->shell);
	
	project_properties_dialog_run (data, shell_co);
}

static void
close_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ProjectWindow *window;

	window = PROJECT_WINDOW (data);
	
	project_window_really_exit (window);
}

static void
exit_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ProjectWindow *window;

	window = PROJECT_WINDOW (data);

	project_window_exit (window);
}

static void
undo_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
}

static void
redo_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
}

static void
preferences_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	preferences_dialog_run (data);
}

static void
about_cb (BonoboUIComponent *component, gpointer data, const char *cname)
{
	ProjectWindow *window;
	GtkWidget     *about, *href, *hbox;

	const gchar *authors[] = {
		"Richard Hult <rhult@codefactory.se>",
		"Mikael Hallendal <micke@codefactory.se>",
		"Thomas Nyberg <thomas@codefactory.se>",
		"Anders Carlsson <andersca@codefactory.se>",
		NULL
 	};

 	window = PROJECT_WINDOW (data);
	
 	about = gnome_about_new ("CodeFactory MrProject", VERSION,
				 _("Copyright (C) 2000-2001 CodeFactory AB"),
 				 authors,
 				 _("A project management application for the GNOME Desktop Environment"),
				 MRPROJECT_IMAGEDIR "cflogo_bw_67x87.png");

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (about)->vbox), hbox, FALSE, FALSE, 0);
	
	href = gnome_href_new ("http://mrproject.codefactory.se", _("The MrProject Homepage"));
	gtk_box_pack_start (GTK_BOX (hbox), href, TRUE, TRUE, 0);

	href= gnome_href_new ("http://mrproject.codefactory.se/contribute.php", _("Contribute to MrProject"));
	gtk_box_pack_start (GTK_BOX (hbox), href, TRUE, TRUE, 0);
	
	gnome_dialog_set_parent (GNOME_DIALOG (about), GTK_WINDOW (window));
 	gnome_dialog_set_close (GNOME_DIALOG (about), TRUE);

 	gtk_widget_show_all (about);
}

static BonoboUIVerb verbs[] = {
	BONOBO_UI_VERB ("FileNew",		new_cb),
	BONOBO_UI_VERB ("FileNewDetails",	new_details_cb),
	BONOBO_UI_VERB ("FileOpen",		open_cb),
	BONOBO_UI_VERB ("FileSave",		save_cb),
	BONOBO_UI_VERB ("FileSaveAs",		save_as_cb),
	BONOBO_UI_VERB ("FileRevert",		revert_cb),
	BONOBO_UI_VERB ("FileProperties",	properties_cb),
	BONOBO_UI_VERB ("FilePrintSetup",	print_setup_cb),
	BONOBO_UI_VERB ("FilePrint",		print_preview_cb),
	BONOBO_UI_VERB ("FilePrintPreview",	print_preview_cb),
	BONOBO_UI_VERB ("FileClose",		close_cb),
	BONOBO_UI_VERB ("FileExit",		exit_cb),

	BONOBO_UI_VERB ("EditUndo",		undo_cb),
	BONOBO_UI_VERB ("EditRedo",		redo_cb),

	BONOBO_UI_VERB ("PreferencesEditPreferences",	preferences_cb),

	BONOBO_UI_VERB ("HelpAbout",		about_cb),

	BONOBO_UI_VERB_END
};

static void
project_window_init (GtkObject *object)
{
	ProjectWindow *window = PROJECT_WINDOW (object);
	window->priv = g_new0 (ProjectWindowPriv, 1);
}

static void
project_window_class_init (GtkObjectClass *object_class)
{
	object_class->destroy = project_window_destroy;
}

static void
pw_shortcut_bar_item_selected (GtkWidget     *widget,
			       GdkEvent      *event,
			       gint           item_num,
			       ProjectWindow *window)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (window->priv->notebook), item_num);
}

/* FIXME: Talk to gal maintainers about making this a public function? */
static void
e_shortcut_bar_set_canvas_style (GtkWidget *canvas)
{
        GtkRcStyle *rc_style;

        rc_style = gtk_rc_style_new ();

        rc_style->color_flags[GTK_STATE_NORMAL] = GTK_RC_FG | GTK_RC_BG;
        rc_style->fg[GTK_STATE_NORMAL].red   = 65535;
        rc_style->fg[GTK_STATE_NORMAL].green = 65535;
        rc_style->fg[GTK_STATE_NORMAL].blue  = 65535;

        rc_style->bg[GTK_STATE_NORMAL].red   = 32512;
        rc_style->bg[GTK_STATE_NORMAL].green = 32512;
        rc_style->bg[GTK_STATE_NORMAL].blue  = 32512;

        gtk_widget_modify_style (GTK_WIDGET (canvas), rc_style);
        gtk_rc_style_unref (rc_style);
}

static void
notebook_switch_page_cb (GtkNotebook     *notebook,
			 GtkNotebookPage *page,
			 gint             page_num,
			 ProjectWindow   *window)
{
	BonoboUIEngine *engine;
	gchar          *id;

	engine = bonobo_window_get_ui_engine (BONOBO_WINDOW (window));       
	bonobo_ui_engine_freeze (engine);
	
	bonobo_ui_component_set_status (window->priv->ui_component,
					" ",
					NULL);

	id = gtk_object_get_data (GTK_OBJECT (page->child), "control_id");
	if (id) {
		shell_activate_control (window->priv->shell, id);
	}

	bonobo_ui_engine_thaw (engine);
}

GtkWidget *
project_window_new (const gchar *uri)
{
	CORBA_Environment  ev;
	ProjectWindow     *window;
	ProjectWindowPriv *priv;
	BonoboUIContainer *ui_container;
	GtkWidget         *ret;
	gint               sx, sy;
	gchar             *title;
	
	window = gtk_type_new (PROJECT_TYPE_WINDOW);

	priv = window->priv;

	ui_container = bonobo_ui_container_new ();

	/* Create the main window. */
	ret = bonobo_window_construct (BONOBO_WINDOW (window), "MrProject", "MrProject");

	gtk_signal_connect (GTK_OBJECT (ret),
			    "delete_event",
			    GTK_SIGNAL_FUNC (project_window_delete_event),
			    NULL);

	bonobo_ui_engine_config_set_path (
		bonobo_window_get_ui_engine (BONOBO_WINDOW (ret)),
		"/MrProject/UIConf/kvps");

	gtk_window_set_policy (GTK_WINDOW (ret), TRUE, TRUE, FALSE);
	title = window_title_from_name (NULL);
	gtk_window_set_title (GTK_WINDOW (ret), title);
	g_free (title);
	
	/* Set the default size of the window window. */
	sx = MAX (gdk_screen_width () - 64, 600);
	sy = MAX (gdk_screen_height () - 64, 480);
	sx = MIN (sx, 1024);
	sy = MIN (sy, 768);
	sx = (sx * 3) / 4;
	sy = (sy * 3) / 4;
	gtk_window_set_default_size (GTK_WINDOW (ret), sx, sy);

	bonobo_ui_container_set_win (ui_container, BONOBO_WINDOW (window));
	
	priv->ui_component = bonobo_ui_component_new ("MrProject");
	bonobo_ui_component_set_container (priv->ui_component,
					   BONOBO_OBJREF (ui_container));

	bonobo_ui_component_freeze (priv->ui_component, NULL);

	bonobo_ui_util_set_ui (priv->ui_component,
			       GNOME_DATADIR,
			       "GNOME_MrProject_Client.ui",
			       "mrproject");

	bonobo_ui_component_add_verb_list_with_data (priv->ui_component,
						     verbs,
						     window);

	set_pixmap (priv->ui_component, "/Toolbar/ToolSave", "24_save.png");
	set_pixmap (priv->ui_component, "/Toolbar/ToolOpen", "24_open.png");
	set_pixmap (priv->ui_component, "/Toolbar/ToolNew", "24_new.png");

	set_pixmap (priv->ui_component, "/menu/File/FileNew", "16_new.png");
	set_pixmap (priv->ui_component, "/menu/File/FileSave", "16_save.png");
	set_pixmap (priv->ui_component, "/menu/File/FileSaveAs", "16_save_as.png");
	set_pixmap (priv->ui_component, "/menu/File/FileRevert", "16_revert.png");
	set_pixmap (priv->ui_component, "/menu/File/FilePrint", "16_print.png");

	set_pixmap (priv->ui_component, "/menu/Edit/EditCut", "16_cut.png");
	set_pixmap (priv->ui_component, "/menu/Edit/EditCopy", "16_copy.png");
	set_pixmap (priv->ui_component, "/menu/Edit/EditPaste", "16_paste.png");

	set_pixmap (priv->ui_component, "/menu/Preferences/PreferencesEditPreferences", "16_properties.png");
	set_pixmap (priv->ui_component, "/menu/Preferences/BonoboCustomize", "16_properties.png");
	
	priv->paned = mr_hpaned_new ();
	
	priv->notebook = gtk_notebook_new ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (priv->notebook), FALSE);
	gtk_signal_connect (GTK_OBJECT (priv->notebook),
			    "switch-page",
			    notebook_switch_page_cb,
			    window);

	priv->shortcut_bar = e_icon_bar_new ();
	gtk_signal_connect (GTK_OBJECT (priv->shortcut_bar),
			    "item_selected",
			    GTK_SIGNAL_FUNC (pw_shortcut_bar_item_selected),
			    window);
	
	e_shortcut_bar_set_canvas_style (priv->shortcut_bar);
	gtk_widget_set_usize (priv->shortcut_bar, 88, -1);
	
	e_paned_add1 (E_PANED (priv->paned), priv->shortcut_bar);
	e_paned_add2 (E_PANED (priv->paned), priv->notebook);
	e_paned_set_position (E_PANED (priv->paned), 86);

	bonobo_window_set_contents (BONOBO_WINDOW (ret), priv->paned);

	priv->shell = shell_new (window, ui_container);

	CORBA_exception_init (&ev);

	shell_open (priv->shell, uri, &ev);
	if (BONOBO_EX (&ev)) {
		g_log_exception (&ev, _("Could not open project."));
		gtk_object_destroy (GTK_OBJECT (ret));
		return NULL;
	}

	/* This is for keeping open projects to be able to save them
	 * when we crash. See main.c.
	 */
	open_projects = g_slist_prepend (open_projects,
					 shell_get_project (priv->shell));

	/* FIXME: eek, fix this hack. */
	get_project_range (window);

	add_shell_listener (window);

	setup_progress_bar (window);
	
	bonobo_ui_component_thaw (priv->ui_component, NULL);	
	
	CORBA_exception_free (&ev);
	
	window_list = g_list_append (window_list, ret);

	gtk_widget_show_all (ret);
	
	return ret;
}

void
project_window_add_control (ProjectWindow  *window,
			    GtkWidget      *widget,
			    gchar          *id,
			    GdkPixbuf      *pixbuf,
			    const gchar    *title)
{
	GtkWidget *page;
	GList     *children;
	gchar     *str;

	gtk_notebook_append_page (GTK_NOTEBOOK (window->priv->notebook),
				  widget, gtk_label_new (title));

	children = gtk_container_children (GTK_CONTAINER (window->priv->notebook));
	page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (window->priv->notebook),
					  g_list_length (children) - 1);

	gtk_object_set_data (GTK_OBJECT (page), "control_id", id);

	str = e_utf8_from_locale_string (title);
	e_icon_bar_add_item (E_ICON_BAR (window->priv->shortcut_bar),
			     pixbuf, 
			     str,
			     -1);
	g_free (str);
}

static gboolean
project_window_delete_event (ProjectWindow *window, gpointer user_data)
{
	if (project_window_really_exit (window)) {
		return FALSE;
	}

	return TRUE;
}

static void
project_window_exit (ProjectWindow *window)
{
	GList *list, *l;

	list = g_list_copy (window_list);
	
	for (l = list; l; l = l->next) {
		project_window_really_exit (l->data);
	}

	g_list_free (list);
}

static gchar *
window_title_from_name (const gchar *str)
{
	if (str && strlen (str) > 0) {
		return g_strdup_printf ("%s - CodeFactory MrProject", str);
	} else {
		return g_strdup ("CodeFactory MrProject");
	}
}

static gchar *
get_project_name (ProjectWindow *window)
{
	GM_Project  project_co;
	gchar      *str, *title;

	project_co = shell_get_project (window->priv->shell);
	if (project_co == CORBA_OBJECT_NIL) {
		return NULL;
	}

	str = bonobo_property_bag_client_get_value_string (project_co,
							   "Name",
							   NULL);
	title = window_title_from_name (str);
	g_free (str);
	
	return title;
}

#define IS_EVENT(event_name, str) (!strncmp (event_name, str, sizeof (str)-1))

static void
listener_callback (BonoboListener    *listener,
		   char              *event_name, 
		   CORBA_any         *any,
		   CORBA_Environment *ev,
		   gpointer           user_data)
{
	ProjectWindow *window;
	gchar         *subtype, *kind;

	window = PROJECT_WINDOW (user_data);

	subtype = bonobo_event_subtype (event_name);
	kind = bonobo_event_kind (event_name);

	if (IS_EVENT (event_name, "Bonobo/Property:")) {
		if (IS_EVENT (kind, "change")) {
			if (!strcmp (subtype, "Name")) {
				gchar *str, *title;

				str = BONOBO_ARG_GET_STRING (any);
				title = window_title_from_name (str);
				gtk_window_set_title (GTK_WINDOW (window), title);
				g_free (title);
			}
		}
	} else {
		g_print ("ProjectWindow: got unhandled event: %s\n", event_name);
	}

	g_free (subtype);
	g_free (kind);
}

static void
add_shell_listener (ProjectWindow *window)
{
	ProjectWindowPriv  *priv;
	CORBA_Environment   ev;
	GM_Shell            shell_co;
	Bonobo_EventSource  es;
	BonoboListener     *listener;

	priv = window->priv;
	
	CORBA_exception_init (&ev);

	shell_co = BONOBO_OBJREF (priv->shell);
	es = GNOME_MrProject_Shell_getProxyEventSource (shell_co, &ev);
	if (BONOBO_EX (&ev) || !es) {
		CORBA_exception_free (&ev);
		return;
	}
	
	/* Listen to events from the shell. */
	listener = bonobo_listener_new (listener_callback, window);
	priv->listener_id = Bonobo_EventSource_addListenerWithMask (
		es,
		BONOBO_OBJREF (listener),
		"Bonobo/Property:change:Start,"	
		"Bonobo/Property:change:Finish,"
		"Bonobo/Property:change:Name",
		&ev);
	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
	}

	bonobo_object_release_unref (es, NULL);
	bonobo_object_unref (BONOBO_OBJECT (listener));

	CORBA_exception_free (&ev);
}

static void
remove_shell_listener (ProjectWindow *window)
{
	ProjectWindowPriv  *priv;
	CORBA_Environment   ev;
	Bonobo_EventSource  es;

	g_return_if_fail (window != NULL);
	g_return_if_fail (IS_PROJECT_WINDOW (window));

	priv = window->priv;

	if (!priv->listener_id) {
		return;
	}

	CORBA_exception_init (&ev);

	es = GNOME_MrProject_Shell_getProxyEventSource (
		BONOBO_OBJREF (priv->shell), &ev);
	if (BONOBO_EX (&ev) || !es) {
		CORBA_exception_free (&ev);
		return;
	}

	Bonobo_EventSource_removeListener (
		es,
		priv->listener_id,
		&ev);
	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
	}

	priv->listener_id = 0;
	
	bonobo_object_release_unref (es, NULL);

	CORBA_exception_free (&ev);
}

static void
get_project_range (ProjectWindow *window)
{
	GM_Project        project_co;
	CORBA_Environment ev;
	time_t            start, finish;

	project_co = shell_get_project (window->priv->shell);
	if (!project_co) {
		return;
	}

	CORBA_exception_init (&ev);
	
	start = bonobo_property_bag_client_get_value_glong (
		project_co, "Start", &ev);
	if (BONOBO_EX (&ev)) {
		start = -1;
		CORBA_exception_free (&ev);
	}		

	finish = bonobo_property_bag_client_get_value_glong (
		project_co, "Finish", &ev);
	if (BONOBO_EX (&ev)) {
		finish = -1;
		CORBA_exception_free (&ev);
	}		

	shell_set_range (window->priv->shell, start, finish);
	CORBA_exception_free (&ev);
}

/* Defines for the quit confirmation dialog. */
#define QUIT_SAVE_AND_QUIT 0
#define QUIT_QUIT          1
#define QUIT_DONT_QUIT     2

static gboolean
project_window_really_exit (ProjectWindow *window)
{
	GM_Project         project_co;
	CORBA_Environment  ev;
	GM_ProjectState    state;
	GtkWidget         *box;
	gint               result;

	project_co = shell_get_project (window->priv->shell);
	CORBA_exception_init (&ev);
	
	state = GNOME_MrProject_Project_getState (project_co, &ev);
	if (BONOBO_EX (&ev)) {
		g_log_exception (&ev, "");
		CORBA_exception_free (&ev);
		result = QUIT_QUIT;
	}
	else if (state == GNOME_MrProject_PROJECT_STATE_SAVED ||
		 state == GNOME_MrProject_PROJECT_STATE_EMPTY) {
		result = QUIT_QUIT;
	}
	else {
		box = gnome_message_box_new (_("You have unsaved information.\n"
					       "Do you really want to quit?"),
					     GNOME_MESSAGE_BOX_WARNING,
					     _("Save and quit"),
					     GNOME_STOCK_PIXMAP_QUIT,
					     _("Don't quit"),
					     NULL);
		
		gtk_window_set_wmclass (GTK_WINDOW (box), "message-box", "MrProject");
		gtk_window_set_modal (GTK_WINDOW (box), TRUE);
		
		result = gnome_dialog_run (GNOME_DIALOG (box));
	}
	
	switch (result) {
	case QUIT_SAVE_AND_QUIT:
		save_cb (NULL, window, NULL);
		
		/* Fall through. */
		
	case QUIT_QUIT:
		gtk_object_destroy (GTK_OBJECT (window));
		return TRUE;
		break;
		
	case QUIT_DONT_QUIT:
	default:
		return FALSE;
		break;
		
	}
}

Shell *
project_window_get_shell (ProjectWindow *window)
{
	return window->priv->shell;
}

static void
setup_progress_bar (ProjectWindow *window)
{
	GtkWidget     *hbox, *progress;
	BonoboControl *control;

	hbox = gtk_hbox_new (FALSE, 0);
	
	progress = gtk_progress_bar_new ();

	gtk_box_pack_start (GTK_BOX (hbox), gtk_label_new (" "), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), progress, TRUE, TRUE, 0);

	gtk_progress_bar_set_orientation (
		GTK_PROGRESS_BAR (progress), GTK_PROGRESS_LEFT_TO_RIGHT);
	gtk_progress_bar_set_bar_style (
		GTK_PROGRESS_BAR (progress), GTK_PROGRESS_CONTINUOUS);

	window->priv->progress_bar = progress;

	gtk_widget_show_all (hbox);

	control = bonobo_control_new (hbox);
	g_return_if_fail (control != NULL);

	bonobo_ui_component_object_set (
		window->priv->ui_component,
		"/status/Progress",
		BONOBO_OBJREF (control),
		NULL);
}

static void
set_pixmap (BonoboUIComponent *uic,
	    const char        *xml_path,
	    const char        *icon)
{
	char      *path;
	GdkPixbuf *pixbuf;

	path = g_concat_dir_and_file (MRPROJECT_IMAGEDIR, icon);

	pixbuf = gdk_pixbuf_new_from_file (path);
	if (pixbuf == NULL) {
		g_warning ("Cannot load image -- %s", path);
		g_free (path);
		return;
	}

	bonobo_ui_util_set_pixbuf (uic, xml_path, pixbuf);

	gdk_pixbuf_unref (pixbuf);

	g_free (path);
}
