/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*----------------------------------------------------------------------*
 * Originally written:
 * 2002   Alexis <materm@tele2.fr>
 *
 *----------------------------------------------------------------------*/

/* $Id: notebook2.c,v 1.19 2004/08/15 16:16:39 alexis Exp $ 
 */

#include "rxvt.h"

#ifdef HAVE_LIBXPM

#include <X11/xpm.h>
#include "close_term.xpm"
#include "term.xpm"
#include "right.xpm"
#include "left.xpm"

#else

#include "close_term.xbm"
#include "term.xbm"
#include "right.xbm"
#include "left.xbm"

#endif /* HAVE_LIBXPM */

#ifdef DEBUG
#define DEBUG_NOT_ALEXIS 1
#else 
#define DEBUG_NOT_ALEXIS 0
#endif

#if DEBUG_NOT_ALEXIS
#define DN_ALEXIS(d,x) if(d <= DEBUG_NOT_ALEXIS) fprintf x
#else
#define DN_ALEXIS(d,x)
#endif

#define THUMB_SPACE 1    /* space between button/thumb */
#define THUMB_WIDTH 60
#define THUMB_HEIGHT 21
#define BUTTON_SIZE 22
#define BUTTON_SPACE 5
#define THUMB_PAD (TOP_SPACE - THUMB_HEIGHT -2)
#define THUMBS_PLACE (width-(4*(BUTTON_SIZE+BUTTON_SPACE)+THUMB_SPACE))

extern Status XAllocColorNew(Display *dpy, Colormap cmap, XColor *xcolor);
static GC whiteGC;
static GC light_greyGC; /* for the active vt*/
static GC dark_greyGC; /* unactive vt*/
static GC dark_grey2GC; /* grey for separation bettween thumbs */

enum {XPM_TERM,XPM_CLOSE,XPM_LEFT,XPM_RIGHT,NB_XPM};


#ifdef HAVE_LIBXPM
static char **xpm_name[] = {
  mini_term_xpm,close_file_xpm,
  bookmark_prev_xpm,bookmark_next_xpm
};
#else
static char *xpm_name[] = {
  term_bits,close_bits,
  left_bits,right_bits
};
#endif
  
static XImage *img[NB_XPM];
static XImage *img_mask[NB_XPM];
  
extern void create_vt();
extern char **cmd_argv;

// width of the thumb bar
static int width;
static int default_tab_title_size;

// widths of the thumbs
static int thumbs_widths[MAX_PAGES];

int width_beetween(int a,int b) {
  int i,w=0;
  
  for(i=a; i <= b; i++) {
    w += thumbs_widths[i] + THUMB_SPACE;
  }
  
  return w;
}
#ifdef DRAW_THUMBBAR

void blank_between_right_and_buttons() {
  int x=width_beetween(TermWin.left_page,TermWin.right_page);
  XFillRectangle(Xdisplay,TermWin.thumb_bar,light_greyGC,
    x,THUMB_PAD,THUMBS_PLACE-x,THUMB_HEIGHT);
}

void compute_size_of_thumbs(void) {
  int i;
  default_tab_title_size = XTextWidth(
    TermWin.font,
    rs_default_tab_title,
    strlen(rs_default_tab_title)) + 20;
  for(i=0; i < MAX_PAGES; i++) {
    thumbs_widths[i] = XTextWidth(TermWin.font,TermWin.vts[i].tab_title,
      strlen(TermWin.vts[i].tab_title)) + 20;
    DN_ALEXIS(2,(stderr,"width %d [%s]: %d\n",i,TermWin.vts[i].tab_title,thumbs_widths[i]));
  }
}
#endif

void init_notebook(void) {
  TermWin.last_page = -1;
  TermWin.active_page = -1;
  TermWin.left_page = 0;
  TermWin.right_page = -1;
  TermWin.last_active_page = 0;
}

#ifdef DRAW_THUMBBAR
void set_thumb_num(int page,int active) {
  // x is the index of visible thumbs : start at 0
  // because TermWin.left_page <= page <= TermWin.right_page <= TermWin.last_page
  int x=page-TermWin.left_page;
  int i;
  char title[TITLE_TAB_MAX+1];
  GC *gc;
  
  DN_ALEXIS(1,(stderr,"set_thumb_num(%d,%s)\n",page, active ? "active" : "unactive"));
  if(active) {
    gc = &light_greyGC;
  } else {
    gc = &dark_greyGC;
  }
  x = THUMB_SPACE;
  for(i=TermWin.left_page; i < page; i++) {
    x += thumbs_widths[i] + THUMB_SPACE;
  }
  
  /* rectangle */
  XFillRectangle(Xdisplay,TermWin.thumb_bar,*gc,
    x,
    THUMB_PAD,thumbs_widths[page],THUMB_HEIGHT);

  /* top horitzontal line */
  XDrawLine(Xdisplay,TermWin.thumb_bar,whiteGC,
    x,THUMB_PAD,
    x+thumbs_widths[page]-1,THUMB_PAD);
  
  if(active) {
    /* left vertical line */
    XDrawLine(Xdisplay,TermWin.thumb_bar,whiteGC,
      x,THUMB_PAD,
      x,THUMB_PAD+THUMB_HEIGHT-1);
    
    /* bottom horitzontal line */
    XDrawLine(Xdisplay,TermWin.thumb_bar,light_greyGC,
      x,THUMB_PAD+THUMB_HEIGHT,
      x+thumbs_widths[page]+THUMB_SPACE,THUMB_PAD+THUMB_HEIGHT);
  } else {
    /* bottom horizontal line */
    XDrawLine(Xdisplay,TermWin.thumb_bar,whiteGC,
      x,THUMB_PAD+THUMB_HEIGHT,
      x+thumbs_widths[page]+THUMB_SPACE,THUMB_PAD+THUMB_HEIGHT);
  }
    
  /* right vertical line */
  XDrawLine(Xdisplay,TermWin.thumb_bar,dark_grey2GC,
    x+thumbs_widths[page],THUMB_PAD,
    x+thumbs_widths[page],THUMB_PAD+THUMB_HEIGHT-1);

  /* draw the text */

  sprintf(title,"%s",TermWin.vts[page].tab_title);
  XDrawString(Xdisplay, TermWin.thumb_bar,
    TermWin.thumbGC,x+10,
    20,title,strlen(title));
    
}
#endif

/* x is the new left_page
 */
void page_left_shift(int x) {
  
  DN_ALEXIS(1,(stderr,"page_left_shift(%d)\n",x));
  if( x >= 0 ) {
    int i;
    int space=THUMB_SPACE;
    
    TermWin.left_page = x;
    for(i=TermWin.left_page; space + thumbs_widths[i] + THUMB_SPACE < THUMBS_PLACE && i <= TermWin.last_page; i++) {
      space += thumbs_widths[i] + THUMB_SPACE;
    }
    
    if( space + thumbs_widths[i] + THUMB_SPACE >= THUMBS_PLACE ) {
      TermWin.right_page = max(0,i-1); // to avoid -1
    } else {
      TermWin.right_page = TermWin.last_page;
    }
    DN_ALEXIS(1,(stderr,"page_left_shift : [%d,%d,%d]\n",
      TermWin.left_page,TermWin.active_page,TermWin.right_page));
#ifdef DRAW_THUMBBAR
    for(i=TermWin.left_page; i <= TermWin.right_page; i++) {
      if(i == TermWin.active_page) {
        set_thumb_num(i,1);
      } else {
        set_thumb_num(i,0);
      }
    }
    blank_between_right_and_buttons();
#endif
  }
}

/* x the the new right page
 */
void page_right_shift(int x) {
  DN_ALEXIS(1,(stderr,"page_right_shift(%d)\n",x));
  if( x <= TermWin.last_page) {
    int i;
    int space=THUMB_SPACE;
    
    TermWin.right_page = x;
    for(i=TermWin.right_page; space + thumbs_widths[i] + THUMB_SPACE < THUMBS_PLACE && i > 0; i--) {
      space += thumbs_widths[i] + THUMB_SPACE;
    }
    
    if( space  + thumbs_widths[i] + THUMB_SPACE >= THUMBS_PLACE) {
      TermWin.left_page = min(i+1,TermWin.last_page);
    } else {
      TermWin.left_page = 0;
    }
    
    DN_ALEXIS(1,(stderr,"page_right_shift : [%d,%d,%d]\n",
      TermWin.left_page,TermWin.active_page,TermWin.right_page));
#ifdef DRAW_THUMBBAR
    for(i=TermWin.left_page; i <= TermWin.right_page; i++) {
      if(i == TermWin.active_page) {
        set_thumb_num(i,1);
      } else {
        set_thumb_num(i,0);
      }
    }
    blank_between_right_and_buttons();
#endif
  }
}

void append_page(int index) {
  /* add keyboard input to the main window
   */
  static int has_keypressed_mask = 0;
  if( !has_keypressed_mask ) {
    XWindowAttributes attr;
    XGetWindowAttributes(Xdisplay,TermWin.parent,&attr);
    if( ! (attr.all_event_masks & KeyPressMask) ) {
      has_keypressed_mask=1;
      XSelectInput(Xdisplay, TermWin.parent,
		    KeyPressMask | FocusChangeMask |
		    StructureNotifyMask | VisibilityChangeMask);
    }
  }

  if( TermWin.last_page < MAX_PAGES-1) {
    int x; /* the top left corner of the new thumb */
    int i;
    int tmp;
    
    DN_ALEXIS(1,(stderr,"append_page(%d)\n",index));
    
    // indicate that we add a new thumb
    TermWin.last_page++;

#ifdef DRAW_THUMBBAR
//    width = (2 * TermWin_internalBorder)+TermWin.width;
    x = THUMB_SPACE;
    for(i=TermWin.left_page; i < TermWin.last_page; i++) {
      x += thumbs_widths[i] + THUMB_SPACE;
    }
    //x = TermWin.last_page*(THUMB_WIDTH+THUMB_SPACE)+THUMB_SPACE;
    
    DN_ALEXIS(2,(stderr,"x=%d,TermWin.width=%d\n",x,TermWin.width));
    if( x+thumbs_widths[TermWin.last_page] > width-(4*(BUTTON_SIZE+BUTTON_SPACE)+THUMB_SPACE) ) {
      // need to shift left the thumbs
      // just compute the new value of TermWin.left_page
    
      DN_ALEXIS(1,(stderr,"shift on left\n"));
      tmp=0;
      // how many thumb do we shift on left ?
      for(i=TermWin.left_page; i < TermWin.last_page ; i++) {
        tmp += thumbs_widths[i];
        if( x + thumbs_widths[TermWin.last_page] - tmp <= width-(4*(BUTTON_SIZE+BUTTON_SPACE)+THUMB_SPACE) ) {
          tmp = width -(4*(BUTTON_SIZE+BUTTON_SPACE)+THUMB_SPACE) - (x + thumbs_widths[TermWin.last_page] - tmp);
          //printf("NOW, left_page = %d, space beetween thumbs and buttons = %d\n",i,tmp);
          break;
        }
      }
      // i is the most left page to shift; i+1 is the new TermWin.left_page !
      TermWin.left_page = i+1;
      //TermWin.left_page = (TermWin.last_page-1) - (TermWin.right_page - TermWin.left_page) + 1;
      x = width_beetween(TermWin.left_page,TermWin.last_page);
      tmp = width - (4*(BUTTON_SIZE+BUTTON_SPACE)+THUMB_SPACE) - x;
      XFillRectangle(Xdisplay,TermWin.thumb_bar,light_greyGC,
        x,THUMB_PAD,tmp,THUMB_HEIGHT
      );
      
    }
#endif    
    TermWin.right_page = TermWin.active_page = TermWin.last_page;

    {
      /* number the page appropriately */
      char *tab_title;
      free(TermWin.vts[TermWin.active_page].tab_title);
      tab_title=strdup(rs_default_tab_title);
      tab_title[strlen(tab_title)-1]=0;
      sprintf(tab_title, "%s%d",tab_title,TermWin.active_page%10);
      TermWin.vts[TermWin.active_page].tab_title = tab_title;
    }

#ifdef DRAW_THUMBBAR
    /* active the new active page */
    /* do a left shift (if necessary) and desactivate the thumb */
    for(i=TermWin.left_page; i < TermWin.right_page; i++) {
      set_thumb_num(i,(i==TermWin.active_page) ? 1:0);
    }
#endif
    
    DN_ALEXIS(1,(stderr,"before create_vt (%s)\n",__FILE__));
    create_vt();

    DN_ALEXIS(1,(stderr,"last page is %d\n",TermWin.last_page));
    f_launch_command(TermWin.last_page,cmd_argv);
    scrollbar_show(1);
  } else {
    DN_ALEXIS(1,(stderr,"sorry, too many tabs\n"));
  }
}

/* consider page is in [ TermWin.left_page , TermWin.right_page ]
 */
void remove_thumb(int page_num) {
  
  if( page_num >= 0 && page_num <= TermWin.last_page) {
    
    int i;
    int minus;
    int x;

    // update thumbs_widths
    for(i=page_num; i < TermWin.last_page; i++) {
      thumbs_widths[i] = thumbs_widths[i+1];
    }
    for(;i < MAX_PAGES; i++) {
      thumbs_widths[i] = default_tab_title_size;
    }

    // compute TermWin.left_page and TermWin.right_page    
    if( TermWin.right_page == TermWin.last_page) {
      TermWin.last_page--;
      TermWin.right_page--;
      x=THUMB_SPACE;
      for(i=TermWin.last_page; x + thumbs_widths[i] + THUMB_SPACE < THUMBS_PLACE && i >= 0; i--) {
        x += thumbs_widths[i] + THUMB_SPACE;
      }
      TermWin.left_page = max(0,i+1);
      
    } else {
      TermWin.last_page--;
      x = width_beetween(TermWin.left_page,page_num-1);
      for(i=page_num; i <= TermWin.last_page && x + thumbs_widths[i] + THUMB_SPACE < THUMBS_PLACE; i++) {
        x += thumbs_widths[i] + THUMB_SPACE;
      }
      TermWin.right_page = min(TermWin.last_page,i-1);
      // may be we can add other thumb 
      if(TermWin.right_page == TermWin.last_page) {
        for(i=TermWin.left_page-1; i >= 0 && x + thumbs_widths[i] + THUMB_SPACE < THUMBS_PLACE; i--) {
          x += x + thumbs_widths[i] + THUMB_SPACE;
        }
        TermWin.left_page = max(0,i+1);
      }
    }
    
#ifdef DRAW_THUMBBAR 
    for(i=TermWin.left_page; i <= TermWin.right_page; i++) {
      set_thumb_num(i,(i == TermWin.active_page)? 1 : 0);
    }
    blank_between_right_and_buttons();
#endif
    DN_ALEXIS(1,(stderr,"right_page=%d\n",TermWin.right_page));
  }
}

// called by the handler of SIGCHLD
void remove_page(int page) {
  int the_end=0;
  
  if( TermWin.last_page == 0) {
    /* destroying the only vt => kill the app
     */
    the_end=1;
  } else if( page == TermWin.active_page) {
    if( page == 0) /* special */
      TermWin.active_page = 1;
    else 
      TermWin.active_page--;
    
    TermWin.vts[page].bMapped = 0;
    TermWin.vts[TermWin.active_page].bMapped = 1;
    XMapRaised(Xdisplay,TermWin.vts[TermWin.active_page].vt);
  }

  pat_destroy_command(page);
  pat_destroy_vt(page);

  if( TermWin.left_page <= page && page <= TermWin.right_page) {
    /* removing a visible vt */
    DN_ALEXIS(2,(stderr,"removing a visible vt\n"));
    remove_thumb(page);
  } else {
    TermWin.last_page--;
    if( page < TermWin.left_page ) {
      TermWin.left_page--;
      TermWin.right_page--;
    }
  }
  
  if(the_end) {
    exit(EXIT_SUCCESS);
  }
  
  if( page == 0 ) {
    TermWin.active_page = 0;
  }
 
#ifdef DRAW_THUMBBAR
  if( TermWin.left_page <= TermWin.active_page && TermWin.active_page <= TermWin.right_page) {
    set_thumb_num(TermWin.active_page,1);
  }
#endif

  DN_ALEXIS(2,(stderr,"left: %d, right: %d, last: %d, active: %d\n",
    TermWin.left_page,TermWin.right_page,TermWin.last_page,TermWin.active_page));
}

// call back appele qd on clique sur effacer un vt
// on envoie un EOF au shell, et c'est dans le sighandler
// qu'on fait le traitement.
void destroy_page(int page,pid_t pid) {
  kill(pid,9);
}

#ifdef DRAW_THUMBBAR
#define MAX_POPUP_STRING_LENGTH 15
#define MAX_MAPPED_STRING_LENGTH 10
void retitle_page(int page) {
	unsigned int width, height, x = 0, y = 0, i;
	unsigned int border_width = 4;
	XEvent report;
	GC gc = TermWin.thumbGC;

   static Window pop_win;
   char buffer[MAX_MAPPED_STRING_LENGTH];
   int bufsize=MAX_MAPPED_STRING_LENGTH;
   int start_x, start_y;
   KeySym keysym;
   XComposeStatus compose;
   int count;
   unsigned int pop_width, pop_height;
   char string[MAX_POPUP_STRING_LENGTH], *strptr;
   int length;
   int done = 0;

  x = THUMB_SPACE;
  for(i=TermWin.left_page; i < page; i++) {
    x += thumbs_widths[i] + THUMB_SPACE;
  }
  y = THUMB_PAD;
	memset(buffer, 0, bufsize);
	memset(string, 0, MAX_POPUP_STRING_LENGTH);
	strncpy(string, TermWin.vts[page].tab_title, MAX_POPUP_STRING_LENGTH -1);
	
      /* Determine pop-up box size from font information */
      pop_width = thumbs_widths[page];
      pop_height = THUMB_HEIGHT;
      pop_win = XCreateSimpleWindow(Xdisplay, TermWin.thumb_bar, x, y,
            pop_width, pop_height, 1,
            BlackPixel(Xdisplay, Xscreen),
                WhitePixel(Xdisplay, Xscreen));
      /* Calculate starting position of string in window */
      start_x = 2;
      start_y = TermWin.font->max_bounds.ascent + 2;
      XSelectInput(Xdisplay, pop_win, ExposureMask | KeyPressMask);
   XMapWindow(Xdisplay, pop_win);
   /* Get events, use first to display text and graphics */
   while (!done)  {
      int redraw_text = 0;
      XNextEvent(Xdisplay, &report);
      switch  (report.type) {
      case Expose:
         if (report.xexpose.window == pop_win)
	   redraw_text = 1;
         else /* It's the main window */
		 process_x_event(&report);
         break;
      case KeyPress:
            /* Get characters until you encounter a
              * carriage return; deal with backspaces, etc. */
            count = XLookupString(&report, buffer, bufsize,
                  &keysym, &compose);
            /* Now do the right thing with as many
             * keysyms as possible */
            if ((keysym == XK_Return) || (keysym == XK_KP_Enter)
                  || (keysym == XK_Linefeed)) {
               XUnmapWindow(Xdisplay, pop_win);
               DN_ALEXIS(1,(stderr,"title is %sd\n",string));
	       done = 1;
               break;
            }
            else if (((keysym >= XK_KP_Space) && (keysym <= XK_KP_9))
                  || ((keysym >= XK_space) && (keysym <= XK_asciitilde))) {
               if (strlen(string) + strlen (buffer) >= MAX_POPUP_STRING_LENGTH)
                  ; //XBell(Xdisplay, 100);
               else
                  strcat(string, buffer);
            }
            else if (keysym >= XK_F1 && keysym <= XK_F35 && buffer != NULL &&
                     strlen(string) + strlen(buffer) < MAX_POPUP_STRING_LENGTH)
                  strcat(string, buffer);
            else if ((keysym == XK_BackSpace) || (keysym == XK_Delete)) {
               if ((length = strlen(string)) > 0) {
                  string[length - 1] = NULL;
                  XClearWindow(Xdisplay, pop_win);
               }
               else
                  ; //XBell(Xdisplay, 100);
            }
	    redraw_text = 1;
	    buffer[1] = NULL;
            break;
      case MappingNotify:
          XRefreshKeyboardMapping(&report);
         break;
      default:
         /* All events selected by StructureNotifyMask
          * except ConfigureNotify are thrown away here,
          * since nothing is done with them */
         break;
      } /* End switch */
      if (redraw_text) {
        strptr = string;
        if (XTextWidth(TermWin.font,string,strlen(string)) > thumbs_widths[i])
        {
          while (strptr-string < strlen(string) -1 && 
                 XTextWidth(TermWin.font,strptr,strlen(strptr)) > thumbs_widths[i]) 
            strptr++;
            XClearWindow(Xdisplay, pop_win);
        }
 
        XDrawString(Xdisplay, pop_win, gc, start_x,
             start_y, strptr, strlen(strptr));
      }
   } /* End while */
   if (strlen(string)) {
     free(TermWin.vts[page].tab_title);
     TermWin.vts[page].tab_title=strdup(string);
     compute_size_of_thumbs();
     for(i=page; i <= TermWin.right_page; i++) {
       set_thumb_num(i,(i == page)? 1 : 0);
     }
     blank_between_right_and_buttons();
   }
}
#endif

void activate_page_num(int page) {
  if(page == TermWin.active_page)
  {
#ifdef DRAW_THUMBBAR
    retitle_page(page);
#endif
    return;
  }
  
  TermWin.last_active_page = TermWin.active_page;
  TermWin.vts[TermWin.last_active_page].bMapped = 0;
  TermWin.active_page = page;
  TermWin.vts[page].bMapped = 1;
  
#ifdef DRAW_THUMBBAR
  if( TermWin.left_page <= page && page <= TermWin.right_page) {
    set_thumb_num(TermWin.active_page,1);
  }
  
  if( TermWin.left_page <= TermWin.last_active_page 
    && TermWin.last_active_page <= TermWin.right_page) {
    
    set_thumb_num(TermWin.last_active_page,0);
  }
#endif
  
  XMapRaised(Xdisplay,TermWin.vts[TermWin.active_page].vt);
//  XUnmapWindow(Xdisplay,TermWin.vts[TermWin.last_active_page].vt);
  f_scr_refresh(TermWin.active_page,FAST_REFRESH);
  DN_ALEXIS(1,(stderr,"active page is %d\n",TermWin.active_page));
  
#if defined(TRANSPARENT) || defined(BACKGROUND_IMAGE)
	refresh_transparent_scrollbar();
#endif
  
}

void resize_thumb_bar(XEvent * ev) {
#ifdef DRAW_THUMBBAR
  static int old_width = 0;
  int x;
  int i;

  if(old_width == (ev->xconfigure).width )
    return;
  
  width = (ev->xconfigure).width;
  //x = (width - (4*(BUTTON_SIZE+BUTTON_SPACE)+THUMB_SPACE)) / (THUMB_WIDTH+THUMB_SPACE);
  x = width - (4*(BUTTON_SIZE+BUTTON_SPACE)+THUMB_SPACE);
  for(i=0; x > THUMB_SPACE; i++) {
    x -= thumbs_widths[i]+THUMB_SPACE;
  }
  
  DN_ALEXIS(1,(stderr,"resize_thumb_bar, x:%d\n",x));
  
  if( (ev->xconfigure).width > old_width ) {
  
    // bigger -> may be we can add thumbs
    if(TermWin.last_page >= 0) {
      TermWin.vts[TermWin.active_page].bMapped = 0;
      TermWin.left_page = 0;
      //TermWin.active_page = 0;
      TermWin.vts[TermWin.active_page].bMapped = 1;
      TermWin.right_page = min(i-1,TermWin.last_page);
    }
    
    XMoveResizeWindow(Xdisplay,TermWin.thumb_bar,0,0,(ev->xconfigure).width,TOP_SPACE);
    
  } else if ((ev->xconfigure).width < old_width) {
  
    // smaller -> may be we need to remove thumbs
    if(TermWin.last_page >= 0) {
      TermWin.vts[TermWin.active_page].bMapped = 0;
      TermWin.left_page = 0;
      //TermWin.active_page = 0;
      TermWin.vts[TermWin.active_page].bMapped = 1;
      TermWin.right_page = min(i-1,TermWin.last_page);
    }
    
    XMoveResizeWindow(Xdisplay,TermWin.thumb_bar,0,0,(ev->xconfigure).width,TOP_SPACE);
    
  }
  
  old_width = width;
#endif
}

#ifdef DRAW_THUMBBAR
void dispatch_button_press(int x,int y,pid_t pid) {
  int z;
  int but;
//  int width = (2 * TermWin_internalBorder)+TermWin.width;
  
  DN_ALEXIS(2,(stderr,"click in (%d,%d)\n",x,y));
  but = -1;
  /* let's decode where the user click
   */
  z = width-x;

  if( z < 4*(BUTTON_SIZE+BUTTON_SPACE) && (z%(BUTTON_SIZE+BUTTON_SPACE)) > BUTTON_SPACE) {
    but = z/(BUTTON_SIZE+BUTTON_SPACE);
    DN_ALEXIS(1,(stderr,"click on button %d\n",but));
    switch(but) {
    case 0 : /* right shift */
      page_right_shift(TermWin.right_page+1);
      break;
    case 1 : /* left shift */
      printf("TermWin.left_page : %d\n",TermWin.left_page);
      page_left_shift(TermWin.left_page-1);
      break;
    case 2 : /* delete the active vt */
      destroy_page(TermWin.active_page,pid);
      break;
    case 3 : /* create a new vt*/
      append_page(0);
      break;
    default :
    break;
  }

  } else if ( x < THUMBS_PLACE && TermWin.last_page >= 0) {
      int w = 0;
      int i;
      for(i=TermWin.left_page; w < x && i <= TermWin.right_page; i++) {
        w += THUMB_SPACE + thumbs_widths[i];
      }
      if(w - THUMB_SPACE >= x) {
        
        but = i-1;
        DN_ALEXIS(2,(stderr,"click on tab %d\n",but));
        activate_page_num(but);
      }
  }
}

void thumb_bar_expose() {
  int i,j,x;
  char title[TITLE_TAB_MAX+1];
  
  DN_ALEXIS(1,(stderr,"thumb_bar_expose [%d,%d,%d]\n",TermWin.left_page,TermWin.active_page,TermWin.right_page));
  
  /* draw the buttons */
  for(i=NB_XPM; i >= 1; i--) {
    XPutImage(Xdisplay,TermWin.thumb_bar,TermWin.thumbGC,img[4-i],
      0,0,
      width-(i*(BUTTON_SIZE+BUTTON_SPACE)),THUMB_PAD,
      BUTTON_SIZE,BUTTON_SIZE);

  }

  if( TermWin.last_page >= 0 ) {
    /* redraw the thumbs and the bevels
     */
    for(i=TermWin.left_page,x=THUMB_SPACE; i <= TermWin.right_page; i++,x += thumbs_widths[i-1]+THUMB_SPACE) {
      
      if( i == TermWin.active_page) {
        
        XFillRectangle(Xdisplay,TermWin.thumb_bar,light_greyGC,
          x,THUMB_PAD,thumbs_widths[i],THUMB_HEIGHT);
        
        /* left vertical line */
        XDrawLine(Xdisplay,TermWin.thumb_bar,whiteGC,
          x,THUMB_PAD,x,THUMB_PAD+THUMB_HEIGHT-1);
        
      } else {
        XFillRectangle(Xdisplay,TermWin.thumb_bar,dark_greyGC,
          x,THUMB_PAD,thumbs_widths[i],THUMB_HEIGHT);
      }
      
      /* top horizontal line*/
      XDrawLine(Xdisplay,TermWin.thumb_bar,whiteGC,
        x,THUMB_PAD,x+thumbs_widths[i]-1,THUMB_PAD);
      
      /* right vertical line */
      XDrawLine(Xdisplay,TermWin.thumb_bar,dark_grey2GC,
        x + thumbs_widths[i],THUMB_PAD,
        x + thumbs_widths[i],THUMB_PAD+THUMB_HEIGHT-1);
    }
    
    x = width_beetween(0,TermWin.active_page-1);
    /* draw a line under thumbs */
    XDrawLine(Xdisplay,TermWin.thumb_bar,whiteGC,0,
      THUMB_PAD+THUMB_HEIGHT/*TOP_SPACE-1*/,
      x+THUMB_SPACE,
      THUMB_PAD+THUMB_HEIGHT/*TOP_SPACE-1*/);
    
    XDrawLine(Xdisplay,TermWin.thumb_bar,light_greyGC,
      x+THUMB_SPACE,
      THUMB_PAD+THUMB_HEIGHT,
      x+thumbs_widths[TermWin.active_page],
      THUMB_PAD+THUMB_HEIGHT);
    
    XDrawLine(Xdisplay,TermWin.thumb_bar,whiteGC,
      x+thumbs_widths[TermWin.active_page]+THUMB_SPACE,
      THUMB_PAD+THUMB_HEIGHT/*TOP_SPACE-1*/,
      width,THUMB_PAD+THUMB_HEIGHT/*TOP_SPACE-1*/);
    
    // part for the text
    x=THUMB_SPACE;
    for(i=TermWin.left_page; i <= TermWin.right_page; i++) {
      sprintf(title,"%s",TermWin.vts[i].tab_title);
      XDrawString(Xdisplay, TermWin.thumb_bar,
        TermWin.thumbGC,x+10,
        20,title,strlen(title));
      x += THUMB_SPACE + thumbs_widths[i];
    }
  }
  
  XMapWindow(Xdisplay,TermWin.thumb_bar);
}
#endif

void create_thumb_bar(XSizeHints *szHint) {
#ifdef DRAW_THUMBBAR 
#ifdef HAVE_LIBXPM
  XpmAttributes xpm_attr;
  XpmColorSymbol xpm_color_sym = {"background",NULL,None};
#endif
  
  XColor color;
  XGCValues gcvalue;
  Pixel darkPixel;
  int i;
  int res;

  DN_ALEXIS(1,(stderr,"Building tumb_bar\n"));
  compute_size_of_thumbs();
  
  width = szHint->width;
  /* create the light_greyGC */
  color.red = 0xd300;
  color.green = 0xd300;
  color.blue = 0xdd00;
  XAllocColorNew(Xdisplay,Xcmap,&color);
  gcvalue.foreground = darkPixel = color.pixel;
  
#ifdef HAVE_LIBXPM
  // this defined the transparent color of the pixmap
  xpm_color_sym.pixel = color.pixel;
#endif
  /* create the window of the thumb bar */
  TermWin.thumb_bar = XCreateSimpleWindow(
    Xdisplay,TermWin.parent,
    0,0,szHint->width,TOP_SPACE,0,
    2,color.pixel
  );
  /* to have transparent notebook ...
   */
/*
#if defined(TRANSPARENT)
  if( Options & Opt_transparent ) {
    XSetWindowBackgroundPixmap(Xdisplay, TermWin.thumb_bar,ParentRelative);
  }
#endif
*/
    
  light_greyGC = XCreateGC(Xdisplay, TermWin.thumb_bar,GCForeground , &gcvalue);
  

  /* create the dark_grey2GC */
  color.red = 0x8300;
  color.green = 0x8300;
  color.blue = 0x9300;
  XAllocColorNew(Xdisplay,Xcmap,&color);
  gcvalue.foreground = color.pixel;

  dark_grey2GC = XCreateGC(Xdisplay, TermWin.thumb_bar,GCForeground , &gcvalue);
  
  /* create the dark_greyGC */
  color.red = 0xc100;
  color.green = 0xc100;
  color.blue = 0xcc00;
  XAllocColorNew(Xdisplay,Xcmap,&color);
  gcvalue.foreground = color.pixel;
  dark_greyGC = XCreateGC(Xdisplay, TermWin.thumb_bar,GCForeground , &gcvalue);

  XSelectInput(Xdisplay,TermWin.thumb_bar,
    ExposureMask|ButtonPressMask|ButtonReleaseMask);

  /* create the GC for the thumb window 
   */
  gcvalue.foreground = PixColors[Color_Black];
	gcvalue.background = PixColors[Color_Grey25];
  gcvalue.line_width = 1;
	TermWin.thumbGC = XCreateGC(Xdisplay, TermWin.thumb_bar,
    GCForeground | GCBackground | GCLineWidth, &gcvalue);
  XSetFont(Xdisplay, TermWin.thumbGC,TermWin.font->fid);

  gcvalue.foreground = WhitePixelOfScreen(DefaultScreenOfDisplay(Xdisplay));
  whiteGC = XCreateGC(Xdisplay, TermWin.thumb_bar,
    GCForeground , &gcvalue);
  
  /* now, create the buttons
   */
#ifdef HAVE_LIBXPM
  xpm_attr.colorsymbols = &xpm_color_sym; /* to have transparent background */
  xpm_attr.numsymbols = 1;
  xpm_attr.visual = Xvisual;
  xpm_attr.colormap = Xcmap;
  xpm_attr.depth = Xdepth;
  xpm_attr.closeness = 30000;
  xpm_attr.valuemask = XpmVisual | XpmColormap | 
    XpmDepth | XpmCloseness | XpmReturnPixels | XpmColorSymbols;
#endif
    
  for(i=0; i < NB_XPM; i++) {
#ifdef HAVE_LIBXPM
    /* create the icon for the button ... */
    res = XpmCreateImageFromData(Xdisplay,xpm_name[i],
      &img[i],&img_mask[i],&xpm_attr);
    DN_ALEXIS(2,(stderr,"XpmCreateImageFromData returns %d\n",res));
#else

    img[i] = XCreateImage(Xdisplay,Xvisual,Xdepth,
      ZPixmap,0,
      xpm_name[i],
      BUTTON_SIZE,BUTTON_SIZE,8,7);
    if(img[i] == NULL) {
      fprintf(stderr,"Null XImage\n");
    }
    
#endif
    
    /* and display it ! */
    XPutImage(Xdisplay,TermWin.thumb_bar,TermWin.thumbGC,img[i],
      0,0,
      width-((4-i)*(BUTTON_SIZE+BUTTON_SPACE)),THUMB_PAD,
      BUTTON_SIZE,BUTTON_SIZE);
    
  }
    
  XMapWindow(Xdisplay,TermWin.thumb_bar);
#endif
}

void page_shift(unsigned int direction,int x) {
  DN_ALEXIS(2,(stderr,"page_shift(%d,%d)\n",direction,x));
  switch(direction) {
    case 0 :
      page_left_shift(x);
    break;
    case 1 :
      page_right_shift(x);
    break;
    default : DN_ALEXIS(1,(stderr,"direction not supported\n"));
  }
}
