//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: audioport.cpp,v 1.1 2002/01/30 14:54:03 muse Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <unistd.h>
#include <math.h>

#include <qpopupmenu.h>
#include <errno.h>
#include "driver/audiodev.h"
#include "audioport.h"
#include "globals.h"
#include "xml.h"
#include "sf/sndfile.h"
#include "song.h"

AudioPort audioPort;

//---------------------------------------------------------
//   AudioPort
//---------------------------------------------------------

AudioPort::AudioPort()
      {
      _device  = 0;
      _state   = "not configured";
      _rwFlags = 1;
      src      = 0;
      }

//---------------------------------------------------------
//   setAudioDevice
//---------------------------------------------------------

void AudioPort::setAudioDevice(AudioDevice* dev)
      {
      if (_device)
            _device->close();
      _device = dev;
      if (_device) {
            _rwFlags &= _device->rwFlags();
            if (_rwFlags == 0)
                  _rwFlags = _device->rwFlags();
            _device->setPort(0);
            _state = _device->open(_rwFlags);
            if (_rwFlags & 2)
                  setup(5, segmentSize * channels());
            }
      else
            _state = "not configured";
      }

//---------------------------------------------------------
//   readAudio
//    ready to read from audio port
//    called from select() loop
//---------------------------------------------------------

void AudioPort::read()
      {
      AudioDevice* dev = device();
      if (dev == 0) {
            printf("AudioPort record: no DEVICE\n");
            return;
            }
      short int buffer[segmentSize * channels()];

      int n  = dev->read((unsigned char*)buffer, segmentSize);
      if (n != segmentSize) {
            printf("Audio Read failed: %d (expected %d): %s\n",
              n, segmentSize, strerror(errno));
            return;
            }
      float* data = getFreeBuffer();
      if (data == 0)
            return;

      //---------------------------------------------------
      //   convert to internal representation
      //    (deinterleave)
      //---------------------------------------------------

      int v[channels()];
      for (int k = 0; k < channels(); ++k) {
            v[k] = 0;
            for (int i = 0; i < segmentSize; i++) {
                  int idata = buffer[i * channels() + k];
                  if (idata > v[k])
                        v[k] = idata;
                  data[i + k * segmentSize] = (idata / 32767.0);
                  }
            }
      writeBuffer();    // put data into fifo
      }

//---------------------------------------------------------
//   writeAudio
//    ready to write to audio port
//    called from select() loop
//---------------------------------------------------------

void AudioPort::write(SndFile* sf, int frame, int idx)
      {
      float* fb = 0;
      float buffer[segmentSize * channels()];

      if (src) {
            memset(buffer, 0, segmentSize * channels() * sizeof(float));
            src->fillBuffer(channels(), buffer, idx);
            fb = buffer;
            }
      else
            printf("====no src\n");
      if (fb && sf) {
            int ch = channels();
            float* b[ch];
            for (int i = 0; i < ch; ++i)
                  b[i] = fb + segmentSize * i;
            sf->seek(frame, 0);
            sf->write(b, segmentSize);
            }
      short obuffer[segmentSize * channels()];
      if (fb == 0)
            memset(obuffer, 0, sizeof(obuffer));
      else {
            //---------------------------------------------------
            //  transform to output buffer float->int
            //---------------------------------------------------

            for (int k = 0; k < channels(); ++k) {
                  for (int i = 0; i < segmentSize; i++) {
                        int val  = lrint(fb[i + segmentSize*k]*32767.0);
                        if (val > 32767)
                              val = 32767;
                        else if (val <= -32768)
                              val = -32768;
                        obuffer[i*channels() + k] = val;
                        }
                  }
            }
      AudioDevice* dev = device();
      if (dev) {
            int rv = dev->write((unsigned char*)obuffer, segmentSize);
            if (rv == -1) {
                  fprintf(stderr, "error writing audio data: %s\n",
                     strerror(errno));
                  exit(-1);
                  }
            }
      }

//---------------------------------------------------------
//   setrwFlags
//---------------------------------------------------------

void AudioPort::setrwFlags(int val)
      {
      int mask = 0x3;
      if (_device)            // mask with device capabilities
            mask = _device->rwFlags();
      if ((_rwFlags & 2) && !(val & mask & 2)) {
            route()->disconnect(this);
            setRoute(0);
            }
      if (!(_rwFlags & 2) && (val & mask & 2)) {
            setRoute(song->master());
            route()->connect(this);
            }
      _rwFlags = val & mask;
      }

