/***************************************************************************
                          controller.cpp  -  description
                             -------------------
    begin                : Mon May 28 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include "mutella.h"
#include "controller.h"
#include "gnudirector.h"
#include "gnushare.h"
#include "gnucache.h"
#include "preferences.h"
#include "gnusearch.h"
#include "property.h"
#include "mui.h"
#include "inifile.h"
#include "dir.h"
#include "asyncfile.h"
#include "event.h"


class MControllerThread: public MThread
{
friend class MController;
friend class MUIThread;
public:
    MControllerThread(MController*);
    void run();
private:
	MController* m_pController;
	bool m_bContinue;
};

class MUIThread: public MThread
{
friend class MController;
public:
    MUIThread(MController*);
    void run();
private:
	MController* m_pController;
};

#define MUTELLARC 			"~/.mutella/mutellarc"
#define MUTELLARC_OLD 		"~/.mutellarc"
#define MUTELLATERMRC 		"~/.mutella/termrc"
#define MUTELLATERMRC_OLD 	"~/.mutella_termrc"
#define MUTELLARC_DIR 		"~/.mutella"
#define MUTELLARC_SEARCH	"~/.mutella/search.save"

MControllerThread::MControllerThread(MController* pC)
{
	m_pController = pC;
	m_bContinue = true;
}

void MControllerThread::run()
{
	time_t last_time = xtime();
	while(m_bContinue)
	{
		MAsyncSocket::Heartbeat(100); //100 ms
#ifdef _DEBUG
		if (last_time+1 < xtime())
			printf("blocking for %d secs detected\n",xtime()-last_time-1);
#endif //_DEBUG
		while(last_time < xtime())
		{
			++last_time;
			if (m_pController->m_pDirector) m_pController->m_pDirector->OnTimer();
		}
		//usleep(1); //1 mks
	}
	TRACE("Leaving MControllerThread::run()...");
}

MUIThread::MUIThread(MController* pC)
{
	m_pController = pC;
}

void MUIThread::run()
{
	m_pController->m_pUI->Do();
	m_pController->m_pNetworkThread->m_bContinue = false;
	TRACE("Leaving MUIThread::run()...");
}

MController::MController()
{
	m_pDirector = NULL;
	m_pPrefs = NULL;
	//
	m_pUI = new MUI(this);
	m_pUIMutex = m_pUI->GetMutex();
	//
	m_pPrefs =  new MGnuPreferences(this);
	//
	new MGnuDirector(this);
	//
	m_pNetworkThread = new MControllerThread(this);
	m_pUIThread = new MUIThread(this);
	//
	m_pPropCont =  new MPropertyContainer;
	// load it with preferences
	
}

MController::~MController()
{
	if (!m_pUIThread->finished())
	{
		POST_WARNING(ES_IMPORTANT, "failed to stop the UI thread");
	}
	else
	{
		delete m_pUIThread;
		delete m_pUI;
	}
	if (!m_pNetworkThread->finished())
	{
		// we still have chaces to finish with no crash...
		POST_WARNING(ES_IMPORTANT, "failed to stop the network thread");
	}
	else
	{
		if (m_pDirector)
			delete m_pDirector;
		// at this point all files must be closed
		TRACE("Waiting for async files...");
		MAsyncFile::WaitForThreadsStop();
		//
		delete m_pNetworkThread;
		delete m_pPropCont;
		delete m_pPrefs;
	}
}

MProperty* MController::GetProperty(LPCSTR szName)
{
	return m_pPropCont->FindProperty(szName);
}

bool MController::GroupResults()
{
	return  (m_pPrefs->m_bGroupbySize || m_pPrefs->m_bGroupbyName ||m_pPrefs->m_bGroupbyFuzzyRules);
}

bool MController::AddSearchUsr(const CString& search, int size, int sizeMode)
{
	return NULL != m_pDirector->AddSearch(search, ST_USER, size, sizeMode);
}

void MController::ForEachSearch(void* pData, tEachSearch callback)
{
	m_pDirector->ForEachSearch(pData, callback);
}

bool MController::GetSearchByID(DWORD dwID, SGnuSearch& gs, std::vector<Result>& rv, std::vector<ResultGroup>& gv)
{
	return m_pDirector->GetSearchByID(dwID, gs, rv, gv);
}

bool MController::GetResultsByID(DWORD dwID, std::vector<Result>& rv)
{
	return m_pDirector->GetResultsByID(dwID, rv);
}

bool MController::ClearSearchByID(DWORD dwID)
{
	return m_pDirector->ClearSearchByID(dwID);
}

bool MController::RemoveSearchByID(DWORD dwID)
{
	return m_pDirector->RemoveSearchByID(dwID);
}

void MController::ForEachConnection(void* pData, tEachConnection callback)
{
	m_pDirector->ForEachConnection(pData, callback);
}

bool MController::CloseConnectionByID(DWORD dwID)
{
	return m_pDirector->CloseConnectionByID(dwID);
}

DWORD MController::GetConnRateRecv()
{
	return m_pDirector->m_dwConnRecvRate;
}
DWORD MController::GetConnRateSend()
{
	return m_pDirector->m_dwConnSendRate;
}

void MController::ForEachUpload(void* pData, tEachUpload callback)
{
	m_pDirector->ForEachUpload(pData, callback);
}

void MController::ForEachDownload(void* pData, tEachDownload callback)
{
	m_pDirector->ForEachDownload(pData, callback);
}

bool MController::RemoveTransferByID(DWORD dwID, bool bDelPart)
{
	return m_pDirector->RemoveTransferByID(dwID, bDelPart);
}

void MController::AddDownload(Result& result)
{
	m_pDirector->AddDownload(result);
}

void MController::AddDownload(const ResultVec& results)
{
	m_pDirector->AddDownload(results);
}

void MController::OpenConnection(const char* host, int port)
{
	m_pDirector->AddNode(host, port);
}

void MController::GetNetStats(int& nHosts, int& nSharingHosts, int& nFiles, int& nSize)
{
	m_pDirector->GetNetStats(nHosts, nSharingHosts, nFiles, nSize);
}

void MController::GetConnStats(int& nConn, int& nUpl, int& nDownl)
{
	nConn = m_pDirector->CountConnections();
	nUpl = m_pDirector->CountUploading();
	nDownl = m_pDirector->CountDownloading();
}

void MController::GetBandwidthStats(int& nConn, int& nUpl, int& nDownl)
{
	nConn = m_pDirector->m_dwConnRecvRate + m_pDirector->m_dwConnSendRate;
	nUpl = (int) m_pDirector->m_dUploadRate;
	nDownl = (int) m_pDirector->m_dDownloadRate;
}

void MController::UpdateHits()
{
}

void MController::Rescan()
{
	m_pDirector->GetShare()->Rescan();
}

void MController::GetSharedFiles(std::vector<SharedFile>& ShareCopy)
{
	m_pDirector->GetShare()->GetShareCopy(ShareCopy);
}

void MController::GetNodeCache(std::list<Node>& list)
{
	m_pDirector->GetCache()->GetNodeList(list);
}

void MController::LoadPartials(const CString& partDir)
{
    DirEntryVec partials;
    if (!ScanDir(partials, ExpandPath(partDir), "*.[[*]]"))
    {
    	cerr << "LoadPartials: error scanning the dir " << partDir << endl;
    	return;
    }
    if (!partials.size())
    {
    	return;
    }
    for (int i = 0; i<partials.size(); ++i)
    {
    	CString path = partials[i].Path;
    	CString name = partials[i].Name;
    	int nB1 = name.find(".[[");
    	int nB2 = name.find("]]");
    	int nSize = atoi(name.substr(nB1+3, nB2-(nB1+3)).c_str());
    	//
    	name = name.substr(0,nB1);
    	CString search = MakeSearchOfFilename(name);

    	//printf("%s %d\n", search.c_str(), nSize);
    	MGnuSearch* pSearch = m_pDirector->AddSearch(search, ST_PARTIAL, nSize, LIMIT_EXACTLY);
    	if (pSearch)
    	{
    		pSearch->m_bAutoget = true;
			pSearch->m_Filename = name;
		}
    }
}

//struct SGnuSearch
//{
//	CString m_Search;
//??	// autoget support
//??	bool    m_bAutoget;
//??	CString m_Filename;
//	// filters
//	DWORD   m_MinSpeed;
//	int     m_SpeedFilterMode;
//	int     m_SpeedFilterValue;
//	// size filters
//	int     m_SizeFilterMode;
//	int     m_SizeFilterValue;

#define PARSE_PARAM(_par) \
	if (0<=(n=sOptions.find(#_par ":")))\
		_par = atol(sOptions.substr(n+strlen(#_par ":")).c_str());


bool MController::AddSearch(const CString& sSearch, const CString& sOptions)
{
	// parse filters
	DWORD   MinSpeed = 0;
	int     SpeedFilterMode = 0;
	int     SpeedFilterValue = 0;
	// size filters
	int     SizeFilterMode = 0;
	int     SizeFilterValue = 0;
	//
	int n;
	PARSE_PARAM(MinSpeed);
	PARSE_PARAM(SpeedFilterMode);
	PARSE_PARAM(SpeedFilterValue);
	PARSE_PARAM(SizeFilterMode);
	PARSE_PARAM(SizeFilterValue);

	MGnuSearch* pSearch = new MGnuSearch(m_pDirector, sSearch, ST_USER, SizeFilterValue, SizeFilterMode);
	pSearch->m_MinSpeed = MinSpeed;
	pSearch->m_SpeedFilterMode = SpeedFilterMode;
	pSearch->m_SpeedFilterValue = SpeedFilterValue;
	//
	if (m_pDirector->AddSearch(pSearch))
		return true;
	//
	delete pSearch;
	return false;
}

void MController::LoadSearches()
{
	TRACE("Loading searches...");
	// searches are saved in the following format:
	//   search string
	//   options
	//   <empty line>
	FILE* f = fopen(ExpandPath(MUTELLARC_SEARCH).c_str(),"r");
	if (!f)
	{
		POST_ERROR(ES_UNIMPORTANT, CString("Failed to open file \'") + MUTELLARC_SEARCH + "\' -- searches will not be loaded");
		TRACE3("Failed to open file \'", MUTELLARC_SEARCH, "\' -- searches will not be loaded");
		return;
	}
	//
	char tmp[1024];
	char* t;
	CString sSearch;
	CString sOptions;
	while (!feof(f) && !ferror(f))
	{
		if (NULL!=fgets(tmp,1024,f))
		{
			tmp[1023] = '\0';
			t = StripWhite(tmp);
			if (strlen(t))
			{
				if (*t != '#')
				{
					sSearch = sOptions;
					sOptions = tmp;
				}
				else
				{
					sSearch.empty();
					sOptions.empty();
				}
			}
			else
			{
				if (sSearch.size() && sOptions.size())
				{
					// place to create a search
					if (!AddSearch(sSearch, sOptions))
						POST_ERROR(ES_GOODTOKNOW, CString("Failed to add search \'") + sSearch + "\' while loading");
				}
				sSearch.empty();
				sOptions.empty();
			}
		}
	}
	fclose(f);
}


void save_search_callback(void* pData, SGnuSearch* pSearch)
{
	FILE* f = (FILE*) pData;
	if (pSearch->IsAutomatic())
		return;
	// search string
	fprintf(f,"%s\n", pSearch->m_Search.c_str());
	// options
	fprintf(f,"MinSpeed:%d  ",         pSearch->m_MinSpeed);
	fprintf(f,"SpeedFilterMode:%d  ",  pSearch->m_SpeedFilterMode);
	fprintf(f,"SpeedFilterValue:%d  ", pSearch->m_SpeedFilterValue);
	fprintf(f,"SizeFilterMode:%d  ",   pSearch->m_SizeFilterMode);
	fprintf(f,"SizeFilterValue:%d  ",  pSearch->m_SizeFilterValue);
	fprintf(f,"\n");
	// empty line
	fprintf(f,"\n");
}

void MController::SaveSearches()
{
	TRACE("Saving searches...");
	// save searches in the following format:
	//   search string
	//   options
	//   <empty line>
	FILE* f = fopen(ExpandPath(MUTELLARC_SEARCH).c_str(),"w");
	if (!f)
	{
		POST_ERROR(ES_IMPORTANT, CString("Failed to create file \'") + MUTELLARC_SEARCH + "\' -- searches will not be stored");
		TRACE3("Failed to create file \'", MUTELLARC_SEARCH, "\' -- searches will not be stored");
		return;
	}
	fprintf(f,"# this is mutella search save file\n");
	fprintf(f,"# it is created automatically on end\n");
	fprintf(f,"# of mutella session\n");
	fprintf(f,"#\n\n");
	
	//for each search...
	ForEachSearch(f, save_search_callback);
	
	fclose(f);
}


bool MController::Init()
{
	// init property environment
	m_pPropCont->Transfer(m_pPrefs->m_pContainer);
	ASSERT(m_pPrefs->m_pContainer->Count()==0);
	// init UIs
	m_pUI->Attach();
	//
	// now we are about to read the configuration
	// so lets check if we have '.mutella' dir
	bool bMigrate = false;
	if (!FileExists(MUTELLARC_DIR))
	{
		POST_MESSAGE(ES_IMPORTANT, CString("Directory \'") + MUTELLARC_DIR + "\' does not exist and will be created");
		if (!CreateDirectory(MUTELLARC_DIR))
		{
			POST_ERROR(ES_IMPORTANT, CString("Failed to create directory \'") + MUTELLARC_DIR + "\' -- configuration will not be read or saved");
		}
		else if (FileExists(MUTELLARC_OLD) || FileExists(MUTELLATERMRC_OLD))
		{
			// profile migration from v0.3.2 or earlier
			POST_MESSAGE(ES_IMPORTANT, "Migrating from version 0.3.2 or earlier");
			bMigrate = true;
			if (FileExists(MUTELLARC_OLD))
			{
				m_pPropCont->Read(MUTELLARC_OLD);
				DeleteFile(MUTELLARC_OLD);
				//
				//MUTELLATERMRC_OLD can be overloaded in MUTELLARC_OLD -- well -- too bad
				MProperty* pP = m_pPropCont->FindProperty("StartupScript");
				if (pP && 0==strcmp(MUTELLATERMRC_OLD, pP->GetStringValue()))
					pP->SetStringValue(MUTELLATERMRC);
				m_pPropCont->Write(MUTELLARC);
			}			
			if (FileExists(MUTELLATERMRC_OLD))
				MoveFile(MUTELLATERMRC_OLD,MUTELLATERMRC);
			POST_MESSAGE(ES_IMPORTANT, CString("Configuration files have been moved to the \'") + MUTELLARC_DIR + "\' directory");
		}
	}
	
	// read config
	m_pPropCont->Read(MUTELLARC);
	// let UIs initialise and load their private settings
	m_pUI->Init();
	
	// create download directories
	if (!FileExists(m_pPrefs->m_szDownloadPath))
	{
		POST_MESSAGE(ES_IMPORTANT, CString("Directory \'") + m_pPrefs->m_szDownloadPath + "\' does not exist and will be created");
		CreateDirectory(m_pPrefs->m_szDownloadPath);
	}
	CString PartialDir = CString(m_pPrefs->m_szDownloadPath) + "/part";
	if (!FileExists(PartialDir))
	{
		POST_MESSAGE(ES_IMPORTANT, CString("Directory \'") + PartialDir + "\' does not exist and will be created");
		CreateDirectory(PartialDir);
	}
	//
	LoadPartials(PartialDir);
	
	//
	if (m_pPrefs->m_bSaveSearches)
		LoadSearches();
	
	// Start sharing and load files
	m_pDirector->GetShare()->InitShare();

	//
	m_pDirector->StartListening();
	return true;
}

// main thread execution loop -- pass control to the UI
void MController::Do()
{
	ASSERT(m_pUI);
	ASSERT(m_pNetworkThread);
	//
	m_pUIThread->start();
	//
	//m_pNetworkThread->run();
	m_pNetworkThread->start();
	//
	m_pUIThread->wait();
	// leaving UI
	m_pUI->Detach();
	//
	m_pNetworkThread->m_bContinue = false;
	//
	// we are closing, save props
	TRACE("MController::Do: storing settings");
	m_pPropCont->Write(MUTELLARC);
	//
	if (m_pPrefs->m_bSaveSearches)
		SaveSearches();
	// stop the share thread
	m_pDirector->GetShare()->CloseShare();
	// wait for the network thread to stop
	m_pNetworkThread->wait(); // timed wait doesnt work //30 sec should be more than enough
	// TODO: stop share thread
	TRACE("Leaving MController::Do");
}

// methods
MGnuPreferences* MController::GetPrefs()
{
	ASSERT(m_pPrefs);
	return m_pPrefs;
}

void MController::Attach(MGnuDirector* pCntr)
{
	ASSERT(pCntr);
	ASSERT(m_pDirector==NULL);
	m_pDirector = pCntr;
}

void MController::Detach(MGnuDirector* pCntr)
{
	ASSERT(m_pDirector==pCntr);
	m_pDirector = NULL;
}

MGnuDirector* MController::GetDirector()
{
	ASSERT(m_pDirector);
	return m_pDirector;
}


