/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 controller.cpp  -  Overall controller class for core and core/UI interaction

    begin                : Mon May 28 2001
    copyright            : (C) 2001 by 
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include "mutella.h"
#include "structures.h"

#include "controller.h"
#include "asyncsocket.h"
#include "asyncdns.h"
#include "packet.h"

#include "event.h"
#include "messages.h"
#include "gnudirector.h"
#include "gnushare.h"
#include "gnucache.h"
#include "property.h"
#include "preferences.h"
#include "gnusearch.h"
#include "mui.h"
#include "uiterminal.h"
#include "uiremote.h"
#include "uilocalsocket.h"
#include "inifile.h"
#include "dir.h"
#include "asyncfile.h"
#include "common.h"
#include "conversions.h"
#include "gnulogcentre.h"
#include "sha1thread.h"
#include "gnumarkedfiles.h"

///////////////////////////////////////////
// MController handles resource persistance
// thats why we define following here
#define MUTELLARC_DIR 		"~/.mutella"
#define MUTELLARC 			"~/.mutella/mutellarc"
#define MUTELLARC_SEARCH	"~/.mutella/search.save"
#define MUTELLARC_HOSTS		"~/.mutella/hosts.save"
#define MUTELLARC_KNOWNFILES "~/.mutella/knownfiles.save"
#define MUTELLARC_WEBCACHES	"~/.mutella/webcaches.save"
#define MUTELLARC_OLD 		"~/.mutellarc"
#define MUTELLATERMRC 		"~/.mutella/termrc"
#define MUTELLATERMRC_OLD 	"~/.mutella_termrc"

/////////////////////////////////////
// this is the thread from which
// MAsyncSocket::Heartbeat() is
// called. De-facto "network" thread
class MGnutellaThread: public MThread
{
friend class MController;
friend class MUIThread;
public:
    MGnutellaThread(MController*);
    void run();
private:
	MController* m_pController;
	bool m_bContinue;
};

/////////////////////////////////
// MGnutellaThread implementation
MGnutellaThread::MGnutellaThread(MController* pC)
{
	m_pController = pC;
	m_bContinue = true;
}

void MGnutellaThread::run()
{
	time_t last_time = xtime();
	while(m_bContinue)
	{
		MAsyncSocket::Heartbeat(100); //100 ms
#ifdef _DEBUG
		if (last_time+1 < xtime())
			printf("blocking for %d secs detected\n",xtime()-last_time-1);
#endif //_DEBUG
		while(last_time < xtime())
		{
			if (m_pController->m_pDirector)
			{
				m_pController->m_pDirector->OnTimer(last_time);
			}
			if (m_pController->m_pPrefs &&
				m_pController->m_pPrefs->m_nSaveStatePeriod > 0 &&
				0 == last_time % (60*m_pController->m_pPrefs->m_nSaveStatePeriod))
			{
				m_pController->SaveState();
			}
			++last_time;
		}
		//xusleep(1); //1 mks
	}
	TRACE("Leaving MControllerThread::run()...");
}

////////////////////////////////////
// each UserInterface (UI) object
// is given a thread of execution
class MUIThread: public MThread
{
public:
    MUIThread(MController*, MUI*);
    void run();
private:
	MController* m_pController;
	MUI*         m_pUI;
};

////////////////////////////
// MUIThread implementation
MUIThread::MUIThread(MController* pC, MUI* pUI)
{
	m_pController = pC;
	m_pUI = pUI;
	setAutoDelete();
}

void MUIThread::run()
{
	m_pUI->Do();
	TRACE("calling MController::RemoveUI()...");
	m_pController->OnStopUI(m_pUI);
	TRACE("Leaving MUIThread::run()...");
}

/////////////////////////////////////
// MPrivPropertyContainer declaration

class MPrivPropertyContainer : public MPropertyContainer {
public:
	MPrivPropertyContainer();
	virtual bool DoMigration(LPCSTR szOldVersion, LPCSTR szCurrentVersion, MIniFile* pInifile);
};

///////////////////////////////////////
// MPrivPropertyContainer imlementation

MPrivPropertyContainer::MPrivPropertyContainer()
{
}

bool MPrivPropertyContainer::DoMigration(LPCSTR szOldVersion, LPCSTR szCurrentVersion, MIniFile* pInifile)
{
	//cout << "Migrating from version `" << szOldVersion << "' to `" << szCurrentVersion << endl;
	char tmp[1024];
	char tmp1[1024];
	bool bModified = false;
	if (strcmp(szOldVersion, "0.4.4")<=0)
	{
		pInifile->SetSection("Bandwidth");
		if (pInifile->ReadStr("BandwidthTransfer", tmp, NULL, true))
		{
			pInifile->WriteStr("BandwidthUpload", tmp);
			pInifile->EraseKey("BandwidthTransfer");
			bModified = true;
		}
		pInifile->SetSection("RemoteUI");
		if (pInifile->ReadStr("AllowedRemoteNet", tmp, NULL, true))
		{
			sprintf(tmp1, "{%s}", tmp);
			pInifile->WriteStr("AllowedRemoteNets", tmp1);
			pInifile->EraseKey("AllowedRemoteNet");
			bModified = true;
		}
	}
	return bModified;
}

/////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////
// MControler

MController::MController()
{
	m_pDirector = NULL;
	m_pPrefs = NULL;
	//
	m_pPrefs =  new MGnuPreferences(this);
	ASSERT(m_pPrefs);
	//
	VERIFY(new MGnuDirector(this));
	//
	m_pNetworkThread = new MGnutellaThread(this);
	ASSERT(m_pNetworkThread);
	//
	m_pPropCont =  new MPrivPropertyContainer;
	ASSERT(m_pPropCont);
	//
	m_pLogCentre = new MGnuLogCentre(this);
	ASSERT(m_pLogCentre);

	m_pSha1Thread = new MSha1Thread();
	ASSERT(m_pSha1Thread);
}

MController::~MController()
{
	if (!m_pLogCentre->finished())
	{
		// we still have chaces to finish with no crash...
		POST_WARNING(ES_IMPORTANT, "failed to stop the logging thread");
	}
	else
	{
		delete m_pLogCentre;
	}

	if (!m_pSha1Thread->finished())
	{
		// we still have chaces to finish with no crash...
		POST_WARNING(ES_IMPORTANT, "failed to stop file-hashing thread");
	}
	else
	{
		delete m_pSha1Thread;
	}

	if (!m_pNetworkThread->finished())
	{
		// we still have chaces to finish with no crash...
		POST_WARNING(ES_IMPORTANT, "failed to stop the network thread");
	}
	else
	{
		if (m_pDirector)
			delete m_pDirector;
		// at this point all files must be closed
		TRACE("Waiting for async files...");
		MAsyncFile::WaitForThreadsStop();
		//
		delete m_pNetworkThread;
		delete m_pPropCont;
		delete m_pPrefs;
	}
}

MProperty* MController::GetProperty(LPCSTR szName)
{
	return m_pPropCont->FindProperty(szName);
}

void MController::ForEachSearch(void* pData, tEachSearch callback)
{
	m_pDirector->ForEachSearch(pData, callback);
}

bool MController::AddSearchUsr(LPCSTR szSearch, int size, int sizeMode, bool bAutoGet /*=false*/ )
{
	return NULL != m_pDirector->AddSearch(szSearch, ST_USER, size, sizeMode, bAutoGet);
}

bool MController::ModifySearch(DWORD dwID, LPCSTR szSearch)
{
	return m_pDirector->ModifySearch(dwID, szSearch);
}

bool MController::GetSearchByID(DWORD dwID, SGnuSearch& gs, vector<Result>& rv, vector<ResultGroup>& gv)
{
	return m_pDirector->GetSearchByID(dwID, gs, rv, gv);
}

int MController::GetSearchesCount()
{
	return m_pDirector->GetSearchesCount();
}

bool MController::GetResultsByID(DWORD dwID, vector<Result>& rv)
{
	return m_pDirector->GetResultsByID(dwID, rv);
}

bool MController::ClearSearchByID(DWORD dwID)
{
	return m_pDirector->ClearSearchByID(dwID);
}

bool MController::RemoveSearchByID(DWORD dwID, bool bDeletePartial)
{
	return m_pDirector->RemoveSearchByID(dwID, bDeletePartial);
}

void MController::ForEachConnection(void* pData, tEachConnection callback)
{
	m_pDirector->ForEachConnection(pData, callback);
}

bool MController::CloseConnectionByID(DWORD dwID)
{
	return m_pDirector->CloseConnectionByID(dwID);
}

DWORD MController::GetConnRateRecv()
{
	return m_pDirector->m_dwConnRecvRate;
}
DWORD MController::GetConnRateSend()
{
	return m_pDirector->m_dwConnSendRate;
}

void MController::ForEachUpload(void* pData, tEachUpload callback)
{
	m_pDirector->ForEachUpload(pData, callback);
}

void MController::ForEachDownload(void* pData, tEachDownload callback)
{
	m_pDirector->ForEachDownload(pData, callback);
}

bool MController::RemoveTransferByID(DWORD dwID, bool bDelPart)
{
	return m_pDirector->RemoveTransferByID(dwID, bDelPart);
}

bool MController::AddDownload(const ResultVec& results)
{
	return m_pDirector->AddDownload(results);
}

bool MController::RenameDownload(DWORD dwID, LPCSTR szNewFilename)
{
	return m_pDirector->RenameDownload(dwID, szNewFilename);
}

void MController::OpenConnection(LPCSTR szHost, int port)
{
	IP ip;
	if (Str2Ip_lookup(szHost, ip))
		m_pDirector->AddNode(ip, port);
}

bool MController::IsUltrapeer()
{
	return m_pDirector->IsUltraPeer();
}

int MController::GetClientStartTime()
{
	return m_pDirector->GetClientStartTime();
}

void MController::GetCacheStats(int& nCatcherHosts, int& nUltraCatcherHosts, int& nStoreHosts, int& nWebCaches)
{
	m_pDirector->GetCacheStats(nCatcherHosts, nUltraCatcherHosts, nStoreHosts, nWebCaches);
}

void MController::GetNetStats(int& nHosts, int& nSharingHosts, int& nFiles, int& nSize)
{
	m_pDirector->GetNetStats(nHosts, nSharingHosts, nFiles, nSize);
}

void MController::GetConnStats(int& nConn, int& nUpl, int& nDownlActive, int& nDownlTotal)
{
	nConn = m_pDirector->CountConnections();
	nUpl = m_pDirector->CountUploads();
	nDownlActive = m_pDirector->CountActiveDownloads();
	nDownlTotal = m_pDirector->CountTotalDownloads();
}

void MController::GetBandwidthStats(int& nConn, int& nUpl, int& nDownl)
{
	nConn = m_pDirector->m_dwConnRecvRate + m_pDirector->m_dwConnSendRate;
	nUpl = (int) m_pDirector->m_dUploadRate;
	nDownl = (int) m_pDirector->m_dDownloadRate;
}

void MController::Rescan()
{
	m_pDirector->GetShare()->Rescan();
}

void MController::GetSharedFiles(vector<SharedFile>& ShareCopy)
{
	m_pDirector->GetShare()->GetShareCopy(ShareCopy);
}

void MController::GetNodeCache(list<Node>& list)
{
	m_pDirector->GetHostCatcher()->GetNodeList(list);
}

void MController::LoadPartials(const CString& partDir)
{
	DirEntryVec partials;
	if (!ScanDir(partials, ExpandPath(partDir), "*.[[*]]"))
	{
		POST_ERROR(ES_IMPORTANT, CString("LoadPartials: error scanning the dir `") + partDir + "'");
		return;
	}
	if (!partials.size())
		return;
	for (int i = 0; i<partials.size(); ++i)
	{
		CString path = partials[i].Path;
		CString name = partials[i].Name;
		int nB1 = name.find(".[[");
		int nB2 = name.find("]]");
		int nSize = atoi(name.substr(nB1+3, nB2-(nB1+3)).c_str());
		//
		name = name.substr(0,nB1);

		// repair stuck downloads
		if (nSize <= partials[i].Size)
		{
			CString sComplPath;
			m_pDirector->BuildDownloadPaths(name, nSize, NULL, &sComplPath, NULL);
			if (MoveFile(path, sComplPath))
			{
				POST_MESSAGE(ES_GOODTOKNOW, CString("Sucessfully repaired stuck download `") + name + "'");
			}
			else
			{
				POST_ERROR(ES_IMPORTANT, CString("Failed to repair stuck download `") + name + "'");
			}
			continue;
		}
     
		CString search = MakeSearchOfFilename(name);
		//
		MGnuSearch* pSearch = m_pDirector->GetSearchByFile(name, nSize);
		if (!pSearch)
		{
			TRACE4("LoadPartials() : creating a new auto-search because we could not find one for ", name, ", ", nSize);
			pSearch = m_pDirector->AddSearch(search.c_str(), ST_ALTERNATIVE, nSize, LIMIT_EXACTLY);
		}
		if (pSearch)
		{
			pSearch->SetAutoget(true);
			pSearch->SetFilename(name);
			//pSearch->SetFileHandle(nFileHandle);
			pSearch->SetType(ST_ALTERNATIVE);
		}
		// register an entry in the "known files" DB
		#warning add a proper knownDB entry
		m_pDirector->GetMarkedFiles()->AddFile(MFT_Partial, nSize, SHA1Hash(), search);
	}
	// clean up the search list -- erase all the searches which are "partial",
	// that are for which we did not find files
	m_pDirector->RemoveSearchesByType(ST_PARTIAL);
}

//struct SGnuSearch
//{
//	CString m_Search;
//??	// autoget support
//??	bool    m_bAutoget;
//??	CString m_Filename;
//	// size filters
//	int     m_SizeFilterMode;
//	int     m_SizeFilterValue;

#define PARSE_PARAM(_par) \
	if (0<=(n=sOptions.find(#_par ":")))\
		_par = atol(sOptions.substr(n+strlen(#_par ":")).c_str());


bool MController::AddSearch(const CString& sSearch, const CString& sOptions)
{
	// size filters
	int     SizeFilterMode = 0;
	int     SizeFilterValue = 0;
	// autoget
	int     Autoget = 0;
	//
	long    Handle = 0;
	//
	int n;
	PARSE_PARAM(SizeFilterMode);
	PARSE_PARAM(SizeFilterValue);
	PARSE_PARAM(Autoget);
	PARSE_PARAM(Handle);
	// parse filename. if it's there we are dealing with a "partial" search
	int nType = ST_USER;
	CString sFilename;
	if (0<=(n=sOptions.find("Filename:")))
	{
		sFilename = sOptions.substr(n+9);
		if (!sFilename.empty())
			nType = ST_PARTIAL;
	}

	// check for Sha1 searches and split the search line if needed
	list<CString> listSha1;
	CString sText = sSearch;
	int nPos = sText.find("Urn:Sha1:");
	if (nPos >= 0)
	{
		CString sSha1 = StripWhite(sText.substr(nPos+9));
		if (nPos)
			sText.cut(nPos);
		else
			sText.clear();
		split_str(sSha1, ' ', listSha1);
	}

	if (sText.empty() && listSha1.empty())
		return false;

	MGnuSearch* pSearch = new MGnuSearch(m_pDirector, sText, SHA1Hash(), nType, SizeFilterValue, SizeFilterMode);
	ASSERT(pSearch);
	pSearch->SetAutoget(Autoget);
	if (!sFilename.empty())
		pSearch->SetFilename(sFilename);
	if (Handle)
		pSearch->SetFileHandle(Handle);
	// sha1
	SHA1Hash sha1;
	for (list<CString>::iterator it = listSha1.begin(); it != listSha1.end(); ++it)
	{
		ASSERT(it->length()==32);
		if (sha1.fromStr(*it))
			pSearch->AddSha1Hash(sha1);
	}
	//
	if (m_pDirector->AddSearch(pSearch))
		return true;
	//
	delete pSearch;
	return false;
}

void MController::LoadSearches()
{
	TRACE("Loading searches...");
	// searches are saved in the following format:
	//   search string
	//   options
	//   <empty line>
	FILE* f = fopen(ExpandPath(MUTELLARC_SEARCH).c_str(),"r");
	if (!f)
	{
		POST_ERROR(ES_UNIMPORTANT, CString("Failed to open file \'") + MUTELLARC_SEARCH + "\' -- searches will not be loaded");
		TRACE3("Failed to open file \'", MUTELLARC_SEARCH, "\' -- searches will not be loaded");
		return;
	}
	//
	char tmp[1024];
	char* t;
	CString sSearch;
	CString sOptions;
	while (!feof(f) && !ferror(f))
	{
		if (NULL!=fgets(tmp,1024,f))
		{
			tmp[1023] = '\0';
			t = StripWhite(tmp);
			if (strlen(t))
			{
				if (*t != '#')
				{
					sSearch = sOptions;
					sOptions = tmp;
				}
				else
				{
					sSearch.clear();
					sOptions.clear();
				}
			}
			else
			{
				if (sSearch.size() && sOptions.size())
				{
					// place to create a search
					if (!AddSearch(sSearch, sOptions))
						POST_ERROR(ES_GOODTOKNOW, CString("Failed to add search \'") + sSearch + "\' while loading");
				}
				sSearch.clear();
				sOptions.clear();
			}
		}
	}
	fclose(f);
}


bool save_search_callback(void* pData, SGnuSearch* pSearch)
{
	FILE* f = (FILE*) pData;
	// search string
	if (!pSearch->m_Search.empty())
		fprintf(f,"%s", pSearch->m_Search.c_str());
	// sha1
	if (!pSearch->m_setSha1.empty())
	{
		if (!pSearch->m_Search.empty())
			fprintf(f," Urn:Sha1:");
		else
			fprintf(f,"Urn:Sha1:");
		for (set<SHA1Hash>::iterator it = pSearch->m_setSha1.begin(); it != pSearch->m_setSha1.end(); ++it)
		{
			fprintf(f," %s", it->toStr().c_str());
		}
	}
	fprintf(f,"\n");
	// options
	fprintf(f,"SizeFilterMode:%d  ",   pSearch->m_SizeFilterMode);
	fprintf(f,"SizeFilterValue:%d  ",  pSearch->m_SizeFilterValue);
	fprintf(f,"Autoget:%d  ",          pSearch->m_bAutoget ? 1 : 0);
	// save filename for automatic searches
	if (pSearch->IsAutomatic())
	{
		if (pSearch->m_FileHandle)
			fprintf(f,"Handle:%d  ",pSearch->m_FileHandle);
		if (pSearch->m_Filename.size())
			fprintf(f,"Filename:%s  ",   pSearch->m_Filename.c_str());
	}
	fprintf(f,"\n");
	// empty line
	fprintf(f,"\n");
	return true;
}

void MController::SaveSearches()
{
	//TRACE("Saving searches...");
	// save searches in the following format:
	//   search string
	//   options
	//   <empty line>
	FILE* f = fopen(ExpandPath(MUTELLARC_SEARCH).c_str(),"w");
	if (!f)
	{
		POST_ERROR(ES_IMPORTANT, CString("Failed to create file \'") + MUTELLARC_SEARCH + "\' -- searches will not be stored");
		TRACE3("Failed to create file \'", MUTELLARC_SEARCH, "\' -- searches will not be stored");
		return;
	}
	fprintf(f,"# this is mutella search save file\n");
	fprintf(f,"# it is created automatically at the end\n");
	fprintf(f,"# of mutella session and updated periodically\n");
	fprintf(f,"# during the session\n");
	fprintf(f,"#\n\n");
	
	//for each search...
	ForEachSearch(f, save_search_callback);
	
	fclose(f);
}

void MController::AddUI(MUI* pUI)
{
	ASSERT(pUI);
	SUIEntry uie;
	uie.pUI     = pUI;
	uie.pThread = new MUIThread(this, pUI);
	ASSERT(uie.pThread);
	// add to the list
	m_mutex.lock();
	m_listUI.push_front(uie);
	m_mutex.unlock();
	// attach this UI
	pUI->Attach(this);
}

bool MController::OnStopUI(MUI* pUI) // called by the UI thread before its termination
{
	MLock lock(m_mutex);
	for (list<SUIEntry>::iterator it = m_listUI.begin();it != m_listUI.end(); ++it)
	{
		if(it->pUI==pUI)
		{
			m_listStoppedUI.push_front(pUI);
			m_listUI.erase(it);
			TRACE("Calling m_waitForChanges.wakeAll()");
			m_waitForChanges.wakeAll();
			return true;
		}
	}
	return m_listStoppedUI.end()!=find(m_listStoppedUI.begin(),m_listStoppedUI.end(),pUI); // it was already in the stopped list -- fine
}

bool MController::Init(CString sConfigFile, bool bStartInteractive)
{
	// init property environment
	m_pPropCont->Transfer(m_pPrefs->m_pContainer);
	ASSERT(m_pPrefs->m_pContainer->Count()==0);
	// create UIs
	if (bStartInteractive)
		AddUI(new MUITerminal());
	AddUI(new MUIRemote());
	AddUI(new MUILocalSocket());
	//
	if (!sConfigFile.empty())
		m_sConfigFile = ExpandPath(sConfigFile);
	else
		m_sConfigFile = ExpandPath(MUTELLARC);
	//
	// now we are about to read the configuration
	// so lets check if we have '.mutella' dir
	bool bMigrate = false;
	if (!FileExists(MUTELLARC_DIR))
	{
		POST_MESSAGE(ES_IMPORTANT, CString("Directory \'") + MUTELLARC_DIR + "\' does not exist and will be created");
		if (!CreateDirectory(MUTELLARC_DIR))
		{
			POST_ERROR(ES_IMPORTANT, CString("Failed to create directory \'") + MUTELLARC_DIR + "\' -- configuration will not be read or saved");
		}
		else if (FileExists(MUTELLARC_OLD) || FileExists(MUTELLATERMRC_OLD))
		{
			// profile migration from v0.3.2 or earlier
			POST_MESSAGE(ES_IMPORTANT, "Migrating from version 0.3.2 or earlier");
			bMigrate = true;
			if (FileExists(MUTELLARC_OLD))
			{
				m_pPropCont->Read(MUTELLARC_OLD);
				DeleteFile(MUTELLARC_OLD);
				//
				//MUTELLATERMRC_OLD can be overloaded in MUTELLARC_OLD -- well -- too bad
				MProperty* pP = m_pPropCont->FindProperty("StartupScript");
				if (pP && 0==strcmp(MUTELLATERMRC_OLD, pP->GetStringValue()))
					pP->SetStringValue(MUTELLATERMRC);
				m_pPropCont->Write(m_sConfigFile.c_str());
			}
			if (FileExists(MUTELLATERMRC_OLD))
				MoveFile(MUTELLATERMRC_OLD,MUTELLATERMRC);
			POST_MESSAGE(ES_IMPORTANT, CString("Configuration files have been moved to the \'") + MUTELLARC_DIR + "\' directory");
		}
	}

	// read config
	m_pPropCont->Read(m_sConfigFile.c_str(), VERSION);
	// let UIs initialise and load their private settings
	m_mutex.lock();
	for (list<SUIEntry>::iterator it = m_listUI.begin();it != m_listUI.end(); ++it)
	{
		it->pUI->Init();
	}
	m_mutex.unlock();
	// create download directories
	if (!FileExists(m_pPrefs->m_szDownloadPath))
	{
		POST_MESSAGE(ES_IMPORTANT, CString("Directory \'") + m_pPrefs->m_szDownloadPath + "\' does not exist and will be created");
		CreateDirectory(m_pPrefs->m_szDownloadPath);
	}
	CString PartialDir = CString(m_pPrefs->m_szDownloadPath) + "/part";
	if (!FileExists(PartialDir))
	{
		POST_MESSAGE(ES_IMPORTANT, CString("Directory \'") + PartialDir + "\' does not exist and will be created");
		CreateDirectory(PartialDir);
	}

	// load marked files database
	m_pDirector->GetMarkedFiles()->LoadList(MUTELLARC_KNOWNFILES);

	//
	if (m_pPrefs->m_bSaveSearches)
		LoadSearches();

	// load partial files (re-generate auto-gets)
	LoadPartials(PartialDir);

	// Start sharing and load files
	m_pDirector->GetShare()->InitShare();

	// load the 'good' host list
	m_pDirector->GetHostStore()->LoadNodes(MUTELLARC_HOSTS);
	
	// load web caches
	m_pDirector->LoadWebCaches(MUTELLARC_WEBCACHES);

	//
	m_pDirector->StartListening();
	return true;
}

// main thread execution loop -- pass control to the UI
void MController::Do()
{
	// here we will start multiple threads, so we have to care about mutexes
	MLock lock(m_mutex);
	//

	ASSERT(m_pLogCentre);
	m_pLogCentre->start();

	ASSERT(m_pSha1Thread);
	m_pSha1Thread->start();

	ASSERT(m_pNetworkThread);

	// because of the bug in gprof which requires
	// non-zero total CPU time to be spent in the main thread of execution
	// the thread model has to be altered when -pg option is used
	#ifndef _PROFILER
	// start the Network thread
	m_pNetworkThread->start();
	#endif //!_PROFILER

	// start all UIs threads
	for (list<SUIEntry>::iterator it = m_listUI.begin();it != m_listUI.end(); ++it)
	{
		it->pThread->start();
	}
	// now we will enter the waiting loop
	// and stay there untill all the UIs finish
	// or somebody sets m_bExplisitStop
	m_bExplisitStop = false;

	#ifdef _PROFILER
	m_mutex.unlock();
	m_pNetworkThread->run();
	m_mutex.lock();
	#endif //_PROFILER

	// UI waiting loop
	while (!m_bExplisitStop && m_listUI.size())
	{
		TRACE("Controller goes to sleep...");
		m_waitForChanges.wait(&m_mutex);
		TRACE("Controller wakes up...");
	}
	//
	// now we are in the exit stage
	if (m_listUI.size())
	{
		// have to stop some UIs
		for (list<SUIEntry>::iterator it = m_listUI.begin();it != m_listUI.end(); ++it)
		{
			it->pUI->Stop(); // this will stop threads and eventually move UIs to the different list
		}
	}
	// this will stop the network thread
	m_pNetworkThread->m_bContinue = false;

	// lets wait for UIs to stop...
	bool bTimeout = false;
	while (!bTimeout && m_listUI.size())
	{
		TRACE("Waiting for UIs...");
		bTimeout = m_waitForChanges.wait(&m_mutex, 5000); //5 sec is more than enough to make up ones mind :-)
		if (bTimeout)
		{
			TRACE("Controller has been waiting for too long...");
		}
		else
		{
			struct timespec rqtp, rmtp;

		 	// TODO:  Note that conditionals may exit early
			// on *many* conditions.  The only way to do this
			// properly is to actually wait on a specific
			// condition and check on that condition.
			// Condition waiting needs to change to wait on
			// specific conditions and test those conditions
			// before moving on.

			// this is kinda hack but wait sometimes returns immediately with no reason
   			rqtp.tv_sec = 1;
    			rqtp.tv_nsec = rmtp.tv_sec = rmtp.tv_nsec = 0;

			// Don't bother doing a locked 1s wait - just try once.
			m_mutex.unlock();

			// Don't bother doing a locked 1s wait - just try once.
			safe_nanosleep(&rqtp, &rmtp);

			m_mutex.lock();
			TRACE("Controller wakes up...");
		}
	}

	SaveState();

	// send a general all-stop message
	ED().SendEvent( new MStringEvent( ET_MESSAGE, ES_NONE, "Mutella is exiting", MSG_MUTELLA_EXIT, MSGSRC_CONTROLLER ) );

	// stop the share thread
	m_pDirector->GetShare()->CloseShare();

	// stop AsyncDns thread
	TRACE("MController::Do: stopping the AsyncDNS thread...");
	MAsyncDnsThread::StopThread();

	// wait for the network thread to stop
	TRACE("MController::Do: waiting for the network thread...");
	m_pNetworkThread->wait(&m_mutex); // timed wait doesnt work //30 sec should be more than enough

	// delete UIs
	for (list<MUI*>::iterator it = m_listStoppedUI.begin(); it != m_listStoppedUI.end(); ++it)
	{
		(*it)->Detach();
		delete (*it);
	}
	if (m_listUI.size())
	{
		TRACE("Some UIs did not respond to the stop request and were not given a chance to store the settings");
	}
	//
	TRACE("Leaving MController::Do");
}

void MController::SaveState()
{
	// we are closing, save props
	//TRACE("MController::SaveState()...");
	if (!m_pPropCont->Write(m_sConfigFile.c_str(), VERSION))
		POST_ERROR(ES_IMPORTANT, "Failed to save settings");
	// save searches
	if (m_pPrefs->m_bSaveSearches)
		SaveSearches();
	// store the 'good' host list
	m_pDirector->GetHostStore()->SaveNodes(MUTELLARC_HOSTS);
	// store known files database
	m_pDirector->GetMarkedFiles()->StoreList(MUTELLARC_KNOWNFILES);
	
	m_pDirector->SaveWebCaches(MUTELLARC_WEBCACHES);
}

void MController::ForceStop()
{
	m_mutex.lock();
	m_bExplisitStop = true;

	#ifdef _PROFILER
	m_pNetworkThread->m_bContinue = false;
	#endif //_PROFILER

	m_waitForChanges.wakeAll();
	m_mutex.unlock();
}

// methods

CString MController::GetRcDirPath()
{
	return ExpandPath(MUTELLARC_DIR);
}

MGnuPreferences* MController::GetPrefs()
{
	ASSERT(m_pPrefs);
	return m_pPrefs;
}

void MController::Attach(MGnuDirector* pCntr)
{
	ASSERT(pCntr);
	ASSERT(m_pDirector==NULL);
	m_pDirector = pCntr;
}

void MController::Detach(MGnuDirector* pCntr)
{
	ASSERT(m_pDirector==pCntr);
	m_pDirector = NULL;
}

MGnuDirector* MController::GetDirector()
{
	ASSERT(m_pDirector);
	return m_pDirector;
}

bool MController::RegisterHttpFolder(MUI* pUI, LPCSTR szFolder)
{
	ASSERT(m_pDirector);
	return m_pDirector->RegisterHttpFolder(pUI, szFolder);
}

bool MController::UnregisterHttpFolder(MUI* pUI, LPCSTR szFolder)
{
	ASSERT(m_pDirector);
	return m_pDirector->UnregisterHttpFolder(pUI, szFolder);
}

bool MController::OnExternalHttpRequest(const IP& ipRemoteHost, LPCSTR szFolder, LPCSTR szHandshake, const char* pBuffRest, HANDLE hSocket, MUI* pUI)
{
	MLock lock(m_mutex);
	//
	for (list<SUIEntry>::iterator it = m_listUI.begin();it != m_listUI.end(); ++it)
	{
		if (it->pUI==pUI)
		{
			lock.unlock();
			return pUI->OnHttpRequest(ipRemoteHost, szFolder, szHandshake, pBuffRest, hSocket);
		}
	}
	return false;
}


