/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_font_manager.cpp 
 * @brief A class that manages shared display lists used for text output
 * 
 */

#ifndef __GC_FONT_MANAGER_H__
#define __GC_FONT_MANAGER_H__

#include <map>
#include <string>
#include "myx_gc_utilities.h"

using namespace std;

//----------------------------------------------------------------------------------------------------------------------

// Default values for text.
const string DefaultFontFamily("Arial");
const string DefaultFontStyle("normal");
const string DefaultFontWeight("400");      // Must be a string as we get it from an attribute that can contain strings.
const string DefaultFontDecoration("none");

#pragma warning(disable: 4251) // Disable warning about DLL interface for template classes.

typedef struct tagFontEntry
{
  GLuint Base;   // The base (first) display list.
  GLsizei Range; // The number of display lists used for that entry.
} TFontEntry;

typedef map<string, TFontEntry, CStringLessThan> CFontList;
typedef map<string, TFontEntry, CStringLessThan>::iterator CFontListIterator;
typedef pair<string, TFontEntry> CFontListElementPair;

// CFontManager is a helper class for text output in the generic canvas. It maps a description string for a font with attributes to a 
// display list. If there is no display for a given font then one is created.
// The font manager is basically a singleton class. We only need one instance of it.
class CFontManager
{
private:
  CFontList FFontList; // A mapper of fonts with specific properties to OpenGL display lists.
  string FDefaultKey;  // The key to the default font entry.
public:
  ~CFontManager(void);

  void Clear(void);
  void CreateDefaultEntry(void);
  TFontEntry CreateFontEntry(const string& Family, const string& Style, int Weight, const string& Decoration);
  string CreateLookupKey(const string& Family, const string& Style, int Weight, const string& Decoration);
  GLuint GetFontBase(const string& Family, const string& Style, int Weight, const string& Decoration);
};

//----------------------------------------------------------------------------------------------------------------------

CFontManager* FontManager(); // The one (and only) font manager instance.

#endif // #ifdef __GC_FONT_MANAGER_H__