--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Strings.Fixed;
with Utils;

package body Shell_Commands is

   use Utils;
   function Raw_Arguments (Shell_Line : String) return Arguments_Array;
   function Join_Arguments (List : Arguments_Array) return Arguments_Array;

   ----------
   -- Find --
   ----------

   function Find (Shell_Line : String) return Command_Type is
      New_Shell_Line : constant String := Cut_Last_Spaces
        (Cut_First_Spaces (Shell_Line));
      Space_Position : constant Natural :=
        Ada.Strings.Fixed.Index (New_Shell_Line, " ");
   begin
      if New_Shell_Line = "" then
         raise No_Command;
      end if;
      if Space_Position = 0 then
         return Command_Type'Value (New_Shell_Line);
      else
         return Command_Type'Value
           (New_Shell_Line (New_Shell_Line'First .. Space_Position - 1));
      end if;
   exception
      when No_Command =>
         raise;
      when others =>
         raise Bad_Command;
   end Find;

   function Find (Shell_Line : Ada.Strings.Unbounded.Unbounded_String)
                 return Command_Type is
   begin
      return Find (Ada.Strings.Unbounded.To_String (Shell_Line));
   end Find;

   ---------------
   -- Arguments --
   ---------------

   function Arguments (Shell_Line : String) return Arguments_Array is
   begin
      return Join_Arguments (Raw_Arguments (Shell_Line));
   end Arguments;

   function Raw_Arguments (Shell_Line : String) return Arguments_Array is
      New_Shell_Line : constant String := Cut_Last_Spaces
        (Cut_First_Spaces (Shell_Line));
      use Ada.Strings.Unbounded;
      Manipulation_String : Unbounded_String := To_Unbounded_String
        (New_Shell_Line);
      First_Space_Position : constant Natural :=
        Ada.Strings.Fixed.Index (New_Shell_Line, " ");
      Number_Of_Argument : Natural := 0;
      Space_Position : Natural;
   begin
      if First_Space_Position = 0 then
         raise No_Argument;
      end if;
      loop
         Space_Position := Index (Manipulation_String, " ");
         exit when Space_Position = 0;
         Number_Of_Argument := Number_Of_Argument + 1;
         loop
            exit when Element (Manipulation_String, Space_Position + 1) /= ' ';
            Space_Position := Space_Position + 1;
         end loop;
         Manipulation_String := To_Unbounded_String
           (Slice (Manipulation_String,
                   Space_Position + 1,
                   Length (Manipulation_String)));
      end loop;
      declare
         Arguments_To_Return : Arguments_Array (1 .. Number_Of_Argument);
      begin
         Manipulation_String :=
           Ada.Strings.Unbounded.To_Unbounded_String (New_Shell_Line);
         Space_Position :=
           Ada.Strings.Unbounded.Index (Manipulation_String, " ");
         loop
            exit when Element (Manipulation_String, Space_Position + 1) /= ' ';
            Space_Position := Space_Position + 1;
         end loop;
         Manipulation_String := To_Unbounded_String
           (Slice (Manipulation_String,
                   Space_Position + 1,
                   Length (Manipulation_String)));
         for I in Arguments_To_Return'Range loop
            Space_Position :=
              Ada.Strings.Unbounded.Index (Manipulation_String, " ");
            if Space_Position = 0 then
               Arguments_To_Return (I) := Manipulation_String;
            else
               Arguments_To_Return (I) :=
                 To_Unbounded_String
                 (Slice (Manipulation_String,
                         1,
                         Space_Position - 1));
               loop
                  exit when Element (Manipulation_String,
                                     Space_Position + 1) /= ' ';
                  Space_Position := Space_Position + 1;
               end loop;
               Manipulation_String := To_Unbounded_String
                 (Slice (Manipulation_String,
                         Space_Position + 1,
                         Length (Manipulation_String)));
            end if;
         end loop;
         return Arguments_To_Return;
      end;
   end Raw_Arguments;

   function Join_Arguments (List : Arguments_Array) return Arguments_Array is
      Number_Of_Action_To_Be_Taken : Natural := 0;
      use Ada.Strings.Unbounded;
   begin
      for I in List'Range loop
         if I /= List'Last then
            if Element (List (I), Length (List (I))) = '\' then
               Number_Of_Action_To_Be_Taken :=
                 Number_Of_Action_To_Be_Taken + 1;
            end if;
         end if;
      end loop;
      if Number_Of_Action_To_Be_Taken = 0 then
         return List;
      end if;
      declare
         New_Length : constant Positive := List'Length -
           Number_Of_Action_To_Be_Taken;
         List_To_Return : Arguments_Array (1 .. New_Length) :=
           (others => Null_Unbounded_String);
         Index_List : Positive := List'First;
         Index_Return : Positive := 1;
         Current_Argument : Unbounded_String := Null_Unbounded_String;
      begin
         loop
            Current_Argument := Current_Argument & List (Index_List);
            if Element (List (Index_List),
                        Length (List (Index_List))) = '\' then
               Replace_Element (Current_Argument,
                                Length (Current_Argument),
                                ' ');
            else
               List_To_Return (Index_Return) := Current_Argument;
               Current_Argument := Null_Unbounded_String;
               Index_Return := Index_Return + 1;
            end if;
            Index_List := Index_List + 1;
            exit when Index_Return > List_To_Return'Length;
         end loop;
         return List_To_Return;
      end;
   end Join_Arguments;

end Shell_Commands;
