/* Various editing functions such as word wrap, to upper, etc.

	Copyright (C) 1993-1998 Sebastiano Vigna 
	Copyright (C) 1999-2001 Todd M. Lewis and Sebastiano Vigna

	This file is part of ne, the nice editor.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2, or (at your option) any
	later version.
	
	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.
	
	You should have received a copy of the GNU General Public License along
	with this program; see the file COPYING.  If not, write to the Free
	Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
	02111-1307, USA.  */



#include "ne.h"

/* The number of type of brackets we recognize. */

#define NUM_BRACKETS 4



/* This function applies a given "to" function to the word the cursor is on.
"to" is usually toupper() or tolower(). */

static int to_something(buffer *b, int (to)(int)) {

	int c, x = b->cur_x;

	assert_buffer(b);

	/* If we are after the end of the line, just return ERROR. */

	if (b->cur_line == b->line_num -1 && b->cur_pos >= b->cur_line_desc->line_len)
		return(ERROR);

	start_undo_chain(b);

	/* ??????? It would be much better to delete and insert the whole word */

	while (b->cur_pos < b->cur_line_desc->line_len && isalpha((unsigned char)b->cur_line_desc->line[b->cur_pos])) {
		c = b->cur_line_desc->line[b->cur_pos];
		delete_char(b, b->cur_line_desc, b->cur_line, b->cur_pos);
		insert_char(b, b->cur_line_desc, b->cur_line, b->cur_pos, to(c));
		char_right(b);
	}

	update_partial_line(b, b->cur_y, x, TRUE);

	end_undo_chain(b);

	char_right(b);

	return(OK);
}



/* These functions upper or lower the case of the word the cursor is on. They
just call to_something(). Note the parentheses around the function names,
which inhibit the possible macros. */

int to_upper(buffer *b) {
	return(to_something(b, (toupper)));
}

int to_lower(buffer *b) {
	return(to_something(b, (tolower)));
}



/* This function upper cases the first letter, and lower cases the following
ones, of the word the cursor is on. The cursor is moved as in to_something(). */

int capitalize(buffer *b) {

	int c, first_char = TRUE, x = b->cur_x;

	assert_buffer(b);

	/* If we are after the end of the line, just return ERROR. */

	if (b->cur_line == b->line_num - 1 && b->cur_pos >= b->cur_line_desc->line_len)
		return(ERROR);

	start_undo_chain(b);

	while (b->cur_pos < b->cur_line_desc->line_len && isalpha((unsigned char)b->cur_line_desc->line[b->cur_pos])) {
		c = b->cur_line_desc->line[b->cur_pos];
		delete_char(b, b->cur_line_desc, b->cur_line, b->cur_pos);
		insert_char(b, b->cur_line_desc, b->cur_line, b->cur_pos, (first_char ? toupper : tolower)(c));
		if (first_char) first_char = FALSE;
		char_right(b);
	}

	update_partial_line(b, b->cur_y, x, TRUE);

	end_undo_chain(b);

	char_right(b);

	return(OK);
}



/* This function tries to find out what bracket matches the bracket under
the cursor, and moves it there. Various error codes can be returned. */

int match_bracket(buffer *b) {

	static char bracket_table[NUM_BRACKETS][2] = { { '(',')' }, { '[',']' }, { '{','}' }, { '<','>' } };

	int i, j, n, y, dir;
	line_desc *ld = b->cur_line_desc;
	char *p;

	if (b->cur_pos >= ld->line_len) return(NOT_ON_A_BRACKET);

	for(i=0; i<NUM_BRACKETS; i++) {
		for(j=0; j<2; j++)
			if (ld->line[b->cur_pos] == bracket_table[i][j]) break;
		if (j < 2) break;
	}

	if (i==NUM_BRACKETS && j==2) return(NOT_ON_A_BRACKET);

	if (j) dir = -1;
	else dir = 1;

	n = 0;
	p = ld->line+b->cur_pos;
	y = b->cur_line;

	while(ld->ld_node.next && ld->ld_node.prev) {

		if (p) {
			while(p >= ld->line && p - ld->line < ld->line_len) {

				if (*p == bracket_table[i][j]) n++;
				else if (*p == bracket_table[i][1-j]) n--;

				if (n == 0) {
					goto_line(b, y);
					goto_pos(b, p-ld->line);
					return(OK);
				}
				p += dir;
			}
		}

		p = NULL;

		if (dir == 1) {
			ld = (line_desc *)ld->ld_node.next;
			if (ld->ld_node.next && ld->line) p = ld->line;
			y++;
		}
		else {
			ld = (line_desc *)ld->ld_node.prev;
			if (ld->ld_node.prev && ld->line) p = ld->line + ld->line_len - 1;
			y--;
		}
	}

	return(CANT_FIND_BRACKET);
}



/* This function breaks a line at the first possible position before the
current cursor position (i.e., at a tab or at a space). The space is
deleted, and a new line is inserted. The cursor is repositioned coherently.
The number of characters existing on the new line is returned, or 0 if
no word wrap was possible. */

int word_wrap(buffer *b) {

	int cur_pos, pos, first_pos;

	if (!(cur_pos = pos = b->cur_pos)) return(ERROR);

	/* Find the first possible position we could break a line on. */
	first_pos = 0;

	/* Skip leading white space */
	while ( first_pos<b->cur_line_desc->line_len && isspace((unsigned char)b->cur_line_desc->line[first_pos]) )
		first_pos++;

	/* Skip non_space after leading white space */
	while ( first_pos<b->cur_line_desc->line_len && !isspace((unsigned char)b->cur_line_desc->line[first_pos]) )
		first_pos++;
	
	/* Now we know that the line shouldn't be broken before line[first_pos]. */
	
   /* Start from the other end now and find a candidate space to break the line on.*/
	while(--pos && !isspace((unsigned char)b->cur_line_desc->line[pos]));

	if (!pos || pos<first_pos) return(ERROR);

	delete_char(b, b->cur_line_desc, b->cur_line, pos);
	insert_lin(b, b->cur_line_desc, b->cur_line, pos);

	return(b->cur_pos-pos-1);
}


/* These functions reformat a paragraph while preserving appropriate
leading white space. The strategy is as follows:

  1. Establish appropriate leading space. This will be taken from the line
     following the current line if it is non-blank. Otherwise it will be
     taken from the current line. Save a copy of it for later as space[].

  2. Start an undo chain.
  
  3. Trim trailing space off the current line.

  4. while the current line is too long (i.e., needs to be split:
       4.1  Find the split point
       4.2  Remove any space at the split point
       4.3  Split the line at the split point.  (We are done with this line)
       4.4  Make the new line the current line
       4.5  Insert the space[] stream we saved in step 1.
       4.6  Trim trailing space off the end of the line.

  5. If the _following_ line is part of this paragraph (i.e., its first
     non-blank character is in the correct position):
       5.1  Add a space to the end of the current line.
       5.2  Copy following line's data starting with the first
            non-blank to the end of the current line.
       5.3  Remove the following line.
       5.4  Goto step 3.

  6. end the undo chain.

  7. Free space[].
  
  8. and refresh the screen

  9. move to the next non-blank after the current line. (We have to do
     this so that commands like "Paragraph 5" will do 5 paragraphs
     instead of only three.)

*/

static char *pa_space     = NULL;  /* Where we keep space for paragraph left offsets */
static int   pa_space_len;         /* How long pa_space is when tabs are expanded */
static int   pa_space_pos;         /* How long pa_space is without expanding tabs */

/* save_space() sets pa_space, pa_space_len, and pa_space_pos to reflect
the space on the left end of the line ld refers to in the context of
the given tab size. If the line contains only space then it is treated
identically to an empty line, in which case save_space() returns 0 and
pa_space, pa_space_len, and pa_space_pos are cleared. Otherwise it
returns 1. The null-terminated string pa_space points to in that case
is appropriate for use with insert_stream(). */

static int save_space( line_desc *ld, int tab_size )
  {
    int pos;
    if ( pa_space ) free(pa_space);
    pa_space     = NULL;
    pa_space_len = 0;
    pa_space_pos = 0;

    if ( !ld->line )
        {
          /* No data on this line. */
          return 0;
        }

    pos = 0;
    while ( pos<ld->line_len && isspace((unsigned char)ld->line[pos]) )
        pos++;

    if (pos == ld->line_len ) return 0; /* Blank lines don't count. */

    pa_space_pos = pos;
    pa_space_len = calc_len(ld, pos, tab_size);
    if ( (pa_space = malloc(pos+1) ) )
        {
          memcpy(pa_space, ld->line, pos);
          pa_space[pos] = '\0';
          return 1;
        }
    return 0;
  }

/* trim_trailing_space() removes spaces from the end of the line
refered to by the line_desc ld. The int line is necessary if you want
to be able to undo later. */

static void trim_trailing_space( buffer *b, line_desc *ld, int line )
  {
    int pos = ld->line_len;
    if ( !ld->line ) return;
    while(pos && isspace((unsigned char)ld->line[pos-1])) pos--;
     
    if (pos < ld->line_len) delete_stream(b, ld, line, pos, ld->line_len-pos);
  }

/* is_part_of_paragraph() determines if the line ld refers to could be
considered part of a paragraph based on its leading spaces compared to
pa_space_len. If they are the same, is_part_of_paragraph() returns 1,
and *first_non_blank is set to the position of the first non-blank
character on the line. Otherwise, *first_non_blank is -1 and
is_part_of_paragraph() returns 0. */
    
static int is_part_of_paragraph( line_desc *ld, int tab_size, int *first_non_blank )
  {
    int pos = 0;
    *first_non_blank = -1;  
    if ( !pa_space ) return 0;
    if ( !ld->line ) return 0;
    while ( pos < ld->line_len && isspace((unsigned char)ld->line[pos]) ) pos++;
    if ( pos < ld->line_len && calc_len(ld, pos, tab_size) == pa_space_len )
        {
          *first_non_blank = pos;
          return 1;
        }
    return 0;
  }

/* paragraph() reformats a paragraph following the current parameters
for right_margin (a value of 0 forces the use of the full screen width).
On completion the cursor is positioned either:

  * on the first non-blank character after the paragraph if there is
one, or

  * on a blank line following the paragraph if there is one, or

  * on the last line of the paragraph.

paragraph() returns OK unless the cursor ends up on the last line of
the file, in which case it returns ERROR. */

int paragraph( buffer *b)
  {
    int pos,
        done,
        line         = b->cur_line,
        right_margin = b->opt.right_margin ? b->opt.right_margin : ne_columns;
    line_desc *ld = b->cur_line_desc;

    if (!ld->line)
        {
          line_down(b);
          return(OK);
        }

                                  /** Step 1 **/
    if ( !(
              ( ld->ld_node.next->next &&
                save_space((line_desc *)ld->ld_node.next, b->opt.tab_size)
              )
            || save_space( ld, b->opt.tab_size)
          )
       )
        {
          line_down(b);
          return(OK);
        }
    
    
                                  /** Step 2 **/
    start_undo_chain(b);

    done = FALSE;
    do{
                                  /** Step 3 **/
        trim_trailing_space( b, ld, line );

                                  /** Step 4 **/
        while (calc_len(ld, ld->line_len, b->opt.tab_size) > right_margin)
            {
              int spaces;
              int split;
              int did_split;
              
              /** 4.1  Find the split point **/

              pos = ld->line_len;
             did_split = split = spaces = 0;
              while ( calc_len(ld, pos, b->opt.tab_size ) > right_margin )
                { int tspaces = 0;
                  while(pos && !isspace((unsigned char)ld->line[pos-1])) pos--;
                  while(pos &&  isspace((unsigned char)ld->line[pos-1]))
                    {
                      pos--;
                      tspaces++;
                    }
                  if ( calc_len(ld, pos, b->opt.tab_size)>pa_space_len)
                    {
                      split  = pos;
                      spaces = tspaces;
                    }
                }

              /** 4.2  Remove any space at the split point **/

              if ( split && spaces )
                  {
                    delete_stream(b, ld, line, split, spaces);
                    /** 4.3  Split the line at the split point.  (We are done with this line) **/
                    if ( ld->line_len > pos )
                        {
                          insert_lin(b, ld, line, split);
                          did_split = 1;
                        }
                  }
                  
              /** 4.4  Make the (new?) next line the current line **/
              if ( ld->ld_node.next->next )
                  {
                    ld = (line_desc *)ld->ld_node.next;
                    line++;

                    /** 4.5  Insert the ps_space[] stream we saved in step 1. Note that  **/
                    /** we only want to do this if this line is the result of a split,   **/
                    /** which is true if did_split != 0.                                 **/

                    if ( did_split && pa_space && pa_space_len )
                        { 
                          insert_stream(b, ld, line, 0, pa_space, pa_space_pos+1);
                        }
    
                    /** 4.6  Trim trailing space off the end of the line. **/
                    trim_trailing_space( b, ld, line );
                  }
                else
                    done = TRUE;  
            }
            
        /** 5. If the _following_ line is part of this paragraph (i.e., its first **/
        /**    non-blank character is in the correct position):                   **/

        if ( ld->ld_node.next->next &&
             is_part_of_paragraph( (line_desc *)ld->ld_node.next, b->opt.tab_size, &pos ))
            {
              /** 5.1  Add a space to the end of the current line.                      **/

              insert_stream( b, ld, line, ld->line_len, " ", 2 );

              /** 5.2  Move following line's data starting with the first               **/
              /**      non-blank to the end of the current line.                        **/
              /**  We do this by first deleting the leading spaces, then we             **/
              /**  delete the newline at the end of the current line.                   **/
              if ( pos > 0 )
                 delete_stream( b, (line_desc *)ld->ld_node.next, line+1, 0, pos );
              delete_stream( b, ld, line, ld->line_len, 1 );
            }
          else
              done = TRUE;
      } while ( !done );

                                  /** Step 6 **/
    end_undo_chain(b);

                                  /** Step 7 **/
    if ( pa_space )
        {
          free( pa_space );
          pa_space = NULL;
        }

                                  /** Step 8 **/
    update_window_lines(b, b->cur_y, ne_lines-2, FALSE);

                                  /** Step 9 **/
    goto_line(b, line);
    line_down(b);

    /* Try to find the first non-blank starting with this line. */
    ld   = b->cur_line_desc;
    line = b->cur_line;  

    do{
        if (ld->line)
            {
              for (pos=0; pos<ld->line_len; pos++)
                {
                  if ( !isspace(ld->line[pos]) )
                      {
                        goto_line(b, line );
                        goto_pos(b, pos );
                        return(ld->ld_node.next ? 0 : -1);
                      }
                }
            }
        ld = (line_desc *)ld->ld_node.next;
        line++;
      } while ( ld->ld_node.next);
         
    return(b->cur_line_desc->ld_node.next ? OK : ERROR);
  }


/* This function centers the current line with respect to the right_margin
parameter. ERROR is returned when centering the last line of text. */

int center(buffer *b) {

	int pos = 0, right_margin = b->opt.right_margin ? b->opt.right_margin : ne_columns;

	while(pos<b->cur_line_desc->line_len && isspace((unsigned char)b->cur_line_desc->line[pos])) pos++;

	if (pos == b->cur_line_desc->line_len) return(OK);

	start_undo_chain(b);

	delete_stream(b, b->cur_line_desc, b->cur_line, 0, pos);
	insert_spaces(b, b->cur_line_desc, b->cur_line, 0, (right_margin - b->cur_line_desc->line_len)/2);

	end_undo_chain(b);

	return(OK);
}


/* This function indents a line of the amount of spaces present of
the previous line. The number of inserted characters is returned. */

int auto_indent_line(buffer *b) {

	int i;
	line_desc *ld = (line_desc *)b->cur_line_desc->ld_node.prev;
	char  *p;

	assert(ld->ld_node.prev != NULL);
	assert_line_desc(ld);

	if (!(p = ld->line)) return(0);

	while(p-ld->line < ld->line_len && isspace((unsigned char)*p)) p++;

	start_undo_chain(b);

	for(i=p-ld->line-1; i>=0; i--)
		insert_char(b, b->cur_line_desc, b->cur_line, 0, ld->line[i]);

	end_undo_chain(b);

	return(p-ld->line);
}
