/*

Copyright (C) 2000, 2001 Christian Kreibich <kreibich@aciri.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <net/if_arp.h>
#include <net/ethernet.h>
#include <netdb.h>
#ifdef FREEBSD
#include <netinet/ip_var.h>
#include <netinet/udp_var.h>
#endif

#include <nd.h>
#include <nd_clipboard.h>
#include <nd_globals.h>
#include <nd_misc.h>
#include <nd_prefs.h>
#include <callbacks.h>
#include <interface.h>
#include <support.h>

#define PBAR_ACTIVITY_BLOCKS 10
#define GTK_CLIST_CLASS_FW(_widget_) GTK_CLIST_CLASS (((GtkObject*) (_widget_))->klass)


static int       nd_misc_ones_complement_checksum(const void *p, int b, u_int32_t sum);
static u_short   nd_misc_in_cksum(register u_short *addr, register int len, u_int preadd);

static GtkProgressBar *pbar;
static int             pbar_total;
static int             pbar_current;
static gboolean        activity_mode;
static int             timeout_set;
static gfloat          old_ratio = -1.0;

static GdkColor bg[5];
static GdkColor red[5];

static GdkPixmap      *incomplete_pmap;
static GdkBitmap      *incomplete_mask;

static int
nd_misc_ones_complement_checksum(const void *p, int b, u_int32_t sum)
{
  const u_short *sp;

  sp = (u_short *) p; /* better be aligned! */
  b /= 2;             /* convert to count of short's */

  /* No need for endian conversions. */
  while ( --b >= 0 )
    sum += *sp++;

  while ( sum > 0xffff )
    sum = (sum & 0xffff) + (sum >> 16);

  return (sum);
}

/*
 * Checksum routine for Internet Protocol family headers (C Version)
 */
static u_short
nd_misc_in_cksum(register u_short *addr, register int len, u_int preadd)
{
  register int nleft = len;
  register u_short *w = addr;
  register u_short answer;
  register int sum = 0;

  sum += preadd;

  /*
   *  Our algorithm is simple, using a 32 bit accumulator (sum),
   *  we add sequential 16 bit words to it, and at the end, fold
   *  back all the carry bits from the top 16 bits into the lower
   *  16 bits.
   */
  while (nleft > 1) {
    sum += *w++;
    nleft -= 2;
  }

  /* mop up an odd byte, if necessary */
  if (nleft == 1)
    sum += *(u_char *)w;

  /*
   * add back carry outs from top 16 bits to low 16 bits
   */
  sum = (sum >> 16) + (sum & 0xffff); /* add hi 16 to low 16 */
  sum += (sum >> 16);                 /* add carry */
  answer = ~sum;                      /* truncate to 16 bits */

  return answer;
}


int   
nd_misc_is_tcpdump_file(char *name)
{
  FILE *f;
  u_int32_t magic = 0;

  if ((f = fopen(name, "r")))
    {
      fread((void*)&magic, sizeof(u_int32_t), 1, f);
      fclose(f);
  
      if ((magic == 0xa1b2c3d4) ||
	  (magic == 0xd4c3b2a1) ||
	  (magic == 0xa1b2cd34) || /* Extended tracefile with Alexey's patches */
	  (magic == 0x34cdb2a1))
	return(1);
    }

  return (0);
}


void  
nd_misc_show_generic_dialog(char *message, char *gui_data, u_int max)
{
  GtkWidget      *w;
  GtkWidget      *w2;
  GtkObject      *adj;
  u_int           value = 0;

  if (!trace.p)
    return;

  w = create_generic_number_dialog();
  gtk_object_set_data(GTK_OBJECT(w), gui_data, (void *)1);

  D_ASSERT(w);
  w2 = gtk_object_get_data(GTK_OBJECT(w), "data_entry_label");
  D_ASSERT(w2);
  gtk_label_set_text(GTK_LABEL(w2), message);

  /* ------------------------------------------------------------- IP */
  if (!strcmp(gui_data, "ip_v"))
    {
      if (trace.p->net_data)
	value = nd_packet_ip(trace.p)->ip_v;
    }
  else if (!strcmp(gui_data, "ip_hl"))
    {
      if (trace.p->net_data)
	value = 4 * nd_packet_ip(trace.p)->ip_hl;
    }
  else if (!strcmp(gui_data, "ip_tos"))
    {
      if (trace.p->net_data)
	value = nd_packet_ip(trace.p)->ip_tos;
    }
  else if (!strcmp(gui_data, "ip_len"))
    {
      if (trace.p->net_data)
	value = ntohs(nd_packet_ip(trace.p)->ip_len);
    }
  else if (!strcmp(gui_data, "ip_id"))
    {
      if (trace.p->net_data)
	value = ntohs(nd_packet_ip(trace.p)->ip_id);
    }
  else if (!strcmp(gui_data, "ip_off"))
    {
      if (trace.p->net_data)
	value = ((ntohs(nd_packet_ip(trace.p)->ip_off) & IP_OFFMASK) << 3);
    }
  else if (!strcmp(gui_data, "ip_ttl"))
    {
      if (trace.p->net_data)
	value = nd_packet_ip(trace.p)->ip_ttl;
    }
  else if (!strcmp(gui_data, "ip_p"))
    {
      if (trace.p->net_data)
	value = nd_packet_ip(trace.p)->ip_p;
    }
  else if (!strcmp(gui_data, "ip_sum"))
    {
      if (trace.p->net_data)
	value = ntohs(nd_packet_ip(trace.p)->ip_sum);
    }
  /* ------------------------------------------------ IP in ICMP error*/
  if (!strcmp(gui_data, "icmp_ip_v"))
    {
      if (trace.p->net_data)
	value = nd_packet_icmp_error_ip(trace.p)->ip_v;
    }
  else if (!strcmp(gui_data, "icmp_ip_hl"))
    {
      if (trace.p->net_data)
	value = 4 * nd_packet_icmp_error_ip(trace.p)->ip_hl;
    }
  else if (!strcmp(gui_data, "icmp_ip_tos"))
    {
      if (trace.p->net_data)
	value = nd_packet_icmp_error_ip(trace.p)->ip_tos;
    }
  else if (!strcmp(gui_data, "icmp_ip_len"))
    {
      if (trace.p->net_data)
	value = ntohs(nd_packet_icmp_error_ip(trace.p)->ip_len);
    }
  else if (!strcmp(gui_data, "icmp_ip_id"))
    {
      if (trace.p->net_data)
	value = ntohs(nd_packet_icmp_error_ip(trace.p)->ip_id);
    }
  else if (!strcmp(gui_data, "icmp_ip_off"))
    {
      if (trace.p->net_data)
	value = ((ntohs(nd_packet_icmp_error_ip(trace.p)->ip_off) & IP_OFFMASK) << 3);
    }
  else if (!strcmp(gui_data, "icmp_ip_ttl"))
    {
      if (trace.p->net_data)
	value = nd_packet_icmp_error_ip(trace.p)->ip_ttl;
    }
  else if (!strcmp(gui_data, "icmp_ip_p"))
    {
      if (trace.p->net_data)
	value = nd_packet_icmp_error_ip(trace.p)->ip_p;
    }
  else if (!strcmp(gui_data, "icmp_ip_sum"))
    {
      if (trace.p->net_data)
	value = ntohs(nd_packet_icmp_error_ip(trace.p)->ip_sum);
    }
  /* ------------------------------------------------------------- TCP */
  else if (!strcmp(gui_data, "th_sport"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_tcp(trace.p)->th_sport);
    }
  else if (!strcmp(gui_data, "th_dport"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_tcp(trace.p)->th_dport);
    }
  else if (!strcmp(gui_data, "th_seqno"))
    {
      if (trace.p->transp_data)
	value = htonl(nd_packet_tcp(trace.p)->th_seq);
    }
  else if (!strcmp(gui_data, "th_ackno"))
    {
      if (trace.p->transp_data)
	value = htonl(nd_packet_tcp(trace.p)->th_ack);
    }
  else if (!strcmp(gui_data, "th_off"))
    {
      if (trace.p->transp_data)
	value = (nd_packet_tcp(trace.p)->th_off << 2);
    }
  else if (!strcmp(gui_data, "th_x2"))
    {
      struct tcphdr *tcphdr;

      tcphdr = nd_packet_tcp(trace.p);

      if (trace.p->transp_data)
	value = (tcphdr->th_x2 << 2) + ((tcphdr->th_flags >> 6) & 0x03);
    }
  else if (!strcmp(gui_data, "th_win"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_tcp(trace.p)->th_win);
    }
  else if (!strcmp(gui_data, "th_sum"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_tcp(trace.p)->th_sum);
    }
  else if (!strcmp(gui_data, "th_urp"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_tcp(trace.p)->th_urp);
    }
  /* ------------------------------------------------------------- UDP */
  else if (!strcmp(gui_data, "uh_sport"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_udp(trace.p)->uh_sport);
    }
  else if (!strcmp(gui_data, "uh_dport"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_udp(trace.p)->uh_dport);
    }
  else if (!strcmp(gui_data, "uh_ulen"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_udp(trace.p)->uh_ulen);
    }
  else if (!strcmp(gui_data, "uh_sum"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_udp(trace.p)->uh_sum);
    }
  /* ------------------------------------------------------------ ICMP */
  else if (!strcmp(gui_data, "icmp_type"))
    {
      if (trace.p->transp_data)
	value = nd_packet_icmp(trace.p)->icmp_type;
    }
  else if (!strcmp(gui_data, "icmp_code"))
    {
      if (trace.p->transp_data)
	value = nd_packet_icmp(trace.p)->icmp_code;
    }
  else if (!strcmp(gui_data, "icmp_cksum"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_icmp(trace.p)->icmp_cksum);
    }
  else if (!strcmp(gui_data, "icmp_error_sport"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_icmp_error_udp(trace.p)->uh_sport);
    }
  else if (!strcmp(gui_data, "icmp_error_dport"))
    {
      if (trace.p->transp_data)
	value = ntohs(nd_packet_icmp_error_udp(trace.p)->uh_dport);
    }

  D_ASSERT(w);
  w2 = gtk_object_get_data(GTK_OBJECT(w), "generic_spin");
  D_ASSERT(w2);
  adj = gtk_adjustment_new(0, 0, max, 1, 1, 1);
  gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(w2),
  				 GTK_ADJUSTMENT(adj));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(w2), value);

  gtk_widget_show(w);
}


void  
nd_misc_show_tcp_seqack_dialog(char *message, char *gui_data)
{
  GtkWidget      *w;
  GtkWidget      *w2;
  char            s[256];

  if (!trace.p)
    return;

  w = create_tcp_seqack_dialog();
  gtk_object_set_data(GTK_OBJECT(w), gui_data, (void *)1);

  D_ASSERT(w);
  w2 = gtk_object_get_data(GTK_OBJECT(w), "tcp_entry_label");
  D_ASSERT(w2);
  gtk_label_set_text(GTK_LABEL(w2), message);

  if (!strcmp(gui_data, "th_seqno"))
    {
      if (trace.p->transp_data)
	sprintf(s, "%u", (u_int32_t)htonl(nd_packet_tcp(trace.p)->th_seq));
    }
  else if (!strcmp(gui_data, "th_ackno"))
    {
      if (trace.p->transp_data)
	sprintf(s, "%u", (u_int32_t)htonl(nd_packet_tcp(trace.p)->th_ack));
    }

  D_ASSERT(w);
  w2 = gtk_object_get_data(GTK_OBJECT(w), "tcp_seqack_entry");
  D_ASSERT(w2);
  gtk_entry_set_text(GTK_ENTRY(w2), s);

  gtk_widget_show(w);
}


void  
nd_misc_show_ip_dialog(char *message, char *gui_data)
{
  GtkWidget       *w;
  GtkWidget       *w2;
  struct in_addr   addr;

  if (!trace.p)
    return;

  w = create_ip_entry_dialog();
  gtk_object_set_data(GTK_OBJECT(w), gui_data, (void *)1);

  D_ASSERT(w);
  w2 = gtk_object_get_data(GTK_OBJECT(w), "ip_entry_label");
  D_ASSERT(w2);
  gtk_label_set_text(GTK_LABEL(w2), message);

  if (!strcmp(gui_data, "ip_src"))
    {
      if (trace.p->net_data)
	addr = nd_packet_ip(trace.p)->ip_src;
    }
  else if (!strcmp(gui_data, "ip_dst"))
    {
      if (trace.p->net_data)
	addr = nd_packet_ip(trace.p)->ip_dst;
    }
  else if (!strcmp(gui_data, "icmp_ip_src"))
    {
      if (trace.p->net_data)
	addr = nd_packet_icmp_error_ip(trace.p)->ip_src;
    }
  else
    {
      if (trace.p->net_data)
	addr = nd_packet_icmp_error_ip(trace.p)->ip_dst;
    }

  if (trace.p->net_data)
    {
      D_ASSERT(w);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "ip_byte1_spin");
      D_ASSERT(w2);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(w2), ((u_char*)&addr.s_addr)[0]);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "ip_byte2_spin");
      D_ASSERT(w2);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(w2), ((u_char*)&addr.s_addr)[1]);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "ip_byte3_spin");
      D_ASSERT(w2);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(w2), ((u_char*)&addr.s_addr)[2]);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "ip_byte4_spin");
      D_ASSERT(w2);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(w2), ((u_char*)&addr.s_addr)[3]);
    }
  
  gtk_widget_show(w);
}


void  
nd_misc_show_mac_dialog(char *message, char *gui_data)
{
  GtkWidget       *w;
  GtkWidget       *w2;
  u_char          *addr = NULL;
  char             s[10];

  if (!trace.p)
    return;

  w = create_mac_entry_dialog();
  gtk_object_set_data(GTK_OBJECT(w), gui_data, (void *)1);

  D_ASSERT(w);
  w2 = gtk_object_get_data(GTK_OBJECT(w), "mac_entry_label");
  D_ASSERT(w2);
  gtk_label_set_text(GTK_LABEL(w2), message);

  if (!strcmp(gui_data, "shost"))
    {
      if (trace.p->link_data)
	addr = ((struct ether_header*)trace.p->link_data)->ether_shost;
    }
  else if (!strcmp(gui_data, "arp_source_hard"))
    {
      if (trace.p->net_data)
	{
	  char *ptr = (char*)nd_packet_arp(trace.p);
	  struct ether_header *eth = (struct ether_header*)(&ptr[8]);

	  addr = (u_char *)eth;
	}
    }
  else if (!strcmp(gui_data, "arp_target_hard"))
    {
      if (trace.p->net_data)
	{
	  struct arphdr* arphdr = nd_packet_arp(trace.p);
	  char *ptr = (char*)arphdr;
	  struct ether_header *eth = (struct ether_header*)(&ptr[8 + arphdr->ar_hln + arphdr->ar_pln]);

	  addr = (u_char *)eth;
	}
    }
  else
    {
      if (trace.p->link_data)
	addr = ((struct ether_header*)trace.p->link_data)->ether_dhost;
    }

  if (trace.p->link_data)
    {
      D_ASSERT(w);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "mac_byte1");
      D_ASSERT(w2);
      sprintf(s, "%x", addr[0]);
      gtk_entry_set_text(GTK_ENTRY(w2), s);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "mac_byte2");
      D_ASSERT(w2);
      sprintf(s, "%x", addr[1]);
      gtk_entry_set_text(GTK_ENTRY(w2), s);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "mac_byte3");
      D_ASSERT(w2);
      sprintf(s, "%x", addr[2]);
      gtk_entry_set_text(GTK_ENTRY(w2), s);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "mac_byte4");
      D_ASSERT(w2);
      sprintf(s, "%x", addr[3]);
      gtk_entry_set_text(GTK_ENTRY(w2), s);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "mac_byte5");
      D_ASSERT(w2);
      sprintf(s, "%x", addr[4]);
      gtk_entry_set_text(GTK_ENTRY(w2), s);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "mac_byte6");
      D_ASSERT(w2);
      sprintf(s, "%x", addr[5]);
      gtk_entry_set_text(GTK_ENTRY(w2), s);
    }
  
  gtk_widget_show(w);
}


void    
nd_misc_set_num_packets_label(void)
{
  char             s[128];
  GtkWidget       *w;

  if (trace.num_packets == 1)
    sprintf(s, _("%i packet."), trace.num_packets);
  else
    sprintf(s, _("%i packets."), trace.num_packets);

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "num_packets_label");
  D_ASSERT(w);
  gtk_label_set_text(GTK_LABEL(w), s);
}


u_int16_t 
nd_misc_ip_checksum(ND_Packet *p)
{
  struct ip   *iphdr;
  u_int16_t    old, result = 0;

  if (!p)
    return (0);
  
  if (nd_packet_has_protocol(p, ND_PROT_IP))
    {
      iphdr = nd_packet_ip(p);

      old = iphdr->ip_sum;
      iphdr->ip_sum = 0;
      result = nd_misc_in_cksum((u_short *)iphdr, iphdr->ip_hl<<2, 0);
      iphdr->ip_sum = old;
    }

  return result;
}


u_int16_t 
nd_misc_icmp_error_ip_checksum(ND_Packet *p)
{
  struct ip   *iphdr;
  u_int16_t    old, result = 0;

  if (!p)
    return (0);
  
  if (nd_packet_has_protocol(p, ND_PROT_ICMP_ERROR_IP))
    {
      iphdr = nd_packet_icmp_error_ip(p);
      
      old = iphdr->ip_sum;
      iphdr->ip_sum = 0;
      result = nd_misc_in_cksum((u_short *)iphdr, iphdr->ip_hl<<2, 0);
      iphdr->ip_sum = old;
    }

  return result;
}


u_int16_t
nd_misc_icmp_checksum(ND_Packet *p)
{
  struct icmp   *icmphdr;
  struct ip     *iphdr;
  u_int16_t      old, result = 0;

  if (!p)
    return (0);
  
  if (nd_packet_has_protocol(p, ND_PROT_ICMP))
    {
      iphdr   = nd_packet_ip(p);
      icmphdr = nd_packet_icmp(p);

      old = icmphdr->icmp_cksum;
      icmphdr->icmp_cksum = 0;
      result = nd_misc_in_cksum((u_short *)icmphdr, ntohs(iphdr->ip_len) - (iphdr->ip_hl * 4), 0);
      icmphdr->icmp_cksum = old;
    }

  return result;
}


u_int16_t 
nd_misc_tcp_checksum(ND_Packet *p)
{
  int             tcp_len, len;
  u_int32_t       sum, addl_pseudo;
  struct ip      *iphdr;
  struct tcphdr  *tcphdr;
  u_int16_t       old, result = 0;

  if (!p)
    return (0);

  if (nd_packet_has_protocol(p, ND_PROT_TCP))
    {
      iphdr  = nd_packet_ip(p);
      tcphdr = nd_packet_tcp(p);

      old = tcphdr->th_sum;
      tcphdr->th_sum = 0;
	  
      /* len is payload length, tcp_len is length of entire tcp packet: */
      tcp_len = IP_PAYLOAD_LENGTH(iphdr);
      len = tcp_len - tcphdr->th_off*4;
      
      if (len % 2 == 1) /* Add in pad byte. */
	sum = htons(((const u_char*) tcphdr)[tcp_len - 1] << 8);
      else
	sum = 0;
      
      sum = nd_misc_ones_complement_checksum((u_short*) &iphdr->ip_src.s_addr, 4, sum);
      sum = nd_misc_ones_complement_checksum((u_short*) &iphdr->ip_dst.s_addr, 4, sum);
      
      addl_pseudo = (htons(IPPROTO_TCP) << 16) | htons((u_short) tcp_len);
      
      sum = nd_misc_ones_complement_checksum((u_short*) &addl_pseudo, 4, sum);
      sum = nd_misc_ones_complement_checksum((u_short*) tcphdr, tcp_len, sum);

      tcphdr->th_sum = old;
      result = ~(sum);
    }
      
  return (result);
}


u_int16_t 
nd_misc_udp_checksum(ND_Packet *p)
{
  struct ip*      iphdr;
  struct udphdr  *udphdr;
  struct ipovly   ipovly;
  u_short        *w;
  u_int           i, preadd;
  u_int16_t       old, result = 0;

  if (!p)
    return (0);

  if (nd_packet_has_protocol(p, ND_PROT_UDP))
    {
      iphdr  = nd_packet_ip(p);
      udphdr = nd_packet_udp(p);
      w  = (u_short*)&ipovly;

      preadd = 0;
      
      bzero(&ipovly, sizeof(struct ipovly));
      ipovly.ih_pr  = iphdr->ip_p;
      ipovly.ih_len = udphdr->uh_ulen;
      ipovly.ih_src = iphdr->ip_src;
      ipovly.ih_dst = iphdr->ip_dst;
      
      for (i = 0; i < sizeof(struct ipovly)/sizeof(u_short) ; i++)
	preadd += *w++;

      old = udphdr->uh_sum;
      udphdr->uh_sum = 0;
      result = nd_misc_in_cksum((u_short*)udphdr, ntohs(udphdr->uh_ulen), preadd);
      udphdr->uh_sum = old;
    }

  return (result);
}


void    
nd_misc_widget_init(void)
{
  GtkWidget  *win;
  GtkWidget  *w, *w2;
  GtkStyle   *gs;
  int         i;

  /* Hide the ICMP error tab. We need that only when we actually
     have an ICMP error packet.
  */
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "notebook1");
  D_ASSERT(w);
  w2 = gtk_notebook_get_nth_page(GTK_NOTEBOOK(w), ND_TAB_ICMP_ERROR);
  if (w2)
    gtk_widget_hide(w2);

  /* Hide the dummy item in the Plugin menu */
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "plugin");
  if (w)
    gtk_widget_hide(w);

  /* Hide the dummy item in the About Plugin menu */
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "aboutdummyitem");
  if (w)
    gtk_widget_hide(w);

  /* Various tcpdump clist initializations I cannot do in Glade */
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);
  GTK_CLIST_CLASS_FW(GTK_CLIST(w))->select_all = nd_misc_gtk_clist_select_all;
  GTK_CLIST_CLASS_FW(GTK_CLIST(w))->unselect_all = nd_misc_gtk_clist_unselect_all;
  gtk_clist_set_column_justification(GTK_CLIST(w), 1, GTK_JUSTIFY_CENTER);
  gtk_clist_set_column_max_width(GTK_CLIST(w), 1, 30);
  gtk_clist_set_column_resizeable(GTK_CLIST(w), 0, FALSE);
  gtk_clist_set_column_width (GTK_CLIST(w), 1, 30);
  gtk_clist_set_column_visibility(GTK_CLIST(w), 1, FALSE);

  /* Initialize the global progress bar pointer */
  pbar = gtk_object_get_data(GTK_OBJECT(toplevel), "progressbar");
  D_ASSERT(pbar);
  gtk_progress_bar_set_activity_blocks(pbar, PBAR_ACTIVITY_BLOCKS);

  /* We want to see the togglebox in the "Apply to all" menu item */
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_apply_to_all");
  D_ASSERT(w);
  gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(w), TRUE);

  /* Pixmap and Color stuff below */
  gs = gtk_widget_get_style(toplevel);

  for (i = 0; i < 5; i++)
    {
      bg[i] = gs->bg[i];
      red[i] = gs->bg[i];
      red[i].red = (2 * red[i].red) > 65535 ? 65535 : 2 * red[i].red;
      if (red[i].red == 0)
	red[i].red = 32768;
      red[i].green /=  2;
      red[i].blue /= 2;
    }
  
  win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_realize(win);
  incomplete_pmap = gdk_pixmap_create_from_xpm(win->window, &incomplete_mask,
					       &gs->bg[GTK_STATE_NORMAL],
					       PACKAGE_DATA_DIR"/pixmaps/incomplete.xpm");
}


void    
nd_misc_widget_set_red(GtkWidget *w, int onoff)
{
  int i;
  GtkRcStyle * rc_style = gtk_rc_style_new();

  if (onoff)
    {
      for (i=0; i<5; i++)
	{
	  rc_style->bg[i] = red[i];
	  rc_style->color_flags[i] = GTK_RC_BG;
	}
    }
  else 
    {
      for (i=0; i<5; i++)
	{
	  rc_style->bg[i] = bg[i];
	  rc_style->color_flags[i] = GTK_RC_BG;
	}
    }
  
  gtk_widget_modify_style(w, rc_style);
  gtk_rc_style_unref(rc_style);
}

void      
nd_misc_gtk_clist_select_all(GtkCList *clist)
{
  int i;
  GtkCListRow *clist_row;
  GList *row_list;
  
  row_list = clist->row_list;

  for (i = 0; row_list; i++)
    {
      clist_row = row_list->data;
      clist_row->state = GTK_STATE_SELECTED;
      if (gtk_clist_row_is_visible (clist, i) != GTK_VISIBILITY_NONE)
	GTK_CLIST_CLASS_FW(clist)->draw_row (clist, NULL, i, clist_row);
      if (!clist->selection)
	{
	  clist->selection = g_list_append (clist->selection, GINT_TO_POINTER(i));
	  clist->selection_end = clist->selection;
	}
      else
	clist->selection_end = 
	  g_list_append (clist->selection_end, GINT_TO_POINTER(i))->next;
      row_list = row_list->next;
    }

  /* XXX need to properly emit a signal here? */
  on_tcpdump_list_select_all(clist, NULL);
}


void
nd_misc_gtk_clist_unselect_all(GtkCList *clist)
{
  int i;
  GtkCListRow *clist_row;
  GList *row_list;
  
  row_list = clist->row_list;

  for (i = 0; row_list; i++)
    {
      clist_row = row_list->data;
      if (clist_row->state == GTK_STATE_SELECTED)
	{
	  clist_row->state = GTK_STATE_NORMAL;
	  if (gtk_clist_row_is_visible (clist, i) != GTK_VISIBILITY_NONE)
	    GTK_CLIST_CLASS_FW(clist)->draw_row (clist, NULL, i, clist_row);
	}
      row_list = row_list->next;
    }

  g_list_free(clist->selection);
  clist->selection = NULL;
  clist->selection_end = NULL;

  /* XXX need to properly emit a signal here? */
  on_tcpdump_list_unselect_all(clist, NULL);
}


void      
nd_misc_statusbar_set(char *text)
{
  static guint timeout_id;
  GtkLabel *l;

  if (!text)
    return;

  l = gtk_object_get_data(GTK_OBJECT(toplevel), "statuslabel");
  D_ASSERT(l);
  gtk_label_set_text(l, text);
  while (gtk_events_pending())
    gtk_main_iteration();

  if (timeout_set)
    gtk_timeout_remove(timeout_id);

  timeout_id = gtk_timeout_add(STATUSBAR_TIMEOUT * 1000, nd_misc_statusbar_clear, NULL);
  timeout_set = TRUE;
}


gint
nd_misc_statusbar_clear(gpointer data)
{
  GtkLabel *l;

  l = gtk_object_get_data(GTK_OBJECT(toplevel), "statuslabel");
  D_ASSERT(l);
  gtk_label_set_text(l, "");

  timeout_set = 0;

  return (0);
  data = NULL;
}


void      
nd_misc_pbar_reset(int num_steps)
{
  pbar_current = 0;
  
  if (num_steps > 0)
    pbar_total = num_steps;
  else
    {
      if (trace.apply_to_all)
	pbar_total = trace.num_packets;
      else
	pbar_total = trace.num_sel;
    }

  gtk_progress_bar_update(pbar, 0.0);
  while (gtk_events_pending())
    gtk_main_iteration();
}


void      
nd_misc_pbar_inc(void)
{
  gfloat ratio;

  pbar_current++;

  if ( (ratio = (gfloat)pbar_current/(gfloat)pbar_total) > old_ratio + 0.01)
    {
      gtk_progress_bar_update(pbar, (gfloat)pbar_current/(gfloat)pbar_total);
      while (gtk_events_pending())
	gtk_main_iteration();
      old_ratio = ratio;
    }
}


void      
nd_misc_pbar_clear(void)
{
  gtk_progress_bar_update(pbar, 0.0);
  old_ratio = -1.0;

  while (gtk_events_pending())
    gtk_main_iteration();
}

gint 
nd_misc_pbar_timeout(gpointer data)
{
  static guint step = 0;

  if (step > PBAR_ACTIVITY_BLOCKS)
    step = 0;

  gtk_progress_set_value(GTK_PROGRESS(pbar), step++);

  while (gtk_events_pending())
    gtk_main_iteration();

  if (activity_mode)
    return (TRUE); /* Let's get called again */

  nd_misc_pbar_clear();

  return (FALSE);
  data = NULL;
}


void      
nd_misc_pbar_start_activity(void)
{
  gtk_progress_set_activity_mode(GTK_PROGRESS(pbar), TRUE);
  activity_mode  = TRUE;
  gtk_timeout_add(100, nd_misc_pbar_timeout, NULL);
}


void      
nd_misc_pbar_stop_activity(void)
{
  gtk_progress_set_activity_mode(GTK_PROGRESS(pbar), FALSE);
  activity_mode = FALSE;
}


void      
nd_misc_menu_adjust(void)
{
  int        sel_size;
  GtkWidget *w2;

  if (nd_clipboard_occupied())
    {
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_paste");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
    }
  else
    {
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_paste");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
    }

  if (trace.p)
    {
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_delete");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_copy");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_cut");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_hilight_fragments");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_dehilight_fragments");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);

      if (trace.p->is_hidden)
	{
	  w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_hide");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
	  w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_show");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
	}
      else
	{
	  w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_show");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
	  w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_hide");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
	}
      
      if (nd_packet_has_protocol(trace.p, ND_PROT_IP))
	{
	  w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_fragment");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
	}
      else
	{
	  w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_fragment");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
	}
    }
  else
    {
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_hide");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_show");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_delete");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_copy");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_cut");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_fragment");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_reassemble");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_hilight_fragments");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_dehilight_fragments");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
    }

  sel_size = trace.num_sel;
  
  if (sel_size > 1)
    {
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_reassemble");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
    }
  else
    {
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_reassemble");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
    }

  if (sel_size > 0)
    {
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_unselect_all");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
    }
  else
    {
      w2 = gtk_object_get_data(GTK_OBJECT(toplevel), "menu_packet_unselect_all");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
    }
}


void      
nd_misc_popupmenu_adjust(GtkWidget *w)
{
  int        sel_size;
  GtkWidget *w2;

  D_ASSERT(w);

  if (nd_clipboard_occupied())
    {
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_paste");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
    }
  else
    {
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_paste");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
    }

  if (trace.p)
    {
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_delete");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_copy");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_cut");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_hilight_fragments");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_dehilight_fragments");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);

      if (trace.p->is_hidden)
	{
	  w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_hide");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
	  w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_show");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
	}
      else
	{
	  w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_show");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
	  w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_hide");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
	}
      
      if (nd_packet_has_protocol(trace.p, ND_PROT_IP))
	{
	  w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_fragment");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
	}
      else
	{
	  w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_fragment");
	  D_ASSERT(w2);
	  gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
	}

    }
  else
    {
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_hide");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_show");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_delete");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_copy");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_cut");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_fragment");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_reassemble");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_hilight_fragments");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_dehilight_fragments");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
    }

  sel_size = trace.num_sel;
  
  if (sel_size > 1)
    {
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_reassemble");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), TRUE);
    }
  else
    {
      w2 = gtk_object_get_data(GTK_OBJECT(w), "packet_reassemble");
      D_ASSERT(w2);
      gtk_widget_set_sensitive(GTK_WIDGET(w2), FALSE);
    }
}


void      
nd_misc_set_windowtitle(char *filename)
{
  char      *sp = filename;
  char       s[1024];
  int        show_full_path;

  /* Set window title */

  if (!nd_prefs_get_item("show_full_path", &show_full_path))
    show_full_path = TRUE;

  if (!show_full_path)
    {
      sp = strrchr(filename, '/');
      if (sp)
	sp++;
      else
	sp = filename;
    }

  snprintf(s, sizeof(s), "NetDude: %s", sp);
  gtk_window_set_title(GTK_WINDOW(toplevel), s);
}


static gint
selection_descend_sort(gconstpointer a, gconstpointer b)
{
  if (GPOINTER_TO_UINT(a) < GPOINTER_TO_UINT(b))
    return 1;
  if (GPOINTER_TO_UINT(a) > GPOINTER_TO_UINT(b))
    return -1;

  return 0;
}


void      
nd_misc_tcpdump_list_remove_selected_rows(void)
{  
  GtkWidget *w;
  GList     *l, *l2 = NULL, *l3 = NULL;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);

  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_tcpdump_list_unselect_row, NULL);
  gtk_clist_freeze(GTK_CLIST(w));

  if (GTK_CLIST(w)->selection)
    {      
      for (l = GTK_CLIST(w)->selection; l; l = l->next)
	l2 = g_list_append(l2, l->data);
      
      l2 = g_list_sort(l2, selection_descend_sort);
      
      for (l3 = l2 ; l3; l3 = l3->next)
	gtk_clist_remove(GTK_CLIST(w), GPOINTER_TO_UINT(l3->data));
    }

  gtk_clist_thaw(GTK_CLIST(w));
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_tcpdump_list_unselect_row, NULL);
  
  if (l2)
    g_list_free(l2);
}


void      
nd_misc_tcpdump_list_incomplete_column_visibility(gboolean visible)
{
  GtkWidget *w;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);

  gtk_clist_set_column_visibility(GTK_CLIST(w), 1, visible);
}


void      
nd_misc_tcpdump_list_set_row_incomplete(int row, gboolean incomplete)
{
  GtkWidget *w;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);

  if (incomplete)
    gtk_clist_set_pixmap(GTK_CLIST(w), row, 1, incomplete_pmap, incomplete_mask);
  else
    gtk_clist_set_text(GTK_CLIST(w), row, 1, "");
}


int       
nd_misc_can_exec(char *filename)
{
  struct stat st;

  if (!filename || filename[0] == '\0')
    return (FALSE);

  /* Stat the file to see if it's executable and
     if the user may run it.
  */
  if (lstat(filename, &st) < 0)
    return (FALSE);

  if ((st.st_mode & S_IXUSR) ||
      (st.st_mode & S_IXGRP) ||
      (st.st_mode & S_IXOTH))
    return (TRUE);

  return (FALSE);
}


