/* dissect.c

   Interface with libwireshark for packet dissection. Packet dissection
   is launched from here but is ultimately performed by libwireshark.

   Copyright (C) 2007-2014 Eloy Paris

   Some code here has been borrowed from the Wireshark source code, and
   is copyright Gerald Combs and others.

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

#include <wiretap/pcap-encap.h> /* For wtap_wtap_encap_to_pcap_encap() */

#include "tcl_ws-ftypes.h"
#include "packets-priv.h"

GHashTable *dissection_vars;

#if !(GLIB_MAJOR_VERSION > 2 || (GLIB_MAJOR_VERSION == 2 && GLIB_MINOR_VERSION >= 12) )
static void
remove_all(gpointer key, gpointer value, gpointer user_data)
{
    return TRUE;
}
#endif

/********************************************************************* 
 *                         pkt_dissect_tcl()                         *
 *********************************************************************/

/*
 * Data structure passed to proto_tree_children_foreach()'s callback
 * function (proto_tree_tcl_node() ).
 */
struct callback_data {
    Tcl_Interp *interp;
    int curr_layer;
    GPtrArray *layers;
    Tcl_Obj *pdu_list;
    GHashTable *variables;
    const char *ns_name; /* Name of the Tcl namespace in which dissection
			    variables will be created. */
    int create_non_ws_vars;
};

static void
proto_tree_tcl_node(proto_node *node, gpointer data)
{
    field_info *fi;
    Tcl_Interp *interp;
    char *varname, *value;
    struct callback_data *cb_data = data;
    Tcl_Obj *obj;
    char tmp[256];
    struct addr a;
    char *format;
#ifdef DEBUG
    gchar *label_ptr;
#endif

    fi = PITEM_FINFO(node);

    interp = cb_data->interp;

    if (fi->hfinfo->id == hf_text_only) {
	/*
	 * Text label. Do nothing since we do not really have a
	 * place to put text labels on.
	 */
#ifdef DEBUG
	/* Get the text */
	label_ptr = fi->rep ? fi->rep->representation : "";

	printf("Text label: %s\n", label_ptr);
#endif
    } else if (!PROTO_ITEM_IS_HIDDEN(node) ) {
	/*
	 * Normal protocols and fields. epan/wslua/wslua_field.c in the
	 * Wireshark source code was helpful when writing this code.
	 */

	/*
	 * Note: "varname" will be g_free()'ed when the hash table
	 * is destroyed.
	 */
	varname = _pkt_varname(node, cb_data->ns_name);

	switch (fi->hfinfo->type) {
	case FT_PROTOCOL:
	    /*
	     * This field is a protocol. We create byte arrays (NetExpect
	     * BArray objects, not TCl Byte Arrays) for protocol fields so
	     * protocols can be accessed from Tcl.
	     */

	    /*
	     * Do no create a BArray if this protocol has no data (expert info.
	     * protocol is one example of a protocol that does not have data in
	     * a tvb).
	     */
	    if (!fi->ds_tvb)
		break;

	    obj = Tcl_NewBArrayObj(tvb_memdup(wmem_epan_scope(),
					      fi->ds_tvb,
					      fi->start,
					      fi->length),
				   fi->length);
	    _pkt_createvar(interp, varname, obj, cb_data->variables);

	    if (cb_data->create_non_ws_vars) {
		/*
		 * Add new PDU to the list of PDUs in this packet.
		 */
		obj = Tcl_NewStringObj(fi->hfinfo->abbrev,
				       strlen(fi->hfinfo->abbrev) );
		Tcl_ListObjAppendElement(interp, cb_data->pdu_list, obj);

		/*
		 * Make protocol of this PDU accessible from Tcl via "pdu(n,proto)".
		 */
		format = _pkt_prepend_ns_name(cb_data->ns_name, "pdu(%d,proto)");
		snprintf(tmp, sizeof(tmp), format, cb_data->curr_layer);
		Tcl_SetVar(interp, tmp, fi->hfinfo->abbrev, 0);
		free(format);

		/*
		 * Make offset into the packet of this PDU accessible from Tcl via
		 * "pdu(n,offset)".
		 */
		obj = Tcl_NewIntObj(fi->start);
		format = _pkt_prepend_ns_name(cb_data->ns_name, "pdu(%d,offset)");
		snprintf(tmp, sizeof(tmp), format, cb_data->curr_layer);
		Tcl_SetVar2Ex(interp, tmp, NULL, obj, 0);
		free(format);

		/*
		 * Make *header* length of the PDU accessible from Tcl via
		 * "pdu(n,hdr_len)".
		 */
		obj = Tcl_NewIntObj(fi->length);
		format = _pkt_prepend_ns_name(cb_data->ns_name, "pdu(%d,hdr_len)");
		snprintf(tmp, sizeof(tmp), format, cb_data->curr_layer);
		Tcl_SetVar2Ex(interp, tmp, NULL, obj, 0);
		free(format);
	    }

	    g_ptr_array_add(cb_data->layers, (gpointer) fi->hfinfo->abbrev);

	    cb_data->curr_layer++;
	    break;
	case FT_NONE:
#ifdef DEBUG
	    printf("fi->hfinfo->type is FT_NONE, fi->hfinfo->abbrev is %s\n",
		   fi->hfinfo->abbrev);
#endif
	    break;
	case FT_BOOLEAN:
	    /* wireshark bug #4049 */
	    obj = Tcl_NewBooleanObj( (int) fvalue_get_uinteger(&fi->value) );
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	case FT_UINT8:
	case FT_UINT16:
	case FT_UINT24:
	case FT_UINT32:
	case FT_FRAMENUM:
	    obj = Tcl_NewUInt32Obj(fvalue_get_uinteger(&fi->value) );
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	case FT_INT8:
	case FT_INT16:
	case FT_INT24:
	case FT_INT32:
	    obj = Tcl_NewInt32Obj(fvalue_get_sinteger(&fi->value) );
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	case FT_FLOAT:
	case FT_DOUBLE:
	    obj = Tcl_NewDoubleObj(fvalue_get_floating(&fi->value) );
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	case FT_INT64:
	    obj = Tcl_NewInt64Obj(fvalue_get_integer64(&fi->value) );
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	case FT_UINT64:
	    obj = Tcl_NewUInt64Obj(fvalue_get_integer64(&fi->value) );
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	case FT_ETHER:
	    a.addr_type = ADDR_TYPE_ETH;
	    a.addr_bits = ETH_ADDR_BITS;
	    memcpy(&a.addr_ip,
		   tvb_memdup(wmem_epan_scope(),
			      fi->ds_tvb,
			      fi->start,
			      fi->length),
		   ETH_ADDR_LEN);
	    obj = Tcl_NewAddrObj(&a);
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	case FT_IPv4:
	    a.addr_type = ADDR_TYPE_IP;
	    a.addr_bits = IP_ADDR_BITS;
	    memcpy(&a.addr_ip,
		   tvb_memdup(wmem_epan_scope(),
			      fi->ds_tvb,
			      fi->start,
			      fi->length),
		   IP_ADDR_LEN);
	    obj = Tcl_NewAddrObj(&a);
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	case FT_IPv6:
	    a.addr_type = ADDR_TYPE_IP6;
	    a.addr_bits = IP6_ADDR_BITS;
	    memcpy(&a.addr_ip,
		   tvb_memdup(wmem_epan_scope(),
			      fi->ds_tvb,
			      fi->start,
			      fi->length),
		   IP6_ADDR_LEN);
	    obj = Tcl_NewAddrObj(&a);
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	case FT_STRING:
	case FT_STRINGZ:
	    value = fvalue_to_string_repr(&fi->value, FTREPR_DISPLAY, NULL);
	    obj = Tcl_NewStringObj(value, strlen(value) );
	    _pkt_createvar(interp, varname, obj, cb_data->variables);

	    g_free(value); /* fvalue_to_string_repr() allocated for us.
			      Needs to be freed. */
	    break;
	case FT_BYTES: 
	case FT_UINT_BYTES: 
	case FT_GUID:
	case FT_OID:
	    obj = Tcl_NewBArrayObj(tvb_memdup(wmem_epan_scope(),
					      fi->ds_tvb,
					      fi->start,
					      fi->length),
				   fi->length);
	    _pkt_createvar(interp, varname, obj, cb_data->variables);
	    break;
	default:
	    value = fvalue_to_string_repr(&fi->value, FTREPR_DISPLAY, NULL);
	    obj = Tcl_NewStringObj(value, strlen(value) );
	    _pkt_createvar(interp, varname, obj, cb_data->variables);

	    g_free(value); /* fvalue_to_string_repr() allocated for us.
			      Needs to be freed. */
	}
    }

    /*
     * What is this assert() in the Wireshark code for, again? I don't
     * know why this condition needs to be satisfied but it was there
     * when I adapted to netexpect this libwireshark code.
     *
     * peloy@netexpect.org
     */
    g_assert(fi->tree_type >= -1 && fi->tree_type < num_tree_types);

    /*
     * This node has children. Since we always make all levels available to the
     * Tcl world we recurse here to process the children nodes.
     */
    if (node->first_child != NULL)
	proto_tree_children_foreach(node, proto_tree_tcl_node, data);
}

static void
proto_tree_tcl(Tcl_Interp *interp, epan_dissect_t *edt, const char *ns_name,
	       int create_non_ws_vars)
{
    struct callback_data data;
    Tcl_Obj *obj;
    int n, hdr_len, payload_len;
    char tmp[256];
    char *varname, *format;

    if (create_non_ws_vars) {
	/*
	 * "pdu" is an important Tcl array that gets populated while going
	 * over the dissection results. We delete the array before creating
	 * it again.
	 */
	Tcl_UnsetVar(interp, "pdu", 0);
    }

    data.interp = interp;
    data.curr_layer = 0;
    data.layers = g_ptr_array_new();
    data.pdu_list = Tcl_NewListObj(0, NULL);
    data.variables = dissection_vars;
    data.ns_name = ns_name;
    data.create_non_ws_vars = create_non_ws_vars;

    /*
     * Remove existing entries in the variables hash table so we can
     * start fresh.
     */
#if GLIB_MAJOR_VERSION > 2 || (GLIB_MAJOR_VERSION == 2 && GLIB_MINOR_VERSION >= 12)
    g_hash_table_remove_all(dissection_vars);
#else
    g_hash_table_foreach_remove(dissection_vars, remove_all, NULL);
#endif

    /*
     * Handle dissection results.
     */
    proto_tree_children_foreach(edt->tree, proto_tree_tcl_node, &data);

    if (create_non_ws_vars) {
	varname = _pkt_prepend_ns_name(ns_name, "pdu(list)");
	Tcl_SetVar2Ex(interp, varname, NULL, data.pdu_list, 0);
	free(varname);

	/*
	 * Create the "pdu(nnn,tot_len)" and "pdu(nnn,payload_len)" variables.
	 * These contain the total length of each PDU, which is header length +
	 * payload length, and the payload length, respectively.  This code is not
	 * pretty or efficient but I can't find a better way of doing this based on
	 * what libwireshark gives us.
	 */

	Tcl_ListObjLength(NULL, data.pdu_list, &n);
	for (n--, payload_len = 0; n; n--) {
	    /* pdu(nnn,payload_len) */
	    obj = Tcl_NewIntObj(payload_len);
	    format = _pkt_prepend_ns_name(ns_name, "pdu(%d,payload_len)");
	    snprintf(tmp, sizeof(tmp), format, n);
	    free(format);
	    Tcl_SetVar2Ex(interp, tmp, NULL, obj, 0);

	    /* Get length of this protocol's header */
	    format = _pkt_prepend_ns_name(ns_name, "pdu(%d,hdr_len)");
	    snprintf(tmp, sizeof(tmp), format, n);
	    free(format);
	    obj = Tcl_GetVar2Ex(interp, tmp, NULL, 0);
	    Tcl_GetIntFromObj(NULL, obj, &hdr_len);

	    /* pdu(nnn,tot_len) */
	    obj = Tcl_NewIntObj(hdr_len + payload_len);
	    format = _pkt_prepend_ns_name(ns_name, "pdu(%d,tot_len)");
	    snprintf(tmp, sizeof(tmp), format, n);
	    free(format);
	    Tcl_SetVar2Ex(interp, tmp, NULL, obj, 0);

	    payload_len += hdr_len;
	}

	/*
	 * Add to the hash table that holds names of Tcl variables that were
	 * created during packet dissection some other variables related to
	 * packet dissection but not really created during the packet
	 * dissection process.
	 */
	g_hash_table_insert(dissection_vars, g_strdup("_"), (gpointer) 1);
	g_hash_table_insert(dissection_vars, g_strdup("pdu"), (gpointer) 1);
    }

    g_ptr_array_free(data.layers, TRUE);
}

/*
 * Dissects a packet by using libwireshark services. Create Tcl variables
 * for the results. This function is very generic since it is called by
 * different parts of Network Expect.
 *
 * Input:
 *
 *	interp: Tcl interpreter
 *	packet: pointer to packet to dissect
 *	fdata: frame_data structure (initialized by caller)
 *	rfcode: a compiled wireshark display filter
 *	ns_name: Tcl namespace name that will store dissection variables
 *	create_non_ws_vars: boolean that controls whether to create the
 *	    non-libwireshark variables (pdu, _, etc.)
 *
 * Returns:
 *
 *	0 if the dissected packet does not pass the wireshark display filter;
 *	!= 0 otherwise.
 */
int
pkt_dissect_tcl(Tcl_Interp *interp, const u_char *packet,
		struct wtap_pkthdr *wphdr, frame_data *fdata,
		dfilter_t *rfcode, const char *ns_name, int create_non_ws_vars)
{
    epan_dissect_t *edt;
    Tcl_Obj *obj;
    int passed;
    struct pcap_pkthdr pkt_hdr;
    char *varname;
    tvbuff_t *tvb;

    /* XXX -- need to properly dispose of this tvb. */
    tvb = tvb_new_real_data(packet, fdata->pkt_len, fdata->cap_len);

    /*
     * Set up the Tcl namespace that will hold all the Tcl variables
     * produced by the libwireshark packet dissection.
     */
    if (ns_name && !Tcl_FindNamespace(interp, ns_name, NULL, TCL_GLOBAL_ONLY) )
	Tcl_CreateNamespace(interp, ns_name, NULL, NULL);

    edt = epan_dissect_new(epan,
			   1 /* create_proto_tree */,
			   1 /* proto_tree_visible */);

    if (rfcode)
	epan_dissect_prime_dfilter(edt, rfcode);

    /* Outsource the heavy lifting to libwireshark */
    epan_dissect_run(edt, 0, wphdr, tvb, fdata, NULL /* cinfo */);

    /* Run the read filter if we have one. */
    passed = rfcode ? dfilter_apply_edt(rfcode, edt) : 1;

    if (passed)
	/*
	 * The packet has been dissected and it passed the wireshark display
	 * filter. Now make the results available to the Tcl world.
	 */
	proto_tree_tcl(interp, edt, ns_name, create_non_ws_vars);

    epan_dissect_free(edt);

    if (create_non_ws_vars) {
	/*
	 * Create the $_(packet) Tcl variable, which contains the dissected
	 * packet's raw data as a Tcl BArray object.
	 */
	obj = Tcl_NewBArrayObj(packet, fdata->cap_len);
	varname = _pkt_prepend_ns_name(ns_name, PACKETDATA_VARNAME);
	Tcl_SetVar2Ex(interp, varname, NULL, obj, 0);
	free(varname);

	/*
	 * Create the $_(raw) Tcl variable, which contains the packet
	 * as a Tcl Packet object.
	 */
	pkt_hdr.len = fdata->pkt_len;
	pkt_hdr.caplen = fdata->cap_len;
	pkt_hdr.ts.tv_sec = fdata->abs_ts.secs;
	pkt_hdr.ts.tv_usec = fdata->abs_ts.nsecs/1000;

	obj = Tcl_NewPacketObj(packet, &pkt_hdr,
			       wtap_wtap_encap_to_pcap_encap(fdata->lnk_t) );
	varname = _pkt_prepend_ns_name(ns_name, PACKETRAW_VARNAME);
	Tcl_SetVar2Ex(interp, varname, NULL, obj, 0);
	free(varname);
    }

    return passed;
}

/********************************************************************* 
 *	     Functions to access dissection Tcl variables            *
 *********************************************************************/

int
_pkt_get_uint8(Tcl_Interp *interp, const char *varname, uint8_t *p)
{
    Tcl_Obj *obj;

    obj = Tcl_GetVar2Ex(interp, varname, NULL, 0);
    if (obj) {
	/* FIXME: check return from Tcl_GetUInt8FromObj() */
	Tcl_GetUInt8FromObj(obj, p);
	return 0;
    } else
	return -1;
}

int
_pkt_get_uint16(Tcl_Interp *interp, const char *varname, uint16_t *p)
{
    Tcl_Obj *obj;

    obj = Tcl_GetVar2Ex(interp, varname, NULL, 0);
    if (obj) {
	/* FIXME: check return from Tcl_GetUInt16FromObj() */
	Tcl_GetUInt16FromObj(obj, p);
	*p = htons(*p);
	return 0;
    } else
	return -1;
}

#if 0
int
_pkt_get_list_uint16(Tcl_Interp *interp, const char *varname, int index,
		     uint16_t *p)
{
    Tcl_Obj *obj, *list_obj;

    list_obj = Tcl_GetVar2Ex(interp, varname, NULL, 0);
    if (list_obj) {
	if (Tcl_ListObjIndex(interp, list_obj, index, &obj) == TCL_ERROR
	    || obj == NULL)
	    return -1;

	if (Tcl_GetUInt16FromObj(obj, p) == TCL_ERROR) return -1;
	*p = htons(*p);
	return 0;
    } else
	return -1;
}
#endif

int
_pkt_get_ipaddr(Tcl_Interp *interp, const char *varname, ip_addr_t *p)
{
    Tcl_Obj *obj;

    obj = Tcl_GetVar2Ex(interp, varname, NULL, 0);
    if (obj) {
	/* FIXME: check return from Tcl_GetIPAddrFromObj() */
	Tcl_GetIPAddrFromObj(obj, p);
	*p = htonl(*p);
	return 0;
    } else
	return -1;
}

int
_pkt_get_boolean(Tcl_Interp *interp, const char *varname, int *p)
{
    Tcl_Obj *obj;

    obj = Tcl_GetVar2Ex(interp, varname, NULL, 0);
    if (obj) {
	/* FIXME: check return from Tcl_GetBooleanFromObj() */
	Tcl_GetBooleanFromObj(interp, obj, p);
	return 0;
    } else
	return -1;
}

/********************************************************************* 
 *			   pkt_dissect_show()                        *
 *********************************************************************/

static int print_hidden = 1;

/*
 * Data structure passed to proto_tree_children_foreach()'s callback
 * function (proto_tree_print_node() ).
 */
struct callback_data_show {
    int level;
    char *prefix;
};

static void
proto_tree_print_node(proto_node *node, gpointer data)
{
    gchar *label_ptr;
    gchar label_str[ITEM_LABEL_LENGTH];
    field_info *fi;
    struct callback_data_show *cb_data = data;

    fi = PITEM_FINFO(node);

    if (PROTO_ITEM_IS_HIDDEN(node) && !print_hidden)
	return;

    /* was a free format label produced? */
    if (fi->rep) {
	label_ptr = fi->rep->representation;
    } else { /* no, make a generic label */
	label_ptr = label_str;
	proto_item_fill_label(fi, label_str);
    }

    if (PROTO_ITEM_IS_GENERATED(node) ) {
	label_ptr = g_strdup_printf("[%s]", label_ptr);
    }

    printf("%s%s\n", cb_data->prefix, label_ptr);

    if (PROTO_ITEM_IS_GENERATED(node) ) {
	g_free(label_ptr);
    }

    g_assert(fi->tree_type >= -1 && fi->tree_type < num_tree_types);

    if (node->first_child != NULL) {
	cb_data->prefix = make_prefix(++cb_data->level);
	proto_tree_children_foreach(node, proto_tree_print_node, data);
	cb_data->prefix = make_prefix(--cb_data->level);
    }
}

static void
proto_tree_print_simple_node(proto_node *node, gpointer data)
{
    gchar *label_ptr;
    field_info *fi;
    char *s;
    struct callback_data_show *cb_data = data;

    fi = PITEM_FINFO(node);

    if (fi->hfinfo->id == hf_text_only) {
	/* Text label */
	label_ptr = fi->rep ? fi->rep->representation : "";

	printf("%sText label: %s\n", cb_data->prefix, label_ptr);
    } else if (!PROTO_ITEM_IS_HIDDEN(node)
	       || (PROTO_ITEM_IS_HIDDEN(node) && print_hidden) ) {
	/*
	 * Normal protocols and fields
	 */

	switch (fi->hfinfo->type) {
	case FT_PROTOCOL:
	    printf("---[ %s ]---\n", fi->hfinfo->abbrev);
#ifdef DEBUG
	    printf("proto = %s, start = %d, len = %d\n",
		   fi->hfinfo->abbrev, fi->start, fi->length);
#endif
	    break;
	case FT_NONE:
#ifdef DEBUG
	    printf("fi->hfinfo->type is FT_NONE\n");
#endif
	    break;
	default:
	    s = fvalue_to_string_repr(&fi->value, FTREPR_DISPLAY, NULL);
	    printf("%s%s: %s\n", cb_data->prefix, fi->hfinfo->abbrev, s);
	    g_free(s); /* fvalue_to_string_repr() allocated for us. Needs to
			  be freed. */
	}
    }

    /*
     * What is this assert() in the Wireshark code for, again? I don't
     * know why this condition needs to be satisfied; I just stole the code
     * and this came with the bounty. EP.-
     */
    g_assert(fi->tree_type >= -1 && fi->tree_type < num_tree_types);

    /*
     * This node has children. We need to recurse to handle them.
     */
    if (node->first_child != NULL) {
	cb_data->prefix = make_prefix(++cb_data->level);
	proto_tree_children_foreach(node, proto_tree_print_simple_node, data);
	cb_data->prefix = make_prefix(--cb_data->level);
    }
}

/*
 * Dissects a packet by using libwireshark services. Print results to
 * stdout.
 *
 * packet: pointer to packet to dissect
 * fdata: frame_data structure (initialized by caller)
 */
void
pkt_dissect_show(const u_char *packet, struct wtap_pkthdr *wphdr,
		 frame_data *fdata, int verbose)
{
    epan_dissect_t *edt;
    struct callback_data_show cb_data;
    tvbuff_t *tvb;

    /* XXX -- need to properly dispose of this tvb. */
    tvb = tvb_new_real_data(packet, fdata->pkt_len, fdata->cap_len);

    edt = epan_dissect_new(epan,
			   1 /* create_proto_tree */,
			   1 /* proto_tree_visible */);

    /* Outsource the heavy lifting to libwireshark */
    epan_dissect_run(edt, 0, wphdr, tvb, fdata, NULL /* cinfo */);

    /*
     * The packet has been dissected. Now print results.
     */
    cb_data.level = 0;
    cb_data.prefix = "";

    proto_tree_children_foreach(edt->tree,
				verbose ? proto_tree_print_node
					: proto_tree_print_simple_node,
				&cb_data);

    epan_dissect_free(edt);
}
