/* pdu-sntp.c
 
   Builder for Simple Network Time Protocol (SNTP) PDUs (RFC 1769)

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"

struct ntp_ts {
    uint32_t int_part;
    uint32_t frac_part;
};

struct ntp_hdr {
    unsigned leap:2;
    unsigned version:3;
    unsigned mode:3;

    uint8_t stratum;
    uint8_t poll;
    uint8_t precision;

    uint32_t root_delay;
    uint32_t root_dispersion;
    uint32_t ref_id;

    struct ntp_ts ref_ts;
    struct ntp_ts orig_ts;
    struct ntp_ts rx_ts;
    struct ntp_ts tx_ts;
};

static const pdu_t pdu_sntp = {
    .name = "sntp",
    .description = "Simple Network Time Protocol",
    .documented_in = "RFC 1769",
    .fields = (field_t []) {
	{
	    .name = "leap",
	    .type = PDU_FTYPE_BITFIELD,
	    .offset = 0,
	    .size = 2,
	    .data = (struct pdu_dict []) { /* Enumeration */
		{"nowarning",	(uint32_t []) {0x00} },
		{"longminute",	(uint32_t []) {0x01} },
		{"shortminute",	(uint32_t []) {0x02} },
		{"nosync",	(uint32_t []) {0x03} },
		{NULL,		NULL}
	    },
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 3} }
	},
	{
	    .name = "version",
	    .type = PDU_FTYPE_BITFIELD,
	    .offset = 2,
	    .size = 3,
	    .defval = (defval_t []) {
		{.type = PDU_DEF_NUM, ._number = 3}
	    }
	},
	{
	    .name = "mode",
	    .type = PDU_FTYPE_BITFIELD,
	    .offset = 5,
	    .size = 3,
	    .data = (struct pdu_dict []) { /* Enumeration */
		{"reserved",	(uint32_t []) {0x00} },
		{"sym_active",	(uint32_t []) {0x01} },
		{"sym_passive",	(uint32_t []) {0x02} },
		{"client",	(uint32_t []) {0x03} },
		{"server",	(uint32_t []) {0x04} },
		{"broadcast",	(uint32_t []) {0x05} },
		{"control",	(uint32_t []) {0x06} },
		{"private",	(uint32_t []) {0x07} },
		{NULL,		NULL}
	    },
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 3} }
	},
	{
	    .name = "stratum",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct ntp_hdr, stratum),
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 2} }
	},
	{
	    .name = "poll",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct ntp_hdr, poll),
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 6} }
	},
	{
	    .name = "precision",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct ntp_hdr, precision)
	},
	{
	    .name = "delay",
	    .type = PDU_FTYPE_FIXEDP32,
	    .offset = offsetof(struct ntp_hdr, root_delay)
	},
	{
	    .name = "dispersion",
	    .type = PDU_FTYPE_FIXEDP32,
	    .offset = offsetof(struct ntp_hdr, root_dispersion)
	},
	{
	    .name = "refid",
	    .type = PDU_FTYPE_UINT32,
	    .offset = offsetof(struct ntp_hdr, ref_id),
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM,
				       ._number = 0x7f000001 /* 127.0.0.1 */} }
	},
	{
	    .name = "ref",
	    .type = PDU_FTYPE_FIXEDP64,
	    .offset = offsetof(struct ntp_hdr, ref_ts)
	},
	{
	    .name = "orig",
	    .type = PDU_FTYPE_FIXEDP64,
	    .offset = offsetof(struct ntp_hdr, orig_ts)
	},
	{
	    .name = "rx",
	    .type = PDU_FTYPE_FIXEDP64,
	    .offset = offsetof(struct ntp_hdr, rx_ts)
	},
	{
	    .name = "tx",
	    .type = PDU_FTYPE_FIXEDP64,
	    .offset = offsetof(struct ntp_hdr, tx_ts)
	},
	{
	    .name = NULL
	}
    },
    .len = sizeof(struct ntp_hdr)
};

void
_pb_register_sntp(void)
{
    _pb_register_protocol(&pdu_sntp);
}
