/*
 * tcmsg_filter.c - traffic control filter message parser
 * Copyright (C) 2014 Tetsumune KISO <t2mune@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "nield.h"
#include "rtnetlink.h"

/*
 * parse traffic control filter messages
 */
int parse_tcmsg_filter(struct nlmsghdr *nlh)
{
    struct tcmsg *tcm;
    int tcm_len;
    struct rtattr *tca[__TCA_MAX];
    char msg[MAX_MSG_SIZE] = "";
    char *mp = msg;
    char ifname[IFNAMSIZ];
    char handle[MAX_STR_SIZE] = "";
    char kind[IFNAMSIZ] = "(unknown)";
    char proto[MAX_STR_SIZE];
    int log_opts = get_log_opts();

    /* debug nlmsghdr */
    if(log_opts & L_DEBUG)
        debug_nlmsg(0, nlh);

    /* get tcmsg */
    tcm_len = NLMSG_PAYLOAD(nlh, 0);
    if(tcm_len < sizeof(*tcm)) {
        rec_log("error: %s: tcmsg: length too short", __func__);
        return(1);
    }
    tcm = (struct tcmsg *)NLMSG_DATA(nlh);

    /* parse traffic control message attributes */
    parse_tc(tca, nlh);

    /* debug tcmsg */
    if(log_opts & L_DEBUG)
        debug_tcmsg(0, nlh, tcm, tca, tcm_len);

    /* kind of message */
    switch(nlh->nlmsg_type) {
        case RTM_NEWTFILTER:
            mp = add_log(msg, mp, "tc filter added: ");
            break;
        case RTM_DELTFILTER:
            mp = add_log(msg, mp, "tc filter deleted: ");
            break;
        default:
            rec_log("error: %s: nlmsg_type: unknown message", __func__);
            return(1);
    }

    /* get interface name */
    if_indextoname_from_lists(tcm->tcm_ifindex, ifname);

    mp = add_log(msg, mp, "interface=%s ", ifname); 

    /* get qdisc kind */
    if(tca[TCA_KIND])
        strncpy(kind, (char *)RTA_DATA(tca[TCA_KIND]), sizeof(kind));

    /* get filter handle */
    if(!strncmp(kind, "u32", sizeof(kind)))
        parse_u32_handle(handle, sizeof(handle), tcm->tcm_handle);
    else
        snprintf(handle, sizeof(handle), "0x%x", tcm->tcm_handle);

    mp = add_log(msg, mp, "handle=%s ", handle);

    /* get priority */
    mp = add_log(msg, mp, "priority=%u ", TC_H_MAJ(tcm->tcm_info)>>16);

    /* get priority */
    strncpy(proto, convert_eth_p(TC_H_MIN(tcm->tcm_info), 0), sizeof(proto));
    if(strlen(proto))
        mp = add_log(msg, mp, "protocol=%s ", proto);
    else
        mp = add_log(msg, mp, "protocol=0x%04x ", ntohs(TC_H_MIN(tcm->tcm_info)));

    /* get filter options */
    mp = add_log(msg, mp, "filter=%s ", kind);

    if(tca[TCA_OPTIONS]) {
        if(!strncmp(kind, "u32", sizeof(kind))) {
            if(parse_tca_options_u32(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
            return(0);
        } else if(!strncmp(kind, "rsvp", sizeof(kind))) {
            if(parse_tca_options_rsvp(msg, &mp, tcm, tca[TCA_OPTIONS]))
                return(1);
            return(0);
        } else if(!strncmp(kind, "route", sizeof(kind))) {
            if(parse_tca_options_route(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
            return(0);
        } else if(!strncmp(kind, "fw", sizeof(kind))) {
            if(parse_tca_options_fw(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
            return(0);
        } else if(!strncmp(kind, "tcindex", sizeof(kind))) {
            if(parse_tca_options_tcindex(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
            return(0);
#if HAVE_DECL_TCA_FLOW_UNSPEC
        } else if(!strncmp(kind, "flow", sizeof(kind))) {
            if(parse_tca_options_flow(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
            return(0);
#endif
        } else if(!strncmp(kind, "basic", sizeof(kind))) {
            if(parse_tca_options_basic(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
            return(0);
#if HAVE_DECL_TCA_CGROUP_UNSPEC
        } else if(!strncmp(kind, "cgroup", sizeof(kind))) {
            if(parse_tca_options_cgroup(msg, &mp, tca[TCA_OPTIONS]))
                return(1);
            return(0);
#endif
        }
    }

    rec_log("%s", msg);

    return(0);
}

/*
 * parse attribute TCA_*_CLASSID
 */
int parse_tca_classid(char *msg, char **mp, struct rtattr *tca)
{
    char classid[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(tca) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    parse_tc_classid(classid, sizeof(classid), *(unsigned *)RTA_DATA(tca));
    *mp = add_log(msg, *mp, "classid=%s ", classid);

    return(0);
}

/*
 * parse attribute TCA_*_INDEV
 */
int parse_tca_indev(char *msg, char **mp, struct rtattr *tca)
{
    char name[IFNAMSIZ];

    if(!RTA_PAYLOAD(tca)) {
        rec_log("error: %s: no payload", __func__);
        return(1);
    } else if(RTA_PAYLOAD(tca) > sizeof(name)) {
        rec_log("error: %s: payload too long", __func__);
        return(1);
    }
    strncpy(name, (char *)RTA_DATA(tca), sizeof(name));
    *mp = add_log(msg, *mp, "in=%s ", name);

    return(0);
}

/*
 * parse attribute TCA_*_MASK
 */
int parse_tca_mask(char *msg, char **mp, struct rtattr *tca)
{
    if(RTA_PAYLOAD(tca) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "mask=0x%08x ", *(unsigned *)RTA_DATA(tca));

    return(0);
}

/*
 * parse filter handle
 */
void parse_u32_handle(char *p, int len, unsigned handle)
{
    if(TC_U32_HTID(handle))
        p += snprintf(p, len, "%x", TC_U32_HTID(handle)>>20);

    p += snprintf(p, len, ":");

    if(TC_U32_HASH(handle))
        p += snprintf(p, len, "%x", TC_U32_HASH(handle));

    p += snprintf(p, len, ":");

    if(TC_U32_NODE(handle))
        p += snprintf(p, len, "%x", TC_U32_NODE(handle));
}

/*
 * parse u32 options
 */
int parse_tca_options_u32(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *u32[__TCA_U32_MAX];
    char *mp_tmp = *mp;

    parse_u32(u32, tca);

    if(u32[TCA_U32_CLASSID])
        if(parse_tca_classid(msg, mp, u32[TCA_U32_CLASSID]))
            return(1);

    if(u32[TCA_U32_HASH])
        if(parse_tca_u32_hash(msg, mp, u32[TCA_U32_HASH]))
            return(1);

    if(u32[TCA_U32_LINK])
        if(parse_tca_u32_link(msg, mp, u32[TCA_U32_LINK]))
            return(1);

    if(u32[TCA_U32_DIVISOR])
        if(parse_tca_u32_divisor(msg, mp, u32[TCA_U32_DIVISOR]))
            return(1);

    if(u32[TCA_U32_INDEV])
        if(parse_tca_indev(msg, mp, u32[TCA_U32_INDEV]))
            return(1);

    if(u32[TCA_U32_MARK])
        if(parse_tca_u32_mark(msg, mp, u32[TCA_U32_MARK]))
            return(1);

    if(*mp != mp_tmp)
        rec_log("%s", msg);

    /* rollback pointer */
    *mp = mp_tmp;

    /* logging for each attribute below */
    if(u32[TCA_U32_SEL])
        if(parse_tca_u32_sel(msg, *mp, u32[TCA_U32_SEL]))
            return(1);

    if(u32[TCA_U32_POLICE])
        if(parse_tca_act_options_police(msg, *mp, u32[TCA_U32_POLICE]))
            return(1);

    if(u32[TCA_U32_ACT])
        if(parse_tca_acts(msg, *mp, u32[TCA_U32_ACT]))
            return(1);

    return(0);
}

/*
 * parse attribute TCA_U32_HASH
 */
int parse_tca_u32_hash(char *msg, char **mp, struct rtattr *u32)
{
    unsigned htid;

    if(RTA_PAYLOAD(u32) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    htid = *(unsigned *)RTA_DATA(u32);

    *mp = add_log(msg, *mp, "hash(table/bucket)=0x%x/0x%x ",
        TC_U32_USERHTID(htid), TC_U32_HASH(htid));

    return(0);
}

/*
 * parse attribute TCA_U32_LINK
 */
int parse_tca_u32_link(char *msg, char **mp, struct rtattr *u32)
{
    char handle[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(u32) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    parse_u32_handle(handle, sizeof(handle), *(unsigned *)RTA_DATA(u32));
    *mp = add_log(msg, *mp, "link=%s ", handle);

    return(0);
}

/*
 * parse attribute TCA_U32_DIVISOR
 */
int parse_tca_u32_divisor(char *msg, char **mp, struct rtattr *u32)
{
    if(RTA_PAYLOAD(u32) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "divisor=%d ", *(unsigned *)RTA_DATA(u32));

    return(0);
}

/*
 * parse attribute TCA_U32_MARK
 */
int parse_tca_u32_mark(char *msg, char **mp, struct rtattr *u32)
{
    struct tc_u32_mark *mark;

    if(RTA_PAYLOAD(u32) < sizeof(*mark)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    mark = (struct tc_u32_mark *)RTA_DATA(u32);
    *mp = add_log(msg, *mp, "mark(value/mask)=0x%04x/0x%04x ", mark->val, mark->mask);

    return(0);
}

/*
 * parse attribute TCA_U32_SEL
 */
int parse_tca_u32_sel(char *msg, char *mp, struct rtattr *u32)
{
    struct tc_u32_sel *sel;
    char flags_list[MAX_STR_SIZE] = "";
    char *mp_tmp = mp;

    if(RTA_PAYLOAD(u32) < sizeof(*sel)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    sel = (struct tc_u32_sel *)RTA_DATA(u32);
    convert_tc_u32_flags(sel->flags, flags_list, sizeof(flags_list),  0),
    add_log(msg, mp, "flags=%s offshift=%d nkeys=%d offmask=0x%04x "
        "off=%hu offoff=%hd hoff=%hd hmask=0x%08x ",
        flags_list, sel->offshift, sel->nkeys, ntohs(sel->offmask),
        sel->off, sel->offoff, sel->hoff, ntohl(sel->hmask));
    rec_log("%s", msg);

    /* rollback pointer */
    mp = mp_tmp;

    int i, len;
    struct tc_u32_key *keys = sel->keys;

    len = sizeof(*sel) + (sizeof(*keys) * sel->nkeys);
    if(RTA_PAYLOAD(u32) < len) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }

    for(i = 0; i < sel->nkeys; i++, keys++) {
        add_log(msg, mp, "key=%d value=0x%08x mask=0x%08x offset=%d offmask=0x%08x ",
            i + 1, ntohl(keys->val), ntohl(keys->mask),
            keys->off, ntohl(keys->offmask));
        rec_log("%s", msg);
    }

    return(0);
}

/*
 * parse rsvp options
 */
int parse_tca_options_rsvp(char *msg, char **mp, struct tcmsg *tcm, struct rtattr *tca)
{
    struct rtattr *rsvp[__TCA_RSVP_MAX];
    char *mp_tmp = *mp;

    parse_rsvp(rsvp, tca);

    if(rsvp[TCA_RSVP_CLASSID])
        if(parse_tca_classid(msg, mp, rsvp[TCA_RSVP_CLASSID]))
            return(1);

    if(rsvp[TCA_RSVP_DST])
        if(parse_tca_rsvp_dst(msg, mp, tcm, rsvp[TCA_RSVP_DST]))
            return(1);

    if(rsvp[TCA_RSVP_SRC])
        if(parse_tca_rsvp_src(msg, mp, tcm, rsvp[TCA_RSVP_SRC]))
            return(1);

    if(rsvp[TCA_RSVP_PINFO])
        if(parse_tca_rsvp_pinfo(msg, mp, rsvp[TCA_RSVP_PINFO]))
            return(1);

    if(*mp != mp_tmp)
        rec_log("%s", msg);

    /* rollback pointer */
    *mp = mp_tmp;

    /* logging for each attribute below */
    if(rsvp[TCA_RSVP_POLICE])
        if(parse_tca_act_options_police(msg, *mp, rsvp[TCA_RSVP_POLICE]))
            return(1);

    if(rsvp[TCA_RSVP_ACT])
        if(parse_tca_acts(msg, *mp, rsvp[TCA_RSVP_ACT]))
            return(1);

    return(0);
}

/*
 * parse attribute TCA_RSVP_DST
 */
int parse_tca_rsvp_dst(char *msg, char **mp, struct tcmsg *tcm, struct rtattr *rsvp)
{
    char addr[INET6_ADDRSTRLEN+1] = "";
    int res;

    res = inet_ntop_tc_addr(tcm, rsvp, addr, sizeof(addr));
    if(res) {
        rec_log("error: %s: %s", __func__,
            (res == 1) ? strerror(errno) : "payload too short");
        return(1);
    }
    *mp = add_log(msg, *mp, "destination=%s ", addr);

    return(0);
}

/*
 * parse attribute TCA_RSVP_SRC
 */
int parse_tca_rsvp_src(char *msg, char **mp, struct tcmsg *tcm, struct rtattr *rsvp)
{
    char addr[INET6_ADDRSTRLEN+1] = "";
    int res;

    res = inet_ntop_tc_addr(tcm, rsvp, addr, sizeof(addr));
    if(res) {
        rec_log("error: %s: %s", __func__,
            (res == 1) ? strerror(errno) : "payload too short");
        return(1);
    }
    *mp = add_log(msg, *mp, "source=%s ", addr);

    return(0);
}

/*
 * parse attribute TCA_RSVP_PINFO
 */
int parse_tca_rsvp_pinfo(char *msg, char **mp, struct rtattr *rsvp)
{
    struct tc_rsvp_pinfo *pinfo;
    struct tc_rsvp_gpi *dpi, *spi;
    struct protoent *proto;

    if(RTA_PAYLOAD(rsvp) < sizeof(*pinfo)) {
        rec_log("error: %s: -- payload too short --", __func__);
        return(1);
    }
    pinfo = (struct tc_rsvp_pinfo *)RTA_DATA(rsvp);
    dpi = &(pinfo->dpi);
    spi = &(pinfo->spi);
    proto = getprotobynumber(pinfo->protocol);

    *mp = add_log(msg, *mp, "dpi(key/mask/offset)=0x%08x/0x%08x/%d ",
        htonl(dpi->key), htonl(dpi->mask), dpi->offset);
    *mp = add_log(msg, *mp, "spi(key/mask/offset)=0x%08x/0x%08x/%d ",
        htonl(spi->key), htonl(spi->mask), spi->offset);
    *mp = add_log(msg, *mp, "tunnel(protocol/id/hdr)=%d(%s)/%d/%d ",
        pinfo->protocol, proto ? proto->p_name : "unknown",
        pinfo->tunnelid, pinfo->tunnelhdr);

    return(0);
}

/*
 * parse route options
 */
int parse_tca_options_route(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *route[__TCA_ROUTE4_MAX];
    char *mp_tmp = *mp;

    parse_route(route, tca);

    if(route[TCA_ROUTE4_CLASSID])
        if(parse_tca_classid(msg, mp, route[TCA_ROUTE4_CLASSID]))
            return(1);

    if(route[TCA_ROUTE4_FROM])
        if(parse_tca_route4_from(msg, mp, route[TCA_ROUTE4_FROM]))
            return(1);

    if(route[TCA_ROUTE4_IIF])
        if(parse_tca_route4_iif(msg, mp, route[TCA_ROUTE4_IIF]))
            return(1);

    if(route[TCA_ROUTE4_TO])
        if(parse_tca_route4_to(msg, mp, route[TCA_ROUTE4_TO]))
            return(1);

    if(*mp != mp_tmp)
        rec_log("%s", msg);

    /* rollback pointer */
    *mp = mp_tmp;

    /* logging for each attribute below */
    if(route[TCA_ROUTE4_POLICE])
        if(parse_tca_act_options_police(msg, *mp, route[TCA_RSVP_POLICE]))
            return(1);

    if(route[TCA_ROUTE4_ACT])
        if(parse_tca_acts(msg, *mp, route[TCA_RSVP_ACT]))
            return(1);

    return(0);
}

/*
 * parse attribute TCA_ROUTE4_FROM
 */
int parse_tca_route4_from(char *msg, char **mp, struct rtattr *route)
{
    if(RTA_PAYLOAD(route) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "from(realm)=%u ", *(unsigned *)RTA_DATA(route));

    return(0);
}

/*
 * parse attribute TCA_ROUTE4_IIF
 */
int parse_tca_route4_iif(char *msg, char **mp, struct rtattr *route)
{
    int ifindex;
    char ifname[IFNAMSIZ];

    if(RTA_PAYLOAD(route) < sizeof(ifindex)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    ifindex = *(int *)RTA_DATA(route);
    if_indextoname_from_lists(ifindex, ifname);

    *mp = add_log(msg, *mp, "from(interface)=%s ", ifname);

    return(0);
}

/*
 * parse attribute TCA_ROUTE4_TO
 */
int parse_tca_route4_to(char *msg, char **mp, struct rtattr *route)
{
    if(RTA_PAYLOAD(route) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "to(realm)=%u ", *(unsigned *)RTA_DATA(route));

    return(0);
}

/*
 * parse fw options
 */
int parse_tca_options_fw(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *fw[__TCA_FW_MAX];
    char *mp_tmp = *mp;

    parse_fw(fw, tca);

    if(fw[TCA_FW_CLASSID])
        if(parse_tca_classid(msg, mp, fw[TCA_FW_CLASSID]))
            return(1);

    if(fw[TCA_FW_INDEV])
        if(parse_tca_indev(msg, mp, fw[TCA_FW_INDEV]))
            return(1);

#if HAVE_DECL_TCA_FW_MASK
    if(fw[TCA_FW_MASK])
        if(parse_tca_mask(msg, mp, fw[TCA_FW_MASK]))
            return(1);
#endif

    if(*mp != mp_tmp)
        rec_log("%s", msg);

    /* rollback pointer */
    *mp = mp_tmp;

    /* logging for each attribute below */
    if(fw[TCA_FW_POLICE])
        if(parse_tca_act_options_police(msg, *mp, fw[TCA_FW_POLICE]))
            return(1);

    if(fw[TCA_FW_ACT])
        if(parse_tca_acts(msg, *mp, fw[TCA_FW_ACT]))
            return(1);

    return(0);
}

/*
 * parse tcindex options
 */
int parse_tca_options_tcindex(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *tcindex[__TCA_TCINDEX_MAX];
    char *mp_tmp = *mp;

    parse_tcindex(tcindex, tca);

    if(tcindex[TCA_TCINDEX_CLASSID])
        if(parse_tca_classid(msg, mp, tcindex[TCA_TCINDEX_CLASSID]))
            return(1);

    if(tcindex[TCA_TCINDEX_HASH])
        if(parse_tca_tcindex_hash(msg, mp, tcindex[TCA_TCINDEX_HASH]))
            return(1);

    if(tcindex[TCA_TCINDEX_MASK])
        if(parse_tca_tcindex_mask(msg, mp, tcindex[TCA_TCINDEX_MASK]))
            return(1);

    if(tcindex[TCA_TCINDEX_SHIFT])
        if(parse_tca_tcindex_shift(msg, mp, tcindex[TCA_TCINDEX_SHIFT]))
            return(1);

    if(tcindex[TCA_TCINDEX_FALL_THROUGH])
        if(parse_tca_tcindex_fall_through(msg, mp, tcindex[TCA_TCINDEX_FALL_THROUGH]))
            return(1);

    if(*mp != mp_tmp)
        rec_log("%s", msg);

    /* rollback pointer */
    *mp = mp_tmp;

    /* logging for each attribute below */
    if(tcindex[TCA_TCINDEX_POLICE])
        if(parse_tca_act_options_police(msg, *mp, tcindex[TCA_TCINDEX_POLICE]))
            return(1);

    if(tcindex[TCA_TCINDEX_ACT])
        if(parse_tca_acts(msg, *mp, tcindex[TCA_TCINDEX_ACT]))
            return(1);

    return(0);
}

/*
 * parse attribute TCA_TCINDEX_HASH
 */
int parse_tca_tcindex_hash(char *msg, char **mp, struct rtattr *tcindex)
{
    if(RTA_PAYLOAD(tcindex) < sizeof(unsigned short)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "hash=%hu ", *(unsigned short *)RTA_DATA(tcindex));

    return(0);
}

/*
 * parse attribute TCA_TCINDEX_MASK
 */
int parse_tca_tcindex_mask(char *msg, char **mp, struct rtattr *tcindex)
{
    if(RTA_PAYLOAD(tcindex) < sizeof(unsigned short)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "mask=0x%04x ", *(unsigned short *)RTA_DATA(tcindex));

    return(0);
}

/*
 * parse attribute TCA_TCINDEX_SHIFT
 */
int parse_tca_tcindex_shift(char *msg, char **mp, struct rtattr *tcindex)
{
    if(RTA_PAYLOAD(tcindex) < sizeof(int)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "shift=%d ", *(int *)RTA_DATA(tcindex));

    return(0);
}

/*
 * parse attribute TCA_TCINDEX_FALL_THROUGH
 */
int parse_tca_tcindex_fall_through(char *msg, char **mp, struct rtattr *tcindex)
{
    if(RTA_PAYLOAD(tcindex) < sizeof(int)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "flag=%s ",
        *(int *)RTA_DATA(tcindex) ? "fall_through" : "pass_on");

    return(0);
}

#if HAVE_DECL_TCA_FLOW_UNSPEC
/*
 * parse flow options
 */
int parse_tca_options_flow(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *flow[__TCA_FLOW_MAX];
    char *mp_tmp = *mp;

    parse_flow(flow, tca);

    if(flow[TCA_FLOW_BASECLASS])
        if(parse_tca_classid(msg, mp, flow[TCA_FLOW_BASECLASS]))
            return(1);

    if(flow[TCA_FLOW_KEYS])
        if(parse_tca_flow_keys(msg, mp, flow[TCA_FLOW_KEYS]))
            return(1);

    if(flow[TCA_FLOW_MODE])
        if(parse_tca_flow_mode(msg, mp, flow[TCA_FLOW_MODE]))
            return(1);

    if(flow[TCA_FLOW_MASK])
        if(parse_tca_mask(msg, mp, flow[TCA_FLOW_MASK]))
            return(1);

    if(flow[TCA_FLOW_XOR])
        if(parse_tca_flow_xor(msg, mp, flow[TCA_FLOW_XOR]))
            return(1);

    if(flow[TCA_FLOW_RSHIFT])
        if(parse_tca_flow_rshift(msg, mp, flow[TCA_FLOW_RSHIFT]))
            return(1);

    if(flow[TCA_FLOW_ADDEND])
        if(parse_tca_flow_addend(msg, mp, flow[TCA_FLOW_ADDEND]))
            return(1);

    if(flow[TCA_FLOW_DIVISOR])
        if(parse_tca_flow_divisor(msg, mp, flow[TCA_FLOW_DIVISOR]))
            return(1);

    if(flow[TCA_FLOW_PERTURB])
        if(parse_tca_flow_perturb(msg, mp, flow[TCA_FLOW_PERTURB]))
            return(1);

    if(*mp != mp_tmp)
        rec_log("%s", msg);

    /* rollback pointer */
    *mp = mp_tmp;

    /* logging for each attribute below */
    if(flow[TCA_FLOW_EMATCHES])
        if(parse_tca_ematch(msg, *mp, flow[TCA_FLOW_EMATCHES]))
            return(1);

    if(flow[TCA_FLOW_POLICE])
        if(parse_tca_act_options_police(msg, *mp, flow[TCA_FLOW_POLICE]))
            return(1);

    if(flow[TCA_FLOW_ACT])
        if(parse_tca_acts(msg, *mp, flow[TCA_FLOW_ACT]))
            return(1);

    return(0);
}

/*
 * parse attribute TCA_FLOW_BASECLASS
 */
int parse_tca_baseclass(char *msg, char **mp, struct rtattr *flow)
{
    char classid[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    parse_tc_classid(classid, sizeof(classid), *(unsigned *)RTA_DATA(flow));
    *mp = add_log(msg, *mp, "baseclass=%s ", classid);

    return(0);
}

/*
 * parse attribute TCA_FLOW_KEYS
 */
int parse_tca_flow_keys(char *msg, char **mp, struct rtattr *flow)
{
    char flags_list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    convert_flow_key(*(unsigned *)RTA_DATA(flow), flags_list, sizeof(flags_list), 0);
    *mp = add_log(msg, *mp, "keys=%s ", flags_list);

    return(0);
}

/*
 * parse attribute TCA_FLOW_MODE
 */
int parse_tca_flow_mode(char *msg, char **mp, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "mode=%s ",
        convert_flow_mode(*(unsigned *)RTA_DATA(flow), 0));

    return(0);
}

/*
 * parse attribute TCA_FLOW_XOR
 */
int parse_tca_flow_xor(char *msg, char **mp, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "xor=0x%08x ", *(unsigned *)RTA_DATA(flow));

    return(0);
}

/*
 * parse attribute TCA_FLOW_RSHIFT
 */
int parse_tca_flow_rshift(char *msg, char **mp, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "rshift=%u ", *(unsigned *)RTA_DATA(flow));

    return(0);
}

/*
 * parse attribute TCA_FLOW_ADDEND
 */
int parse_tca_flow_addend(char *msg, char **mp, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "addend=0x%x ", *(unsigned *)RTA_DATA(flow));

    return(0);
}

/*
 * parse attribute TCA_FLOW_DIVISOR
 */
int parse_tca_flow_divisor(char *msg, char **mp, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "divisor=%u ", *(unsigned *)RTA_DATA(flow));

    return(0);
}

/*
 * parse attribute TCA_FLOW_PERTURB
 */
int parse_tca_flow_perturb(char *msg, char **mp, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    *mp = add_log(msg, *mp, "perturb=%u(sec) ", *(unsigned *)RTA_DATA(flow));

    return(0);
}
#endif

/*
 * parse basic options
 */
int parse_tca_options_basic(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *basic[__TCA_BASIC_MAX];
    char *mp_tmp = *mp;

    parse_basic(basic, tca);

    if(basic[TCA_BASIC_CLASSID])
        if(parse_tca_classid(msg, mp, basic[TCA_BASIC_CLASSID]))
            return(1);

    if(*mp != mp_tmp)
        rec_log("%s", msg);

    /* rollback pointer */
    *mp = mp_tmp;

    /* logging for each attribute below */
    if(basic[TCA_BASIC_EMATCHES])
        if(parse_tca_ematch(msg, *mp, basic[TCA_BASIC_EMATCHES]))
            return(1);

    if(basic[TCA_BASIC_POLICE])
        if(parse_tca_act_options_police(msg, *mp, basic[TCA_BASIC_POLICE]))
            return(1);

    if(basic[TCA_BASIC_ACT])
        if(parse_tca_acts(msg, *mp, basic[TCA_BASIC_ACT]))
            return(1);

    return(0);
}

#if HAVE_DECL_TCA_CGROUP_UNSPEC
/*
 * parse cgroup options
 */
int parse_tca_options_cgroup(char *msg, char **mp, struct rtattr *tca)
{
    struct rtattr *cgroup[__TCA_CGROUP_MAX];

    parse_cgroup(cgroup, tca);

    /* logging for each attribute below */
    if(cgroup[TCA_CGROUP_EMATCHES])
        if(parse_tca_ematch(msg, *mp, cgroup[TCA_CGROUP_EMATCHES]))
            return(1);

    if(cgroup[TCA_CGROUP_POLICE])
        if(parse_tca_act_options_police(msg, *mp, cgroup[TCA_CGROUP_POLICE]))
            return(1);

    if(cgroup[TCA_CGROUP_ACT])
        if(parse_tca_acts(msg, *mp, cgroup[TCA_CGROUP_ACT]))
            return(1);

    return(0);
}
#endif

/*
 * parse attribute TCA_EMATCH_*
 */
int parse_tca_ematch(char *msg, char *mp, struct rtattr *tca)
{
    struct rtattr *em_tree[__TCA_EMATCH_TREE_MAX];
    int num = -1;

    parse_ematch_tree(em_tree, tca);

    if(em_tree[TCA_EMATCH_TREE_HDR])
        if(parse_tca_ematch_tree_hdr(em_tree[TCA_EMATCH_TREE_HDR], &num))
            return(1);

    if(em_tree[TCA_EMATCH_TREE_LIST])
        if(parse_tca_ematch_tree_list(msg, mp, em_tree[TCA_EMATCH_TREE_LIST], num))
            return(1);

    return(0);
}

/*
 * parse attribute TCA_EMATCH_TREE_HDR
 */
int parse_tca_ematch_tree_hdr(struct rtattr *em_tree, int *num)
{
    struct tcf_ematch_tree_hdr *hdr;

    if(RTA_PAYLOAD(em_tree) < sizeof(*hdr)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    hdr = (struct tcf_ematch_tree_hdr *)RTA_DATA(em_tree);
    *num = hdr->nmatches;

    return(0);
}

/*
 * parse attribute TCA_EMATCH_TREE_LIST
 */
int parse_tca_ematch_tree_list(char *msg, char *mp, struct rtattr *em_tree, int num)
{
    struct rtattr *em_list[num+1];
    struct tcf_ematch_hdr *hdr;
    int i;
    char *mp_tmp = mp;

    parse_ematch_list(em_list, em_tree, num);

    /* no exist em_list[0] */
    for(i = 1; i < num + 1; i++, mp = mp_tmp) {
        if(!em_list[i])
            return(0);

        if(RTA_PAYLOAD(em_list[i]) < sizeof(*hdr)) {
            rec_log("error: %s: payload too short", __func__);
            return(1);
        }
        hdr = (struct tcf_ematch_hdr *)RTA_DATA(em_list[i]);
        mp = add_log(msg, mp, "ematch=%s ", convert_tcf_em_kind(hdr->kind, 0));

        /* use (char*)hdr in order to count by one byte */
        switch(hdr->kind) {
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_CMP_H
            case TCF_EM_CMP:
                if(parse_ematch_cmp(msg, mp, (char *)hdr + sizeof(*hdr),
                    RTA_PAYLOAD(em_list[i]) - sizeof(*hdr)))
                    return(1);
                break;
#endif
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_NBYTE_H
            case TCF_EM_NBYTE:
                if(parse_ematch_nbyte(msg, mp, (char *)hdr + sizeof(*hdr),
                    RTA_PAYLOAD(em_list[i]) - sizeof(*hdr)))
                    return(1);
                break;
#endif
            case TCF_EM_U32:
                if(parse_ematch_u32(msg, mp, (char *)hdr + sizeof(*hdr),
                    RTA_PAYLOAD(em_list[i]) - sizeof(*hdr)))
                    return(1);
                break;
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
            case TCF_EM_META:
                if(parse_ematch_meta(msg, mp, (char *)hdr + sizeof(*hdr),
                    RTA_PAYLOAD(em_list[i]) - sizeof(*hdr)))
                    return(1);
                break;
#endif
            /* not implemented yet
            case TCF_EM_TEXT:
            case TCF_EM_VLAN:
            case TCF_EM_CANID:
            case TCF_EM_IPSET:
            */
            default:
                rec_log("%s", msg);
        }
    }

    return(0);
}

#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_CMP_H
/*
 * parse ematch cmp
 */
int parse_ematch_cmp(char *msg, char *mp, void *p, int len)
{
    struct tcf_em_cmp *cmp;

    if(len < sizeof(struct tcf_em_cmp)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    cmp = (struct tcf_em_cmp *)p;

    mp = add_log(msg, mp,
        "layer=%d align=%s flag=%s operand=%s value=0x%08x mask=0x%08x offset=%hu ",
        cmp->layer, convert_tcf_em_align(cmp->align, 0),
        cmp->flags ? "trans" : "none", convert_tcf_em_opnd(cmp->opnd, 0),
        cmp->val, cmp->mask, cmp->off);

    rec_log("%s", msg);

    return(0);
}
#endif

#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_NBYTE_H
/*
 * parse ematch nbyte
 */
int parse_ematch_nbyte(char *msg, char *mp, void *p, int len)
{
    struct tcf_em_nbyte *nbyte;
    int i;
    char *data;

    if(len < sizeof(struct tcf_em_nbyte)) {
        rec_log("error: %s: tcf_em_nbyte: payload too short", __func__);
        return(1);
    }
    nbyte = (struct tcf_em_nbyte *)p;

    if(len - sizeof(*nbyte) < nbyte->len) {
        rec_log("error: %s: len: payload too short", __func__);
        return(1);
    }

    data = (char *)nbyte + sizeof(*nbyte);

    mp = add_log(msg, mp, "layer=%d ", nbyte->layer);

    for(i = 0; i < nbyte->len; i++) {
        if(!i)
            mp = add_log(msg, mp, "value=\"");

        mp = add_log(msg, mp, "%c", isprint(data[i]) ? data[i] : '.');

        if(nbyte->len - i == 1)
            mp = add_log(msg, mp, "\"");
    }

    mp = add_log(msg, mp, " offset=%hu ", nbyte->off);

    rec_log("%s", msg);

    return(0);
}
#endif

/*
 * parse ematch u32
 */
int parse_ematch_u32(char *msg, char *mp, void *p, int len)
{
    struct tc_u32_key *key;

    if(len < sizeof(*key)) {
        rec_log("error: %s: payload too short", __func__);
        return(1);
    }
    key = (struct tc_u32_key *)p;

    mp = add_log(msg, mp, "value=0x%08x mask=0x%08x offset=%d offmask=0x%08x",
        ntohl(key->val), ntohl(key->mask), key->off, key->offmask);

    rec_log("%s", msg);

    return(0);
}

#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
/*
 * parse ematch meta
 */
int parse_ematch_meta(char *msg, char *mp, void *p, int len)
{
    struct rtattr *meta[__TCA_EM_META_MAX];
    struct tcf_meta_hdr *hdr = NULL;
    struct tcf_meta_val *left = NULL, *right = NULL;

    parse_meta(meta, p, len);

    if(meta[TCA_EM_META_HDR]) {
        if(RTA_PAYLOAD(meta[TCA_EM_META_HDR]) < sizeof(*hdr)) {
            rec_log("error: %s: TCA_EM_META_HDR: payload too short", __func__);
            return(1);
        }
        hdr = (struct tcf_meta_hdr *)RTA_DATA(meta[TCA_EM_META_HDR]);
        left = &(hdr->left);
        right = &(hdr->right);
    } else {
        rec_log("error: %s: TCA_EM_META_HDR: no attribute", __func__);
        return(1);
    }

    mp = add_log(msg, mp, "match=(");

    if(meta[TCA_EM_META_LVALUE]) {
        if(parse_tca_em_meta_value(msg, &mp, left, meta[TCA_EM_META_LVALUE]))
            return(1);
    } else {
        rec_log("error: %s: TCA_EM_META_LVALUE: no attribute", __func__);
        return(1);
    }

    mp = add_log(msg, mp, "%s ", convert_tcf_em_opnd(left->op, 0));

    if(meta[TCA_EM_META_RVALUE]) {
        if(parse_tca_em_meta_value(msg, &mp, right, meta[TCA_EM_META_RVALUE]))
            return(1);
    } else {
        rec_log("error: %s: TCA_EM_META_RVALUE: no attribute", __func__);
        return(1);
    }
    mp = add_log(msg, mp, ") ");

    rec_log("%s", msg);

    return(0);
}

/*
 * parse ematch meta value
 */
int parse_tca_em_meta_value(char *msg, char **mp, struct tcf_meta_val *val, struct rtattr *meta)
{
    int id = TCF_META_ID(val->kind);
    int type = TCF_META_TYPE(val->kind);
    char *data = (char *)RTA_DATA(meta);
    int i;

    if(id != TCF_META_ID_VALUE) {
        *mp = add_log(msg, *mp, "%s ", convert_tcf_meta_id(id, 0));
        if(val->shift)
            *mp = add_log(msg, *mp, "shift %d ", val->shift);
        if(type == TCF_META_TYPE_INT && *(unsigned *)RTA_DATA(meta)) {
            if(RTA_PAYLOAD(meta) < sizeof(__u32)) {
                rec_log("error: %s: payload too short", __func__);
                return(1);
            }
            *mp = add_log(msg, *mp, "mask 0x%08x ", *(unsigned *)RTA_DATA(meta));
        }

        return(0);
    }

    switch(type) {
        case TCF_META_TYPE_VAR:
            for(i = 0; i < RTA_PAYLOAD(meta); i++)
                *mp = add_log(msg, *mp, "%c", isprint(data[i]) ? data[i] : '.');
            break;
        case TCF_META_TYPE_INT:
            if(RTA_PAYLOAD(meta) < sizeof(__u32)) {
                rec_log("error: %s: payload too short", __func__);
                return(1);
            }
            *mp = add_log(msg, *mp, "%d", *(int *)RTA_DATA(meta));
            break;
        default:
            rec_log("error: %s: unknown type(%d)", __func__, type);
            return(1);
    }

    return(0);
}
#endif

/*
 * convert interface address from binary to text
 */
int inet_ntop_tc_addr(struct tcmsg *tcm, struct rtattr *tca, char *saddr, int slen)
{
    int af = -1;
    unsigned char *addr = (unsigned char *)RTA_DATA(tca);
    int len = RTA_PAYLOAD(tca);

    switch(ntohs(TC_H_MIN(tcm->tcm_info))) {
        case ETH_P_IP:
            af = AF_INET;
            if(len < 4)
                return(2);
            break;
        case ETH_P_IPV6:
            af = AF_INET6;
            if(len < 16)
                return(2);
            break;
    }

    if(!inet_ntop(af, addr, saddr, slen))
        return(1);

    return(0);
}

/*
 * debug u32 options
 */
void debug_tca_options_u32(int lev, struct rtattr *tca)
{
    struct rtattr *u32[__TCA_U32_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_u32(u32, tca);

    if(u32[TCA_U32_CLASSID])
        debug_tca_u32_classid(lev+1, u32[TCA_U32_CLASSID]);

    if(u32[TCA_U32_HASH])
        debug_tca_u32_hash(lev+1, u32[TCA_U32_HASH]);

    if(u32[TCA_U32_LINK])
        debug_tca_u32_link(lev+1, u32[TCA_U32_LINK]);

    if(u32[TCA_U32_DIVISOR])
        debug_tca_u32_divisor(lev+1, u32[TCA_U32_DIVISOR]);

    if(u32[TCA_U32_SEL])
        debug_tca_u32_sel(lev+1, u32[TCA_U32_SEL]);

    if(u32[TCA_U32_POLICE])
        debug_tca_u32_police(lev+1, u32[TCA_U32_POLICE]);

    if(u32[TCA_U32_ACT])
        debug_tca_u32_act(lev+1, u32[TCA_U32_ACT]);

    if(u32[TCA_U32_INDEV])
        debug_tca_u32_indev(lev+1, u32[TCA_U32_INDEV]);

    if(u32[TCA_U32_PCNT])
        debug_tca_u32_pcnt(lev+1, u32);

    if(u32[TCA_U32_MARK])
        debug_tca_u32_mark(lev+1, u32[TCA_U32_MARK]);
}

/*
 * debug attribute TCA_U32_CLASSID
 */
void debug_tca_u32_classid(int lev, struct rtattr *u32)
{
    unsigned n_classid;
    char s_classid[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(u32) < sizeof(n_classid)) {
        rec_dbg(lev, "TCA_U32_CLASSID(%hu): -- payload too short --",
            RTA_ALIGN(u32->rta_len));
        return;
    }
    n_classid = *(unsigned *)RTA_DATA(u32);
    parse_tc_classid(s_classid, sizeof(s_classid), n_classid);

    rec_dbg(lev, "TCA_U32_CLASSID(%hu): 0x%08x(%s)",
        RTA_ALIGN(u32->rta_len), n_classid, s_classid);
}

/*
 * debug attribute TCA_U32_HASH
 */
void debug_tca_u32_hash(int lev, struct rtattr *u32)
{
    unsigned htid;

    if(RTA_PAYLOAD(u32) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_U32_HASH(%hu): -- payload too short --",
            RTA_ALIGN(u32->rta_len));
        return;
    }
    htid = *(unsigned *)RTA_DATA(u32);

    rec_dbg(lev, "TCA_U32_HASH(%hu): 0x%08x(0x%x/0x%x)",
        RTA_ALIGN(u32->rta_len), htid, TC_U32_USERHTID(htid), TC_U32_HASH(htid));
}

/*
 * debug attribute TCA_U32_LINK
 */
void debug_tca_u32_link(int lev, struct rtattr *u32)
{
    unsigned n_handle;
    char s_handle[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(u32) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_U32_LINK(%hu): -- payload too short --",
            RTA_ALIGN(u32->rta_len));
        return;
    }
    n_handle = *(unsigned *)RTA_DATA(u32);
    parse_u32_handle(s_handle, sizeof(s_handle), n_handle);

    rec_dbg(lev, "TCA_U32_LINK(%hu): 0x%08x(%s)",
        RTA_ALIGN(u32->rta_len), n_handle, s_handle);
}

/*
 * debug attribute TCA_U32_DIVISOR
 */
void debug_tca_u32_divisor(int lev, struct rtattr *u32)
{
    if(RTA_PAYLOAD(u32) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_U32_DIVISOR(%hu): -- payload too short --",
            RTA_ALIGN(u32->rta_len));
        return;
    }

    rec_dbg(lev, "TCA_U32_DIVISOR(%hu): %u",
        RTA_ALIGN(u32->rta_len), *(unsigned *)RTA_DATA(u32));
}

/*
 * debug attribute TCA_U32_SEL
 */
void debug_tca_u32_sel(int lev, struct rtattr *u32)
{
    struct tc_u32_sel *sel;
    struct tc_u32_key *keys;
    char flags_list[MAX_STR_SIZE] = "";
    int i, len;

    if(RTA_PAYLOAD(u32) < sizeof(*sel)) {
        rec_dbg(lev, "TCA_U32_SEL(%hu): -- payload too short --",
            RTA_ALIGN(u32->rta_len));
        return;
    }
    sel = (struct tc_u32_sel *)RTA_DATA(u32);
    convert_tc_u32_flags(sel->flags, flags_list, sizeof(flags_list), 1);

    rec_dbg(lev, "TCA_U32_SEL(%hu):", RTA_ALIGN(u32->rta_len));
    rec_dbg(lev, "    [ tc_u32_sel(%d) ]", sizeof(*sel));
    rec_dbg(lev, "        flags(%d): %d(%s)", sizeof(sel->flags), sel->flags, flags_list);
    rec_dbg(lev, "        offshift(%d): %d", sizeof(sel->offshift), sel->offshift);
    rec_dbg(lev, "        nkeys(%d): %d", sizeof(sel->nkeys), sel->nkeys);
    rec_dbg(lev, "        offmask(%d): 0x%04x", sizeof(sel->offmask), sel->offmask);
    rec_dbg(lev, "        off(%d): %hu", sizeof(sel->off), sel->off);
    rec_dbg(lev, "        offoff(%d): %hd", sizeof(sel->offoff), sel->offoff);
    rec_dbg(lev, "        hoff(%d): %hd", sizeof(sel->hoff), sel->hoff);
    rec_dbg(lev, "        hmask(%d): 0x%08x", sizeof(sel->hmask), sel->hmask);

    len = sizeof(*sel) + (sizeof(*keys) * sel->nkeys);
    if(RTA_PAYLOAD(u32) < len) {
        rec_dbg(lev, "        keys[0](%d): -- payload too short --",
            RTA_PAYLOAD(u32) - sizeof(*sel));
        return;
    }

    keys = sel->keys;
    for(i = 0; i < sel->nkeys; i++, keys++) {
        rec_dbg(lev+2, "[ tc_u32_key keys[%d](%d) ]", i, sizeof(*keys));
        rec_dbg(lev+2, "    mask(%d): 0x%08x", sizeof(keys->mask), keys->mask); /* AND */
        rec_dbg(lev+2, "    val(%d): 0x%08x", sizeof(keys->val), keys->val); /* XOR */
        rec_dbg(lev+2, "    off(%d): %d", sizeof(keys->off), keys->off); /* Offset */
        rec_dbg(lev+2, "    offmask(%d): 0x%08x", sizeof(keys->offmask), keys->offmask);
    }
}

/*
 * debug attribute TCA_U32_POLICE
 */
void debug_tca_u32_police(int lev, struct rtattr *u32)
{
    rec_dbg(lev, "TCA_U32_POLICE(%hu):", RTA_ALIGN(u32->rta_len));

    debug_tca_act_options_police(lev, u32);
}

/*
 * debug attribute TCA_U32_ACT
 */
void debug_tca_u32_act(int lev, struct rtattr *u32)
{
    rec_dbg(lev, "TCA_U32_ACT(%hu):", RTA_ALIGN(u32->rta_len));

    debug_tca_acts(lev+1, u32);
}

/*
 * debug attribute TCA_U32_INDEV
 */
void debug_tca_u32_indev(int lev, struct rtattr *u32)
{
    char name[IFNAMSIZ];

    if(!RTA_PAYLOAD(u32)) {
        rec_dbg(lev, "TCA_U32_INDEV(%hu): -- no payload --",
            RTA_ALIGN(u32->rta_len));
        return;
    } else if(RTA_PAYLOAD(u32) > sizeof(name)) {
        rec_dbg(lev, "TCA_U32_INDEV(%hu): -- payload too long --",
            RTA_ALIGN(u32->rta_len));
        return;
    }
    strncpy(name, (char *)RTA_DATA(u32), sizeof(name));

    rec_dbg(lev, "TCA_U32_INDEV(%hu): %s", RTA_ALIGN(u32->rta_len), name);
}

/*
 * debug attribute TCA_U32_PCNT
 */
void debug_tca_u32_pcnt(int lev, struct rtattr *u32[])
{
    struct tc_u32_pcnt *pcnt;
    struct tc_u32_sel *sel;
    unsigned long *kcnts;
    int i, len;

    if(RTA_PAYLOAD(u32[TCA_U32_PCNT]) < sizeof(*pcnt)) {
        rec_dbg(lev, "TCA_U32_PCNT(%hu): -- payload too short --",
            RTA_ALIGN(u32[TCA_U32_PCNT]->rta_len));
        return;
    }
    pcnt = (struct tc_u32_pcnt *)RTA_DATA(u32[TCA_U32_PCNT]);

    rec_dbg(lev, "TCA_U32_PCNT(%hu):", RTA_ALIGN(u32[TCA_U32_PCNT]->rta_len));
    rec_dbg(lev, "    [ tc_u32_pcnt(%d) ]", sizeof(*pcnt));
    rec_dbg(lev, "        rcnt(%d): %lu", sizeof(pcnt->rcnt), pcnt->rcnt);
    rec_dbg(lev, "        rhit(%d): %lu", sizeof(pcnt->rhit), pcnt->rhit);

    if(RTA_PAYLOAD(u32[TCA_U32_SEL]) < sizeof(*sel))
        return;

    sel = RTA_DATA(u32[TCA_U32_SEL]);

    len = sizeof(*pcnt) + (sel->nkeys * sizeof(unsigned long));
    if(RTA_PAYLOAD(u32[TCA_U32_PCNT]) < len) {
        rec_dbg(lev, "        kcnts[0](%d): -- payload too short --",
            RTA_PAYLOAD(u32[TCA_U32_PCNT]) - sizeof(*pcnt));
        return;
    }

    kcnts = (unsigned long *)pcnt->kcnts;
    for(i = 0; i < sel->nkeys; i++, kcnts++)
        rec_dbg(lev, "        kcnts[%d](%d): %lu", i, sizeof(*kcnts), *kcnts);
}

/*
 * debug attribute TCA_U32_MARK
 */
void debug_tca_u32_mark(int lev, struct rtattr *u32)
{
    struct tc_u32_mark *mark;

    if(RTA_PAYLOAD(u32) < sizeof(*mark)) {
        rec_dbg(lev, "TCA_U32_MARK(%hu): -- payload too short --",
            RTA_ALIGN(u32->rta_len));
        return;
    }
    mark = (struct tc_u32_mark *)RTA_DATA(u32);

    rec_dbg(lev, "TCA_U32_MARK(%hu):", RTA_ALIGN(u32->rta_len));
    rec_dbg(lev, "    [ tc_u32_mark(%d) ]", sizeof(*mark));
    rec_dbg(lev, "        val(%d): 0x%08x", sizeof(mark->val), mark->val);
    rec_dbg(lev, "        mask(%d): 0x%08x", sizeof(mark->mask), mark->mask);
    rec_dbg(lev, "        success(%d): 0x%08x", sizeof(mark->success), mark->success);
}

/*
 * debug rsvp options
 */
void debug_tca_options_rsvp(int lev, struct tcmsg *tcm, struct rtattr *tca)
{
    struct rtattr *rsvp[__TCA_RSVP_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_rsvp(rsvp, tca);

    if(rsvp[TCA_RSVP_CLASSID])
        debug_tca_rsvp_classid(lev+1, rsvp[TCA_RSVP_CLASSID]);

    if(rsvp[TCA_RSVP_DST])
        debug_tca_rsvp_dst(lev+1, tcm, rsvp[TCA_RSVP_DST]);

    if(rsvp[TCA_RSVP_SRC])
        debug_tca_rsvp_src(lev+1, tcm, rsvp[TCA_RSVP_SRC]);

    if(rsvp[TCA_RSVP_PINFO])
        debug_tca_rsvp_pinfo(lev+1, rsvp[TCA_RSVP_PINFO]);

    if(rsvp[TCA_RSVP_POLICE])
        debug_tca_rsvp_police(lev+1, rsvp[TCA_RSVP_POLICE]);

    if(rsvp[TCA_RSVP_ACT])
        debug_tca_rsvp_act(lev+1, rsvp[TCA_RSVP_ACT]);
}

/*
 * debug attribute TCA_RSVP_CLASSID
 */
void debug_tca_rsvp_classid(int lev, struct rtattr *rsvp)
{
    unsigned n_classid;
    char s_classid[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(rsvp) < sizeof(n_classid)) {
        rec_dbg(lev, "TCA_RSVP_CLASSID(%hu): -- payload too short --",
            RTA_ALIGN(rsvp->rta_len));
        return;
    }
    n_classid = *(unsigned *)RTA_DATA(rsvp);
    parse_tc_classid(s_classid, sizeof(s_classid), n_classid);

    rec_dbg(lev, "TCA_RSVP_CLASSID(%hu): 0x%08x(%s)",
        RTA_ALIGN(rsvp->rta_len), n_classid, s_classid);
}

/*
 * debug attribute TCA_RSVP_DST
 */
void debug_tca_rsvp_dst(int lev, struct tcmsg *tcm, struct rtattr *rsvp)
{
    char addr[INET6_ADDRSTRLEN+1] = "";
    int res;

    res = inet_ntop_tc_addr(tcm, rsvp, addr, sizeof(addr));
    if(res) {
        rec_dbg(lev, "TCA_RSVP_DST(%hu): -- %s --",
            RTA_ALIGN(rsvp->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "TCA_RSVP_DST(%hu): %s", RTA_ALIGN(rsvp->rta_len), addr);
}

/*
 * debug attribute TCA_RSVP_SRC
 */
void debug_tca_rsvp_src(int lev, struct tcmsg *tcm, struct rtattr *rsvp)
{
    char addr[INET6_ADDRSTRLEN+1];
    int res;

    res = inet_ntop_tc_addr(tcm, rsvp, addr, sizeof(addr));
    if(res) {
        rec_dbg(lev, "TCA_RSVP_SRC(%hu): -- %s --",
            RTA_ALIGN(rsvp->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "TCA_RSVP_SRC(%hu): %s", RTA_ALIGN(rsvp->rta_len), addr);
}

/*
 * debug attribute TCA_RSVP_PINFO
 */
void debug_tca_rsvp_pinfo(int lev, struct rtattr *rsvp)
{
    struct tc_rsvp_pinfo *pinfo;
    struct tc_rsvp_gpi *dpi, *spi;

    if(RTA_PAYLOAD(rsvp) < sizeof(*pinfo)) {
        rec_dbg(lev, "TCA_RSVP_PINFO(%hu): -- payload too short --",
            RTA_ALIGN(rsvp->rta_len));
        return;
    }
    pinfo = (struct tc_rsvp_pinfo *)RTA_DATA(rsvp);
    dpi = &(pinfo->dpi);
    spi = &(pinfo->spi);

    rec_dbg(lev, "TCA_RSVP_PINFO(%hu):", RTA_ALIGN(rsvp->rta_len));
    rec_dbg(lev, "    [ tc_rsvp_pinfo(%d) ]", sizeof(*pinfo));
    rec_dbg(lev, "        [ tc_rsvp_gpi dpi(%d) ]", sizeof(*dpi));
    rec_dbg(lev, "            key(%d): 0x%08x", sizeof(dpi->key), dpi->key);
    rec_dbg(lev, "            mask(%d): 0x%08x", sizeof(dpi->mask), dpi->mask);
    rec_dbg(lev, "            offset(%d): %d", sizeof(dpi->offset), dpi->offset);
    rec_dbg(lev, "        [ tc_rsvp_gpi spi(%d) ]", sizeof(*spi));
    rec_dbg(lev, "            key(%d): 0x%08x", sizeof(spi->key), spi->key);
    rec_dbg(lev, "            mask(%d): 0x%08x", sizeof(spi->mask), spi->mask);
    rec_dbg(lev, "            offset(%d): %d", sizeof(spi->offset), spi->offset);
    rec_dbg(lev, "        protocol(%d): %d", sizeof(pinfo->protocol), pinfo->protocol);
    rec_dbg(lev, "        tunnelid(%d): %d", sizeof(pinfo->tunnelid), pinfo->tunnelid);
    rec_dbg(lev, "        tunnelhdr(%d): %d", sizeof(pinfo->tunnelhdr), pinfo->tunnelhdr);
    rec_dbg(lev, "        pad(%d): %d", sizeof(pinfo->pad), pinfo->pad);
}

/*
 * debug attribute TCA_RSVP_POLICE
 */
void debug_tca_rsvp_police(int lev, struct rtattr *rsvp)
{
    rec_dbg(lev, "TCA_RSVP_POLICE(%hu):", RTA_ALIGN(rsvp->rta_len));

    debug_tca_act_options_police(lev, rsvp);
}

/*
 * debug attribute TCA_RSVP_ACT
 */
void debug_tca_rsvp_act(int lev, struct rtattr *rsvp)
{
    rec_dbg(lev, "TCA_RSVP_ACT(%hu):", RTA_ALIGN(rsvp->rta_len));

    debug_tca_acts(lev+1, rsvp);
}

/*
 * debug route options
 */
void debug_tca_options_route(int lev, struct rtattr *tca)
{
    struct rtattr *route[__TCA_ROUTE4_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_route(route, tca);

    if(route[TCA_ROUTE4_CLASSID])
        debug_tca_route4_classid(lev+1, route[TCA_ROUTE4_CLASSID]);

    if(route[TCA_ROUTE4_TO])
        debug_tca_route4_to(lev+1, route[TCA_ROUTE4_TO]);

    if(route[TCA_ROUTE4_FROM])
        debug_tca_route4_from(lev+1, route[TCA_ROUTE4_FROM]);

    if(route[TCA_ROUTE4_IIF])
        debug_tca_route4_iif(lev+1, route[TCA_ROUTE4_IIF]);

    if(route[TCA_ROUTE4_POLICE])
        debug_tca_route4_police(lev+1, route[TCA_ROUTE4_POLICE]);

    if(route[TCA_ROUTE4_ACT])
        debug_tca_route4_act(lev+1, route[TCA_ROUTE4_ACT]);
}

/*
 * debug attribute TCA_ROUTE4_CLASSID
 */
void debug_tca_route4_classid(int lev, struct rtattr *route)
{
    unsigned n_classid;
    char s_classid[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(route) < sizeof(n_classid)) {
        rec_dbg(lev, "TCA_ROUTE4_CLASSID(%hu): -- payload too short --",
            RTA_ALIGN(route->rta_len));
        return;
    }
    n_classid = *(unsigned *)RTA_DATA(route);
    parse_tc_classid(s_classid, sizeof(s_classid), n_classid);

    rec_dbg(lev, "TCA_ROUTE4_CLASSID(%hu): 0x%08x(%s)",
        RTA_ALIGN(route->rta_len), n_classid, s_classid);
}

/*
 * debug attribute TCA_ROUTE4_TO
 */
void debug_tca_route4_to(int lev, struct rtattr *route)
{
    if(RTA_PAYLOAD(route) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_ROUTE4_TO(%hu): -- payload too short --",
            RTA_ALIGN(route->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_ROUTE4_TO(%hu): %u",
        RTA_ALIGN(route->rta_len), *(unsigned *)RTA_DATA(route));
}

/*
 * debug attribute TCA_ROUTE4_FROM
 */
void debug_tca_route4_from(int lev, struct rtattr *route)
{
    if(RTA_PAYLOAD(route) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_ROUTE4_FROM(%hu): -- payload too short --",
            RTA_ALIGN(route->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_ROUTE4_FROM(%hu): %u",
        RTA_ALIGN(route->rta_len), *(unsigned *)RTA_DATA(route));
}

/*
 * debug attribute TCA_ROUTE4_IIF
 */
void debug_tca_route4_iif(int lev, struct rtattr *route)
{
    int ifindex;
    char ifname[IFNAMSIZ];

    if(RTA_PAYLOAD(route) < sizeof(ifindex)) {
        rec_dbg(lev, "TCA_ROUTE4_IIF(%hu): -- payload too short --",
            RTA_ALIGN(route->rta_len));
        return;
    }
    ifindex = *(int *)RTA_DATA(route);
    if_indextoname_from_lists(ifindex, ifname);

    rec_dbg(lev, "TCA_ROUTE4_IIF(%hu): %d(%s)",
        RTA_ALIGN(route->rta_len), ifindex, ifname);
}

/*
 * debug attribute TCA_ROUTE4_POLICE
 */
void debug_tca_route4_police(int lev, struct rtattr *route)
{
    rec_dbg(lev, "TCA_ROUTE4_POLICE(%hu):", RTA_ALIGN(route->rta_len));

    debug_tca_act_options_police(lev, route);
}

/*
 * debug attribute TCA_ROUTE4_ACT
 */
void debug_tca_route4_act(int lev, struct rtattr *route)
{
    rec_dbg(lev, "TCA_ROUTE4_ACT(%hu):", RTA_ALIGN(route->rta_len));

    debug_tca_acts(lev+1, route);
}

/*
 * debug fw options
 */
void debug_tca_options_fw(int lev, struct rtattr *tca)
{
    struct rtattr *fw[__TCA_FW_MAX];

    parse_fw(fw, tca);

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    if(fw[TCA_FW_CLASSID])
        debug_tca_fw_classid(lev+1, fw[TCA_FW_CLASSID]);

    if(fw[TCA_FW_POLICE])
        debug_tca_fw_police(lev+1, fw[TCA_FW_POLICE]);

    if(fw[TCA_FW_INDEV])
        debug_tca_fw_indev(lev+1, fw[TCA_FW_INDEV]);

    if(fw[TCA_FW_ACT])
        debug_tca_fw_act(lev+1, fw[TCA_FW_ACT]);

#if HAVE_DECL_TCA_FW_MASK
    if(fw[TCA_FW_MASK])
        debug_tca_fw_mask(lev+1, fw[TCA_FW_MASK]);
#endif
}

/*
 * debug attribute TCA_FW_CLASSID
 */
void debug_tca_fw_classid(int lev, struct rtattr *fw)
{
    unsigned n_classid;
    char s_classid[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(fw) < sizeof(n_classid)) {
        rec_dbg(lev, "TCA_FW_CLASSID(%hu): -- payload too short --",
            RTA_ALIGN(fw->rta_len));
        return;
    }
    n_classid = *(unsigned *)RTA_DATA(fw);
    parse_tc_classid(s_classid, sizeof(s_classid), n_classid);

    rec_dbg(lev, "TCA_FW_CLASSID(%hu): 0x%08x(%s)",
        RTA_ALIGN(fw->rta_len), n_classid, s_classid);
}

/*
 * debug attribute TCA_FW_POLICE
 */
void debug_tca_fw_police(int lev, struct rtattr *fw)
{
    rec_dbg(lev, "TCA_FW_POLICE(%hu):", RTA_ALIGN(fw->rta_len));

    debug_tca_act_options_police(lev, fw);
}

/*
 * debug attribute TCA_FW_INDEV
 */
void debug_tca_fw_indev(int lev, struct rtattr *fw)
{
    char name[IFNAMSIZ];

    if(!RTA_PAYLOAD(fw)) {
        rec_dbg(lev, "TCA_FW_INDEV(%hu): -- no payload --",
            RTA_ALIGN(fw->rta_len));
        return;
    } else if(RTA_PAYLOAD(fw) > sizeof(name)) {
        rec_dbg(lev, "TCA_FW_INDEV(%hu): -- payload too long --",
            RTA_ALIGN(fw->rta_len));
        return;
    }
    strncpy(name, (char *)RTA_DATA(fw), sizeof(name));

    rec_dbg(lev, "TCA_FW_INDEV(%hu): %s", RTA_ALIGN(fw->rta_len), name);
}

/*
 * debug attribute TCA_FW_ACT
 */
void debug_tca_fw_act(int lev, struct rtattr *fw)
{
    rec_dbg(lev, "TCA_FW_ACT(%hu):", RTA_ALIGN(fw->rta_len));

    debug_tca_acts(lev+1, fw);
}

#if HAVE_DECL_TCA_FW_MASK
/*
 * debug attribute TCA_FW_MASK
 */
void debug_tca_fw_mask(int lev, struct rtattr *fw)
{
    if(RTA_PAYLOAD(fw) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FW_MASK(%hu): -- payload too short --",
            RTA_ALIGN(fw->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FW_MASK(%hu): 0x%08x",
        RTA_ALIGN(fw->rta_len), *(unsigned *)RTA_DATA(fw));
}
#endif

/*
 * debug tcindex options
 */
void debug_tca_options_tcindex(int lev, struct rtattr *tca)
{
    struct rtattr *tcindex[__TCA_TCINDEX_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_tcindex(tcindex, tca);

    if(tcindex[TCA_TCINDEX_HASH])
        debug_tca_tcindex_hash(lev+1, tcindex[TCA_TCINDEX_HASH]);

    if(tcindex[TCA_TCINDEX_MASK])
        debug_tca_tcindex_mask(lev+1, tcindex[TCA_TCINDEX_MASK]);

    if(tcindex[TCA_TCINDEX_SHIFT])
        debug_tca_tcindex_shift(lev+1, tcindex[TCA_TCINDEX_SHIFT]);

    if(tcindex[TCA_TCINDEX_FALL_THROUGH])
        debug_tca_tcindex_fall_through(lev+1, tcindex[TCA_TCINDEX_FALL_THROUGH]);

    if(tcindex[TCA_TCINDEX_CLASSID])
        debug_tca_tcindex_classid(lev+1, tcindex[TCA_TCINDEX_CLASSID]);

    if(tcindex[TCA_TCINDEX_POLICE])
        debug_tca_tcindex_police(lev+1, tcindex[TCA_TCINDEX_POLICE]);

    if(tcindex[TCA_TCINDEX_ACT])
        debug_tca_tcindex_act(lev+1, tcindex[TCA_TCINDEX_ACT]);
}

/*
 * debug attribute TCA_TCINDEX_HASH
 */
void debug_tca_tcindex_hash(int lev, struct rtattr *tcindex)
{
    if(RTA_PAYLOAD(tcindex) < sizeof(unsigned short)) {
        rec_dbg(lev, "TCA_TCINDEX_HASH(%hu): -- payload too short --",
            RTA_ALIGN(tcindex->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_TCINDEX_HASH(%hu): %hu",
        RTA_ALIGN(tcindex->rta_len), *(unsigned short *)RTA_DATA(tcindex));
}

/*
 * debug attribute TCA_TCINDEX_MASK
 */
void debug_tca_tcindex_mask(int lev, struct rtattr *tcindex)
{
    if(RTA_PAYLOAD(tcindex) < sizeof(unsigned short)) {
        rec_dbg(lev, "TCA_TCINDEX_MASK(%hu): -- payload too short --",
            RTA_ALIGN(tcindex->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_TCINDEX_MASK(%hu): 0x%04x",
        RTA_ALIGN(tcindex->rta_len), *(unsigned short *)RTA_DATA(tcindex));
}

/*
 * debug attribute TCA_TCINDEX_SHIFT
 */
void debug_tca_tcindex_shift(int lev, struct rtattr *tcindex)
{
    if(RTA_PAYLOAD(tcindex) < sizeof(int)) {
        rec_dbg(lev, "TCA_TCINDEX_SHIFT(%hu): -- payload too short --",
            RTA_ALIGN(tcindex->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_TCINDEX_SHIFT(%hu): %d",
        RTA_ALIGN(tcindex->rta_len), *(int *)RTA_DATA(tcindex));
}

/*
 * debug attribute TCA_TCINDEX_FALL_THROUGH
 */
void debug_tca_tcindex_fall_through(int lev, struct rtattr *tcindex)
{
    if(RTA_PAYLOAD(tcindex) < sizeof(int)) {
        rec_dbg(lev, "TCA_TCINDEX_FALL_THROUGH(%hu): -- payload too short --",
            RTA_ALIGN(tcindex->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_TCINDEX_FALL_THROUGH(%hu): 0x%08x",
        RTA_ALIGN(tcindex->rta_len), *(int *)RTA_DATA(tcindex));
}

/*
 * debug attribute TCA_TCINDEX_CLASSID
 */
void debug_tca_tcindex_classid(int lev, struct rtattr *tcindex)
{
    unsigned n_classid;
    char s_classid[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(tcindex) < sizeof(n_classid)) {
        rec_dbg(lev, "TCA_TCINDEX_CLASSID(%hu): -- payload too short --",
            RTA_ALIGN(tcindex->rta_len));
        return;
    }
    n_classid = *(unsigned *)RTA_DATA(tcindex);
    parse_tc_classid(s_classid, sizeof(s_classid), n_classid);

    rec_dbg(lev, "TCA_TCINDEX_CLASSID(%hu): 0x%08x(%s)",
        RTA_ALIGN(tcindex->rta_len), n_classid, s_classid);
}

/*
 * debug attribute TCA_TCINDEX_POLICE
 */
void debug_tca_tcindex_police(int lev, struct rtattr *tcindex)
{
    rec_dbg(lev, "TCA_TCINDEX_POLICE(%hu):", RTA_ALIGN(tcindex->rta_len));

    debug_tca_act_options_police(lev, tcindex);
}

/*
 * debug attribute TCA_TCINDEX_ACT
 */
void debug_tca_tcindex_act(int lev, struct rtattr *tcindex)
{
    rec_dbg(lev, "TCA_TCINDEX_ACT(%hu):", RTA_ALIGN(tcindex->rta_len));

    debug_tca_acts(lev+1, tcindex);
}

#if HAVE_DECL_TCA_FLOW_UNSPEC
/*
 * debug flow options
 */
void debug_tca_options_flow(int lev, struct rtattr *tca)
{
    struct rtattr *flow[__TCA_FLOW_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_flow(flow, tca);

    if(flow[TCA_FLOW_KEYS])
        debug_tca_flow_keys(lev+1, flow[TCA_FLOW_KEYS]);

    if(flow[TCA_FLOW_MODE])
        debug_tca_flow_mode(lev+1, flow[TCA_FLOW_MODE]);

    if(flow[TCA_FLOW_BASECLASS])
        debug_tca_flow_baseclass(lev+1, flow[TCA_FLOW_BASECLASS]);

    if(flow[TCA_FLOW_RSHIFT])
        debug_tca_flow_rshift(lev+1, flow[TCA_FLOW_RSHIFT]);

    if(flow[TCA_FLOW_ADDEND])
        debug_tca_flow_addend(lev+1, flow[TCA_FLOW_ADDEND]);

    if(flow[TCA_FLOW_MASK])
        debug_tca_flow_mask(lev+1, flow[TCA_FLOW_MASK]);

    if(flow[TCA_FLOW_XOR])
        debug_tca_flow_xor(lev+1, flow[TCA_FLOW_XOR]);

    if(flow[TCA_FLOW_DIVISOR])
        debug_tca_flow_divisor(lev+1, flow[TCA_FLOW_DIVISOR]);

    if(flow[TCA_FLOW_ACT])
        debug_tca_flow_act(lev+1, flow[TCA_FLOW_ACT]);

    if(flow[TCA_FLOW_POLICE])
        debug_tca_flow_police(lev+1, flow[TCA_FLOW_POLICE]);

    if(flow[TCA_FLOW_EMATCHES])
        debug_tca_flow_ematches(lev+1, flow[TCA_FLOW_EMATCHES]);

    if(flow[TCA_FLOW_PERTURB])
        debug_tca_flow_perturb(lev+1, flow[TCA_FLOW_PERTURB]);
}

/*
 * debug attribute TCA_FLOW_KEYS
 */
void debug_tca_flow_keys(int lev, struct rtattr *flow)
{
    char flags_list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FLOW_KEYS(%hu): -- payload too short --",
            RTA_ALIGN(flow->rta_len));
        return;
    }
    convert_flow_key(*(unsigned *)RTA_DATA(flow), flags_list, sizeof(flags_list), 1);

    rec_dbg(lev, "TCA_FLOW_KEYS(%hu): %u(%s)",
        RTA_ALIGN(flow->rta_len), *(unsigned *)RTA_DATA(flow), flags_list);
}

/*
 * debug attribute TCA_FLOW_MODE
 */
void debug_tca_flow_mode(int lev, struct rtattr *flow)
{
    unsigned mode;

    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FLOW_MODE(%hu): -- payload too short --",
            RTA_ALIGN(flow->rta_len));
        return;
    }
    mode = *(unsigned *)RTA_DATA(flow);

    rec_dbg(lev, "TCA_FLOW_MODE(%hu): %u(%s)",
        RTA_ALIGN(flow->rta_len), mode, convert_flow_mode(mode, 1));
}

/*
 * debug attribute TCA_FLOW_BASECLASS
 */
void debug_tca_flow_baseclass(int lev, struct rtattr *flow)
{
    unsigned n_classid;
    char s_classid[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(flow) < sizeof(n_classid)) {
        rec_dbg(lev, "TCA_FLOW_BASECLASS(%hu): -- payload too short --",
            RTA_ALIGN(flow->rta_len));
        return;
    }
    n_classid = *(unsigned *)RTA_DATA(flow);
    parse_tc_classid(s_classid, sizeof(s_classid), n_classid);

    rec_dbg(lev, "TCA_FLOW_BASECLASS(%hu): 0x%08x(%s)",
        RTA_ALIGN(flow->rta_len), n_classid, s_classid);
}

/*
 * debug attribute TCA_FLOW_RSHIFT
 */
void debug_tca_flow_rshift(int lev, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FLOW_RSHIFT(%hu): -- payload too short --",
            RTA_ALIGN(flow->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FLOW_RSHIFT(%hu): %u",
        RTA_ALIGN(flow->rta_len), *(unsigned *)RTA_DATA(flow));
}

/*
 * debug attribute TCA_FLOW_ADDEND
 */
void debug_tca_flow_addend(int lev, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FLOW_ADDEND(%hu): -- payload too short --",
            RTA_ALIGN(flow->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FLOW_ADDEND(%hu): 0x%08x",
        RTA_ALIGN(flow->rta_len), *(unsigned *)RTA_DATA(flow));
}

/*
 * debug attribute TCA_FLOW_MASK
 */
void debug_tca_flow_mask(int lev, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FLOW_MASK(%hu): -- payload too short --",
            RTA_ALIGN(flow->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FLOW_MASK(%hu): 0x%08x",
        RTA_ALIGN(flow->rta_len), *(unsigned *)RTA_DATA(flow));
}

/*
 * debug attribute TCA_FLOW_XOR
 */
void debug_tca_flow_xor(int lev, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FLOW_XOR(%hu): -- payload too short --",
            RTA_ALIGN(flow->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FLOW_XOR(%hu): 0x%08x",
        RTA_ALIGN(flow->rta_len), *(unsigned *)RTA_DATA(flow));
}

/*
 * debug attribute TCA_FLOW_DIVISOR
 */
void debug_tca_flow_divisor(int lev, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FLOW_DIVISOR(%hu): -- payload too short --",
            RTA_ALIGN(flow->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FLOW_DIVISOR(%hu): 0x%u",
        RTA_ALIGN(flow->rta_len), *(unsigned *)RTA_DATA(flow));
}

/*
 * debug attribute TCA_TCA_FLOW_ACT
 */
void debug_tca_flow_act(int lev, struct rtattr *flow)
{
    rec_dbg(lev, "TCA_TCA_FLOW_ACT(%hu):", RTA_ALIGN(flow->rta_len));

    debug_tca_acts(lev+1, flow);
}

/*
 * debug attribute TCA_FLOW_POLICE
 */
void debug_tca_flow_police(int lev, struct rtattr *flow)
{
    rec_dbg(lev, "TCA_FLOW_POLICE(%hu):", RTA_ALIGN(flow->rta_len));

    debug_tca_act_options_police(lev, flow);
}

/*
 * debug attribute TCA_FLOW_EMATCHES
 */
void debug_tca_flow_ematches(int lev, struct rtattr *flow)
{
    rec_dbg(lev, "TCA_FLOW_EMATCHES(%hu):", RTA_ALIGN(flow->rta_len));

    debug_tca_ematch(lev, flow);
}

/*
 * debug attribute TCA_FLOW_PERTURB
 */
void debug_tca_flow_perturb(int lev, struct rtattr *flow)
{
    if(RTA_PAYLOAD(flow) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_FLOW_PERTURB(%hu): -- payload too short --",
            RTA_ALIGN(flow->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_FLOW_PERTURB(%hu): 0x%u",
        RTA_ALIGN(flow->rta_len), *(unsigned *)RTA_DATA(flow));
}
#endif

/*
 * debug basic options
 */
void debug_tca_options_basic(int lev, struct rtattr *tca)
{
    struct rtattr *basic[__TCA_BASIC_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_basic(basic, tca);

    if(basic[TCA_BASIC_CLASSID])
        debug_tca_basic_classid(lev+1, basic[TCA_BASIC_CLASSID]);

    if(basic[TCA_BASIC_EMATCHES])
        debug_tca_basic_ematches(lev+1, basic[TCA_BASIC_EMATCHES]);

    if(basic[TCA_BASIC_ACT])
        debug_tca_basic_act(lev+1, basic[TCA_BASIC_ACT]);

    if(basic[TCA_BASIC_POLICE])
        debug_tca_basic_police(lev+1, basic[TCA_BASIC_POLICE]);
}

/*
 * debug attribute TCA_BASIC_CLASSID
 */
void debug_tca_basic_classid(int lev, struct rtattr *basic)
{
    unsigned n_classid;
    char s_classid[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(basic) < sizeof(n_classid)) {
        rec_dbg(lev, "TCA_BASIC_CLASSID(%hu): -- payload too short --",
            RTA_ALIGN(basic->rta_len));
        return;
    }
    n_classid = *(unsigned *)RTA_DATA(basic);
    parse_tc_classid(s_classid, sizeof(s_classid), n_classid);

    rec_dbg(lev, "TCA_BASIC_CLASSID(%hu): 0x%08x(%s)",
        RTA_ALIGN(basic->rta_len), n_classid, s_classid);
}

/*
 * debug attribute TCA_BASIC_EMATCHES
 */
void debug_tca_basic_ematches(int lev, struct rtattr *basic)
{
    rec_dbg(lev, "TCA_BASIC_EMATCHES(%hu):", RTA_ALIGN(basic->rta_len));

    debug_tca_ematch(lev, basic);
}

/*
 * debug attribute TCA_TCA_BASIC_ACT
 */
void debug_tca_basic_act(int lev, struct rtattr *basic)
{
    rec_dbg(lev, "TCA_TCA_BASIC_ACT(%hu):", RTA_ALIGN(basic->rta_len));

    debug_tca_acts(lev+1, basic);
}

/*
 * debug attribute TCA_BASIC_POLICE
 */
void debug_tca_basic_police(int lev, struct rtattr *basic)
{
    rec_dbg(lev, "TCA_BASIC_POLICE(%hu):", RTA_ALIGN(basic->rta_len));

    debug_tca_act_options_police(lev, basic);
}

#if HAVE_DECL_TCA_CGROUP_UNSPEC
/*
 * debug cgroup options
 */
void debug_tca_options_cgroup(int lev, struct rtattr *tca)
{
    struct rtattr *cgroup[__TCA_CGROUP_MAX];

    rec_dbg(lev, "TCA_OPTIONS(%hu):", RTA_ALIGN(tca->rta_len));

    parse_cgroup(cgroup, tca);

    if(cgroup[TCA_CGROUP_ACT])
        debug_tca_cgroup_act(lev+1, cgroup[TCA_CGROUP_ACT]);

    if(cgroup[TCA_CGROUP_POLICE])
        debug_tca_cgroup_police(lev+1, cgroup[TCA_CGROUP_POLICE]);

    if(cgroup[TCA_CGROUP_EMATCHES])
        debug_tca_cgroup_ematches(lev+1, cgroup[TCA_CGROUP_EMATCHES]);
}

/*
 * debug attribute TCA_TCA_CGROUP_ACT
 */
void debug_tca_cgroup_act(int lev, struct rtattr *cgroup)
{
    rec_dbg(lev, "TCA_TCA_CGROUP_ACT(%hu):", RTA_ALIGN(cgroup->rta_len));

    debug_tca_acts(lev+1, cgroup);
}

/*
 * debug attribute TCA_CGROUP_POLICE
 */
void debug_tca_cgroup_police(int lev, struct rtattr *cgroup)
{
    rec_dbg(lev, "TCA_CGROUP_POLICE(%hu):", RTA_ALIGN(cgroup->rta_len));

    debug_tca_act_options_police(lev, cgroup);
}

/*
 * debug attribute TCA_CGROUP_EMATCHES
 */
void debug_tca_cgroup_ematches(int lev, struct rtattr *cgroup)
{
    rec_dbg(lev, "TCA_CGROUP_EMATCHES(%hu):", RTA_ALIGN(cgroup->rta_len));

    debug_tca_ematch(lev, cgroup);
}
#endif

/*
 * debug attribute TCA_EMATCH_*
 */
void debug_tca_ematch(int lev, struct rtattr *tca)
{
    struct rtattr *em_tree[__TCA_EMATCH_TREE_MAX];
    int num = -1;

    parse_ematch_tree(em_tree, tca);

    if(em_tree[TCA_EMATCH_TREE_HDR])
        num = debug_tca_ematch_tree_hdr(lev+1, em_tree[TCA_EMATCH_TREE_HDR]);

    if(em_tree[TCA_EMATCH_TREE_LIST])
        debug_tca_ematch_tree_list(lev+1, em_tree[TCA_EMATCH_TREE_LIST], num);
}

/*
 * debug attribute TCA_EMATCH_TREE_HDR
 */
int debug_tca_ematch_tree_hdr(int lev, struct rtattr *em_tree)
{
    struct tcf_ematch_tree_hdr *hdr;

    if(RTA_PAYLOAD(em_tree) < sizeof(*hdr)) {
        rec_dbg(lev, "TCA_EMATCH_TREE_HDR(%hu): -- payload too short --",
            RTA_ALIGN(em_tree->rta_len));
        return(-1);
    }
    hdr = (struct tcf_ematch_tree_hdr *)RTA_DATA(em_tree);

    rec_dbg(lev, "TCA_EMATCH_TREE_HDR(%hu):", RTA_ALIGN(em_tree->rta_len));
    rec_dbg(lev, "    [ tcf_ematch_tree_hdr(%d) ]", sizeof(*hdr));
    rec_dbg(lev, "        nmatches(%d): %hu", sizeof(hdr->nmatches), hdr->nmatches);
    rec_dbg(lev, "        progid(%d): %hu", sizeof(hdr->progid), hdr->progid);

    return(hdr->nmatches);
}

/*
 * debug attribute TCA_EMATCH_TREE_LIST
 */
void debug_tca_ematch_tree_list(int lev, struct rtattr *em_tree, int num)
{
    struct rtattr *em_list[num+1];
    struct tcf_ematch_hdr *hdr;
    int i;

    rec_dbg(lev, "TCA_EMATCH_TREE_LIST(%hu):", RTA_ALIGN(em_tree->rta_len));

    parse_ematch_list(em_list, em_tree, num);

    /* no exist em_list[0] */
    for(i = 1; i < num + 1; i++) {
        if(!em_list[i])
            return;

        if(RTA_PAYLOAD(em_list[i]) < sizeof(*hdr)) {
            rec_dbg(lev+1, "[ tcf_ematch_hdr[%d](%d) ] -- payload too short --",
                i, sizeof(*hdr));
            return;
        }
        hdr = (struct tcf_ematch_hdr *)RTA_DATA(em_list[i]);

        rec_dbg(lev+1, "[ tcf_ematch_hdr[%d](%d) ]", i, sizeof(*hdr));
        rec_dbg(lev+1, "    matchid(%d): %hu", sizeof(hdr->matchid), hdr->matchid);
        rec_dbg(lev+1, "    kind(%d): %hu(%s)",
            sizeof(hdr->kind), hdr->kind, convert_tcf_em_kind(hdr->kind, 1));
        rec_dbg(lev+1, "    flags(%d): %hu(%s)",
            sizeof(hdr->flags), hdr->flags, convert_tcf_em_flag(hdr->flags, 1));
        rec_dbg(lev+1, "    pad(%d): %hu", sizeof(hdr->pad), hdr->pad);

        /* use (char*)hdr in order to count by one byte */
        switch(hdr->kind) {
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_CMP_H
            case TCF_EM_CMP:
                debug_ematch_cmp(lev+1, (char *)hdr + sizeof(*hdr),
                    RTA_PAYLOAD(em_list[i]) - sizeof(*hdr));
                break;
#endif
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_NBYTE_H
            case TCF_EM_NBYTE:
                debug_ematch_nbyte(lev+1, (char *)hdr + sizeof(*hdr),
                    RTA_PAYLOAD(em_list[i]) - sizeof(*hdr));
                break;
#endif
            case TCF_EM_U32:
                debug_ematch_u32(lev+1, (char *)hdr + sizeof(*hdr),
                    RTA_PAYLOAD(em_list[i]) - sizeof(*hdr));
                break;
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
            case TCF_EM_META:
                debug_ematch_meta(lev+1, (char *)hdr + sizeof(*hdr),
                    RTA_PAYLOAD(em_list[i]) - sizeof(*hdr));
                break;
#endif
            /* not implemented yet
            case TCF_EM_TEXT:
            case TCF_EM_VLAN:
            case TCF_EM_CANID:
            case TCF_EM_IPSET:
            */
            default:
                break;
        }
    }
}

#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_CMP_H
/*
 * debug ematch cmp
 */
void debug_ematch_cmp(int lev, void *p, int len)
{
    struct tcf_em_cmp *cmp;

    if(len < sizeof(struct tcf_em_cmp)) {
        rec_dbg(lev, "[ tcf_em_cmp(%d) ] -- payload too short --", len);
        return;
    }
    cmp = (struct tcf_em_cmp *)p;

    rec_dbg(lev, "[ tcf_em_cmp(%d) ]", len);
    rec_dbg(lev, "    val(%d): 0x%08x", sizeof(cmp->val), cmp->val);
    rec_dbg(lev, "    mask(%d): 0x%08x", sizeof(cmp->mask), cmp->mask);
    rec_dbg(lev, "    off(%d): %hu", sizeof(cmp->off), cmp->off);
    rec_dbg(lev, "    align:4(%d): 0x%x(%s)",
        sizeof(__u8), cmp->align, convert_tcf_em_align(cmp->align, 1));
    rec_dbg(lev, "    flags:4(%d): 0x%x(%s)",
        sizeof(__u8), cmp->flags, TCF_EM_CMP_TRANS ? "TRANS" : "none");
    rec_dbg(lev, "    layer:4(%d): 0x%x", sizeof(__u8), cmp->layer);
    rec_dbg(lev, "    opnd:4(%d): 0x%x(%s)",
        sizeof(__u8), cmp->opnd, convert_tcf_em_opnd(cmp->opnd, 1));
}
#endif

#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_NBYTE_H
/*
 * debug ematch nbyte
 */
void debug_ematch_nbyte(int lev, void *p, int len)
{
    struct tcf_em_nbyte *nbyte;
    int i;
    char *data;

    if(len < sizeof(struct tcf_em_nbyte)) {
        rec_dbg(lev, "[ tcf_em_nbyte(%d) ] -- payload too short --", len);
        return;
    }
    nbyte = (struct tcf_em_nbyte *)p;

    rec_dbg(lev, "[ tcf_em_nbyte(%d) ]", len);
    rec_dbg(lev, "    off(%d): %hu", sizeof(nbyte->off), nbyte->off);
    rec_dbg(lev, "    len:12(%d): %hu", sizeof(__u16), nbyte->len);
    rec_dbg(lev, "    layer:4(%d): %hu", sizeof(__u8), nbyte->layer);

    if(len - sizeof(*nbyte) < nbyte->len) {
        rec_dbg(lev, "    data(%d): -- payload too short --", nbyte->len);
        return;
    }

    data = (char *)nbyte + sizeof(*nbyte);

    for(i = 0; i < nbyte->len; i++)
        data[i] = isprint(data[i]) ? data[i] : '.';

    data[i] = '\0';

    rec_dbg(lev, "    data(%d): %s", nbyte->len, data);
}
#endif

/*
 * debug ematch u32
 */
void debug_ematch_u32(int lev, void *p, int len)
{
    struct tc_u32_key *key;

    if(len < sizeof(*key)) {
        rec_dbg(lev, "[ tc_u32_key(%d) ] -- payload too short --", len);
        return;
    }
    key = (struct tc_u32_key *)p;

    rec_dbg(lev, "[ tc_u32_key(%d) ]", sizeof(*key));
    rec_dbg(lev, "    mask(%d): 0x%08x", sizeof(key->mask), key->mask); /* AND */
    rec_dbg(lev, "    val(%d): 0x%08x", sizeof(key->val), key->val); /* XOR */
    rec_dbg(lev, "    off(%d): %d", sizeof(key->off), key->off); /* Offset */
    rec_dbg(lev, "    offmask(%d): 0x%08x", sizeof(key->offmask), key->offmask);
}

#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
/*
 * debug ematch meta
 */
void debug_ematch_meta(int lev, void *p, int len)
{
    struct rtattr *meta[__TCA_EM_META_MAX];
    struct tcf_meta_hdr *hdr = NULL;

    parse_meta(meta, p, len);

    if(meta[TCA_EM_META_HDR])
        hdr = debug_tca_em_meta_hdr(lev, meta[TCA_EM_META_HDR]);

    if(meta[TCA_EM_META_LVALUE])
        debug_tca_em_meta_lvalue(lev, meta[TCA_EM_META_LVALUE], hdr);

    if(meta[TCA_EM_META_RVALUE])
        debug_tca_em_meta_rvalue(lev, meta[TCA_EM_META_RVALUE], hdr);
}

/*
 * debug attribute TCA_EM_META_HDR
 */
struct tcf_meta_hdr *debug_tca_em_meta_hdr(int lev, struct rtattr *meta)
{
    struct tcf_meta_hdr *hdr;
    struct tcf_meta_val *left, *right;

    if(RTA_PAYLOAD(meta) < sizeof(*hdr)) {
        rec_dbg(lev, "TCA_EM_META_HDR(%hu): -- payload too short --",
            RTA_ALIGN(meta->rta_len));
        return(NULL);
    }
    hdr = (struct tcf_meta_hdr *)RTA_DATA(meta);
    left = &(hdr->left);
    right = &(hdr->right);

    rec_dbg(lev, "TCA_EM_META_HDR(%hu):", RTA_ALIGN(meta->rta_len));
    rec_dbg(lev, "    [ tcf_meta_hdr(%d) ]", sizeof(*hdr));
    rec_dbg(lev, "        [ tcf_meta_val left(%d) ]", sizeof(*left));
    rec_dbg(lev, "            kind(%d): 0x%04x(%s,%s)",
        sizeof(left->kind), left->kind,
        convert_tcf_meta_type(TCF_META_TYPE(left->kind), 1),
        convert_tcf_meta_id(TCF_META_ID(left->kind), 1));
    rec_dbg(lev, "            shift(%d): %d", sizeof(left->shift), left->shift);
    rec_dbg(lev, "            op(%d): %d(%s)",
        sizeof(left->op), left->op, convert_tcf_em_opnd(left->op, 1));
    rec_dbg(lev, "        [ tcf_meta_val right(%d) ]", sizeof(*right));
    rec_dbg(lev, "            kind(%d): 0x%04x(%s,%s)",
        sizeof(right->kind), right->kind,
        convert_tcf_meta_type(TCF_META_TYPE(right->kind), 1),
        convert_tcf_meta_id(TCF_META_ID(right->kind), 1));
    rec_dbg(lev, "            shift(%d): %d", sizeof(right->shift), right->shift);
    rec_dbg(lev, "            op(%d): %d(%s)",
        sizeof(right->op), right->op, convert_tcf_em_opnd(right->op, 1));

    return(hdr);
}

/*
 * debug attribute TCA_EM_META_LVALUE
 */
void debug_tca_em_meta_lvalue(int lev, struct rtattr *meta, struct tcf_meta_hdr *hdr)
{
    int i, type, len = RTA_PAYLOAD(meta) + 1;
    char *data = (char *)RTA_DATA(meta), *value;

    if(!hdr) {
        rec_dbg(lev, "TCA_EM_META_LVALUE(%hu): -- unknown type --",
            RTA_ALIGN(meta->rta_len));
        return;
    }
    type = TCF_META_TYPE(hdr->left.kind);

    switch(type) {
        case TCF_META_TYPE_VAR:
            value = malloc(len);
            if(!value) {
                rec_dbg(lev, "TCA_EM_META_LVALUE(%hu): -- %s --",
                    RTA_ALIGN(meta->rta_len), strerror(errno));
                return;
            }
            memset(value, 0, len);

            for(i = 0; i < RTA_PAYLOAD(meta); i++)
                value[i] = isprint(data[i]) ? data[i] : '.';
            data[i] = '\0';

            rec_dbg(lev, "TCA_EM_META_LVALUE(%hu): %s",
                RTA_ALIGN(meta->rta_len), data);

            free(value);
            break;
        case TCF_META_TYPE_INT:
            if(RTA_PAYLOAD(meta) < sizeof(__u32)) {
                rec_dbg(lev, "TCA_EM_META_LVALUE(%hu): -- payload too short --",
                    RTA_ALIGN(meta->rta_len));
                return;
            }
            rec_dbg(lev, "TCA_EM_META_LVALUE(%hu): %d",
                RTA_ALIGN(meta->rta_len), *(int *)RTA_DATA(meta));
            break;
        default:
            rec_dbg(lev, "TCA_EM_META_LVALUE(%hu): -- unknown type(%d) --",
                RTA_ALIGN(meta->rta_len), type);
            return;
    }

}

/*
 * debug attribute TCA_EM_META_RVALUE
 */
void debug_tca_em_meta_rvalue(int lev, struct rtattr *meta, struct tcf_meta_hdr *hdr)
{
    int i, type, len = RTA_PAYLOAD(meta) + 1;
    char *data = (char *)RTA_DATA(meta), *value;

    if(!hdr) {
        rec_dbg(lev, "TCA_EM_META_RVALUE(%hu): -- unknown type --",
            RTA_ALIGN(meta->rta_len));
        return;
    }
    type = TCF_META_TYPE(hdr->left.kind);

    switch(type) {
        case TCF_META_TYPE_VAR:
            value = malloc(len);
            if(!value) {
                rec_dbg(lev, "TCA_EM_META_RVALUE(%hu): -- %s --",
                    RTA_ALIGN(meta->rta_len), strerror(errno));
                return;
            }
            memset(value, 0, len);

            for(i = 0; i < RTA_PAYLOAD(meta); i++)
                value[i] = isprint(data[i]) ? data[i] : '.';
            data[i] = '\0';

            rec_dbg(lev, "TCA_EM_META_RVALUE(%hu): %s",
                RTA_ALIGN(meta->rta_len), data);

            free(value);
            break;
        case TCF_META_TYPE_INT:
            if(RTA_PAYLOAD(meta) < sizeof(__u32)) {
                rec_dbg(lev, "TCA_EM_META_RVALUE(%hu): -- payload too short --",
                    RTA_ALIGN(meta->rta_len));
                return;
            }
            rec_dbg(lev, "TCA_EM_META_RVALUE(%hu): %d",
                RTA_ALIGN(meta->rta_len), *(int *)RTA_DATA(meta));
            break;
        default:
            rec_dbg(lev, "TCA_EM_META_RVALUE(%hu): -- unknown type(%d) --",
                RTA_ALIGN(meta->rta_len), type);
            return;
    }
}
#endif

/*
 *  convert TC_U32_* flags from number to string
 */
void convert_tc_u32_flags(int flags, char *flags_list, int len, int debug)
{
    if(!flags) {
        strncpy(flags_list, debug ? "NONE" : "none", len);
        return;
    }
#define _TC_U32_FLAGS(s1, s2) \
    if((flags & TC_U32_##s1) && (len - strlen(flags_list) - 1 > 0)) \
        (flags &= ~ TC_U32_##s1) ? \
            strncat(flags_list, debug ? #s1 "," : #s2 ",", \
                len - strlen(flags_list) - 1) : \
            strncat(flags_list, debug ? #s1 : #s2, \
                len - strlen(flags_list) - 1);
    _TC_U32_FLAGS(TERMINAL, terminal);
    _TC_U32_FLAGS(OFFSET, offset);
    _TC_U32_FLAGS(VAROFFSET, varoffset);
    _TC_U32_FLAGS(EAT, eat);
#undef _TC_U32_FLAGS
    if(!strlen(flags_list))
        strncpy(flags_list, debug ? "UNKNOWN" : "unknown", len);
}

/*
 * convert ETH_P_* protocol from number to string
 */
const char *convert_eth_p(int proto, int debug)
{
    proto = ntohs(proto);

#define _ETH_P(s1, s2) \
    if(proto == ETH_P_##s1) \
        return(debug ? #s1 : #s2);
    _ETH_P(LOOP, loop)
    _ETH_P(PUP, pup)
    _ETH_P(PUPAT, PUPAT)
    _ETH_P(IP, ip)
    _ETH_P(X25, x25)
    _ETH_P(ARP, arp)
    _ETH_P(BPQ, bpq)
    _ETH_P(IEEEPUP, ieeepup)
    _ETH_P(IEEEPUPAT, ieeeupat)
    _ETH_P(DEC, dec)
    _ETH_P(DNA_DL, dna_dl)
    _ETH_P(DNA_RC, dna_rc)
    _ETH_P(DNA_RT, dna_rt)
    _ETH_P(LAT, lat)
    _ETH_P(DIAG, diag)
    _ETH_P(CUST, cust)
    _ETH_P(SCA, sca)
#ifdef ETH_P_TEB
    _ETH_P(TEB, teb)
#endif
    _ETH_P(RARP, rarp)
    _ETH_P(ATALK, atalk)
    _ETH_P(AARP, aarp)
    _ETH_P(8021Q, 8021q)
    _ETH_P(IPX, ipx)
    _ETH_P(IPV6, ipv6)
#ifdef ETH_P_PAUSE
    _ETH_P(PAUSE, pause)
#endif
    _ETH_P(SLOW, slow)
    _ETH_P(WCCP, wccp)
    _ETH_P(PPP_DISC, ppp_disc)
    _ETH_P(PPP_SES, ppp_ses)
    _ETH_P(MPLS_UC, mpls_uc)
    _ETH_P(MPLS_MC, mpls_mc)
    _ETH_P(ATMMPOA, atmmpoa)
#ifdef ETH_P_LINK_CTL
    _ETH_P(LINK_CTL, link_ctl)
#endif
    _ETH_P(ATMFATE, atmfate)
#ifdef ETH_P_PAE
    _ETH_P(PAE, pae)
#endif
    _ETH_P(AOE, aoe)
#ifdef ETH_P_8021AD
    _ETH_P(8021AD, 8021ad)
#endif
#ifdef ETH_P_802_EX1
    _ETH_P(802_EX1, 802_ex1)
#endif
    _ETH_P(TIPC, tipc)
#ifdef ETH_P_8021AH
    _ETH_P(8021AH, 8021ah)
#endif
#ifdef ETH_P_1588
    _ETH_P(1588, 1588)
#endif
    _ETH_P(FCOE, fcoe)
#ifdef ETH_P_TDLS
    _ETH_P(TDLS, tdls)
#endif
    _ETH_P(FIP, fip)
#ifdef ETH_P_QINQ1
    _ETH_P(QINQ1, qinq1)
#endif
#ifdef ETH_P_QINQ2
    _ETH_P(QINQ2, qinq2)
#endif
#ifdef ETH_P_QINQ3
    _ETH_P(QINQ3, qinq3)
#endif
#ifdef ETH_P_EDSA
    _ETH_P(EDSA, edsa)
#endif
#ifdef ETH_P_AF_IUCV
    _ETH_P(AF_IUCV, af_iucv)
#endif
    _ETH_P(802_3, 802_3)
    _ETH_P(AX25, ax25)
    _ETH_P(ALL, all)
    _ETH_P(802_2, 802_2)
    _ETH_P(SNAP, snap)
    _ETH_P(DDCMP, ddcmp)
    _ETH_P(WAN_PPP, wan_ppp)
    _ETH_P(PPP_MP, ppp_mp)
    _ETH_P(LOCALTALK, localtalk)
#ifdef ETH_P_CAN
    _ETH_P(CAN, can)
#endif
#ifdef ETH_P_CANFD
    _ETH_P(CANFD, canfd)
#endif
    _ETH_P(PPPTALK, ppptalk)
    _ETH_P(TR_802_2, tr_802_2)
    _ETH_P(MOBITEX, mobitex)
    _ETH_P(CONTROL, control)
    _ETH_P(IRDA, irda)
    _ETH_P(ECONET, econet)
    _ETH_P(HDLC, hdlc)
    _ETH_P(ARCNET, arcnet)
#ifdef ETH_P_DSA
    _ETH_P(DSA, dsa)
#endif
#ifdef ETH_P_TRAILER
    _ETH_P(TRAILER, trailer)
#endif
#ifdef ETH_P_PHONET
    _ETH_P(PHONET, phonet)
#endif
#ifdef ETH_P_IEEE802154
    _ETH_P(IEEE802154, ieee802154)
#endif
#ifdef ETH_P_CAIF
    _ETH_P(CAIF, caif)
#endif
#undef _ETH_P
    return("UNKNOWN");
}

#if HAVE_DECL_TCA_FLOW_UNSPEC
/*
 *  convert FLOW_KEY_* flags from number to string
 */
void convert_flow_key(int flags, char *flags_list, int len, int debug)
{
    if(!flags) {
        strncpy(flags_list, debug ? "NONE" : "none", len);
        return;
    }
#define _FLOW_KEY(s1, s2) \
    if((flags & (1 << FLOW_KEY_##s1)) && (len - strlen(flags_list) - 1 > 0)) \
        (flags &= ~(1 << FLOW_KEY_##s1)) ? \
            strncat(flags_list, debug ? #s1 "," : #s2 ",", \
                len - strlen(flags_list) - 1) : \
            strncat(flags_list, debug ? #s1 : #s2, \
                len - strlen(flags_list) - 1);
        _FLOW_KEY(SRC, src)
        _FLOW_KEY(DST, dst)
        _FLOW_KEY(PROTO, proto)
        _FLOW_KEY(PROTO_SRC, proto-src)
        _FLOW_KEY(PROTO_DST, proto-dst)
        _FLOW_KEY(IIF, iif)
        _FLOW_KEY(PRIORITY, priority)
        _FLOW_KEY(MARK, mark)
        _FLOW_KEY(NFCT, nfct)
        _FLOW_KEY(NFCT_SRC, nfct-src)
        _FLOW_KEY(NFCT_DST, nfct-dst)
        _FLOW_KEY(NFCT_PROTO_SRC, nfct-proto-src)
        _FLOW_KEY(NFCT_PROTO_DST, nfct-proto-dst)
        _FLOW_KEY(RTCLASSID, rt-classid)
        _FLOW_KEY(SKUID, sk-uid)
        _FLOW_KEY(SKGID, sk-gid)
        _FLOW_KEY(VLAN_TAG, vlan-tag)
#if HAVE_DECL_FLOW_KEY_RXHASH
        _FLOW_KEY(RXHASH, rxhash)
#endif
#undef _FLOW_KEY
    if(!strlen(flags_list))
        strncpy(flags_list, debug ? "UNKNOWN" : "unknown", len);
}

/*
 * convert FLOW_MODE_* protocol from number to string
 */
const char *convert_flow_mode(int mode, int debug)
{
#define _FLOW_MODE(s1, s2) \
    if(mode == FLOW_MODE_##s1) \
        return(debug ? #s1 : #s2);
    _FLOW_MODE(MAP, map)
    _FLOW_MODE(HASH, hash)
#undef _FLOW_MODE
    return(debug ? "UNKNOWN" : "unknown");
}
#endif

/*
 * convert TCF_EM_* kind from number to string
 */
const char *convert_tcf_em_kind(int kind, int debug)
{
#define _TCF_EM(s1, s2) \
    if(kind == TCF_EM_##s1) \
        return(debug ? #s1 : #s2);
    _TCF_EM(CONTAINER, container)
    _TCF_EM(CMP, cmp)
    _TCF_EM(NBYTE, nbyte)
    _TCF_EM(U32, u32)
    _TCF_EM(META, meta)
    _TCF_EM(TEXT, text)
#if HAVE_DECL_TCF_EM_VLAN
    _TCF_EM(VLAN, vlan)
#endif
#if HAVE_DECL_TCF_EM_CANID
    _TCF_EM(CANID, canid)
#endif
#if HAVE_DECL_TCF_EM_IPSET
    _TCF_EM(IPSET, ipset)
#endif
#undef _TCF_EM
    return(debug ? "UNKNOWN" : "unknown");
}

/*
 * convert TCF_EM_* flags from number to string
 */
const char *convert_tcf_em_flag(int flag, int debug)
{
#define _TCF_EM(s1, s2) \
    if(flag == TCF_EM_##s1) \
        return(debug ? #s1 : #s2);
    _TCF_EM(REL_END, end)
    _TCF_EM(REL_AND, and)
    _TCF_EM(REL_OR, or)
    _TCF_EM(INVERT, invert)
    _TCF_EM(SIMPLE, simple)
#undef _TCF_EM
    return(debug ? "UNKNOWN" : "unknown");
}

#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_CMP_H
/*
 * convert TCF_EM_ALIGN_* from number to string
 */
const char *convert_tcf_em_align(int align, int debug)
{
#define _TCF_EM_ALIGN(s1, s2) \
    if(align == TCF_EM_ALIGN_##s1) \
        return(debug ? #s1 : #s2);
    _TCF_EM_ALIGN(U8, u8)
    _TCF_EM_ALIGN(U16, u16)
    _TCF_EM_ALIGN(U32, u32)
#undef _TCF_EM_ALIGN
    return(debug ? "UNKNOWN" : "unknown");
}
#endif

/*
 * convert TCF_EM_OPND_* from number to string
 */
const char *convert_tcf_em_opnd(int opnd, int debug)
{
#define _TCF_EM_OPND(s1, s2) \
    if(opnd == TCF_EM_OPND_##s1) \
        return(debug ? #s1 : #s2);
    _TCF_EM_OPND(EQ, eq)
    _TCF_EM_OPND(GT, gt)
    _TCF_EM_OPND(LT, lt)
#undef _TCF_EM_OPND
    return(debug ? "UNKNOWN" : "unknown");
}

#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
/*
 * convert TCF_META_TYPE_* from number to string
 */
const char *convert_tcf_meta_type(int type, int debug)
{
#define _TCF_META_TYPE(s1, s2) \
    if(type == TCF_META_TYPE_##s1) \
        return(debug ? #s1 : #s2);
    _TCF_META_TYPE(VAR, var)
    _TCF_META_TYPE(INT, int)
#undef _TCF_META_TYPE
    return(debug ? "UNKNOWN" : "unknown");
}

/*
 * convert TCF_META_ID_* from number to string
 */
const char *convert_tcf_meta_id(int id, int debug)
{
#define _TCF_META_ID(s1, s2) \
    if(id == TCF_META_ID_##s1) \
        return(debug ? #s1 : #s2);
        _TCF_META_ID(VALUE, value)
        _TCF_META_ID(RANDOM, random)
        _TCF_META_ID(LOADAVG_0, loadavg_0)
        _TCF_META_ID(LOADAVG_1, loadavg_1)
        _TCF_META_ID(LOADAVG_2, loadavg_2)
        _TCF_META_ID(DEV, dev)
        _TCF_META_ID(PRIORITY, priority)
        _TCF_META_ID(PROTOCOL, protocol)
        _TCF_META_ID(PKTTYPE, pkttype)
        _TCF_META_ID(PKTLEN, pktlen)
        _TCF_META_ID(DATALEN, datalen)
        _TCF_META_ID(MACLEN, maclen)
        _TCF_META_ID(NFMARK, nfmark)
        _TCF_META_ID(TCINDEX, tcindex)
        _TCF_META_ID(RTCLASSID, rtclassid)
        _TCF_META_ID(RTIIF, rtiif)
        _TCF_META_ID(SK_FAMILY, sk_family)
        _TCF_META_ID(SK_STATE, sk_state)
        _TCF_META_ID(SK_REUSE, sk_reuse)
        _TCF_META_ID(SK_BOUND_IF, sk_bound_if)
        _TCF_META_ID(SK_REFCNT, sk_refcnt)
        _TCF_META_ID(SK_SHUTDOWN, sk_shutdown)
        _TCF_META_ID(SK_PROTO, sk_proto)
        _TCF_META_ID(SK_TYPE, sk_type)
        _TCF_META_ID(SK_RCVBUF, sk_rcvbuf)
        _TCF_META_ID(SK_RMEM_ALLOC, sk_rmem_alloc)
        _TCF_META_ID(SK_WMEM_ALLOC, sk_wmem_alloc)
        _TCF_META_ID(SK_OMEM_ALLOC, sk_omem_alloc)
        _TCF_META_ID(SK_WMEM_QUEUED, sk_wmem_queued)
        _TCF_META_ID(SK_RCV_QLEN, sk_rcv_qlen)
        _TCF_META_ID(SK_SND_QLEN, sk_snd_qlen)
        _TCF_META_ID(SK_ERR_QLEN, sk_err_qlen)
        _TCF_META_ID(SK_FORWARD_ALLOCS, sk_forward_allocs)
        _TCF_META_ID(SK_SNDBUF, sk_sndbug)
        _TCF_META_ID(SK_ALLOCS, sk_allocs)
        /* __TCF_META_ID_SK_ROUTE_CAPS */
        _TCF_META_ID(SK_HASH, sk_hash)
        _TCF_META_ID(SK_LINGERTIME, sk_lingertime)
        _TCF_META_ID(SK_ACK_BACKLOG, sk_ack_backlog)
        _TCF_META_ID(SK_MAX_ACK_BACKLOG, sk_max_ack_backlog)
        _TCF_META_ID(SK_PRIO, sk_prio)
        _TCF_META_ID(SK_RCVLOWAT, sk_rcvlowat)
        _TCF_META_ID(SK_RCVTIMEO, sk_rcvtimeo)
        _TCF_META_ID(SK_SNDTIMEO, sk_sndtimeo)
        _TCF_META_ID(SK_SENDMSG_OFF, sk_sendmsg_off)
        _TCF_META_ID(SK_WRITE_PENDING, sk_write_pending)
        _TCF_META_ID(VLAN_TAG, vlan_tag)
#if HAVE_DECL_TCF_META_ID_RXHASH
        _TCF_META_ID(RXHASH, rxhash)
#endif
#undef _TCF_META_ID
    return(debug ? "UNKNOWN" : "unknown");
}
#endif
