/*
 *  Copyright (C) 1998-2004 Luca Deri <deri@ntop.org>
 *                      
 *  			    http://www.ntop.org/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ntop.h"
#include "globals-report.h"

/* #define DEBUG */

typedef struct nfsEntries {
  HostTraffic* host;
  Counter bytesSent, bytesRcvd;
  Counter lastBytesSent, lastBytesRcvd;
  float sentThpt, rcvdThpt;
} NfsEntries;

static NfsEntries *nfsEntries[MAX_NFS_NAME_HASH];
static time_t nextNfsUpdate;

/* **** f o r w a r d **** */
static void termNfsFunct(void);
static void handleNFSPacket(u_char *_deviceId, const struct pcap_pkthdr *h, const u_char *p);
static void handleNfsWatchHTTPrequest(char* url);

/* ********************* */

static PluginInfo nfsPluginInfo[] = {
  {
    VERSION, /* current ntop version */
    "nfsWatchPlugin",
    "This plugin both handles NFS packets and produces a report about them. "
      "(This allows sites without nfs to avoid the processing overhead).",
    "2.4",           /* version */
    "<a href=\"http://luca.ntop.org/\" alt=\"Luca's home page\">L.Deri</a>",
    "nfsWatch",      /* http://<host>:<port>/plugins/nfsWatch */
    0, /* Active by default */
    0, /* Inactive setup */
    NULL, /* no special startup after init */
    termNfsFunct,    /* TermFunc   */
    handleNFSPacket, /* PluginFunc */
    handleNfsWatchHTTPrequest,
    "port 2049", /* BPF filter: filter all the nfs (nfsd = port 2049) packets */
    NULL /* no status */
  }
};

/* ********************* */

static NfsEntries* findNfsEntry(HostAddr addr) {
  unsigned long key;
  unsigned int numSearches = 0;

  if (addr.hostFamily == AF_INET)
    key = addr.Ip4Address.s_addr % MAX_NFS_NAME_HASH;
#ifdef INET6
  else if (addr.hostFamily == AF_INET6)
    key = *(u_int32_t *)&addr.Ip6Address.s6_addr[0] % MAX_NFS_NAME_HASH;
#endif

  while((nfsEntries[key] != NULL) 
	&& (numSearches++ < MAX_NFS_NAME_HASH)
	&& (addrcpy(&nfsEntries[key]->host->hostIpAddress,&addr)))
    break;

  if(nfsEntries[key] == NULL) {
    HostTraffic* host = findHostByNumIP(addr, myGlobals.actualReportDeviceId);

    if(host == NULL)
      return(NULL); /* This shouldn't happen */

    nfsEntries[key] = (NfsEntries*)malloc(sizeof(NfsEntries));
    memset(nfsEntries[key], 0, sizeof(NfsEntries));
    nfsEntries[key]->host = host;
    return(nfsEntries[key]);
  } else if(addrcmp(&nfsEntries[key]->host->hostIpAddress,&addr) == 0)
    return(nfsEntries[key]);
  else
    return(NULL);
}

/* ********************* */

static void updateNfsThpt(void) {
  unsigned long now, secsDiff;
  int i;

  now = time(NULL);
  secsDiff = now-nextNfsUpdate+PARM_THROUGHPUT_REFRESH_INTERVAL;

#ifdef DEBUG
  traceEvent(CONST_TRACE_INFO, "Updating NFS thpt...");
#endif

  for(i=0; i<MAX_NFS_NAME_HASH; i++)
    if(nfsEntries[i] != NULL) {
      float diff;

      diff = (float)(nfsEntries[i]->bytesSent - nfsEntries[i]->lastBytesSent);
      nfsEntries[i]->sentThpt = diff/secsDiff;
#ifdef DEBUG
      traceEvent(CONST_TRACE_INFO, "%s) %.2f/", 
		 nfsEntries[i]->host->hostResolvedName, 
		 nfsEntries[i]->sentThpt);
#endif

      diff = (float)(nfsEntries[i]->bytesRcvd-nfsEntries[i]->lastBytesRcvd);
      nfsEntries[i]->rcvdThpt = diff/secsDiff;
#ifdef DEBUG
      traceEvent(CONST_TRACE_INFO, "%.2f", nfsEntries[i]->rcvdThpt);
#endif

      nfsEntries[i]->lastBytesSent = nfsEntries[i]->bytesSent;
      nfsEntries[i]->lastBytesRcvd = nfsEntries[i]->bytesRcvd;
    }

  nextNfsUpdate = now+PARM_THROUGHPUT_REFRESH_INTERVAL;
}

/* ********************* */

static void handleNFSPacket(u_char *_deviceId,
			    const struct pcap_pkthdr *h, const u_char *p) {
  struct ip ip;
#ifdef INET6
  struct ip6_hdr *ip6;
  u_int isipv6;
#endif
  struct udphdr nfsPktHdr;
  u_int hlen, length;
#ifdef DEBUG
  u_short sport, dport;
#endif
  NfsEntries *srcHost, *dstHost;
  HostAddr srcAddr, dstAddr;
  int deviceId, actualDeviceId;

#ifdef WIN32
  deviceId = 0;
#else
  deviceId = *_deviceId; /* Fix courtesy of Berthold Gunreben <bg@suse.de> */
#endif

  actualDeviceId = getActualInterface(deviceId);

  if ( (myGlobals.device[deviceId].datalink >= MAX_DLT_ARRAY) ||
       (myGlobals.headerSize[myGlobals.device[deviceId].datalink] == 0) ) {
      return;  /* Sorry, but without the headerSize, we can't do this... */
  }

  length = h->len-myGlobals.headerSize[myGlobals.device[deviceId].datalink];

  memcpy(&ip, (p+myGlobals.headerSize[myGlobals.device[deviceId].datalink]), sizeof(struct ip));
#ifdef INET6
  isipv6 = (ip.ip_v == 6)?1:0;
  if (isipv6){
    hlen = sizeof(struct ip6_hdr);
    ip6 = (struct ip6_hdr *)(p+myGlobals.headerSize[myGlobals.device[deviceId].datalink]);
  }else{
    ip6 = NULL;
#endif
    hlen = (u_int)ip.ip_hl * 4;
#ifdef INET6
  }
#endif

  memcpy(&nfsPktHdr, (p+myGlobals.headerSize[myGlobals.device[deviceId].datalink]+hlen),sizeof(struct udphdr));  

#ifdef DEBUG
  sport = ntohs(nfsPktHdr.uh_sport);
  dport = ntohs(nfsPktHdr.uh_dport);

  traceEvent(CONST_TRACE_INFO, "NFS %s:%d ", intoa(ip.ip_src), sport);
  traceEvent(CONST_TRACE_INFO, "-> %s:%d [len=%d]", intoa(ip.ip_dst), dport, length);
#endif

#ifdef INET6
  if (isipv6){
    addrput(AF_INET6, &srcAddr, &ip6->ip6_src);
    addrput(AF_INET6, &dstAddr, &ip6->ip6_dst);
  }else {
#endif
    ip.ip_src.s_addr = ntohl(ip.ip_src.s_addr);
    ip.ip_dst.s_addr = ntohl(ip.ip_dst.s_addr);
    addrput(AF_INET, &srcAddr,&ip.ip_src.s_addr);
    addrput(AF_INET, &dstAddr,&ip.ip_dst.s_addr); 
#ifdef INET6
  }
#endif
  srcHost = findNfsEntry(srcAddr);
  if(srcHost != NULL) srcHost->bytesSent += length;

  dstHost = findNfsEntry(dstAddr);
  if(dstHost != NULL) dstHost->bytesRcvd += length;

  if(time(NULL) > nextNfsUpdate)
    updateNfsThpt();
}

/* ******************************* */

static int sortNFShostsIP(const void *_a, const void *_b) {
  NfsEntries **a = (NfsEntries **)_a;
  NfsEntries **b = (NfsEntries **)_b;
  int rc;

  if((a == NULL) && (b != NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsIP() (1)");
    return(1);
  } else if((a != NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsIP() (2)");
    return(-1);
  } else if((a == NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsIP() (3)");
    return(0);
  }

  rc = cmpFctnResolvedName(a, b);
  return(rc);
}

static int sortNFShostsSent(const void *_a, const void *_b) {
  NfsEntries **a = (NfsEntries **)_a;
  NfsEntries **b = (NfsEntries **)_b;

  if((a == NULL) && (b != NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsSent() (1)");
    return(1);
  } else if((a != NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsSent() (2)");
    return(-1);
  } else if((a == NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsSent() (3)");
    return(0);
  }

  if((*a)->bytesSent < (*b)->bytesSent)
    return(1);
  else if ((*a)->bytesSent > (*b)->bytesSent)
    return(-1);
  else
    return(0);
}

static int sortNFShostsRcvd(const void *_a, const void *_b) {
  NfsEntries **a = (NfsEntries **)_a;
  NfsEntries **b = (NfsEntries **)_b;

  if((a == NULL) && (b != NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsRcvd() (1)");
    return(1);
  } else if((a != NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsRcvd() (2)");
    return(-1);
  } else if((a == NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsRcvd() (3)");
    return(0);
  }

  if((*a)->bytesRcvd < (*b)->bytesRcvd)
    return(1);
  else if ((*a)->bytesRcvd > (*b)->bytesRcvd)
    return(-1);
  else
    return(0);
}

static int sortNFShostsSentThpt(const void *_a, const void *_b) {
  NfsEntries **a = (NfsEntries **)_a;
  NfsEntries **b = (NfsEntries **)_b;
  int rc;

  if((a == NULL) && (b != NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsSentThpt() (1)");
    return(1);
  } else if((a != NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsSentThpt() (2)");
    return(-1);
  } else if((a == NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsSentThpt() (3)");
    return(0);
  }

  if((*a)->sentThpt < (*b)->sentThpt)
    return(1);
  else if ((*a)->sentThpt > (*b)->sentThpt)
    return(-1);
  else
    return(0);
}

static int sortNFShostsRcvdThpt(const void *_a, const void *_b) {
  NfsEntries **a = (NfsEntries **)_a;
  NfsEntries **b = (NfsEntries **)_b;
  int rc;

  if((a == NULL) && (b != NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsRcvdThpt() (1)");
    return(1);
  } else if((a != NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsRcvdThpt() (2)");
    return(-1);
  } else if((a == NULL) && (b == NULL)) {
    traceEvent(CONST_TRACE_WARNING, "sortNFShostsRcvdThpt() (3)");
    return(0);
  }

  if((*a)->rcvdThpt < (*b)->rcvdThpt)
    return(1);
  else if ((*a)->rcvdThpt > (*b)->rcvdThpt)
    return(-1);
  else
    return(0);
}

/* ********************* */

#define CONST_NFS_SORT_HOSTIP          0
#define CONST_NFS_SORT_BYTESSENT       1
#define CONST_NFS_SORT_BYTESRCVD       2
#define CONST_NFS_SORT_SENTTHPT        3
#define CONST_NFS_SORT_RCVDTHPT        4


static void* cmpFctnNFS[] = { &sortNFShostsIP,
                              &sortNFShostsSent,
                              &sortNFShostsSentThpt,
                              &sortNFShostsRcvd,
                              &sortNFShostsRcvdThpt};

static int cmpFctnNFSmax = sizeof(cmpFctnNFS) / sizeof(cmpFctnNFS[0]);

/* ****************************** */

static void handleNfsWatchHTTPrequest(char* url) {
  static short everUpdated = 0;
  char tmpStr[2*LEN_GENERAL_WORK_BUFFER];
  int numEntries = 0, i, revertOrder=0;
  char *pluginName = "<A HREF=/plugins/nfsWatch", *sign[16], *arrow[6];
  NfsEntries *tmpNfsEntries[MAX_NFS_NAME_HASH];
  float maxSentThpt=-1, maxRcvdThpt=-1;
  char formatBuf[32], formatBuf1[32], hostLinkBuf[LEN_GENERAL_WORK_BUFFER];
  int nfsColumnSort = 0;
  char *arrowGif;

  sendHTTPHeader(FLAG_HTTP_TYPE_HTML, 0, 1);
  printHTMLheader("Welcome to nfsWatch", NULL, 0);

  if(!everUpdated) {
    updateNfsThpt();
    everUpdated = 1;
  }

  if(url[0] == '\0') {
    nfsColumnSort = 0;
  } else {
    if(url[0] == '-') {
      revertOrder = 1;
      nfsColumnSort = atoi(&url[1]);
    } else {
      nfsColumnSort = atoi(url);
    }
  }

  if(nfsColumnSort<0) nfsColumnSort=0;
  if(nfsColumnSort>cmpFctnNFSmax) nfsColumnSort=0;

  for(i=0; i<16; i++) sign[i] = "";
  if(!revertOrder) sign[nfsColumnSort] = "-";

  for(i=0; i<MAX_NFS_NAME_HASH; i++) {
    if(nfsEntries[i] != NULL) {
      tmpNfsEntries[numEntries++] = nfsEntries[i];
      if(nfsEntries[i]->sentThpt > maxSentThpt) maxSentThpt = nfsEntries[i]->sentThpt;
      if(nfsEntries[i]->sentThpt > maxRcvdThpt) maxRcvdThpt = nfsEntries[i]->rcvdThpt;
    }
  }

  if(numEntries > 0) {

    sendString("<CENTER>\n");
    sendString("<TABLE BORDER=1 "TABLE_DEFAULTS"><TR "TR_ON" "DARK_BG">");

    if(revertOrder)
      arrowGif = "&nbsp;<IMG ALT=\"Ascending order, click to reverse\" SRC=/arrow_up.gif BORDER=0>";
    else
      arrowGif = "&nbsp;<IMG ALT=\"Descending order, click to reverse\" SRC=/arrow_down.gif BORDER=0>";

    for(i=0; i<=5; i++)
      if(abs(nfsColumnSort) == i) arrow[i] = arrowGif; else arrow[i] = "";

    if(snprintf(tmpStr, sizeof(tmpStr),
                "<TR>"
                "<TH "TH_BG" rowspan=\"2\" valign=\"bottom\">%s?%s%d>Host %s</A></TH>\n"
                "<TH "TH_BG" colspan=\"3\">Data Sent</TH>\n"
                "<TH "TH_BG" colspan=\"3\">Data Received</TH>\n"
                "</TR>\n",
                pluginName, sign[1], CONST_NFS_SORT_HOSTIP, arrow[1]) < 0) BufferTooShort();
    sendString(tmpStr);

    if(snprintf(tmpStr, sizeof(tmpStr),
                "<TR>"
                "<TH "TH_BG">%s?%s%d>Bytes %s</A></TH>\n"
                "<TH "TH_BG" COLSPAN=2>%s?%s%d>Throughput %s</A></TH>\n"
                "<TH "TH_BG">%s?%s%d>Bytes %s</A></TH>\n"
                "<TH "TH_BG" COLSPAN=2>%s?%s%d>Throughput %s</A></TH>\n"
                "</TR>\n",
                pluginName, sign[2], CONST_NFS_SORT_BYTESSENT, arrow[2],
                pluginName, sign[3], CONST_NFS_SORT_SENTTHPT,  arrow[3],
                pluginName, sign[4], CONST_NFS_SORT_BYTESRCVD, arrow[4],
                pluginName, sign[5], CONST_NFS_SORT_RCVDTHPT,  arrow[5]) < 0) BufferTooShort();
    sendString(tmpStr);

    qsort(tmpNfsEntries, numEntries, sizeof(NfsEntries*), cmpFctnNFS[nfsColumnSort]);

    for(i=0; i<numEntries; i++) {
      NfsEntries *theEntry;
      char bar[512];

      if(revertOrder)
        theEntry = tmpNfsEntries[numEntries-i-1];
      else
        theEntry = tmpNfsEntries[i];
      if(snprintf(tmpStr, sizeof(tmpStr), "<TR "TR_ON" %s>%s"
                  "<TD "TD_BG" ALIGN=RIGHT>%s</TD>"
                  "<TD "TD_BG" ALIGN=RIGHT>%s</TD>",
                  getRowColor(), makeHostLink(theEntry->host, 1, 1, 0, hostLinkBuf, sizeof(hostLinkBuf)),
                  formatBytes(theEntry->bytesSent, 1, formatBuf, sizeof(formatBuf)),
                  formatThroughput(theEntry->sentThpt, 1, formatBuf1, sizeof(formatBuf1))) < 0)
        BufferTooShort();
      sendString(tmpStr);
      printBar(bar, sizeof(bar), (unsigned short)((theEntry->sentThpt*100)/maxSentThpt), FLAG_NONSPLITBAR, 100, 1);

      if(snprintf(tmpStr, sizeof(tmpStr), "<TD "TD_BG" ALIGN=RIGHT>%s</TD>"
                  "<TD "TD_BG" ALIGN=RIGHT>%s</TD>\n",
                  formatBytes(theEntry->bytesRcvd, 1, formatBuf, sizeof(formatBuf)),
                  formatThroughput(theEntry->rcvdThpt, 1, formatBuf1, sizeof(formatBuf))) < 0)
        BufferTooShort();
      sendString(tmpStr);

      printBar(bar, sizeof(bar), (unsigned short)((theEntry->rcvdThpt*100)/maxRcvdThpt), FLAG_NONSPLITBAR, 100, 1);

      sendString("</TR>\n");
    }

    sendString("</TABLE></CENTER><p>\n");

    sendString("<p><b>NOTE</b>:<ul>"
	       "<li>Click <a href=\"" CONST_HOST_SORT_NOTE_HTML "\">here</a> for more information about host sorting."
	       "</ul><p>\n");    
  } else {
    printNoDataYet();
  }

  printPluginTrailer(nfsPluginInfo->pluginURLname, NULL);

  printHTMLtrailer();
}

/* ****************************** */

static void termNfsFunct(void) {
  traceEvent(CONST_TRACE_INFO, "NFS: Thanks for using nfsWatchPlugin"); 
  traceEvent(CONST_TRACE_ALWAYSDISPLAY, "NFS: Done"); 
  fflush(stdout);
}

/* ****************************** */

/* Plugin entry fctn */
#ifdef MAKE_STATIC_PLUGIN
PluginInfo* nfsPluginEntryFctn(void) {
#else
PluginInfo* PluginEntryFctn(void) {
#endif

  traceEvent(CONST_TRACE_ALWAYSDISPLAY, "NFS: Welcome to %s. (C) 1999-2004 by Luca Deri",  
	     nfsPluginInfo->pluginName);

  memset(nfsEntries, 0, sizeof(NfsEntries*));
  nextNfsUpdate = time(NULL)+PARM_THROUGHPUT_REFRESH_INTERVAL;

  return(nfsPluginInfo);
}
