
/******************************************************************************
**
**  Copyright (C) 2005 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    mod_if.c
**  Date:    September 23, 2005
**
**  Author:  Brian Wotring
**  Purpose: platform specific methods for monitoring network devices.
**
******************************************************************************/


/* CODE USED IN THIS MODULE WAS ORIGINALLY TAKEN FROM: 
*
*   http://mail.nl.linux.org/kernelnewbies/2003-05/msg00090.html
*/

static const char *MODULE_NAME = "mod_if";


#ifndef WIN32
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>

#include "libosiris.h"
#include "libfileapi.h"
#include "rootpriv.h"
#include "common.h"
#include "version.h"

#include "scanner.h"
#include "logging.h"

#if defined(SYSTEM_LINUX)
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include <sys/socket.h>
#include <sys/types.h>
#include <net/if.h>

#include <sys/ioctl.h>
#include <net/if_arp.h>
#include <arpa/inet.h>

#define inaddrr(x) (*(struct in_addr *) &ifr->x[sizeof sa.sin_port])
#define IFRSIZE   ((int)(size * sizeof (struct ifreq)))
#endif

#if defined(SYSTEM_LINUX)
void process_if_unix( SCANNER *scanner )
{
    unsigned char*u;
    int    sockfd, size  = 1;
    struct ifreq *ifr;
    struct ifconf ifc;
    struct sockaddr_in sa;

    SCAN_RECORD_TEXT_1 record;

    /* Make sure we are able to create sockets */
    
    if ( (sockfd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP)) < 0 )
    {
        log_error( "mod_if unable to create socket!" );
        return;
    }

    ifc.ifc_len = IFRSIZE;
    ifc.ifc_req = NULL;

    do
    {
        ++size;

        /* realloc buffer size until no overflow occurs  */
        
        if ((ifc.ifc_req = realloc(ifc.ifc_req, IFRSIZE)) == NULL )
        {
            log_error( "out of memory!!!" );
            return;
        }

        ifc.ifc_len = IFRSIZE;

        if (ioctl(sockfd, SIOCGIFCONF, &ifc))
        {
            log_error("ioctl failure: SIOCFIFCONF");
            return;
        }

    } while (IFRSIZE <= ifc.ifc_len);

    ifr = ifc.ifc_req;

    for (;(char *) ifr < (char *) ifc.ifc_req + ifc.ifc_len; ++ifr)
    {
        if (ifr->ifr_addr.sa_data == (ifr+1)->ifr_addr.sa_data)
        {
            continue;  /* duplicate, skip it */
        }

        if (ioctl(sockfd, SIOCGIFFLAGS, ifr))
        {
            continue;  /* failed to get flags, skip it */
        }

        initialize_scan_record( (SCAN_RECORD *)&record,
                                SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

        osi_snprintf( record.name, sizeof( record.name ),
                      "if:%s:IP", ifr->ifr_name );

        osi_snprintf( record.data, sizeof( record.data ),
                      "%s", inet_ntoa(inaddrr(ifr_addr.sa_data)));

        send_scan_data( scanner, (SCAN_RECORD *)&record );

    /*
     * This won't work on HP-UX 10.20 as there's no SIOCGIFHWADDR ioctl. You'll
     * need to use DLPI or the NETSTAT ioctl on /dev/lan0, etc (and you'll need
     *  to be root to use the NETSTAT ioctl. Also this is deprecated and doesn't
     *     work on 11.00).
     *
     * On Digital Unix you can use the SIOCRPHYSADDR ioctl according to an old
     * utility I have. Also on SGI I think you need to use a raw socket, e.g. s
     * = socket(PF_RAW, SOCK_RAW, RAWPROTO_SNOOP)
     *
     * Dave
     *
     * From: David Peter <dave.peter@eu.citrix.com>
     **/

        if ( ioctl(sockfd, SIOCGIFHWADDR, ifr) == 0 )
        {
            /* Select which  hardware types to process.
             **
             **    See list in system include file included from
             **    /usr/include/net/if_arp.h  (For example, on
             **    Linux see file /usr/include/linux/if_arp.h to
             **    get the list.)
             **/

            switch (ifr->ifr_hwaddr.sa_family)
            {
                default:
                    continue;

                case  ARPHRD_NETROM:
                case  ARPHRD_ETHER:
                case  ARPHRD_PPP:
                case  ARPHRD_EETHER:
                case  ARPHRD_IEEE802:
                    break;
            }

            u = (unsigned char *) &ifr->ifr_addr.sa_data;

            /* send record for MAC for this interface */

            if (u[0] + u[1] + u[2] + u[3] + u[4] + u[5])
            {
                initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

                osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

                osi_snprintf( record.name, sizeof( record.name ),
                    "if:%s:MAC", ifr->ifr_name );

                osi_snprintf( record.data, sizeof( record.data ),
                        "%2.2x.%2.2x.%2.2x.%2.2x.%2.2x.%2.2x",
                        u[0], u[1], u[2], u[3], u[4], u[5]);

                send_scan_data( scanner, (SCAN_RECORD *)&record );
            }
        }

        if ( ioctl(sockfd, SIOCGIFNETMASK, ifr) == 0 &&
             strcmp("255.255.255.255", inet_ntoa(inaddrr(ifr_addr.sa_data))))
        {
            initialize_scan_record( (SCAN_RECORD *)&record,
                                SCAN_RECORD_TYPE_TEXT_1 );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            osi_snprintf( record.name, sizeof( record.name ),
                "if:%s:NETMASK", ifr->ifr_name );

            osi_snprintf( record.data, sizeof( record.data ),
                "%s", inet_ntoa(inaddrr(ifr_addr.sa_data)));

            send_scan_data( scanner, (SCAN_RECORD *)&record );
        }

        if (ifr->ifr_flags & IFF_BROADCAST)
        {
            if ( ioctl(sockfd, SIOCGIFBRDADDR, ifr) == 0 &&
                 strcmp("0.0.0.0", inet_ntoa(inaddrr(ifr_addr.sa_data))))
            {

                initialize_scan_record( (SCAN_RECORD *)&record,
                                        SCAN_RECORD_TYPE_TEXT_1 );

                osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

                osi_snprintf( record.name, sizeof( record.name ),
                    "if:%s:BROADCAST", ifr->ifr_name );

                osi_snprintf( record.data, sizeof( record.data ),
                    "%s",inet_ntoa(inaddrr(ifr_addr.sa_data)));

                send_scan_data( scanner, (SCAN_RECORD *)&record );
            }
        }

        /* Added by David Vasil to check for Promiscuous mode */

        initialize_scan_record( (SCAN_RECORD *)&record,
                                SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );


        osi_snprintf( record.name, sizeof( record.name ),
                      "if:%s:PROMISC", ifr->ifr_name );

        if ( ioctl(sockfd, SIOCGIFFLAGS, ifr) == 0 &&
             ifr->ifr_flags & IFF_PROMISC)
        {
            osi_strlcpy( record.data, "ENABLED", sizeof( record.data ) );
        }

        else
        {
            osi_strlcpy( record.data, "DISABLED", sizeof( record.data ) );
        }

        send_scan_data( scanner, (SCAN_RECORD *)&record );


        if ( ioctl(sockfd, SIOCGIFMTU, ifr) == 0 )
        {
            initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            osi_snprintf( record.name, sizeof( record.name ),
                "if:%s:MTU", ifr->ifr_name );

            osi_snprintf( record.data, sizeof( record.data ),
                "%u", ifr->ifr_mtu );

            send_scan_data( scanner, (SCAN_RECORD *)&record );
        }

        if ( ioctl(sockfd, SIOCGIFMETRIC, ifr) == 0 )
        {
            initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            osi_snprintf( record.name, sizeof( record.name ),
                "if:%s:METRIC", ifr->ifr_name );

            osi_snprintf( record.data, sizeof( record.data ),
                "%u", ifr->ifr_metric );

            send_scan_data( scanner, (SCAN_RECORD *)&record );
        }
    }

    close(sockfd);
}

#endif

void mod_if( SCANNER *scanner )
{
#if defined(SYSTEM_LINUX)
    process_if_unix( scanner );
#endif

}
