/***************************************************************************
  grain.cpp
  -------------------
  A grain class
  -------------------
  begin         October 9th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include "grain.h"
#include "resource.h"

#include <cmath>
#include <qstring.h>
#include <qstringlist.h>

//////////////////////////////////////////////////////////////////////////////
// Construction, destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Grain()
// -------
// Constructor

Grain::Grain(const QString &name, const double &quantity, const double &extract,
    const double &color, const int &use)
    : name_(name), quantity_(quantity), extract_(extract), color_(color), use_(use),
    yield_(quantity * (extract - 1.000)), hcu_(quantity * color)
{ ; }

Grain::Grain(const QString &name, const double &quantity, const double &extract,
    const double &color, const QString &use)
    : name_(name), quantity_(quantity), extract_(extract), color_(color),
    use_(useStringToEnum(use)),
    yield_(quantity * (extract - 1.000)), hcu_(quantity * color)
{ ; }

//////////////////////////////////////////////////////////////////////////////
// Grain(const Grain &)
// --------------------
// Copy Constructor

Grain::Grain(const Grain &g)
    : name_(g.name_), quantity_(g.quantity_), extract_(g.extract_),
    color_(g.color_), use_(g.use_), yield_(g.yield_), hcu_(g.hcu_)
{ ; }

//////////////////////////////////////////////////////////////////////////////
// ~Grain()
// --------
// Copy Constructor

Grain::~Grain() {}

//////////////////////////////////////////////////////////////////////////////
// Operators                                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// operator=()
// -----------
// Assignment operator

Grain Grain::operator=(const Grain &g)
{
    if (this != &g) {
        name_ = g.name_;
        quantity_ = g.quantity_;
        extract_ = g.extract_;
        color_ = g.color_;
        use_ = g.use_;
        yield_ = g.yield_;
        hcu_ = g.hcu_;
    }
    return *this;
}

//////////////////////////////////////////////////////////////////////////////
// operator==()
// ------------
// Equivalence operator

bool Grain::operator==(const Grain &g)
{
    return (name_ == g.name_)
        && (quantity_ == g.quantity_)
        && (extract_ == g.extract_)
        && (color_ == g.color_)
        && (use_ == g.use_);
}

//////////////////////////////////////////////////////////////////////////////
// Serialization (deprecated)                                               //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// obsoleteSerializeOut()
// ----------------------
// Return class as a serializable string. This class is deprecated...

QString Grain::obsoleteSerializeOut()
{
    // converting on the fly from the deprecated file format
    QString text = QString::number(quantity_ * 100.0) + ", " + QString::number(extract_ * 1000.0);
    text += ", " + QString::number(color_ * 10.0) + ", " + useString();
    return text;
}

//////////////////////////////////////////////////////////////////////////////
// obsoleteSerializeIn()
// ---------------------
// Initialize class from a serialized name/value string pair. Deprecated...

void Grain::obsoleteSerializeIn(const QString &name, const QString &value)
{
    QString valuestring = value;
    QStringList tokens;
    int pos;

    // get tokens
    while (valuestring.length() > 0) {
        pos = valuestring.find(',');
        if (pos > 0) {
            tokens += valuestring.left(pos);
            valuestring.remove(0, pos+1);
            valuestring = valuestring.stripWhiteSpace();
        } else {
            // get last token
            tokens += valuestring;
            valuestring = "";
        }
    }
    // create new style
    name_ = name;
    // converting on the fly from the deprecated file format
    quantity_    = tokens[0].toDouble() / 100.0;
    extract_    = tokens[1].toDouble() / 1000.0;
    color_    = tokens[2].toDouble() / 10.0;
    /** do a bit more work on use */
    use_ = useStringToEnum(tokens[3]);
    yield_ = quantity_ * (extract_ - 1.000);
    hcu_ = quantity_ * color_;
}

///////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                             //
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// colorString()
// -------------
// Return color as a string

QString Grain::colorString()
{
    return QString::number(color_) + AppResource::CHAR_LATIN_DEGREE;
}

///////////////////////////////////////////////////////////////////////////////
// useString()
// -----------
// Return usage as a string

QString Grain::useString()
{
    switch (use_) {
        case GRAIN_EXTRACT: return QObject::tr("extract");
        case GRAIN_MASHED: return QObject::tr("mashed");
        case GRAIN_STEEPED:return QObject::tr("steeped");
    }
    return QObject::tr("other");
}

///////////////////////////////////////////////////////////////////////////////
// useStringList()
// -----------
// Return string list of usage types

QStringList Grain::useStringList()
{
    QStringList stemp = QObject::tr("extract");
    stemp += QObject::tr("mashed");
    stemp += QObject::tr("steeped");
    stemp += QObject::tr("other");
    return stemp;
}

///////////////////////////////////////////////////////////////////////////////
// useStringToEnum()
// -----------------
// Convert use string to enumeration type

int Grain::useStringToEnum(const QString &name)
{
    if ((name.lower() == "mashed") || (name.lower() == "mash")) return GRAIN_MASHED;
    if ((name.lower() == "steeped") || (name.lower() == "steep")) return GRAIN_STEEPED;
    if (name.lower() == "extract") return GRAIN_EXTRACT;
    return GRAIN_OTHER;
}

//////////////////////////////////////////////////////////////////////////////
// GrainListItem class                                                      //
//////////////////////////////////////////////////////////////////////////////

GrainListItem::GrainListItem(QListView * parent, QString a, QString b,
    QString c, QString d, QString e, QString f, QString g, QString h )
    : QListViewItem(parent, a, b, c, d, e, f, g, h), git_(0) { ; }

GrainListItem::~GrainListItem() {}

void GrainListItem::setToIterator()
{
    setText(0, (*git_).name());
    setText(1, (*git_).quantityString());
    setText(2, (*git_).extractString());
    setText(3, (*git_).colorString());
    setText(4, (*git_).useString());
}
