/*
 * Copyright 1995,96,97 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: samuel.y,v 2.10 1997/04/18 15:57:26 bousch Exp $
 *
 * The yacc file for Samuel's parsing.
 */

%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "saml.h"
#include "samuel.h"

#define yyerror(s)	fprintf(stderr, "samuel: %s\n", s)

static void repeat_subs (mref_t expr, mref_t e1, mref_t e2)
{
	int error;
	mref_t save = mref_new();
	
	do {
		mref_copy(save, expr);
		error = mref_subs(expr, expr, e1, e2);
	}
	while (!error && mref_differ(save,expr));
	mref_free(save);
}

static void do_binary_op (char c)
{
	mref_t e2 = pop_mref();
	mref_t e1 = pop_mref();

	switch(c) {
	    case '+':	mref_add(e1, e1, e2); break;
	    case '-':	mref_sub(e1, e1, e2); break;
	    case '*':	mref_mul(e1, e1, e2); break;
	    case '/':	mref_div(e1, e1, e2); break;
	    case '&':	mref_gcd(e1, e1, e2); break;
	    case ':':	mref_diff(e1, e1, e2); break;
	    default:	panic("impossible binary operator");
	}
	push_mref(e1); mref_free(e2);
}

%}

%union {
	int integer;
	char* string;
}

%token TOK_ARROW TOK_SHIFT TOK_DBL_ARROW TOK_SC_NL
%token TOK_MODULO TOK_PRINT_VARS
%token <string> IDENT INTEGER QSTRING
%type <integer> expr_list optional_expr_list

%right '='
%left TOK_MODULO
%left '&'
%left '+' '-'
%left '*' '.' '/'
%left UNARY_OP
%left '^' ':'

%%

lines:	/* empty */
	| lines TOK_PRINT_VARS	{ list_variables(); }
	| lines line semicolon;

semicolon: ';'		{ /* no action */    }
	| TOK_SC_NL	{ print_cpu_usage(); }
	;

line:	/* empty */
	| printable_expr
	    {
	    	mref_t top = pop_mref();
	    	FILE *fp;

		if (post_processor && (fp = popen(post_processor,"w"))) {
		    	fputs(mref_string(top), fp);
		    	fputc('\n', fp);
			pclose(fp);
		}
		else {
			puts(mref_string(top));
			fflush(stdout);
		}
	    	assign_to_variable("last",top);
	    	mref_free(top);
	    }
	| assignment_expr
	    {
	    	mref_t top = pop_mref();
	    	assign_to_variable("last",top);
	    	mref_free(top);
	    }
	| error		{ clean_stack(); }
	;

nested_expr: expr
	| nested_expr ',' expr TOK_ARROW expr
	    {
	    	mref_t e3 = pop_mref();
	    	mref_t e2 = pop_mref();
	    	mref_t e1 = pop_mref();
	    	
	    	mref_subs(e1, e1, e2, e3);
	    	push_mref(e1); mref_free(e2); mref_free(e3);
	    }
	| nested_expr ',' expr TOK_SHIFT expr
	    {
	    	mref_t e3 = pop_mref();
	    	mref_t e2 = pop_mref();
	    	mref_t e1 = pop_mref();
	    	
	    	repeat_subs(e1, e2, e3);
	    	push_mref(e1); mref_free(e2); mref_free(e3);
	    }
	| nested_expr ',' IDENT TOK_DBL_ARROW IDENT
	    {
	    	mref_t l1 = mref_new();
	    	mref_t l2 = mref_new();
	    	mref_t ne = pop_mref();

	    	mref_build(l1, ST_LITERAL, $3); free($3);
	    	mref_build(l2, ST_LITERAL, $5); free($5);
	    	mref_move_lit(ne ,ne, l1, l2);
	    	push_mref(ne); mref_free(l1); mref_free(l2);
	    }
	;

expr_list: expr			{ $$=1; }
	| expr_list ',' expr	{ ++$$; }
	;

integer_expr: INTEGER
	    {
	    	/* Returns the integer as a polynomial */
		mref_t number = mref_new();
		
		mref_build(number, ST_INTEGER, $1);
		mref_promote(number, mr_model);
		free($1); push_mref(number);
	    }
	;

ident_expr: IDENT
	    {
	    	mref_t mr = mref_new();
	    	read_identifier(mr,$1);
	    	free($1); push_mref(mr);
	    }
	| '\\' IDENT
	    {
	    	mref_t mr = mref_new();
	    	mref_build(mr, ST_LITERAL, $2);
	    	mref_promote(mr, mr_model);
	    	free($2); push_mref(mr);
	    }
	| '?' IDENT
	    {
	    	mref_t mr = mref_new();
	    	read_variable(mr,$2);
	    	free($2); push_mref(mr);
	    }
	| '^' IDENT
	    {
	    	mref_t mr = mref_new();
	    	read_variable(mr,$2);
	    	destroy_variable($2);
	    	free($2); push_mref(mr);
	    }
	;

file_expr: QSTRING
	    {
	    	mref_t mr = mref_new();
	    	read_mref_from_file(mr,$1);
	    	free($1); push_mref(mr);
	    }
	;

det_expr: '|' expr_list '|'
	    {
	    	mref_t top = make_square_matrix($2);
	    	mref_det(top, top);
	    	push_mref(top);
	    }
	;

simple_tensor: IDENT '!' INTEGER
	    {
	    	mref_t mr = mref_new();
	    	char *buff = realloc($1, strlen($1)+strlen($3)+2);

	    	if (!buff) panic("out of memory");
	    	strcat(buff, "!");
	    	strcat(buff, $3); free($3);
	    	mref_build(mr, ST_TENSOR, buff); free(buff);
	    	mref_promote(mr, mr_model);
	    	push_mref(mr);
	    }
	;

optional_expr_list: /* empty */		{ $$ = 0; }
	| expr_list			{ $$ = $1; }
	;

primary: integer_expr
	| file_expr
	| ident_expr
	| det_expr
	| '(' nested_expr ')'		{ }
	;

expr: printable_expr | assignment_expr ;

printable_expr: primary
	| simple_tensor
	| '~' primary '(' expr ',' expr ')'
	    {
	    	mref_t e3 = pop_mref();
	    	mref_t e2 = pop_mref();
	    	mref_t e1 = pop_mref();

	    	mref_elim(e1, e1, e2, e3);
	    	push_mref(e1); mref_free(e2); mref_free(e3);
	    }
	| IDENT '(' optional_expr_list ')'	{ execute_function($1,$3); }
	| expr '+' expr		{ do_binary_op('+'); }
	| expr '-' expr		{ do_binary_op('-'); }
	| expr '*' expr		{ do_binary_op('*'); }
	| expr '.' expr		{ do_binary_op('*'); }
	| expr '/' expr		{ do_binary_op('/'); }
	| expr '&' expr		{ do_binary_op('&'); }
	| expr ':' primary	{ do_binary_op(':'); }
	| expr '^' INTEGER
	    {
	    	mref_t top = pop_mref();
	    	mref_power(top, top, atoi($3));
	    	free($3); push_mref(top);
	    }
	| expr TOK_MODULO '(' expr ',' expr ')'
	    { execute_function("modulo", 3); }
	| '-' expr				%prec UNARY_OP
	    {
	    	mref_t top = pop_mref();
	    	mref_negate(top, top);
	    	push_mref(top);
	    }
	| '+' expr				%prec UNARY_OP
	    { /* Nothing to do */ }
	;

assignment_expr: IDENT '=' expr
	    {
	    	mref_t top = pop_mref();
	    	assign_to_variable($1, top);
	    	free($1); push_mref(top);
	    }
	| QSTRING '=' expr
	    {
	    	mref_t top = pop_mref();
	    	write_mref_to_file($1, top);
	    	free($1); push_mref(top);
	    }
	;
