
/*
#    Sfront, a SAOL to C translator    
#    This file: Network library -- samidi receiver journal functions
#    Copyright (C) 1999  Regents of the University of California
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License (Version 2) as
#    published by the Free Software Foundation.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Maintainer: John Lazzaro, lazzaro@cs.berkeley.edu
*/


#ifndef NSYS_NET
#include "net_include.h"
#endif


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*     high-level functions: receiving recovery journals        */
/*______________________________________________________________*/


/****************************************************************/
/*          main routine for parsing recovery journal           */
/****************************************************************/

int nsys_netin_journal_recovery(source * sptr, int rtpcode, 
				unsigned char * packet,
				int numbytes, unsigned char * buff,  
				long * fill, long size)

{
  netout_jrecv_state * jrecv;
  int numchan, bflag, i, j;
  short chanlen, loglen;
  unsigned char chapters, chan, low, high, many;
  unsigned char * checkptr, * p;

  if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
    {
      printf("\ndoing recovery for mset%i (fill: %i)\n", sptr->mset, 
	     (j = (*fill))); 

#if (NSYS_LATENOTES_DEBUG == NSYS_LATENOTES_DEBUG_ON)
      if (sptr->tm_margin)
	printf("Time: %f\n", (nsys_netout_tstamp - sptr->tm_first)/ARATE);
#endif

      fflush(stdout);
    }

  if ((numbytes -= (1 + ((NSYS_SM_MLENMAX & (*packet))))) < NSYS_SM_JH_SIZE) 
    return NSYS_JOURNAL_CORRUPTED;

  packet += 1 + (NSYS_SM_MLENMAX & (*packet));
  many = (rtpcode == NSYS_RTPCODE_LOSTMANY);

  if ((packet[NSYS_SM_JH_LOC_FLAGS] & NSYS_SM_JH_CHKA) || 
      ((packet[NSYS_SM_JH_LOC_FLAGS] & NSYS_SM_CHKS) && (!many)))
    return NSYS_JOURNAL_RECOVERED;

  numchan =  (packet[NSYS_SM_JH_LOC_FLAGS] & NSYS_SM_JH_CHANMASK) + 1;
  checkptr = &(packet[NSYS_SM_JH_LOC_CHECK]);

  numbytes -= NSYS_SM_JH_SIZE;
  packet += NSYS_SM_JH_SIZE;

  while (numchan--)
    {
      if (numbytes < NSYS_SM_CH_SIZE)
	return NSYS_JOURNAL_CORRUPTED;

      chan = ((packet[NSYS_SM_CH_LOC_FLAGS] & NSYS_SM_CH_CHANMASK)
	      >> NSYS_SM_CH_CHANSHIFT);

      if ((jrecv = sptr->jrecv[chan]) == NULL)
	jrecv = sptr->jrecv[chan] = nsys_netin_newrecv(chan);

      chapters = packet[NSYS_SM_CH_LOC_TOC];
      chanlen = *((short *)&(packet[NSYS_SM_CH_LOC_FLAGS]));
      chanlen = ntohs((unsigned short)chanlen) & NSYS_SM_CH_LENMASK;
      
      if ((numbytes -= chanlen) < 0)
	return NSYS_JOURNAL_CORRUPTED;

      p = packet + NSYS_SM_CH_SIZE;
      packet += chanlen;

      /*****************************/
      /* chapter P: Program Change */
      /*****************************/

      if (chapters & NSYS_SM_CH_TOC_SETP)
	{
	  if ((chanlen -= NSYS_SM_CP_SIZE) < 0)
	    return NSYS_JOURNAL_CORRUPTED;

	  if (((!(p[NSYS_SM_CP_LOC_PROGRAM] & NSYS_SM_CHKS)) || many) && 
	      nsys_netin_jrec_program(sptr, p, jrecv, buff, fill, size))
	    return NSYS_JOURNAL_FILLEDBUFF;

	  p += NSYS_SM_CP_SIZE;
	}

      /**************************/
      /* chapter W: Pitch Wheel */
      /**************************/

      if (chapters & NSYS_SM_CH_TOC_SETW)
	{
	  if ((chanlen -= NSYS_SM_CW_SIZE) < 0)
	    return NSYS_JOURNAL_CORRUPTED;

	  if (((!(p[NSYS_SM_CW_LOC_FIRST] & NSYS_SM_CHKS)) || many) &&
	      nsys_netin_jrec_wheel(sptr, p, jrecv, buff, fill, size))
	    return NSYS_JOURNAL_FILLEDBUFF;

	  p += NSYS_SM_CW_SIZE;
	}

      /**************************/
      /* chapter N: Note On/Off */
      /**************************/

      if (chapters & NSYS_SM_CH_TOC_SETN)
	{
	  if ((chanlen -= NSYS_SM_CN_HDRSIZE) < 0)
	    return NSYS_JOURNAL_CORRUPTED;

	  loglen = p[NSYS_SM_CN_LOC_LENGTH] & NSYS_SM_CN_CLRB;
	  bflag = !(p[NSYS_SM_CN_LOC_LENGTH] & NSYS_SM_CN_CHKB);
	  low = ((p[NSYS_SM_CN_LOC_LOWHIGH] & NSYS_SM_CN_LOWMASK) >>
		 NSYS_SM_CN_LOWSHIFT);
	  high = (p[NSYS_SM_CN_LOC_LOWHIGH] & NSYS_SM_CN_HIGHMASK);

	  if ((chanlen -= (loglen*NSYS_SM_CN_LOGSIZE + 
			   ((low <= high) ? (high - low + 1) : 0))) < 0)
	    return NSYS_JOURNAL_CORRUPTED;

	  p += NSYS_SM_CN_HDRSIZE;

	  if (loglen)
	    {
	      if (nsys_netin_jrec_notelog(sptr, p, jrecv, many, loglen,
					  checkptr, buff, fill, size))
		return NSYS_JOURNAL_FILLEDBUFF;
	      p += loglen*NSYS_SM_CN_LOGSIZE;
	    }
	  if ((bflag || many) && (low <= high))
	    {	      
	      if (nsys_netin_jrec_bitfield(sptr, p, jrecv, low, high,
					   buff, fill, size))
		return NSYS_JOURNAL_FILLEDBUFF;
	    }

	  p += (low <= high) ? (high - low + 1) : 0;
	}

      /**************************/
      /* chapter A: Poly Touch */
      /**************************/

      if (chapters & NSYS_SM_CH_TOC_SETA)
	{
	  if ((chanlen -= NSYS_SM_CA_HDRSIZE) < 0)
	    return NSYS_JOURNAL_CORRUPTED;
	  
	  loglen = (p[NSYS_SM_CA_LOC_LENGTH] & NSYS_SM_CLRS) + 1;

	  if ((chanlen -= loglen*NSYS_SM_CA_LOGSIZE) < 0)
	    return NSYS_JOURNAL_CORRUPTED;

	  if (((!(p[NSYS_SM_CA_LOC_LENGTH] & NSYS_SM_CHKS)) || many) &&
	      nsys_netin_jrec_ptouch(sptr, p, jrecv, loglen, many,
				     buff, fill, size))
	    return NSYS_JOURNAL_FILLEDBUFF;

	  p += (NSYS_SM_CA_HDRSIZE + loglen*NSYS_SM_CA_LOGSIZE);
	}

      /****************************/
      /* chapter T: Channel Touch */
      /****************************/

      if (chapters & NSYS_SM_CH_TOC_SETT)
	{
	  if ((chanlen -= NSYS_SM_CT_SIZE) < 0)
	    return NSYS_JOURNAL_CORRUPTED;

	  if (((!(p[NSYS_SM_CT_LOC_PRESSURE] & NSYS_SM_CHKS)) || many) &&
	      nsys_netin_jrec_ctouch(sptr, p, jrecv, buff, fill, size))
	    return NSYS_JOURNAL_FILLEDBUFF;

	  p += NSYS_SM_CT_SIZE;
	}

      /**************************/
      /* chapter C: Controllers */
      /**************************/

      if (chapters & NSYS_SM_CH_TOC_SETC)
	{
	  if ((chanlen -= NSYS_SM_CC_HDRSIZE) < 0)
	    return NSYS_JOURNAL_CORRUPTED;

	  loglen = (p[NSYS_SM_CC_LOC_LENGTH] & NSYS_SM_CLRS) + 1;

	  if ((chanlen -= loglen*NSYS_SM_CC_LOGSIZE) < 0)
	    return NSYS_JOURNAL_CORRUPTED;
	  
	  if (((!(p[NSYS_SM_CC_LOC_LENGTH] & NSYS_SM_CHKS)) || many) &&
	      nsys_netin_jrec_control(sptr, p, jrecv, loglen, many,
				      buff, fill, size))
	    return NSYS_JOURNAL_FILLEDBUFF;

	  p += (NSYS_SM_CC_HDRSIZE + loglen*NSYS_SM_CC_LOGSIZE);
	}
    }

  if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
    {
      printf("recovery (fill: %li):", (*fill));
      for (i = 0; j + i < (*fill); i++)
	printf(" %hhu", buff[j - (*fill) + i]);
      printf("\n");
      fflush(stdout);
    }

  return NSYS_JOURNAL_RECOVERED;
}

/****************************************************************/
/*            receiver state: add a new MIDI event              */
/****************************************************************/

void nsys_netin_journal_trackstate(source * sptr, unsigned char cmd,
				   unsigned char ndata, unsigned char vdata)

{
  netout_jrecv_state * jrecv;

  if ((jrecv = sptr->jrecv[cmd & 0x0F]) == NULL)
    jrecv = sptr->jrecv[cmd & 0x0F] = nsys_netin_newrecv(cmd & 0x0F);

  switch (cmd & 0xF0) {

  case CSYS_MIDI_PROGRAM:
    jrecv->chapterp_program = vdata | NSYS_SM_RV_SETF;
    jrecv->chapterp_bankc = NSYS_SM_RV_CLRF & 
      jrecv->chapterc_value[CSYS_MIDI_CC_BANKSELECT_MSB];
    jrecv->chapterp_bankf = NSYS_SM_RV_CLRF & 
      jrecv->chapterc_value[CSYS_MIDI_CC_BANKSELECT_LSB];
    break;

  case CSYS_MIDI_WHEEL:
    jrecv->chapterw_first = ndata | NSYS_SM_RV_SETF;
    jrecv->chapterw_second = vdata;
    break;

  case CSYS_MIDI_NOTEOFF:
    jrecv->chaptern_vel[ndata] = NSYS_SM_RV_SETF;
    break;

  case CSYS_MIDI_NOTEON:
    if (vdata)
      {
	jrecv->chaptern_vel[ndata] = vdata | NSYS_SM_RV_SETF;
	jrecv->chaptern_tstamp[ndata] = nsys_netout_tstamp;
	jrecv->chaptern_extseq[ndata] = sptr->hi_ext;
      }
    else
      jrecv->chaptern_vel[ndata] = NSYS_SM_RV_SETF;
    break;

  case CSYS_MIDI_PTOUCH:
    jrecv->chaptera_pressure[ndata] = vdata | NSYS_SM_RV_SETF;
    break;

  case CSYS_MIDI_CTOUCH:
    jrecv->chaptert_pressure = ndata | NSYS_SM_RV_SETF;
    break;

  case CSYS_MIDI_CC:
    switch (ndata) {
    case CSYS_MIDI_CC_ALLSOUNDOFF:
    case CSYS_MIDI_CC_ALLNOTESOFF:
      jrecv->chapterc_value[ndata] = ((jrecv->chapterc_value[ndata] + 1) |   
				      NSYS_SM_RV_SETF);
      if (jrecv->chapterc_value[ndata] == NSYS_SM_RV_SETF)
	jrecv->chapterc_value[ndata]++;
      break;
    case CSYS_MIDI_CC_SUSTAIN:
      if (vdata)
	{
	  if ((jrecv->chapterc_value[ndata] & NSYS_SM_CC_CLRF) == 0)
	    {
	      if (!(jrecv->chapterc_sustain = 
		    (jrecv->chapterc_sustain + 1) & NSYS_SM_CC_MODULO))
		jrecv->chapterc_sustain++;
	    }
	  vdata = jrecv->chapterc_sustain;
	}
      /* falls through */
    default:
      jrecv->chapterc_value[ndata] = vdata | NSYS_SM_RV_SETF;
      break;
    }
    break;

  }

}


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                 second-level receiver functions              */
/*______________________________________________________________*/

/****************************************************************/
/*               process chapter P (program change)             */
/****************************************************************/


int nsys_netin_jrec_program(source * sptr, unsigned char * p,
			  netout_jrecv_state * jrecv,
			  unsigned char * buff,  
			  long * fill, long size)

{
  unsigned char newx, newy, vel, yflag;

  vel = (p[NSYS_SM_CP_LOC_PROGRAM] & NSYS_SM_CLRS);
  newx = (p[NSYS_SM_CP_LOC_BANKCOARSE] & NSYS_SM_CP_CLRC);
  newy = (p[NSYS_SM_CP_LOC_BANKFINE] & NSYS_SM_CP_CLRF);
  
  if ((jrecv->chapterp_program == 0) || 
      (newx != jrecv->chapterp_bankc) || 
      (newy != jrecv->chapterp_bankf) ||
      (vel != (jrecv->chapterp_program & NSYS_SM_RV_CLRF)))
    {
      yflag = ((newx != (NSYS_SM_CLRS & jrecv->chapterc_value
			 [CSYS_MIDI_CC_BANKSELECT_MSB])) ||
	       (newy != (NSYS_SM_CLRS & jrecv->chapterc_value
			 [CSYS_MIDI_CC_BANKSELECT_LSB])) ||
	       (jrecv->chapterc_value
		[CSYS_MIDI_CC_BANKSELECT_MSB] == 0) ||
	       (jrecv->chapterc_value
		[CSYS_MIDI_CC_BANKSELECT_LSB] == 0));
      
      /* if needed, change bank-switch values */
      
      if (yflag)
	{			  
	  if (nsys_netin_journal_addcmd(sptr, buff, fill, size,
					jrecv->chan | CSYS_MIDI_CC,
					CSYS_MIDI_CC_BANKSELECT_MSB
					, newx)) 
	    return NSYS_JOURNAL_FILLEDBUFF;
	  if (nsys_netin_journal_addcmd(sptr, buff, fill, size,
					jrecv->chan | CSYS_MIDI_CC,
					CSYS_MIDI_CC_BANKSELECT_LSB
					, newy))
	    return NSYS_JOURNAL_FILLEDBUFF;
	}
      
      /* do program change */
      
      if (nsys_netin_journal_addcmd(sptr, buff, fill, size,
				    jrecv->chan | CSYS_MIDI_PROGRAM,
				    vel, 0))
	return NSYS_JOURNAL_FILLEDBUFF;
      
      /* if changed, reset bank-switch values */
      
      if (yflag)
	{
	  if (nsys_netin_journal_addcmd(sptr, buff, fill, size,
					jrecv->chan | CSYS_MIDI_CC,
					CSYS_MIDI_CC_BANKSELECT_MSB
					, jrecv->chapterc_value
					[CSYS_MIDI_CC_BANKSELECT_MSB])) 
	    return NSYS_JOURNAL_FILLEDBUFF;
	  if (nsys_netin_journal_addcmd(sptr, buff, fill, size,
					jrecv->chan | CSYS_MIDI_CC,
					CSYS_MIDI_CC_BANKSELECT_LSB
					, jrecv->chapterc_value
					[CSYS_MIDI_CC_BANKSELECT_LSB]))
	    return NSYS_JOURNAL_FILLEDBUFF;
	}
      
      if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
	printf("PChange %hhu [%hhu,%hhu] --> %hhu [%hhu, %hhu]\n",
	       jrecv->chapterp_program & NSYS_SM_RV_CLRF,
	       jrecv->chapterp_bankc, jrecv->chapterp_bankf,
	       vel, newx, newy);
      
      jrecv->chapterp_program = vel | NSYS_SM_RV_SETF;
      jrecv->chapterp_bankc = newx ;
      jrecv->chapterp_bankf = newy ;
    }
  
  return NSYS_JOURNAL_RECOVERED;
}

/****************************************************************/
/*               process chapter W (pitch wheel)                */
/****************************************************************/


int nsys_netin_jrec_wheel(source * sptr, unsigned char * p,
			  netout_jrecv_state * jrecv,
			  unsigned char * buff,  
			  long * fill, long size)

{
  unsigned char newx, newy;

  newx = p[NSYS_SM_CW_LOC_FIRST] & NSYS_SM_CLRS;
  newy = p[NSYS_SM_CW_LOC_SECOND] & NSYS_SM_CLRS;
  
  if ((newx != (jrecv->chapterw_first & NSYS_SM_RV_CLRF)) || 
      (newy != jrecv->chapterw_second) || 
      (jrecv->chapterw_first == 0))
    {
      if (nsys_netin_journal_addcmd(sptr, buff, fill, size,
				    jrecv->chan | CSYS_MIDI_WHEEL,
				    newx, newy))
	return NSYS_JOURNAL_FILLEDBUFF;
      
      if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
	printf("PWheel (%hhu, %hhu) --> (%hhu, %hhu)\n", 
	       jrecv->chapterw_first & NSYS_SM_CLRS,
	       jrecv->chapterw_second, newx, newy);
      
      jrecv->chapterw_first = newx | NSYS_SM_RV_SETF;
      jrecv->chapterw_second = newy;
    }

  return NSYS_JOURNAL_RECOVERED;
}


/****************************************************************/
/*               process Note Logs of chapter N                 */
/****************************************************************/

int nsys_netin_jrec_notelog(source * sptr, unsigned char * p,
			    netout_jrecv_state * jrecv, unsigned char many, 
			    short loglen, unsigned char * checkptr,
			    unsigned char * buff, 
			    long * fill, long size)

{
  unsigned char newx, newy, vel;
  int yflag, noteon, noteoff;
  unsigned long check_ext = 0;

  while (loglen)
    {
      if ((!(p[NSYS_SM_CN_LOC_NUM] & NSYS_SM_CHKS)) || many)
	{		      
	  newx = p[NSYS_SM_CN_LOC_NUM] & NSYS_SM_CLRS;
	  newy = p[NSYS_SM_CN_LOC_VEL] & NSYS_SM_CN_CLRY;
	  yflag = (p[NSYS_SM_CN_LOC_VEL] & NSYS_SM_CN_CHKY);
	  vel = jrecv->chaptern_vel[newx] & NSYS_SM_RV_CLRF;
	  
	  /* todo: tstamp and exseq checks for vel == newy case,  */
	  /*       NoteOn cull if journal packet arrived too late */
	  
	  /* logic for deciding to do a NoteOff command 
	   *
	   * Note is presently on, and logged note is a NoteOff
	   * or a NoteOn with a different velocity. Later add
	   * other tests.
	   */
	  
	  noteoff = ((vel != 0) && (vel != newy));
	  
	  /* logic for deciding to do a NoteOn command 
	   *
	   * Log command is a NoteOn and has arrived recently, and
	   * has a different value than currently playing note.
	   *
	   */
	  
	  noteon = (newy != 0) && (vel != newy) && yflag && sptr->ontime;
	  
	  /* tests for NoteOn with same value as current note */
	  
	  if ((vel == newy) && newy)
	    {
	      
	      /* prepare extended checkpoint packet number */
	      
	      if (checkptr)
		{
		  check_ext = ntohs(*((unsigned short *)checkptr)); 
		  
		  if (check_ext < sptr->hi_lobits)
		    check_ext |= (sptr->hi_ext & NSYS_RTPSEQ_EXMASK);
		  else
		    check_ext |= ((sptr->hi_ext & NSYS_RTPSEQ_EXMASK)
				  - NSYS_RTPSEQ_EXINCR);
		  checkptr = NULL;
		}
	      
	      /* 
	       * Test one: is NoteOn recent, and the current
	       * playing note dated? If so, NoteOff and NoteOn
	       */
	      
	      if (yflag && (NSYS_SM_CN_RECDELAY < 
			    (nsys_netout_tstamp - 
			     jrecv->chaptern_tstamp[newx])))
		{
		  noteon = 1;
		  noteoff = 1;
		  if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
		    printf("TS-");
		}
	      
	      else
		{
		  
		  /* 
		   * Test two: does playing note predate checkpoint? 
		   * If so, they are different: do NoteOff & perhaps
		   * NoteOn
		   */
		  
		  if (jrecv->chaptern_extseq[newx] < check_ext)
		    {
		      noteoff = 1;
		      noteon = (yflag != 0);
		      if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
			printf("EX-");
		    }
		}
	    }
	  
	  if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
	    printf("NoteLog %hhu: %s(%hhu, %hhu); Actions:%s%s%s\n", 
		   newx, (p[NSYS_SM_CN_LOC_VEL] & NSYS_SM_CN_CHKY) 
		   ? "#" : "-", vel, newy, 
		   (noteoff) ? " NoteOff" : "",
		   (noteon) ? " NoteOn" : "",
		   !(noteon || noteoff) ? " None" : "");
	  
	  /* do NoteOff, then NoteOn, if needed */
	  
	  if (noteoff && (nsys_netin_journal_addcmd
			  (sptr, buff, fill, size,
			   jrecv->chan | CSYS_MIDI_NOTEOFF, newx, 0)))
	    return NSYS_JOURNAL_FILLEDBUFF;
	  
	  if (noteon && (nsys_netin_journal_addcmd
			 (sptr, buff, fill, size,
			  jrecv->chan | CSYS_MIDI_NOTEON, newx, newy)))
	    return NSYS_JOURNAL_FILLEDBUFF;
	  
	  /* update state if needed */
	  
	  if (noteoff || (vel != newy)) 
	    {
	      jrecv->chaptern_vel[newx] = newy | NSYS_SM_RV_SETF;
	      jrecv->chaptern_tstamp[newx] = nsys_netout_tstamp;
	      jrecv->chaptern_extseq[newx] = sptr->hi_ext; 
	    }
	}
      loglen--;
      p += NSYS_SM_CN_LOGSIZE;
    }

  return NSYS_JOURNAL_RECOVERED;
}


/****************************************************************/
/*               process bitfields of chapter N                */
/****************************************************************/

int nsys_netin_jrec_bitfield(source * sptr, unsigned char * p,
			     netout_jrecv_state * jrecv, unsigned char low, 
			     unsigned char high, unsigned char * buff, 
			     long * fill, long size)
{
  int i;
  unsigned char newx, bitfield;

  while (low <= high)
    {	      
      i = 0;
      bitfield = 128;
      while (bitfield)
	{
	  if ((*p) & bitfield)
	    {
	      newx = (low << NSYS_SM_CN_BFSHIFT) + i;
	      
	      if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
		printf("Bitfield %hhu: (%hhu); ", newx,  
		       jrecv->chaptern_vel[newx]);
	      
	      if (jrecv->chaptern_vel[newx] != NSYS_SM_RV_SETF)
		{
		  if (nsys_netin_journal_addcmd 
		      (sptr, buff, fill, size, 
		       jrecv->chan | CSYS_MIDI_NOTEOFF, newx, 0))
		    return NSYS_JOURNAL_FILLEDBUFF;
		  jrecv->chaptern_vel[newx] = NSYS_SM_RV_SETF;
		  
		  if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
		    printf(" Actions: NoteOff");
		  
		}
	      
	      if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
		printf("\n");
	    }
	  bitfield >>= 1;
	  i++;
	}
      low++;
      p++;
    }
  
  return NSYS_JOURNAL_RECOVERED;
}


/****************************************************************/
/*               process chapter P (poly touch)                 */
/****************************************************************/

int nsys_netin_jrec_ptouch(source * sptr, unsigned char * p,
			   netout_jrecv_state * jrecv, short loglen,
			   unsigned char many, unsigned char * buff, 
			   long * fill, long size)

{
  unsigned char newx, newy;

  p += NSYS_SM_CA_HDRSIZE;
  while (loglen)
    {
      if ((!(p[NSYS_SM_CA_LOC_NUM] & NSYS_SM_CA_CHKF)) || many)
	{
	  newx = p[NSYS_SM_CA_LOC_NUM] & NSYS_SM_CC_CLRF;
	  newy = p[NSYS_SM_CA_LOC_PRESSURE] & NSYS_SM_CC_CLRD;
	  if (((jrecv->chaptera_pressure[newx] & NSYS_SM_RV_CLRF)
	       != newy) || (jrecv->chaptera_pressure[newx] == 0))
	    {
	      if (nsys_netin_journal_addcmd(sptr, buff, fill, size,
					    jrecv->chan | CSYS_MIDI_PTOUCH,
					    newx, newy))
		  return NSYS_JOURNAL_FILLEDBUFF;
	      
	      if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
		printf("PTouchLog %hhu: %hhu --> %hhu\n", newx, 
		       jrecv->chaptera_pressure[newx] &
		       NSYS_SM_RV_CLRF, newy);
	      
	      jrecv->chaptera_pressure[newx] = (newy |
						NSYS_SM_RV_SETF);
	    }
	}
      loglen--;
      p += NSYS_SM_CA_LOGSIZE;
    }

  return NSYS_JOURNAL_RECOVERED;
}


/****************************************************************/
/*               process chapter T (channel touch)              */
/****************************************************************/

int nsys_netin_jrec_ctouch(source * sptr, unsigned char * p,
			   netout_jrecv_state * jrecv,
			   unsigned char * buff,  
			   long * fill, long size)

{
  unsigned char newx;

  newx = p[NSYS_SM_CT_LOC_PRESSURE] & NSYS_SM_CLRS;
  if ((newx != (jrecv->chaptert_pressure & NSYS_SM_RV_CLRF)) ||
      (jrecv->chaptert_pressure == 0))
    {
      if (nsys_netin_journal_addcmd(sptr, buff, fill, size,
				    jrecv->chan | CSYS_MIDI_CTOUCH,
				    newx, 0))
	  return NSYS_JOURNAL_FILLEDBUFF;
      
      if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
	printf("CTouch (%hhu) --> (%hhu)\n", 
	       jrecv->chaptert_pressure & NSYS_SM_RV_CLRF, newx);
      
      jrecv->chaptert_pressure = newx | NSYS_SM_RV_SETF;
    }

  return NSYS_JOURNAL_RECOVERED;
}


/****************************************************************/
/*               process chapter C (controllers)                */
/****************************************************************/

int nsys_netin_jrec_control(source * sptr, unsigned char * p,
			   netout_jrecv_state * jrecv, short loglen,
			   unsigned char many, unsigned char * buff, 
			   long * fill, long size)

{
  unsigned char newx, newy;

  p += NSYS_SM_CC_HDRSIZE;
  while (loglen)
    {
      if ((!(p[NSYS_SM_CC_LOC_LNUM] & NSYS_SM_CC_CHKF)) || many)
	{
	  newx = p[NSYS_SM_CC_LOC_LNUM] & NSYS_SM_CC_CLRF;
	  newy = p[NSYS_SM_CC_LOC_LVAL] & NSYS_SM_CC_CLRD;
	  if (((jrecv->chapterc_value[newx] & NSYS_SM_RV_CLRF)
	       != newy) || (jrecv->chapterc_value[newx] == 0))
	    {

	      switch (newx) {    
	      case CSYS_MIDI_CC_ALLSOUNDOFF:
	      case CSYS_MIDI_CC_ALLNOTESOFF:
		newy = 0;
		break;
	      case CSYS_MIDI_CC_SUSTAIN:
		if (newy)
		  {
		    jrecv->chapterc_sustain = newy;
		    if (jrecv->chapterc_value[newx] & NSYS_SM_RV_CLRF)
		      newy = 0;  /* later, recover full off->on cycle */
		    else
		      newy = 64; /* normal case, recover the lost pedal on */
		  }
		break;
	      }

	      if (nsys_netin_journal_addcmd(sptr, buff, fill, size,
					    jrecv->chan | CSYS_MIDI_CC,
					    newx, newy))
		  return NSYS_JOURNAL_FILLEDBUFF;
	      
	      if (NSYS_JOURNAL_DEBUG == NSYS_JOURNAL_DEBUG_ON)
		printf("CntrlLog %hhu: %hhu --> %hhu\n", newx, 
		       jrecv->chapterc_value[newx] & NSYS_SM_CLRS,
		       newy);
	      
	      jrecv->chapterc_value[newx] = (p[NSYS_SM_CC_LOC_LVAL] |
					     NSYS_SM_RV_SETF);
	    }
	}
      loglen--;
      p += NSYS_SM_CC_LOGSIZE;
    }

  return NSYS_JOURNAL_RECOVERED;
}


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                  low-level journal functions                 */
/*______________________________________________________________*/

/****************************************************************/
/*           adds a new MIDI command to the buffer              */
/****************************************************************/

int nsys_netin_journal_addcmd(source * sptr, unsigned char * buff, 
			      long * fill, long size,
			      unsigned char cmd, unsigned char ndata,
			      unsigned char vdata)
			      
{
  int threebyte;
  long idx;

  threebyte = (((cmd & 0xF0) != CSYS_MIDI_PROGRAM) && 
	       ((cmd & 0xF0) != CSYS_MIDI_CTOUCH));

  if ((size - (*fill)) < (3 + threebyte))
    return NSYS_JOURNAL_FILLEDBUFF;

  idx = *fill;

  buff[idx++] = cmd;
  buff[idx++] = ndata;

  if (threebyte)
    buff[idx++] = vdata;

  buff[idx++] = (unsigned char)(sptr->mset);

  *fill = idx;

  return NSYS_JOURNAL_RECOVERED;

}

/* end Network library -- receiver journal functions */
