/* Copyright (C) 1993,1994 by the author(s).
 
 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with ShapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 * AtFS -- Attribute Filesystem
 *
 * afsys.h - Internal type and Constant-Definitions for the 
 *	     Attribute-Filesystem
 *
 * Author: Andreas Lampen (Andreas.Lampen@cs.tu-berlin.de)
 *
 * $Header: afsys.h[7.0] Fri Jan 14 13:35:40 1994 andy@cs.tu-berlin.de frozen $
 */

#ifndef _ATFSYS_
#define _ATFSYS_

/*=========================================================================
 * Installation dependent constants 
 *=========================================================================*/

#define AF_SYMHASHSIZE 507       /* size of hashtable for symbols */
#define AF_UDAHASHSIZE  61       /* size of hashtable for user defined attrs */
#define AF_UDANUM        8       /* number of user defined attributes in */
                                 /* linear list */
#define AF_SEGLEN       32       /* size of segment for (re-)allocation */

/*** Derived Object Cache Limits and Defaults ***/
#define AF_MAX_CACHED_PER_NAME		52
#define AF_DEFAULT_CACHED_PER_NAME	3
#define AF_DEFAULT_CACHED_PER_ATTR	1
#define AF_DEFAULT_CACHED_MAX		1024


/*=========================================================================
 * general constants 
 *=========================================================================*/

#ifndef TRUE
#define TRUE		1
#endif
#ifndef FALSE
#define FALSE		0
#endif

#define ERROR		-1
#define LOCAL		static
#define EXPORT
#define bool		short

/*=========================================================================
 * Hash stuff
 *=========================================================================*/

typedef struct Af_hshent Af_hashent;

/*=========================================================================
 * Internal Type Definitions
 *=========================================================================*/

/**** buffer for version-independent attributes ****/
typedef struct {
		char	*af_host;               /* hostname */  
		char	*af_syspath;    	/* system path */
		char	*af_globname;		/* global filename */
		char	*af_globtype;		/* global filename ext.(type)*/
		char	*af_ownname;		/* name of owner */
		char	*af_ownhost;		/* host of owner */
		char	*af_owndomain;		/* domain of owner */
	       } Af_cattrs;

/**** buffer for version-dependent attributes ****/
typedef struct {
		char	*af_name;	        /* filename */
		char	*af_type;	        /* filename extension (type) */
		int	af_gen;			/* generation number */
		int	af_rev;			/* revision number */
		short	af_state;		/* version state (see below) */
		short   af_class;               /* file class */
		char 	*af_auname;		/* name of author */
		char    *af_auhost;		/* host of author */
		char    *af_audomain;		/* domain of author */
		mode_t  af_mode;		/* protection (from inode) */
		char    *af_lckname;            /* name of locker */
		char    *af_lckhost;            /* host of locker */
		char    *af_lckdomain;          /* domain of locker */
		time_t	af_mtime;		/* date of last modification */
		time_t	af_atime;		/* date of last access */
		time_t	af_ctime;		/* date of last status change*/
		time_t	af_stime;		/* save date */
		time_t	af_ltime;		/* date of last lock change */
		size_t	af_notesize;		/* size of note */
		char	*af_note;		/* modification note */
		int     af_udanum;              /* number of uda entries */
		Af_hashent *af_uhtab;           /* hash table for udefattrs */
		char    *af_udalist[AF_UDANUM+1]; /* list of udattrs if # <= 8 */
		short	af_repr;		/* kind of representation */
		size_t	af_fsize;		/* size of file */
		size_t	af_dsize;		/* size of delta */
		char	*af_data;		/* ptr to chunk or delta */
		char    *af_hashname;           /* name of associated file */
		short   af_nrefs;               /* # of references to rev. */
		int	af_succgen,		/* physical 	  */
			af_succrev;		/*	successor */
		int	af_predgen,		/* physical 	    */
			af_predrev;		/*	predecessor */
	       } Af_vattrs;

/**** Descriptor for revision list ****/
typedef struct rvlist Af_revlist;

struct rvlist {
                char	  *af_arpath;        /* pathname of archive dir */
		gid_t	  af_owngid;	     /* gid of archive's owner */
		time_t    af_lastmod;        /* last mod. of archive file */
		char	  *af_busyfilename;  /* filename of busy version */
		short	  af_nrevs;	     /* number of revs in list */
		short	  af_listlen;	     /* total length of list */
		size_t	  af_datasize;	     /* size of data-segment */
		short	  af_extent;	     /* extent of revision list */
		Af_cattrs af_cattrs;	     /* version-independent attrs */
		Af_vattrs *af_list;	     /* pointer to revision list */
		int       af_access;         /* access number */
		char      *af_mem;           /* list of ptrs to alloc. mem. */
		int       af_totalmem;       /* size of memory used */
		Af_revlist *af_next;         /* index of next freelist entry */
	      };

/* Constants for af_extent */

/**** extent and type of revision list ****/

#define AF_SEGMASK      0007   /* Mask for data flags */

#define AF_ATTRS	0001   /* Attributes are loaded */
#define AF_DATA		0002   /* Data are loaded */
#define AF_COMPLETE     0004   /* List is complete (derived object caches only) */

#define AF_CACHE        0010   /* Derived Object Cache */
#define AF_ARCHIVE      0020   /* Line of development */

#define AF_UXWRITE      0100   /* Unix write permission on AtFS subdir */
#define AF_LISTBUSY     0200   /* List is currently in use */

/*=========================================================================
 * More Hash stuff
 *=========================================================================*/

struct Af_hshent { char          *symbol;
		   Af_hashent    *next;
		 };
 
/*=========================================================================
 * Internal Installation dependent constants
 *=========================================================================*/

/**** "syntactical sugar" for user defined attributes ****/
#define AF_UDANAMDEL    '='		/* Delimiter for UDA name in archive */
#define AF_UDAVALDEL	'\01'		/* Delimiter for UDA values in arch. */

/**** UNIX Environment ****/
#define AF_TMPDIR	"/tmp"		/* location of temporary files */
#define AF_SUBDIR	"AtFS"		/* subdirectory for archives */
#define AF_ATTRDIR	"Attr"		/* subdirectory for attr. archives */
#define AF_DATADIR	"Data"		/* subdirectory for data archives */
#define AF_LOCKDIR	"Lock"		/* subdirectory for locks */
#define AF_ATTRBACKUPDIR "Attr.bak"	/* subdir for backups of attr. archives */
#define AF_DATABACKUPDIR "Data.bak"	/* the same for data archives */
#define AF_CACHENAME    "ObjCache"
#define AF_CACHELCKNAME "ObjCache.lck"
#define AF_CACHEFILEID  "OC"

/**** error message logging */
#define AF_ERRLOG	"/tmp/AtFSerrlog"

/*=========================================================================
 * Internal Constant Definitions
 *=========================================================================*/

/**** general ****/

#define AF_READ         0
#define AF_WRITE        1
#define AF_RDWR         2

/**** representation types ****/

#define AF_CHUNK	0
#define AF_DELTA	1
#define AF_FILE         2 /* version resides in an own file (busy version) */

/**** Version numbering ****/

#define AF_INITGEN	1
#define AF_INITREV	0

/**** Modes for archive manipulation ****/

#define AF_CHANGE      01
#define AF_ALLVERS     02

/**** Environment interaction ****/

/**** Permissions for checkperm ****/

#define AF_OWNER        0001
#define AF_AUTHOR       0002
#define AF_LOCKHOLDER   0004
#define AF_WORLD        0010

#define AF_REMOTE       -2

/*=========================================================================
 * Useful macros
 *=========================================================================*/

#define CATTR(keyp)      keyp->af_ldes->af_cattrs
#define VATTR(keyp)      keyp->af_ldes->af_list[keyp->af_lpos]

/* compare filekeys -- returnes 0 if equal (like strcmp) */
#define af_keycmp(key1,key2) (((key1)->af_ldes != (key2)->af_ldes) || ((key1)->af_lpos != (key2)->af_lpos))

/* report error and return */
#define FATAL(msg1,msg2,errcd,retcd) { af_err (msg1, msg2, errcd); return (retcd); }
#define FAIL(msg1,msg2,errcd,retcd) { af_serr (msg1, msg2, errcd); return (retcd); }

/* convert nil pointer to empty string */
#define NOTNIL(str) (str ? str : "")

/* regex pattern matching all strings */
#define AF_PATTERN_ALL	"^.*$"

/*=========================================================================
 * Declarations
 *=========================================================================*/

#ifdef __STDC__
#define A(alist) alist
#else
#define A(alist) ()
#endif

/*** afarchive.c ***/
char*	afFirstItem	A((char *line));
char*	afNextItem	A((char *line));
int	afReadData	A((Af_revlist *list));
int	afReadAttrs	A((Af_revlist *list));
int	afWriteArchive	A((Af_revlist *list));

/*** afarlock.c ***/
FILE	*afOpenLock	A((char *filename, int mode, Af_revlist *list));
int	afCloseUnlock	A((FILE *file, int mode, Af_revlist *list));

/*** afcache.c ***/
Af_revlist	*afInitList	A((char *pathsym, char *namesym, char *typesym));
Af_revlist	*afInitObjCache	A((char *pathsym));
Af_revlist	*afTestList	A((char *pathsym, char *namesym, char *typesym));
int		afRefreshList	A((Af_revlist *list, int extent));
int		afDetachList	A((Af_revlist *list));
int		afListSpace	A((int size));

/*** afcompar.c ***/
int	af_cmpatime	A((Af_key *key1, Af_key *key2));
int	af_cmpauthor	A((Af_key *key1, Af_key *key2));
int	af_cmpctime	A((Af_key *key1, Af_key *key2));
int	af_cmpgen	A((Af_key *key1, Af_key *key2));
int	af_cmphost	A((Af_key *key1, Af_key *key2));
int	af_cmpbound	A((Af_key *key1, Af_key *key2));
int	af_cmplocker	A((Af_key *key1, Af_key *key2));
int	af_cmpltime	A((Af_key *key1, Af_key *key2));
int	af_cmpmode	A((Af_key *key1, Af_key *key2));
int	af_cmpmtime	A((Af_key *key1, Af_key *key2));
int	af_cmpname	A((Af_key *key1, Af_key *key2));
int	af_cmpowner	A((Af_key *key1, Af_key *key2));
int	af_cmprev	A((Af_key *key1, Af_key *key2));
int	af_cmpsize	A((Af_key *key1, Af_key *key2));
int	af_cmpstate	A((Af_key *key1, Af_key *key2));
int	af_cmpsvtime	A((Af_key *key1, Af_key *key2));
int	af_cmpspath	A((Af_key *key1, Af_key *key2));
int	af_cmptype	A((Af_key *key1, Af_key *key2));
int	af_cmpversion	A((Af_key *key1, Af_key *key2));
int	af_cmpuda	A((Af_key *key1, Af_key *key2));

/*** afdelta.c ***/
int	af_nodelta	A((Af_key *busykey, Af_key *savekey));
int	af_dodelta	A((Af_key *busykey, Af_key *predkey, Af_key *savekey));
int	af_undodelta	A((Af_key *deltakey, char *filename));
int	af_rmdelta	A((Af_key *deltakey));

/*** afdeltaproc.c ***/
char	*afConvertDelta	A((char *deltaStr, size_t deltaSize, size_t *newDeltaSize));
int	afReconsData	A((char *srcStr, char *deltaStr, size_t srcSize, size_t deltaSize, char *targetfn));
int	afMakeDelta	A((char *srcStr, char *targetStr, size_t srcSize, size_t targetSize, char *deltafn));

/*** afenviron.c ***/
uid_t	af_getuid	A((char *name, char *host, char *domain));
gid_t	af_getgid	A((char *name, char *host, char *domain));
int 	af_checkread	A((Af_key *key));
int	af_checkperm	A((Af_key *key, int mode));
char 	*af_gethostname	A((void));
char 	*af_getdomain	A((void));

/*** aferror.c ***/
void	af_serr		A((char *routine, char *called, int errcd));
void	af_err		A((char *routine, char *called, int errcd));
void	af_wng		A((char *routine, char *comment));

/*** afkeys.c ***/
int	af_buildkey	A((Af_revlist *list, int gen, int rev, Af_key *key));
Af_key *af_gbuskey	A((Af_revlist *list));
Af_key *af_glastkey	A((Af_revlist *list));
Af_key *af_glastgenkey	A((Af_revlist *list, int gen));
int	af_gfreepos	A((Af_revlist *list));
int	af_glastgenpos	A((Af_revlist *list, int gen));
void	afIncreaseAccess A((Af_key *key));
void	afDecreaseAccess A((Af_key *key));

/*** aflib.c ***/
size_t	af_retfsize	A((char *fileName));
int	af_cpfile	A((char *source, size_t size, char *dest));
int	af_bsearch	A((char **list, int size, char *target));

/*** afmemory.c***/
char	*af_malloc	A((Af_revlist *list, unsigned size));
char    *af_realloc	A((Af_revlist *list, char *ptr, unsigned size));
void	af_free		A((Af_revlist *list, char *ptr));
void	af_frmemlist	A((Af_revlist *list));

/*** afnames.c ***/
char	*af_uniqpath	A((char *path));
Af_user *afArchiveOwner A((char *archDir, bool *writeok, gid_t *gid));
mode_t	afArchiveMode   A((char *archDir));
char	*afArchivePath	A((char *path));
char	*afArchiveName	A((char *archivePath, char *arClass, char *name, char *type, int mod));
char	*afCacheFileName A((char *archPath, char *name));
char	*afCacheUniqueName A((char *unixName, char uniqChar));
char	*af_gtmpname	A((char *path));
char	*af_gbusname	A((char *path, char *name, char *type));

/*** afobjcache.c ***/
int	afObjCacheRead	A((Af_revlist *list));
int	afObjCacheWrite	A((Af_revlist *list));
int	afObjCacheAdd	A((Af_key *aso, Af_key *outKey, char *uniqAttr));
int	afObjCacheRestore A((Af_key *aso, char *fileName));

/*** afstore.c ***/
int	afAccessAso	A((Af_key *key, int mode));
int	afUpdateAso	A((Af_key *key, int mode));
int	afNewAso	A((Af_revlist *list, Af_key *key, int mode, int gen));
int	afAddAso	A((Af_key *key));
int	afDeleteAso	A((Af_key *key));
int	afGetAso	A((char *syspath, char *name, char *type, int gen, int rev, Af_key *key));
int	afTestAso	A((char *path, char *name, char *type, int mode));
int	afBuildFile	A((Af_key *key, char *name));

/*** afsymtab.c ***/
int	afHashval	A((char *symbol, int size));
char	*af_entersym	A((char *symbol));
char	*af_enterhost	A((char *hostname));
char	*af_enterdomain A((char *domain));

/*** aftime.c ***/
char	*af_asctime	A((void));
time_t  af_acttime	A((void));

/*** aftmpfiles.c ***/
void	af_reglckfile	A((char *name));
void	af_unregtmpfile	A((char *name));
void	af_regtmpfile	A((char *name));

/*** aftransaction.c ***/
int	afAddToTransactionList A((Af_revlist *list));

/*** afudattrs.c ***/
void	afInitUdas	A((Af_key *key));
int	afEnterUda	A((Af_key *key, char *symbol));
int	afReplUda	A((Af_key *key, char *symbol));
int	afDelUda	A((Af_key *key, char *symbol));
char	*afLookupUda	A((Af_key *key, char *symbol));
int	afMatchUda	A((Af_key *key, char *entry));
int	afListUdas	A((Af_key *key, char **symbollist));
int	afDropUdas	A((Af_key *key));
int	afCopyUdas	A((Af_key *src, Af_key *dest));
int	afAttrNameCmp	A((char *attr1, char *attr2));

#endif
