/*
 * Copyright (c) 1998, 1999, 2000, 2001
 *	Phil Thompson <phil@river-bank.demon.co.uk>
 *
 * The code generator module for SIP.
 */


#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "sip.h"


/* Macro expansions for template code. */

#define	MACRO_ESCAPE	'$'
#define	MACRO_CLASS	'C'			/* Class name. */
#define	MACRO_SOURCES	'S'			/* Source files. */
#define	MACRO_OBJECTS	'O'			/* Object files. */
#define	MACRO_CPPSUFF	'c'			/* C++ file suffix. */
#define	MACRO_OBJSUFF	'o'			/* Object file suffix. */
#define	MACRO_CPPMODULE	'm'			/* C++ module name. */
#define	MACRO_PERCENT	'P'			/* Percent sign. */


/* The maximum number of nested version qualifiers in the generated Python. */

#define	MAX_VQ_STACK	10


/* Control what generateSingleArg() actually generates. */

typedef enum {
	Call,
	Declaration,
	Definition
} funcArgType;


static int currentLineNr;		/* Current output line number. */
static char *currentFileName;		/* Current output file name. */
static versionDef vqstack[MAX_VQ_STACK];	/* Nested version qualifiers. */
static int vqstackptr = 0;		/* Version qualifier stack pointer. */


static void generateDocumentation(sipSpec *,char *);
static void generateMakefile(sipSpec *,char *,char *);
static void generatePython(sipSpec *,char *);
static void generateModuleProxy(sipSpec *,FILE *);
static void generateModuleProxyHeader(sipSpec *,char *);
static void generatePackageHeader(sipSpec *,char *);
static void generateCpp(sipSpec *,char *,char *);
static void generateVersionProgram(sipSpec *,char *,char *);
static void generateIfaceCpp(sipSpec *,ifaceFileDef *,char *,char *);
static void generateMappedTypeCpp(mappedTypeDef *,sipSpec *,FILE *);
static void generateMappedTypeHeader(mappedTypeDef *,char *,sipSpec *,FILE *);
static void generateClassVersCpp(classVersDef *,sipSpec *,FILE *);
static void generateClassVersHeader(classVersDef *,char *,sipSpec *,FILE *);
static int generatePythonPushVersion(versionDef *,int,FILE *);
static versionQual *pushVersion(versionDef *);
static void popVersion();
static int generatePythonClassVersWrapper(sipSpec *,classVersDef *,int,FILE *);
static void generatePythonClassVers(sipSpec *,classVersDef *,int,FILE *);
static void generateClassFunctions(sipSpec *,classVersDef *,FILE *);
static void generateComplexCode(sipSpec *,classVersDef *,FILE *);
static void generateFunction(sipSpec *,memberDef *,overDef *,classVersDef *,classVersDef *,int,FILE *);
static void generateFunctionBody(sipSpec *,overDef *,classVersDef *,FILE *);
static void generatePythonConstructor(sipSpec *,classVersDef *,FILE *);
static void generateCppCodeBlock(codeBlock *,classVersDef *,FILE *);
static void generateMacroCode(codeBlock *,classVersDef *,sipSpec *,char *,char *,FILE *);
static void generateMacroSource(sipSpec *,char *,FILE *);
static void generateMacroObject(sipSpec *,char *,FILE *);
static void generateUsedIncludes(ifaceFileList *,FILE *);
static void generateIfaceHeader(sipSpec *,ifaceFileDef *,char *);
static void generateWrapperClassDeclaration(sipSpec *,classVersDef *,FILE *);
static void generateLimitedWrapperClassDeclaration(sipSpec *,classVersDef *,FILE *);
static void generateResultType(argDef *,FILE *);
static void generateCallTidyUp(funcArgs *,FILE *);
static void generateArgs(funcArgs *,funcArgType,FILE *);
static void generateVariable(argDef *,int,int,FILE *);
static void generateValueType(argDef *,FILE *);
static void generateSingleArg(argDef *,int,funcArgType,FILE *);
static void generateBaseType(argDef *,FILE *);
static void generateExpression(valueDef *,FILE *);
static void generateTupleBuilder(sipSpec *,funcArgs *,int,FILE *);
static void generateEmitter(sipSpec *,classVersDef *,visibleList *,FILE *);
static void generateVirtualHandlerDeclaration(overDef *,FILE *);
static void generateVirtualHandlers(sipSpec *,classVersDef *,FILE *);
static void generateVirtualHandler(sipSpec *,virtOverDef *,FILE *);
static void generateVirtualCatcher(classVersDef *,int,virtOverDef *,FILE *);
static void generateProtectedEnums(sipSpec *,classVersDef *,FILE *);
static void generateProtectedDeclarations(sipSpec *,classVersDef *,FILE *);
static void generateProtectedDefinitions(sipSpec *,classVersDef *,FILE *);
static void generateArgCallList(funcArgs *,FILE *fp);
static void generateConstructorCall(sipSpec *,classVersDef *,ctorDef *,int,FILE *);
static void generateHandleResult(sipSpec *,overDef *,int,FILE *);
static void generateSimpleFunctionCall(fcallDef *,FILE *);
static void generateFunctionCall(sipSpec *,classVersDef *,overDef *,int,FILE *);
static void generateVariableHandler(sipSpec *,varDef *,FILE *);
static void generateObjToCppConversion(argDef *,char *,char *,FILE *);
static void generateObjConversion(varDef *,FILE *);
static void generateVarClassConversion(sipSpec *,varDef *,FILE *);
static void generateArgConvertors(sipSpec *,funcArgs *,int,int,char *,FILE *);
static void generateOredVersionStart(versionOrList *,FILE *);
static void generateOredVersionEnd(versionOrList *,FILE *);
static void generateVersionStart(versionDef *,FILE *);
static void generateVersionEnd(versionDef *,FILE *);
static void generateVoidPointers(sipSpec *,classVersDef *,FILE *);
static void generateChars(sipSpec *,classVersDef *,FILE *);
static void generateStrings(sipSpec *,classVersDef *,FILE *);
static void generateEnums(sipSpec *,classVersDef *,FILE *);
static void generateLongs(sipSpec *,classVersDef *,FILE *);
static void generateDoubles(sipSpec *,classVersDef *,FILE *);
static void generateClasses(sipSpec *,classVersDef *,FILE *);
static void generateAccessFunctions(sipSpec *,FILE *);
static void generateSpecialMethodStart(classVersDef *,specialPyMethod *,FILE *);
static void generateSpecialMethodEnd(specialPyMethod *,FILE *);
static void generateConvertToDeclarations(mappedTypeDef *,classVersDef *,char *,FILE *);
static void generateConvertToDefinitions(mappedTypeDef *,classVersDef *,FILE *);
static void freeVersionOrList(versionOrList *);
static int generateArgParser(funcArgs *,int,int,FILE *);
static int countVirtuals(classVersDef *);
static int skipOverload(overDef *,memberDef *,classVersDef *,classVersDef *,int);
static char *createIfaceFileName(char *,ifaceFileDef *,char *);
static FILE *createFile(sipSpec *,char *,char *,char *);
static void closeFile(FILE *);
static void prcode(FILE *,char *,...);
static void prScopedName(FILE *,scopedNameDef *,char *);
static void prTypeName(FILE *,argDef *,int);
static int needsSpecialHandling(sipSpec *,classVersDef *,overDef *);
static void generateSpecialHandlingStart(sipSpec *,classVersDef *,overDef *,
					 FILE *);
static void generateSpecialHandlingEnd(sipSpec *,classVersDef *,overDef *,
				       FILE *);


/*
 * Generate the code from a specification.
 */

void generateCode(sipSpec *pt,char *codeDir,stringList *makefiles,char *docFile,char *cppSuffix)
{
	/* Generate the documentation. */

	if (docFile != NULL)
		generateDocumentation(pt,docFile);

	/* Generate the Makefiles. */

	while (makefiles != NULL)
	{
		generateMakefile(pt,makefiles -> s,cppSuffix);

		makefiles = makefiles -> next;
	}

	/* Generate the code. */

	if (codeDir != NULL)
	{
		generateVersionProgram(pt,codeDir,cppSuffix);
		generatePackageHeader(pt,codeDir);
		generateCpp(pt,codeDir,cppSuffix);
		generatePython(pt,codeDir);
	}
}


/*
 * Generate the version generator program.  This program generates a header
 * file that is included by other C++ code and contains #defines corresponding
 * to each of the different (secondary) versions.
 *
 * Why not just #include the necessary file that provides the version
 * information and use the C++ test specified as part of the version definition
 * in the .sip files?
 *
 * The answer is that moc can't handle C++ pre-processor commands so the proxy
 * header file must be run through the C++ pre-processor beforehand.  The code
 * generated by moc is then #included by the main module code.  The net result
 * is that the header file specifying the version information is #included by
 * the main module code and #included a second time - but the second time is a
 * version that has already been run through the C++ pre-processor and has
 * therefore lost it's usual means of protecting itself from being #included
 * twice.  Unless the file follows certain rules (like having no function
 * definitions) it is likely to make the C++ compiler complain.  Therefore the
 * solution is to use a generated file that isn't going to cause complaints.
 */

static void generateVersionProgram(sipSpec *pt,char *codeDir,char *cppSuffix)
{
	char *vfile;
	FILE *fp;
	versionQual *vq;

	vfile = concat(codeDir,"/sip_helper",cppSuffix,NULL);
	fp = createFile(pt,vfile,"//","Generate a header file containing the version information.");

	prcode(fp,
"\n"
"\n"
"#include <iostream.h>\n"
"\n"
		);

	generateCppCodeBlock(pt -> versioncode,NULL,fp);

	prcode(fp,
"\n"
"\n"
"int main(int argc,char **argv)\n"
"{\n"
"	cout << \"// Define simple constants for each version qualifier.\\n\\n\";\n"
		);

	for (vq = pt -> secondaries; vq != NULL; vq = vq -> next)
		prcode(fp,
"\n"
"	cout << \"#undef SIP_VERS_%s\\n\";\n"
"#if %s\n"
"	cout << \"#define SIP_VERS_%s\\n\";\n"
"#endif\n"
			,vq -> name,vq -> vqcpp,vq -> name);

	prcode(fp,
"\n"
"	return 0;\n"
"}\n"
		);

	closeFile(fp);
	free(vfile);
}


/*
 * Generate the module proxy header file.
 */

static void generateModuleProxyHeader(sipSpec *pt,char *codeDir)
{
	char *hfile, *mname = pt -> module -> name -> text;
	FILE *fp;
	funcArgsList *fl;

	hfile = concat(codeDir,"/sip",mname,"Proxy",mname,".h",NULL);
	fp = createFile(pt,hfile,"//","Module proxy signal/slot handler class.");

	prcode(fp,
"\n"
"#include \"sip%sVersion.h\"\n"
"\n"
"class sipProxy%s: public sipProxy\n"
"{\n"
"	Q_OBJECT\n"
"\n"
"public:\n"
"	char *getProxySlot(char *);\n"
"\n"
"public slots:\n"
		,mname,mname);

	for (fl = pt -> sigargs; fl != NULL; fl = fl -> next)
	{
		generateOredVersionStart(fl -> vol,fp);

		prcode(fp,
"	void proxySlot(");

		generateArgs(fl -> fa,Declaration,fp);

		prcode(fp,");\n"
			);

		generateOredVersionEnd(fl -> vol,fp);
	}

	prcode(fp,
"};\n"
		);

	closeFile(fp);
	free(hfile);
}


/*
 * Generate the documentation.
 */

static void generateDocumentation(sipSpec *pt,char *docFile)
{
	FILE *fp;
	codeBlock *cb;

	fp = createFile(pt,docFile,NULL,NULL);

	for (cb = pt -> docs; cb != NULL; cb = cb -> next)
		fputs(cb -> frag,fp);

	closeFile(fp);
}


/*
 * Generate the code Makefile.
 */

static void generateMakefile(sipSpec *pt,char *makeFile,char *cppSuffix)
{
	char *tn;
	FILE *fp;
	codeBlock *cb;
	mkTemplateDef *mtd;

	/* Find the template. */

	if ((tn = strrchr(makeFile,'/')) != NULL)
		++tn;
	else
		tn = makeFile;

	for (mtd = pt -> mktemplates; mtd != NULL; mtd = mtd -> next)
		if (strcmp(mtd -> name,tn) == 0)
			break;

	if (mtd == NULL)
		fatal("Unknown Makefile template: %s\n",tn);

	fp = createFile(pt,makeFile,NULL,NULL);

	for (cb = mtd -> templ; cb != NULL; cb = cb -> next)
		generateMacroCode(cb,NULL,pt,cppSuffix,mtd -> objext,fp);

	closeFile(fp);
}


/*
 * Generate the Python wrapper code.
 */

static void generatePython(sipSpec *pt,char *codeDir)
{
	char *pyfile, *mname = pt -> module -> name -> text;
	FILE *fp;
	codeBlock *cb;
	moduleDef *mod;
	classVersDef *cvd;
	classVersList *cvl, *impcvdl;

	pyfile = concat(codeDir,"/",mname,".py",NULL);
	fp = createFile(pt,pyfile,"#","Python wrapper code.");

	/* Generate the signal/slot support if needed. */

	if (pt -> qobjclass >= 0)
		prcode(fp,
"\n"
"# Support for signals.\n"
"\n"
"def SLOT(slot):\n"
"	return '1' + slot\n"
"\n"
"def SIGNAL(sig):\n"
"	return '2' + sig\n"
"\n"
"def PYSIGNAL(sig):\n"
"	return '9' + sig\n"
			);

	/*
	 * Now get the different modules.  The first job is to load the C++
	 * code for any imported modules.  If we left it to the dynamic loader
	 * to do when importing the C++ code for this module it would
	 * probably fail because the C++ modules are unlikely to be in a
	 * directory in the standard path.  We can't leave it to any imports
	 * of classes from those modules because a) they may be dependent on
	 * particular versions, and b) the version tests may need this module
	 * loaded to determine the version information.
	 */

	prcode(fp,
"\n"
		);

	for (mod = pt -> moduleList; mod != NULL; mod = mod -> next)
	{
		/*
		 * Handle the case where the C++ module has a different name.
		 * This make use of "as" which is only in newer Pythons, but as
		 * this feature is for BlackAdder (which has the latest Python)
		 * then we should be Ok.
		 */

		if (mod == pt -> module && pt -> cppmname != mname)
			prcode(fp,
"import lib%sc as lib%sc\n"
				,pt -> cppmname,mod -> name -> text);
		else
			prcode(fp,
"import lib%sc\n"
				,mod -> name -> text);
	}

	/*
	 * Go through all the classes defined in this module and build up a
	 * list of all the super classes needed from other modules.
	 */

	impcvdl = NULL;

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
	{
		classList *cl;

		if (cvd -> common -> iff -> module != pt -> module)
			continue;

		/* Check the super-classes for any not in this module. */

		for (cl = cvd -> supers; cl != NULL; cl = cl -> next)
		{
			classVersList *impcv;

			if (cl -> c -> iff -> module == pt -> module)
				continue;

			/* We need the particular version. */

			for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
				if (cvl -> cv -> common == cl -> c)
					break;

			/* Check it's not already in the list. */

			for (impcv = impcvdl; impcv != NULL; impcv = impcv -> next)
				if (impcv -> cv == cvl -> cv)
					break;

			if (impcv == NULL)
			{
				impcv = sipMalloc(sizeof (versionQual));

				impcv -> cv = cvl -> cv;

				impcv -> next = impcvdl;
				impcvdl = impcv;
			}
		}
	}

	/* Generate the imports of any classes we need. */

	if (impcvdl != NULL)
		prcode(fp,
"\n"
			);

	for (cvl = impcvdl; cvl != NULL; cvl = cvl -> next)
	{
		int indent;

		indent = generatePythonPushVersion(&cvl -> cv -> version,0,fp);

		prcode(fp,
"%Ifrom %s import %s\n"
			,indent,cvl -> cv -> common -> iff -> module -> name -> text,classVersBaseName(cvl -> cv));

		popVersion();
	}

	prcode(fp,
"\n"
"lib%sc.sipInitModule()\n"
		,mname);

	/* Generate the pre-Python code blocks. */

	for (cb = pt -> prepycode; cb != NULL; cb = cb -> next)
		prcode(fp,"\n%s",cb -> frag);

	prcode(fp,
"\n"
		);

	/* Generate the Python class objects. */

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
		if (cvd -> common -> iff -> module == pt -> module)
			generatePythonClassVersWrapper(pt,cvd,0,fp);

	/* Register the classes. */

	prcode(fp,
"# Register the classes with the C++ module.\n"
"\n"
"lib%sc.sipRegisterClasses()\n"
		,mname);

	/* Generate the Python code blocks. */

	for (cb = pt -> pycode; cb != NULL; cb = cb -> next)
		prcode(fp,"\n%s",cb -> frag);

	closeFile(fp);
	free(pyfile);
}


/*
 * Generate the Python tests for a particular version.
 */

static int generatePythonPushVersion(versionDef *vd,int indent,FILE *fp)
{
	versionQual *vq;

	vq = pushVersion(vd);

	if (vq != NULL)
	{
		prcode(fp,
"%Iif %s:\n"
			,indent,vq -> vqpy);

		++indent;
	}

	/* Return the new indent level. */

	return indent;
}


/*
 * Push a new version qualifier onto the stack and return the specific
 * qualifier that needs to be applied to achieve it.  This is done only to
 * eliminate redundant #if...#endif directives to make the generated code
 * prettier.
 */

static versionQual *pushVersion(versionDef *vd)
{
	versionQual *vq;

	if (vqstackptr >= MAX_VQ_STACK)
		fatal("Internal error: increase the value of MAX_VQ_STACK\n");

	/* Assume the new version is the specific version. */

	vq = vd -> secondary;

	if (vqstackptr > 0)
	{
		int i;

		/* We apply the secondary if it hasn't already been applied. */

		for (i = 0; i < vqstackptr; ++i)
			if (vqstack[i].secondary == vd -> secondary)
			{
				/* It's already been applied. */

				vq = NULL;
				break;
			}
	}

	vqstack[vqstackptr++] = *vd;

	return vq;
}


/*
 * Pop a version qualifier off the stack.
 */

static void popVersion()
{
	--vqstackptr;
}


/*
 * Generate the Python class object for a particular version.
 */

static int generatePythonClassVersWrapper(sipSpec *pt,classVersDef *gcvd,
					  int indent,FILE *fp)
{
	classVersDef *cvd;

	/* It may already have been done. */

	if (isGenerated(gcvd))
		return indent;

	if (indent >= 0)
		setIsGenerated(gcvd);
	else if (isReady(gcvd))
		return indent;
	else
		setIsReady(gcvd);

	/* Classes in separate modules have already been done. */

	if (gcvd -> common -> iff -> module == pt -> module)
	{
		classVersList *cvl;

		/* Make sure the super-classes have been generated. */

		for (cvl = gcvd -> hierachy; cvl != NULL; cvl = cvl -> next)
		{
			classList *cl;

			for (cl = gcvd -> supers; cl != NULL; cl = cl -> next)
				if (cl -> c == cvl -> cv -> common)
					generatePythonClassVersWrapper(pt,cvl -> cv,0,fp);
		}

		/* Make sure the enclosing scope has been generated. */

		if (gcvd -> ecvd != NULL)
			indent = generatePythonClassVersWrapper(pt,gcvd -> ecvd,indent,fp);

		if (indent >= 0)
		{
			generatePythonClassVers(pt,gcvd,indent,fp);

			/* Any "children" must be indented further. */

			++indent;
		}


		/*
		 * Handle any class version for which this one is the enclosing
		 * scope.
		 */

		for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
			if (cvd -> ecvd == gcvd)
				generatePythonClassVersWrapper(pt,cvd,indent,fp);
	}

	return indent;
}


/*
 * Generate an expression in C++.
 */

static void generateExpression(valueDef *vd,FILE *fp)
{
	while (vd != NULL)
	{
		if (vd -> vunop != '\0')
			prcode(fp,"%c",vd -> vunop);

		switch (vd -> vtype)
		{
		case qchar_value:
			prcode(fp,"'%c'",vd -> u.vqchar);
			break;

		case string_value:
			prcode(fp,"%N",vd -> u.vstr);
			break;

		case numeric_value:
			prcode(fp,"%l",vd -> u.vnum);
			break;

		case real_value:
			prcode(fp,"%g",vd -> u.vreal);
			break;

		case scoped_value:
			prcode(fp,"%S",vd -> u.vscp);
			break;

		case fcall_value:
			generateSimpleFunctionCall(vd -> u.fcd,fp);
			break;
		}
 
		if (vd -> vbinop != '\0')
			prcode(fp," %c ",vd -> vbinop);
 
		vd = vd -> next;
	}
}


/*
 * Generate the wrapper for a Python class.
 */

static void generatePythonClassVers(sipSpec *pt,classVersDef *cvd,int indent,FILE *fp)
{
	char *mname = pt -> module -> name -> text;
	int noIntro;
	classList *cl;
	varDef *vd;

	indent = generatePythonPushVersion(&cvd -> version,indent,fp);

	prcode(fp,
"%Iclass %s",indent,classVersBaseName(cvd));

	/* Inherit from all immediate super-classes. */

	noIntro = TRUE;

	for (cl = cvd -> supers; cl != NULL; cl = cl -> next)
	{
		if (noIntro)
		{
			prcode(fp,"(");
			noIntro = FALSE;
		}
		else
			prcode(fp,",");

		prcode(fp,"%s",cl -> c -> iff -> name -> text);
	}

	if (!noIntro)
		prcode(fp,")");

	prcode(fp,":\n");

	/* Generate the standard methods. */

	prcode(fp,
"%I	def __init__(self,*args):\n"
"%I		lib%sc.sipCallCtor(%d,self,args)\n"
		,indent
		,indent,mname,cvd -> common -> classnr);

	if (!isNamespace(cvd) && cvd -> supers == NULL)
		prcode(fp,
"%I	def __del__(self):\n"
"%I		__sipDtor__(self)\n"
			,indent
			,indent);

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
		if (vd -> ecvd == cvd && !isStaticVar(vd))
		{
			prcode(fp,
"%I	def __getattr__(self,name):\n"
"%I		return lib%sc.sipCallGetVar(%d,self,name)\n"
"%I	def __setattr__(self,name,value):\n"
"%I		return lib%sc.sipCallSetVar(%d,self,name,value)\n"
				,indent
				,indent,mname,cvd -> common -> classnr
				,indent
				,indent,mname,cvd -> common -> classnr);

			break;
		}

	/* Generate the special methods. */

	if (cvd -> combmeth & SPM_METH_REPR)
		prcode(fp,
"%I	def __repr__(self):\n"
"%I		return repr(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combmeth & SPM_METH_STR)
		prcode(fp,
"%I	def __str__(self):\n"
"%I		return str(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combmeth & SPM_METH_CMP)
		prcode(fp,
"%I	def __cmp__(self,other):\n"
"%I		return cmp(self.sipThis,other)\n"
			,indent,indent);

	if (cvd -> combmeth & SPM_METH_HASH)
		prcode(fp,
"%I	def __hash__(self):\n"
"%I		return hash(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combmeth & SPM_METH_CALL)
		prcode(fp,
"%I	def __call__(self,*args):\n"
"%I		return self.sipThis(args)\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_ADD || cvd -> combseqmeth & SPM_SEQ_CONCAT)
		prcode(fp,
"%I	def __add__(self,other):\n"
"%I		return self.sipThis + other\n"
"%I	def __radd__(self,other):\n"
"%I		return other + self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_SUB)
		prcode(fp,
"%I	def __sub__(self,other):\n"
"%I		return self.sipThis - other\n"
"%I	def __rsub__(self,other):\n"
"%I		return other - self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_MUL || cvd -> combseqmeth & SPM_SEQ_REPEAT)
		prcode(fp,
"%I	def __mul__(self,other):\n"
"%I		return self.sipThis * other\n"
"%I	def __rmul__(self,other):\n"
"%I		return other * self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_DIV)
		prcode(fp,
"%I	def __div__(self,other):\n"
"%I		return self.sipThis / other\n"
"%I	def __rdiv__(self,other):\n"
"%I		return other / self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_MOD)
		prcode(fp,
"%I	def __mod__(self,other):\n"
"%I		return self.sipThis % other\n"
"%I	def __rmod__(self,other):\n"
"%I		return other % self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_DIVMOD)
		prcode(fp,
"%I	def __divmod__(self,other):\n"
"%I		return divmod(self.sipThis,other)\n"
"%I	def __rdivmod__(self,other):\n"
"%I		return divmod(other,self.sipThis)\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_POW)
		prcode(fp,
"%I	def __pow__(self,other,modulo = None):\n"
"%I		return pow(self.sipThis,other,modulo)\n"
"%I	def __rpow__(self,other):\n"
"%I		return other ** self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_NEG)
		prcode(fp,
"%I	def __neg__(self):\n"
"%I		return -self.sipThis\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_POS)
		prcode(fp,
"%I	def __pos__(self):\n"
"%I		return +self.sipThis\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_ABS)
		prcode(fp,
"%I	def __abs__(self):\n"
"%I		return abs(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_NONZERO)
		prcode(fp,
"%I	def __nonzero__(self):\n"
"%I		return not self.sipThis\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_INVERT)
		prcode(fp,
"%I	def __invert__(self):\n"
"%I		return ~self.sipThis\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_LSHIFT)
		prcode(fp,
"%I	def __lshift__(self,other):\n"
"%I		return self.sipThis << other\n"
"%I	def __rlshift__(self,other):\n"
"%I		return other << self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_RSHIFT)
		prcode(fp,
"%I	def __rshift__(self,other):\n"
"%I		return self.sipThis >> other\n"
"%I	def __rrshift__(self,other):\n"
"%I		return other >> self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_AND)
		prcode(fp,
"%I	def __and__(self,other):\n"
"%I		return self.sipThis & other\n"
"%I	def __rand__(self,other):\n"
"%I		return other & self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_XOR)
		prcode(fp,
"%I	def __xor__(self,other):\n"
"%I		return self.sipThis ^ other\n"
"%I	def __rxor__(self,other):\n"
"%I		return other ^ self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_OR)
		prcode(fp,
"%I	def __or__(self,other):\n"
"%I		return self.sipThis | other\n"
"%I	def __ror__(self,other):\n"
"%I		return other | self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_COERCE)
		prcode(fp,
"%I	def __coerce__(self,other):\n"
"%I		return coerce(self.sipThis,other)\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_INT)
		prcode(fp,
"%I	def __int__(self):\n"
"%I		return int(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_LONG)
		prcode(fp,
"%I	def __long__(self):\n"
"%I		return long(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_FLOAT)
		prcode(fp,
"%I	def __float__(self):\n"
"%I		return float(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_OCT)
		prcode(fp,
"%I	def __oct__(self):\n"
"%I		return oct(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_HEX)
		prcode(fp,
"%I	def __hex__(self):\n"
"%I		return hex(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_IADD || cvd -> combseqmeth & SPM_SEQ_ICONCAT)
		prcode(fp,
"%I	def __iadd__(self,other):\n"
"%I		self.sipThis += other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_ISUB)
		prcode(fp,
"%I	def __isub__(self,other):\n"
"%I		self.sipThis -= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_IMUL || cvd -> combseqmeth & SPM_SEQ_IREPEAT)
		prcode(fp,
"%I	def __imul__(self,other):\n"
"%I		self.sipThis *= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_IDIV)
		prcode(fp,
"%I	def __idiv__(self,other):\n"
"%I		self.sipThis /= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_IMOD)
		prcode(fp,
"%I	def __imod__(self,other):\n"
"%I		self.sipThis %= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_IPOW)
		prcode(fp,
"%I	def __ipow__(self,other,modulo = None):\n"
"%I		self.sipThis **= other)\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_ILSHIFT)
		prcode(fp,
"%I	def __lshift__(self,other):\n"
"%I		self.sipThis <<= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_IRSHIFT)
		prcode(fp,
"%I	def __rshift__(self,other):\n"
"%I		self.sipThis >>= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_IAND)
		prcode(fp,
"%I	def __and__(self,other):\n"
"%I		self.sipThis &= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_IXOR)
		prcode(fp,
"%I	def __xor__(self,other):\n"
"%I		self.sipThis ^= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combnummeth & SPM_NUM_IOR)
		prcode(fp,
"%I	def __or__(self,other):\n"
"%I		self.sipThis |= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (cvd -> combseqmeth & SPM_SEQ_LEN || cvd -> combmapmeth & SPM_MAP_LEN)
		prcode(fp,
"%I	def __len__(self):\n"
"%I		return len(self.sipThis)\n"
			,indent,indent);

	if (cvd -> combseqmeth & SPM_SEQ_GETITEM || cvd -> combmapmeth & SPM_MAP_GETITEM)
		prcode(fp,
"%I	def __getitem__(self,key):\n"
"%I		return self.sipThis[key]\n"
			,indent,indent);

	if (cvd -> combseqmeth & SPM_SEQ_SETITEM || cvd -> combmapmeth & SPM_MAP_SETITEM)
		prcode(fp,
"%I	def __setitem__(self,key,value):\n"
"%I		return self.sipThis[key] = value\n"
"%I	def __delitem__(self,key):\n"
"%I		return del self.sipThis[key]\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combseqmeth & SPM_SEQ_GETSLICE)
		prcode(fp,
"%I	def __getslice__(self,i,j):\n"
"%I		return self.sipThis[i:j]\n"
			,indent,indent);

	if (cvd -> combseqmeth & SPM_SEQ_SETSLICE)
		prcode(fp,
"%I	def __setslice__(self,i,j,sequence):\n"
"%I		return self.sipThis[i:j] = sequence\n"
"%I	def __delslice__(self,i,j):\n"
"%I		return del self.sipThis[i:j]\n"
			,indent,indent
			,indent,indent);

	if (cvd -> combseqmeth & SPM_SEQ_CONTAINS)
		prcode(fp,
"%I	def __contains__(self,member):\n"
"%I		return member in self.sipThis\n"
			,indent,indent);

	if (cvd -> combmeth & SPM_METH_RICH)
	{
		prcode(fp,
"%I	def __lt__(self,other):\n"
"%I		return self.sipThis < other\n"
			,indent,indent);

		prcode(fp,
"%I	def __le__(self,other):\n"
"%I		return self.sipThis <= other\n"
			,indent,indent);

		prcode(fp,
"%I	def __eq__(self,other):\n"
"%I		return self.sipThis == other\n"
			,indent,indent);

		prcode(fp,
"%I	def __ne__(self,other):\n"
"%I		return self.sipThis != other\n"
			,indent,indent);

		prcode(fp,
"%I	def __gt__(self,other):\n"
"%I		return self.sipThis > other\n"
			,indent,indent);

		prcode(fp,
"%I	def __ge__(self,other):\n"
"%I		return self.sipThis >= other\n"
			,indent,indent);
	}

	prcode(fp,
"\n"
		);

	popVersion();
}


/*
 * Generate the C++ package specific header file.
 */

static void generatePackageHeader(sipSpec *pt,char *codeDir)
{
	char *hfile, *mname = pt -> module -> name -> text;
	FILE *fp;
	nameDef *nd;

	hfile = concat(codeDir,"/sip",mname,"Decl",mname,".h",NULL);
	fp = createFile(pt,hfile,"//","C++ package header file.");

	/* Include files. */

	prcode(fp,
"\n"
"#ifndef _%sDECL%s_H\n"
"#define	_%sDECL%s_H\n"
"\n"
"#ifdef HAVE_CONFIG_H\n"
"#include \"config.h\"\n"
"#endif\n"
"\n"
"#include <sip%s.h>\n"
"\n"
"#include \"sip%sVersion.h\"\n"
"\n"
		,mname,mname,mname,mname,(pt -> sigslots ? "Qt" : ""),mname);

	generateCppCodeBlock(pt -> exphdrcode,NULL,fp);
	generateCppCodeBlock(pt -> hdrcode,NULL,fp);

	/* The names declarations (for all modules). */

	prcode(fp,
"\n"
		);

	for (nd = pt -> namecache; nd != NULL; nd = nd -> next)
		prcode(fp,
"extern SIP_%s char %N[];\n"
			,(nd -> module == pt -> module) ? "EXPORT" : "IMPORT"
			,nd);

	if (pt -> sigslots)
		prcode(fp,
"\n"
"sipProxy *sipNewProxy_%s Py_PROTO(());\n"
			,mname);

	prcode(fp,
"\n"
"#endif\n"
		);

	closeFile(fp);
	free(hfile);
}


/*
 * Generate the C++ code.
 */

static void generateCpp(sipSpec *pt,char *codeDir,char *cppSuffix)
{
	char *cppfile, *mname = pt -> module -> name -> text;
	int nrclasses, noIntro;
	FILE *fp;
	classDef *cd;
	classVersDef *cvd;
	nameDef *nd;
	memberDef *md;
	expFuncDef *ef;
	ifaceFileDef *iff;

	/* Generate any module proxy header file. */

	if (pt -> sigargs != NULL)
		generateModuleProxyHeader(pt,codeDir);

	cppfile = concat(codeDir,"/",mname,"cmodule",cppSuffix,NULL);
	fp = createFile(pt,cppfile,"//","C++ wrapper code.");

	prcode(fp,
"\n"
"#include \"sip%sDecl%s.h\"\n"
		,mname,mname);

	generateUsedIncludes(pt -> used,fp);

	/* The names definitions. */

	for (nd = pt -> namecache; nd != NULL; nd = nd -> next)
		if (nd -> module == pt -> module)
			prcode(fp,
"char %N[] = \"%s\";\n"
				,nd,nd -> text);

	/* The class objects for any namespaces. */

	noIntro = TRUE;

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
		if (isNamespace(cvd))
		{
			if (noIntro)
			{
				prcode(fp,
"\n"
"// Class objects for namespaces.\n"
"\n"
					);

				noIntro = FALSE;
			}

			generateVersionStart(&cvd -> version,fp);

			prcode(fp,
"static PyObject *sipClass_%C;\n"
				,classVersFQName(cvd));

			generateVersionEnd(&cvd -> version,fp);
		}

	/* Generate the module proxy. */

	if (pt -> sigargs != NULL)
	{
		generateModuleProxy(pt,fp);

		prcode(fp,
"\n"
"sipProxy *sipNewProxy_%s()\n"
"{\n"
"	return new sipProxy%s();\n"
"}\n"
			,mname,mname);
	}

	/* Generate the C++ code blocks. */

	generateCppCodeBlock(pt -> cppcode,NULL,fp);

	/* Generate the global functions. */

	for (md = pt -> othfuncs; md != NULL; md = md -> next)
		if (md -> module == pt -> module)
		{
			overDef *od;

			prcode(fp,
"\n"
"static PyObject *sipDo_%s(PyObject *,PyObject *sipArgs)\n"
"{\n"
"	int sipArgsParsed = 0;\n"
				,md -> pyname -> text);

			for (od = pt -> overs; od != NULL; od = od -> next)
				if (od -> common == md)
					generateFunctionBody(pt,od,NULL,fp);

			prcode(fp,
"\n"
"	// Report an error if the arguments couldn't be parsed.\n"
"\n"
"	sipNoFunction(sipArgsParsed,%N);\n"
"\n"
"	return NULL;\n"
"}\n"
				,md -> pyname);
		}

	/* Generate the access functions. */

	generateAccessFunctions(pt,fp);

	/* Generate the module data structures. */

	prcode(fp,
"\n"
"static sipClassDef classesTable[] = {\n"
		);

	nrclasses = 0;

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		int nrcond;
		classVersDef *cvd;

		if (cd -> iff -> module != pt -> module)
			continue;

		++nrclasses;

		/*
		 * Version control is handled a little differently to make
		 * sure there is a table entry for every possible class.
		 */

		nrcond = 0;

		for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
		{
			if (cvd -> common != cd)
				continue;

			if (cvd -> version.secondary != NULL)
			{
				++nrcond;

				prcode(fp,
"%s %V\n"
					,(nrcond < 2 ? "#if " : "#elif ")
					,cvd -> version.secondary);
			}

			prcode(fp,
"	{%N, "
				,cd -> iff -> name);

			if (isNamespace(cvd))
				prcode(fp,"NULL, &sipClass_%C, NULL, NULL,"
					,classVersFQName(cvd));
			else
			{
				varDef *vd;

				prcode(fp,"sipNew_%C, &sipClass_%C, sipClassAttrTab_%C,"
					,classVersFQName(cvd)
					,classVersFQName(cvd)
					,classVersFQName(cvd));

				for (vd = pt -> vars; vd != NULL; vd = vd -> next)
				{
					if (vd -> ecvd != cvd)
						continue;

					if (!isStaticVar(vd))
						break;
				}

				if (vd != NULL)
					prcode(fp," sipClassVarHierTab_%C,"
						,classVersFQName(cvd));
				else
					prcode(fp," NULL,");
			}

			prcode(fp," %d},\n"
				,cvd -> ecvd != NULL ? cvd -> ecvd -> common -> classnr : -1);
		}

		if (nrcond > 0)
			prcode(fp,
"#else\n"
"	{NULL, NULL, NULL, NULL, NULL, -1},\n"
"#endif\n"
				);
	}

	prcode(fp,
"};\n"
"\n"
"static sipModuleDef sipModule = {\n"
"	%N,\n"
"	%d,\n"
"	classesTable\n"
"};\n"
		,pt -> module -> name,nrclasses);


	/* Generate the call class ctor function. */

	prcode(fp,
"\n"
"// The entry point into the bindings for constructors.\n"
"\n"
"static PyObject *callCtor(PyObject *,PyObject *args)\n"
"{\n"
"	return sipCallCtor(&sipModule,args);\n"
"}\n"
		);

	/* Generate the get and set class attribute functions. */

	prcode(fp,
"\n"
"// The entry point into the bindings for getting class variables.\n"
"\n"
"static PyObject *callGetVar(PyObject *,PyObject *args)\n"
"{\n"
"	return sipGetVar(&sipModule,args);\n"
"}\n"
"\n"
"// The entry point into the bindings for setting class variables.\n"
"\n"
"static PyObject *callSetVar(PyObject *,PyObject *args)\n"
"{\n"
"	return sipSetVar(&sipModule,args);\n"
"}\n"
		);

	/* Generate the register Python globals function. */

	prcode(fp,
"\n"
"// Initialise the module.\n"
"\n"
"static PyObject *initModule(PyObject *,PyObject *)\n"
"{\n"
"	if (sipRegisterModule(&sipModule) < 0)\n"
"		return NULL;\n"
		);

	/* Generate the global functions. */

	noIntro = TRUE;

	for (md = pt -> othfuncs; md != NULL; md = md -> next)
		if (md -> module == pt -> module)
		{
			if (noIntro)
			{
				prcode(fp,
"\n"
"	// Add the global functions to the dictionary.\n"
"\n"
"	static PyMethodDef globfuncs[] = {\n"
					);

				noIntro = FALSE;
			}

			prcode(fp,
"		{%N, sipDo_%s, METH_VARARGS, NULL},\n"
				,md -> pyname,md -> pyname -> text);
		}

	if (!noIntro)
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddFunctions(sipModule.md_dict,globfuncs) < 0)\n"
"		return NULL;\n"
			);

	/* Generate the global variables. */

	generateVoidPointers(pt,NULL,fp);
	generateChars(pt,NULL,fp);
	generateStrings(pt,NULL,fp);
	generateLongs(pt,NULL,fp);
	generateDoubles(pt,NULL,fp);
	generateEnums(pt,NULL,fp);

	prcode(fp,
"\n"
"	Py_INCREF(Py_None);\n"
"	return Py_None;\n"
"}\n"
		);

	/* Generate the register Python classes function. */

	prcode(fp,
"\n"
"// Final stage after the Python classes have been created.\n"
"\n"
"static PyObject *registerClasses(PyObject *,PyObject *)\n"
"{\n"
"	if (sipRegisterClasses(&sipModule,%d) < 0)\n"
"		return NULL;\n"
		,pt -> qobjclass);

	/* Add the global class instances. */

	generateClasses(pt,NULL,fp);

	/* Generate all static class variables. */

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
	{
		if (cvd -> common -> iff -> module != pt -> module)
			continue;

		generateVoidPointers(pt,cvd,fp);
		generateChars(pt,cvd,fp);
		generateStrings(pt,cvd,fp);
		generateLongs(pt,cvd,fp);
		generateDoubles(pt,cvd,fp);
		generateEnums(pt,cvd,fp);
		generateClasses(pt,cvd,fp);
	}

	prcode(fp,
"\n"
"	Py_INCREF(Py_None);\n"
"	return Py_None;\n"
"}\n"
		);

	/* Generate the Python module initialisation function. */

	prcode(fp,
"\n"
"// The Python module initialisation function.\n"
"\n"
"extern \"C\" SIP_EXPORT void initlib%sc()\n"
"{\n"
"	static PyMethodDef methods[] = {\n"
"		{\"sipCallCtor\", callCtor, METH_VARARGS, NULL},\n"
"		{\"sipCallGetVar\", callGetVar, METH_VARARGS, NULL},\n"
"		{\"sipCallSetVar\", callSetVar, METH_VARARGS, NULL},\n"
"		{\"sipInitModule\", initModule, METH_VARARGS, NULL},\n"
"		{\"sipRegisterClasses\", registerClasses, METH_VARARGS, NULL},\n"
		,pt -> cppmname);

	for (ef = pt -> exposed; ef != NULL; ef = ef -> next)
		prcode(fp,
"		{\"%s\", %s, METH_VARARGS, NULL},\n"
			,ef -> name,ef -> name);

	prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	// Initialise the module.\n"
"\n"
"	Py_InitModule(\"lib%sc\",methods);\n"
"}\n"
		,pt -> cppmname);

	closeFile(fp);
	free(cppfile);

	/* Generate the interface source and header files. */

	for (iff = pt -> ifacefiles; iff != NULL; iff = iff -> next)
	{
		/* Make sure at least one version is not a namespace. */

		if (hasMappedType(iff) || hasRealClassVers(iff))
		{
			if (iff -> module == pt -> module)
				generateIfaceCpp(pt,iff,codeDir,cppSuffix);

			if (isUsed(iff))
				generateIfaceHeader(pt,iff,codeDir);
		}
	}
}


/*
 * Generate the access functions for the variables.
 */

static void generateAccessFunctions(sipSpec *pt,FILE *fp)
{
	varDef *vd;

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		if (vd -> module != pt -> module || vd -> accessfunc == NULL)
			continue;

		if (vd -> ecvd != NULL && !isStaticVar(vd))
			continue;

		prcode(fp,
"\n"
			);

		generateVersionStart(&vd -> version,fp);

		prcode(fp,
"// Access function.\n"
"\n"
"static const void *sipAccess_%C()\n"
"{\n"
			,vd -> fqname);

		generateCppCodeBlock(vd -> accessfunc,NULL,fp);

		prcode(fp,
"}\n"
			);

		generateVersionEnd(&vd -> version,fp);
	}
}


/*
 * Generate the code to add a set of class instances to a dictionary.
 */

static void generateClasses(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		scopedNameDef *vcname;

		if (vd -> ecvd != cvd || vd -> module != pt -> module)
			continue;

		if (vd -> type.atype != class_type)
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				prcode(fp,
"\n"
					);

				generateVersionStart(&cvd -> version,fp);

				prcode(fp,
"	// Add the class instances to the class dictionary.\n"
"\n"
"	static sipClassInstanceDef %C_classInstances[] = {\n"
					,classVersFQName(cvd));
			}
			else
				prcode(fp,
"\n"
"	// Add the class instances to the module dictionary.\n"
"\n"
"	static sipClassInstanceDef classInstances[] = {\n"
					);

			noIntro = FALSE;
		}

		generateVersionStart(&vd -> version,fp);

		vcname = classVersFQName(vd -> type.u.cvd);

		if (vd -> accessfunc != NULL)
		{
			prcode(fp,
"		{%N, sipAccess_%C, sipClass_%C, SIP_SIMPLE | SIP_ACCFUNC},\n"
				,vd -> name,vd -> fqname,vcname);
		}
		else if (vd -> type.nrderefs != 0)
		{
			prcode(fp,
"		{%N, &%S, sipClass_%C, SIP_SIMPLE | SIP_INDIRECT},\n"
				,vd -> name,vd -> fqname,vcname);
		}
		else
		{
			prcode(fp,
"		{%N, &%S, sipClass_%C, SIP_SIMPLE},\n"
				,vd -> name,vd -> fqname,vcname);
		}

		generateVersionEnd(&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		NULL\n"
"	};\n"
"\n"
"	if (sipAddClassInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%C) -> cl_dict,%C_",classVersFQName(cvd),classVersFQName(cvd));
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"classInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of void pointers to a dictionary.
 */

static void generateVoidPointers(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		if (vd -> ecvd != cvd || vd -> module != pt -> module)
			continue;

		if (vd -> type.atype != voidptr_type && vd -> type.atype != struct_type)
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				prcode(fp,
"\n"
					);

				generateVersionStart(&cvd -> version,fp);

				prcode(fp,
"	// Add the void pointers to the class dictionary.\n"
"\n"
"	static sipVoidPtrInstanceDef %C_voidPtrInstances[] = {\n"
					,classVersFQName(cvd));
			}
			else
				prcode(fp,
"\n"
"	// Add the void pointers to the module dictionary.\n"
"\n"
"	static sipVoidPtrInstanceDef voidPtrInstances[] = {\n"
					);

			noIntro = FALSE;
		}

		generateVersionStart(&vd -> version,fp);

		prcode(fp,
"		{%N, %S},\n"
			,vd -> name,vd -> fqname);

		generateVersionEnd(&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddVoidPtrInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%C) -> cl_dict,%C_",classVersFQName(cvd),classVersFQName(cvd));
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"voidPtrInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of characters to a dictionary.
 */

static void generateChars(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		argType vtype = vd -> type.atype;

		if (vd -> ecvd != cvd || vd -> module != pt -> module)
			continue;

		if (!((vtype == ustring_type || vtype == string_type) && vd -> type.nrderefs == 0))
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				prcode(fp,
"\n"
					);

				generateVersionStart(&cvd -> version,fp);

				prcode(fp,
"	// Add the chars to the class dictionary.\n"
"\n"
"	static sipCharInstanceDef %C_charInstances[] = {\n"
					,classVersFQName(cvd));
			}
			else
				prcode(fp,
"\n"
"	// Add the chars to the module dictionary.\n"
"\n"
"	static sipCharInstanceDef charInstances[] = {\n"
					);

			noIntro = FALSE;
		}

		generateVersionStart(&vd -> version,fp);

		prcode(fp,
"		{%N, %S},\n"
			,vd -> name,vd -> fqname);

		generateVersionEnd(&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddCharInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%C) -> cl_dict,%C_",classVersFQName(cvd),classVersFQName(cvd));
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"charInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of strings to a dictionary.
 */

static void generateStrings(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		argType vtype = vd -> type.atype;

		if (vd -> ecvd != cvd || vd -> module != pt -> module)
			continue;

		if (!((vtype == ustring_type || vtype == string_type) && vd -> type.nrderefs != 0))
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				prcode(fp,
"\n"
					);

				generateVersionStart(&cvd -> version,fp);

				prcode(fp,
"	// Add the strings to the class dictionary.\n"
"\n"
"	static sipStringInstanceDef %C_stringInstances[] = {\n"
					,classVersFQName(cvd));
			}
			else
				prcode(fp,
"\n"
"	// Add the strings to the module dictionary.\n"
"\n"
"	static sipStringInstanceDef stringInstances[] = {\n"
					);

			noIntro = FALSE;
		}

		generateVersionStart(&vd -> version,fp);

		prcode(fp,
"		{%N, %S},\n"
			,vd -> name,vd -> fqname);

		generateVersionEnd(&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddStringInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%C) -> cl_dict,%C_",classVersFQName(cvd),classVersFQName(cvd));
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"stringInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of enums to a dictionary.
 */

static void generateEnums(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	enumDef *ed;

	/* First the tables of values for each enum. */

	noIntro = TRUE;

	for (ed = pt -> enums; ed != NULL; ed = ed -> next)
	{
		enumValueDef *evd;

		if (ed -> ecvd != cvd || ed -> module != pt -> module)
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				prcode(fp,
"\n"
					);

				generateVersionStart(&cvd -> version,fp);

				prcode(fp,
"	// Add the enums to the class dictionary.\n"
"\n"
"	static sipEnumValueInstanceDef %C_enumValues[] = {\n"
					,classVersFQName(cvd));
			}
			else
				prcode(fp,
"\n"
"	// Add the enums to the module dictionary.\n"
"\n"
"	static sipEnumValueInstanceDef enumValues[] = {\n"
				);

			noIntro = FALSE;
		}

		generateVersionStart(&ed -> version,fp);

		for (evd = ed -> values; evd != NULL; evd = evd -> next)
		{
			prcode(fp,
"		{%N, ",evd -> name);

			if (cvd != NULL)
				prcode(fp,"%s%S::",(isProtectedEnum(ed) ? "sip" : ""),classVersFQName(cvd));

			prcode(fp,"%s},\n"
				,evd -> name -> text);
		}

		generateVersionEnd(&ed -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddEnumInstances("
			);

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%C) -> cl_dict,%C_",classVersFQName(cvd),classVersFQName(cvd));
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"enumValues) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of longs to a dictionary.
 */

static void generateLongs(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		argType vtype = vd -> type.atype;

		if (vd -> ecvd != cvd || vd -> module != pt -> module)
			continue;

		if (!(vtype == enum_type || vtype == ushort_type || vtype == short_type || vtype == uint_type || vtype == cint_type || vtype == int_type || vtype == ulong_type || vtype == long_type || vtype == bool_type))
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				prcode(fp,
"\n"
					);

				generateVersionStart(&cvd -> version,fp);

				prcode(fp,
"	// Add the longs to the class dictionary.\n"
"\n"
"	static sipLongInstanceDef %C_longInstances[] = {\n"
					,classVersFQName(cvd));
			}
			else
				prcode(fp,
"\n"
"	// Add the longs to the module dictionary.\n"
"\n"
"	static sipLongInstanceDef longInstances[] = {\n"
					);

			noIntro = FALSE;
		}

		generateVersionStart(&vd -> version,fp);

		prcode(fp,
"		{%N, %S},\n"
			,vd -> name,vd -> fqname);

		generateVersionEnd(&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddLongInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%C) -> cl_dict,%C_",classVersFQName(cvd),classVersFQName(cvd));
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"longInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of doubles to a dictionary.
 */

static void generateDoubles(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		argType vtype = vd -> type.atype;

		if (vd -> ecvd != cvd || vd -> module != pt -> module)
			continue;

		if (!(vtype == float_type || vtype == double_type))
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				prcode(fp,
"\n"
					);

				generateVersionStart(&cvd -> version,fp);

				prcode(fp,
"	// Add the doubles to the class dictionary.\n"
"\n"
"	static sipDoubleInstanceDef %C_doubleInstances[] = {\n"
					,classVersFQName(cvd));
			}
			else
				prcode(fp,
"\n"
"	// Add the doubles to the module dictionary.\n"
"\n"
"	static sipDoubleInstanceDef doubleInstances[] = {\n"
					);

			noIntro = FALSE;
		}

		generateVersionStart(&vd -> version,fp);

		prcode(fp,
"		{%N, %S},\n"
			,vd -> name,vd -> fqname);

		generateVersionEnd(&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddDoubleInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%C) -> cl_dict,%C_",classVersFQName(cvd),classVersFQName(cvd));
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"doubleInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(&cvd -> version,fp);
	}
}


/*
 * Generate the implementation of the module proxy.
 */

static void generateModuleProxy(sipSpec *pt,FILE *fp)
{
	char *mname = pt -> module -> name -> text;
	funcArgsList *fl;

	prcode(fp,
"\n"
"#include \"sip%sProxy%s.moc\"\n"
"\n"
"char *sipProxy%s::getProxySlot(char *sigargs)\n"
"{\n"
"	static char *tab[] = {\n"
		,mname,mname,mname);

	for (fl = pt -> sigargs; fl != NULL; fl = fl -> next)
	{
		generateOredVersionStart(fl -> vol,fp);

		prcode(fp,
"		SLOT(proxySlot(");

		generateArgs(fl -> fa,Declaration,fp);

		prcode(fp,")),\n"
			);

		generateOredVersionEnd(fl -> vol,fp);
	}

	prcode(fp,
"		NULL\n"
"	};\n"
"\n"
"	return searchProxySlotTable(tab,sigargs);\n"
"}\n"
		);

	for (fl = pt -> sigargs; fl != NULL; fl = fl -> next)
	{
		prcode(fp,
"\n"
			);

		generateOredVersionStart(fl -> vol,fp);

		prcode(fp,
"void sipProxy%s::proxySlot(",mname);

		generateArgs(fl -> fa,Definition,fp);

		prcode(fp,")\n"
"{\n"
"	sipSender = sender();\n"
"\n"
"	int relLock = sipCondAcquireLock();\n"
			);

		generateTupleBuilder(pt,fl -> fa,FALSE,fp);

		prcode(fp,
"\n"
"	if (sipArgs == NULL || sipEmitToSlot(&sipRealSlot,sipArgs) < 0)\n"
"		PyErr_Print();\n"
"\n"
"	Py_XDECREF(sipArgs);\n"
"\n"
"	sipCondReleaseLock(relLock);\n"
"}\n"
			);

		generateOredVersionEnd(fl -> vol,fp);
	}
}


/*
 * Generate the C++ code for an interface.
 */

static void generateIfaceCpp(sipSpec *pt,ifaceFileDef *iff,char *codeDir,
			     char *cppSuffix)
{
	char *cppfile, *cmname = iff -> module -> name -> module -> name -> text;
	classVersDef *cvd;
	mappedTypeDef *mtd;
	FILE *fp;

	cppfile = createIfaceFileName(codeDir,iff,cppSuffix);
	fp = createFile(pt,cppfile,"//","C++ interface wrapper code.");

	prcode(fp,
"\n"
"#include \"sip%sDecl%s.h\"\n"
"#include \"sip%s%F.h\"\n"
		,cmname,cmname,cmname,iff -> fqname);

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
		if (cvd -> common -> iff == iff)
			generateClassVersCpp(cvd,pt,fp);

	for (mtd = pt -> mappedtypes; mtd != NULL; mtd = mtd -> next)
		if (mtd -> iff == iff)
			generateMappedTypeCpp(mtd,pt,fp);

	closeFile(fp);
	free(cppfile);
}


/*
 * Return a filename for an interface C++ or header file on the heap.
 */

static char *createIfaceFileName(char *codeDir,ifaceFileDef *iff,char *suffix)
{
	char *fn;
	scopedNameDef *snd;

	fn = concat(codeDir,"/sip",iff -> module -> name -> module -> name -> text,NULL);

	for (snd = iff -> fqname; snd != NULL; snd = snd -> next)
		append(&fn,snd -> name);

	append(&fn,suffix);

	return fn;
}


/*
 * Generate the C++ code for a mapped type version.
 */

static void generateMappedTypeCpp(mappedTypeDef *mtd,sipSpec *pt,FILE *fp)
{
	prcode(fp,
"\n"
		);

	generateVersionStart(&mtd -> version,fp);

	generateConvertToDefinitions(mtd,NULL,fp);

	/* Generate the from type convertor. */

	prcode(fp,
"\n"
"PyObject *sipConvertFrom_%T(const %B *sipCpp)\n"
"{\n"
		,&mtd -> type,&mtd -> type);

	generateCppCodeBlock(mtd -> convfromcode,NULL,fp);

	prcode(fp,
"}\n"
		);

	generateVersionEnd(&mtd -> version,fp);
}


/*
 * Generate the C++ code for a class version.
 */

static void generateClassVersCpp(classVersDef *cvd,sipSpec *pt,FILE *fp)
{
	varDef *vd;
	classVersList *cvl;
	visibleList *vl;
	spmDef *sd;
	int noIntro;

	prcode(fp,
"\n"
		);

	generateVersionStart(&cvd -> version,fp);

	prcode(fp,
"\n"
"PyObject *sipClass_%C;\n"
		,classVersFQName(cvd));

	/* Generate the Python object structures. */

	prcode(fp,
"\n"
"static void sipDealloc_%C(sipThisType *);\n"
		,classVersFQName(cvd));

	for (sd = cvd -> spms; sd != NULL; sd = sd -> next)
	{
		generateSpecialMethodStart(cvd,sd -> spm,fp);

		prcode(fp,";\n"
			);

		generateSpecialMethodEnd(sd -> spm,fp);
	}

	/* Define the table for number methods if any. */

	if (cvd -> combnummeth || cvd -> combnummethx)
	{
		prcode(fp,
"\n"
"static PyNumberMethods sipNumMethods_%C = {\n"
			,classVersFQName(cvd));

		if (cvd -> combnummeth & SPM_NUM_ADD)
			prcode(fp,
"	sip__add__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_SUB)
			prcode(fp,
"	sip__sub__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_MUL)
			prcode(fp,
"	sip__mul__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_DIV)
			prcode(fp,
"	sip__div__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_MOD)
			prcode(fp,
"	sip__mod__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_DIVMOD)
			prcode(fp,
"	sip__divmod__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_POW)
			prcode(fp,
"	sip__pow__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_NEG)
			prcode(fp,
"	sip__neg__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_POS)
			prcode(fp,
"	sip__pos__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_ABS)
			prcode(fp,
"	sip__abs__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_NONZERO)
			prcode(fp,
"	sip__nonzero__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_INVERT)
			prcode(fp,
"	sip__invert__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_LSHIFT)
			prcode(fp,
"	sip__lshift__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_RSHIFT)
			prcode(fp,
"	sip__rshift__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_AND)
			prcode(fp,
"	sip__and__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_XOR)
			prcode(fp,
"	sip__xor__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_OR)
			prcode(fp,
"	sip__or__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_COERCE)
			prcode(fp,
"	sip__coerce__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_INT)
			prcode(fp,
"	sip__int__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_LONG)
			prcode(fp,
"	sip__long__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_FLOAT)
			prcode(fp,
"	sip__float__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_OCT)
			prcode(fp,
"	sip__oct__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_HEX)
			prcode(fp,
"	sip__hex__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0\n"
				);

		/* The inplace methods came in with Python 2.0. */

		prcode(fp,
"#if PY_VERSION_HEX >= 0x02000000\n"
			);

		if (cvd -> combnummeth & SPM_NUM_IADD)
			prcode(fp,
"	sip__iadd__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_ISUB)
			prcode(fp,
"	sip__isub__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_IMUL)
			prcode(fp,
"	sip__imul__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_IDIV)
			prcode(fp,
"	sip__idiv__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_IMOD)
			prcode(fp,
"	sip__imod__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_IPOW)
			prcode(fp,
"	sip__ipow__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_ILSHIFT)
			prcode(fp,
"	sip__ilshift__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_IRSHIFT)
			prcode(fp,
"	sip__irshift__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth & SPM_NUM_IAND)
			prcode(fp,
"	sip__iand__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummethx & SPM_NUM_IXOR)
			prcode(fp,
"	sip__ixor__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummethx & SPM_NUM_IOR)
			prcode(fp,
"	sip__ior__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		prcode(fp,
"#endif\n"
"};\n"
			);
	}

	/* Define the table for sequence methods if any. */

	if (cvd -> combseqmeth || cvd -> combseqmethx)
	{
		prcode(fp,
"\n"
"static PySequenceMethods sipSeqMethods_%C = {\n"
			,classVersFQName(cvd));

		if (cvd -> combseqmeth & SPM_SEQ_LEN)
			prcode(fp,
"	sip__len__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combseqmeth & SPM_SEQ_CONCAT)
			prcode(fp,
"	sip__add__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combseqmeth & SPM_SEQ_REPEAT)
			prcode(fp,
"	sip__mul__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combseqmeth & SPM_SEQ_GETITEM)
			prcode(fp,
"	sip__getitem__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combseqmeth & SPM_SEQ_SETITEM)
			prcode(fp,
"	sip__setitem__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combseqmeth & SPM_SEQ_GETSLICE)
			prcode(fp,
"	sip__getslice__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combseqmeth & SPM_SEQ_SETSLICE)
			prcode(fp,
"	sip__setslice__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		/* __contains__ came in with Python 1.6. */

		prcode(fp,
"#if PY_VERSION_HEX >= 0x01060000\n"
			);

		if (cvd -> combseqmeth & SPM_SEQ_CONTAINS)
			prcode(fp,
"	sip__contains__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		prcode(fp,
"#endif\n"
			);

		/* __iadd__ and __imul__ came in with Python 2.0. */

		prcode(fp,
"#if PY_VERSION_HEX >= 0x02000000\n"
			);

		if (cvd -> combseqmeth & SPM_SEQ_ICONCAT)
			prcode(fp,
"	sip__iadd__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combseqmeth & SPM_SEQ_IREPEAT)
			prcode(fp,
"	sip__imul__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		prcode(fp,
"#endif\n"
"};\n"
			);
	}

	/* Define the table for mapping methods if any. */

	if (cvd -> combmapmeth || cvd -> combmapmethx)
	{
		prcode(fp,
"\n"
"static PyMappingMethods sipMapMethods_%C = {\n"
			,classVersFQName(cvd));

		if (cvd -> combmapmeth & SPM_MAP_LEN)
			prcode(fp,
"	sip__len__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combmapmeth & SPM_MAP_GETITEM)
			prcode(fp,
"	sip__getitem__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combmapmeth & SPM_MAP_SETITEM)
			prcode(fp,
"	sip__setitem__%C,\n"
				,classVersFQName(cvd));
		else
			prcode(fp,
"	0\n"
				);

		prcode(fp,
"};\n"
			);
	}

	prcode(fp,
"\n"
"static PyTypeObject sipType_%C = {\n"
"	PyObject_HEAD_INIT(&PyType_Type)\n"
"	0,\n"
"	%N,\n"
"	sizeof (sipThisType),\n"
"	0,\n"
"	(destructor)sipDealloc_%C,\n"
"	0,\n"
"	0,\n"
"	0,\n"
		,classVersFQName(cvd)
		,cvd -> common -> iff -> name
		,classVersFQName(cvd));

	if (cvd -> combmeth & SPM_METH_CMP)
		prcode(fp,
"	sip__cmp__%C,\n"
			,classVersFQName(cvd));
	else
		prcode(fp,
"	0,\n"
			);

	if (cvd -> combmeth & SPM_METH_REPR)
		prcode(fp,
"	sip__repr__%C,\n"
			,classVersFQName(cvd));
	else
		prcode(fp,
"	0,\n"
			);

	if (cvd -> combnummeth || cvd -> combnummethx)
		prcode(fp,
"	&sipNumMethods_%C,\n"
			,classVersFQName(cvd));
	else
		prcode(fp,
"	0,\n"
			);

	if (cvd -> combseqmeth || cvd -> combseqmethx)
		prcode(fp,
"	&sipSeqMethods_%C,\n"
			,classVersFQName(cvd));
	else
		prcode(fp,
"	0,\n"
			);

	if (cvd -> combmapmeth || cvd -> combmapmethx)
		prcode(fp,
"	&sipMapMethods_%C,\n"
			,classVersFQName(cvd));
	else
		prcode(fp,
"	0,\n"
			);

	if (cvd -> combmeth & SPM_METH_HASH)
		prcode(fp,
"	sip__hash__%C,\n"
			,classVersFQName(cvd));
	else
		prcode(fp,
"	0,\n"
			);

	if (cvd -> combmeth & SPM_METH_CALL)
		prcode(fp,
"	sip__call__%C,\n"
			,classVersFQName(cvd));
	else
		prcode(fp,
"	0,\n"
			);

	if (cvd -> combmeth & SPM_METH_STR)
		prcode(fp,
"	sip__str__%C,\n"
			,classVersFQName(cvd));
	else
		prcode(fp,
"	0,\n"
			);

	prcode(fp,
"	0,\n"
"	0,\n"
"	0,\n"
"	Py_TPFLAGS_DEFAULT,\n"
"	0,\n"
"	0,\n"
"	0,\n"
		);

	if (cvd -> combmeth == SPM_METH_RICH)
		prcode(fp,
"#if PY_VERSION_HEX >= 0x02010000\n"
"	sip__richcompare__%C,\n"
"#endif\n"
			,classVersFQName(cvd));

	prcode(fp,
"};\n"
		);

	/* Generate any local class code. */

	generateCppCodeBlock(cvd -> cppcode,cvd,fp);

	generateClassFunctions(pt,cvd,fp);

	/* Generate the variable handlers. */

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		if (vd -> ecvd != cvd || isStaticVar(vd))
			continue;

		generateVariableHandler(pt,vd,fp);
	}

	/* Generate the attribute table. */

	prcode(fp,
"\n"
"PyMethodDef sipClassAttrTab_%C[] = {\n"
		,classVersFQName(cvd));

	/*
	 * Only provide an entry point if there is at least one overload that
	 * is defined in this class and is a non-abstract function or slot.  We
	 * allow private (even though we don't actually generate code) because
	 * we need to intercept the name before it reaches a more public
	 * version further up the class hierachy.
	 */

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;
		versionOrList *vol;

		vol = NULL;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			if (od -> common != vl -> m ||
			    !versionsOverlap(&od -> version,&cvd -> version))
				continue;

			if (!isSignal(od) && !isAbstract(od) && (isProtected(od) || vl -> cv == cvd))
				orVersionQual(pt,&vol,&od -> version);
		}

		if (vol != NULL)
		{
			generateOredVersionStart(vol,fp);

			prcode(fp,
"	{%N, sipDo_%C_%s, METH_VARARGS, NULL},\n"
				,vl -> m -> pyname,classVersFQName(cvd),vl -> m -> pyname -> text);

			generateOredVersionEnd(vol,fp);
			freeVersionOrList(vol);
		}
	}

	prcode(fp,
"	{NULL}\n"
"};\n"
		);

	/* Generate the variable tables. */

	noIntro = TRUE;

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
	{
		if (vd -> ecvd != cvd || isStaticVar(vd))
			continue;

		if (noIntro)
		{
			noIntro = FALSE;

			prcode(fp,
"\n"
"PyMethodDef sipClassVarTab_%C[] = {\n"
				,classVersFQName(cvd));
		}

		generateVersionStart(&vd -> version,fp);

		prcode(fp,
"	{%N, sipGetSetVar_%C, 0, NULL},\n"
			,vd -> name,vd -> fqname);

		generateVersionEnd(&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"	{NULL}\n"
"};\n"
"\n"
"PyMethodDef *sipClassVarHierTab_%C[] = {\n"
			,classVersFQName(cvd));

		for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
			for (vd = pt -> vars; vd != NULL; vd = vd -> next)
			{
				if (vd -> ecvd != cvl -> cv || isStaticVar(vd))
					continue;

				prcode(fp,
"	sipClassVarTab_%C,\n"
					,classVersFQName(cvl -> cv));

				break;
			}

		prcode(fp,
"	NULL\n"
"};\n"
			);
	}

	generateConvertToDefinitions(NULL,cvd,fp);

	generateVersionEnd(&cvd -> version,fp);
}


/*
 * Generate the "to type" convertor definitions.
 */

static void generateConvertToDefinitions(mappedTypeDef *mtd,classVersDef *cvd,
					 FILE *fp)
{
	codeBlock *canconvtocode, *convtocode;
	ifaceFileDef *iff;
	argDef type;

	if (cvd != NULL)
	{
		canconvtocode = cvd -> canconvtocode;
		convtocode = cvd -> convtocode;

		iff = cvd -> common -> iff;

		type.atype = class_type;
		type.u.cvd = cvd;
	}
	else
	{
		canconvtocode = mtd -> canconvtocode;
		convtocode = mtd -> convtocode;

		iff = mtd -> iff;

		type.atype = mapped_type;
		type.u.mtd = mtd;
	}

	type.argflags = 0;
	type.nrderefs = 0;
	type.defval = NULL;

	/* Generate the type convertors. */

	prcode(fp,
"\n"
"int sipCanConvertTo_%T(PyObject *sipPy)\n"
"{\n"
		,&type);

	if (canconvtocode != NULL)
		generateCppCodeBlock(canconvtocode,cvd,fp);
	else
		prcode(fp,
"	return sipIsSubClassInstance(sipPy,sipClass_%T);\n"
			,&type);

	prcode(fp,
"}\n"
		);

	prcode(fp,
"\n"
"%s sipConvertTo_%T(PyObject *sipPy,%B **sipCppPtr,int sipWillDeref,int *sipIsErr)\n"
"{\n"
"	if (*sipIsErr || sipPy == NULL)\n"
"		return%s;\n"
"\n"
		,(convtocode != NULL ? "int" : "void"),&type,&type
		,(convtocode != NULL ? " 0" : ""));

	if (convtocode != NULL)
		generateCppCodeBlock(convtocode,cvd,fp);
	else
		prcode(fp,
"	if (sipPy == Py_None)\n"
"	{\n"
"		sipCheckNone(sipWillDeref,sipIsErr,%N);\n"
"		*sipCppPtr = NULL;\n"
"\n"
"		return;\n"
"	}\n"
"\n"
"	*sipCppPtr = (%B *)sipConvertToCpp(sipPy,sipClass_%T,sipIsErr);\n"
			,iff -> name
			,&type,&type);

	prcode(fp,
"}\n"
		);

	prcode(fp,
"\n"
"%B *sipForceConvertTo_%T(PyObject *valobj,int *iserrp)\n"
"{\n"
"	if (*iserrp || valobj == NULL || valobj == Py_None)\n"
"		return NULL;\n"
"\n"
"	if (sipCanConvertTo_%T(valobj))\n"
"	{\n"
"		%B *val;\n"
"\n"
"		sipConvertTo_%T(valobj,&val,0,iserrp);\n"
"\n"
"		return val;\n"
"	}\n"
"\n"
"	sipBadClass(%N);\n"
"\n"
"	*iserrp = 1;\n"
"\n"
"	return NULL;\n"
"}\n"
		,&type,&type
		,&type
		,&type
		,&type
		,iff -> name);
}


/*
 * Generate the start of the declaration for a special method handler.
 */

static void generateSpecialMethodStart(classVersDef *cvd,specialPyMethod *spm,FILE *fp)
{
	if (spm -> pyvers != 0)
	{
		char buf[20];

		sprintf(buf,"%08x",spm -> pyvers);

		prcode(fp,
"#if PY_VERSION_HEX >= 0x%s\n"
			,buf);
	}

	prcode(fp,
"static %s sip%s%C(",spm -> rettype,spm -> name,classVersFQName(cvd));

	if (spm -> nrargs >= 0)
	{
		int i;

		for (i = 0; i < spm -> nrargs; ++i)
		{
			if (i > 0)
				prcode(fp,",");

			prcode(fp,"PyObject *a%d",i);
		}
	}
	else
	{
		char *args;

		switch (spm -> id)
		{
		case SPM_NUM_COERCE:
			args = "PyObject **a0,PyObject **a1";
			break;

		case SPM_SEQ_REPEAT:
		case SPM_SEQ_IREPEAT:
		case SPM_SEQ_GETITEM:
			args = "PyObject *a0,int a1";
			break;

		case SPM_SEQ_SETITEM:
			args = "PyObject *a0,int a1,PyObject *a2";
			break;

		case SPM_SEQ_GETSLICE:
			args = "PyObject *a0,int a1,int a2";
			break;

		case SPM_SEQ_SETSLICE:
			args = "PyObject *a0,int a1,int a2,PyObject *a3";
			break;

		case SPM_METH_RICH:
			args = "PyObject *a0,PyObject *a1,int a2";
			break;
		}

		prcode(fp,"%s",args);
	}

	prcode(fp,")");
}


/*
 * Generate the end of the declaration for a special method handler.
 */

static void generateSpecialMethodEnd(specialPyMethod *spm,FILE *fp)
{
	if (spm -> pyvers != 0)
		prcode(fp,
"#endif\n"
			);
}


/*
 * Generate a variable handler.
 */

static void generateVariableHandler(sipSpec *pt,varDef *vd,FILE *fp)
{
	char *deref;
	argType atype = vd -> type.atype;

	prcode(fp,
"\n"
		);

	generateVersionStart(&vd -> version,fp);

	prcode(fp,
"static PyObject *sipGetSetVar_%C(PyObject *sipThisObj,PyObject *valobj)\n"
"{\n"
		,vd -> fqname);

	if (atype == class_type || atype == mapped_type)
		prcode(fp,
"	int iserr = 0;\n"
			);

	prcode(fp,
"	");

	generateValueType(&vd -> type,fp);

	prcode(fp,"val;\n"
"	%S *ptr;\n"
"\n"
"	if ((ptr = (%S *)sipGetCppPtr((sipThisType *)sipThisObj,sipClass_%C)) == NULL)\n"
"		return NULL;\n"
"\n"
"	if (valobj == NULL)\n"
"	{\n"
		,classVersFQName(vd -> ecvd)
		,classVersFQName(vd -> ecvd)
		,classVersFQName(vd -> ecvd));

	/* Generate the get handler part. */

	prcode(fp,
"		val = %sptr -> %s;\n"
"\n"
					,(((atype == class_type || atype == mapped_type) && vd -> type.nrderefs == 0) ? "&" : ""),vd -> name -> text);

	switch (atype)
	{
	case mapped_type:
		prcode(fp,
"		valobj = sipConvertFrom_%T(val);\n"
			,&vd -> type);

		break;

	case class_type:
		generateVarClassConversion(pt,vd,fp);
		break;

	case bool_type:
		prcode(fp,
"		valobj = sipConvertFromBool((int)val);\n"
			);

		break;

	case ustring_type:
	case string_type:
		if (vd -> type.nrderefs == 0)
			prcode(fp,
"		valobj = PyString_FromStringAndSize(%s&val,1);\n"
				,(atype == ustring_type) ? "(char *)" : "");
		else
			prcode(fp,
"		valobj = PyString_FromString(val);\n"
				);

		break;

	case float_type:
		prcode(fp,
"		valobj = PyFloat_FromDouble((double)val);\n"
			);
		break;

	case double_type:
		prcode(fp,
"		valobj = PyFloat_FromDouble(val);\n"
			);
		break;

	case enum_type:
	case ushort_type:
	case short_type:
	case uint_type:
	case cint_type:
	case int_type:
	case ulong_type:
		prcode(fp,
"		valobj = PyInt_FromLong((long)val);\n"
			);
		break;

	case long_type:
		prcode(fp,
"		valobj = PyInt_FromLong(val);\n"
			);
		break;

	case struct_type:
	case voidptr_type:
		prcode(fp,
"		valobj = sipConvertFromVoidPtr(val);\n"
			);
		break;
	}

	prcode(fp,
"\n"
"		return valobj;\n"
"	}\n"
		);

	/* Generate the set handler part. */

	prcode(fp,
"\n"
		);

	generateObjToCppConversion(&vd -> type,"val","\t",fp);

	deref = "";

	if (atype == class_type || atype == mapped_type)
	{
		if (vd -> type.nrderefs == 0)
			deref = "*";

		prcode(fp,
"\n"
"	if (iserr)\n"
			);
	}
	else if (atype == ustring_type || atype == string_type)
	{
		if (vd -> type.nrderefs == 0)
			deref = "*";

		prcode(fp,
"\n"
"	if (val == NULL)\n"
			);
	}
	else
		prcode(fp,
"\n"
"	if (PyErr_Occurred() != NULL)\n"
			);

	prcode(fp,
"	{\n"
"		sipBadSetType(%N,%N);\n"
"		return NULL;\n"
"	}\n"
"\n"
"	ptr -> %s = %sval;\n"
"\n"
"	Py_INCREF(Py_None);\n"
"	return Py_None;\n"
"}\n"
		,vd -> ecvd -> common -> iff -> name,vd -> name
		,vd -> name -> text,deref);

	generateVersionEnd(&vd -> version,fp);
}


/*
 * Generate an object conversion fragment.
 */

static void generateObjConversion(varDef *vd,FILE *fp)
{
	prcode(fp,
"			");

	generateValueType(&vd -> type,fp);

	prcode(fp,"vc;\n"
"\n"
		);

	generateObjToCppConversion(&vd -> type,"vc","\t\t\t",fp);

	prcode(fp,
"\n"
"			if (val == vc)\n"
		);
}


/*
 * Generate an variable class conversion fragment.
 */

static void generateVarClassConversion(sipSpec *pt,varDef *vd,FILE *fp)
{
	classVersDef *cvd = vd -> type.u.cvd;

	prcode(fp,
"		valobj = sipMapCppToSelf(val,");

	if (cvd -> convtosubcode == NULL)
		prcode(fp,"sipClass_%C",classVersFQName(cvd));
	else
		prcode(fp,"sipSubClass_%C(val)",classVersFQName(cvd));

	prcode(fp,");\n"
		);
}


/*
 * Generate the declaration of a variable that is initialised from a Python
 * object.
 */

static void generateObjToCppConversion(argDef *ad,char *name,char *indent,FILE *fp)
{
	if (ad -> atype == class_type)
	{
		prcode(fp,
"%s%s = sipForceConvertTo_%C(%sobj,&iserr);\n"
			,indent,name,classVersFQName(ad -> u.cvd),name);
	}
	else if (ad -> atype == mapped_type)
	{
		prcode(fp,
"%s%s = sipForceConvertTo_%C(%sobj,&iserr);\n"
			,indent,name,ad -> u.mtd -> iff -> fqname,name);
	}
	else if (ad -> atype == enum_type)
		prcode(fp,
"%s%s = (%S)PyInt_AsLong(%sobj);\n"
			,indent,name,ad -> u.ed -> fqname,name);
	else
	{
		char *fmt;

		switch (ad -> atype)
		{
		case ustring_type:
			if (ad -> nrderefs == 0)
				fmt = "%s%s = *(unsigned char *)PyString_AsString(%sobj);\n";
			else
				fmt = "%s%s = (unsigned char *)PyString_AsString(%sobj);\n";
			break;

		case string_type:
			if (ad -> nrderefs == 0)
				fmt = "%s%s = *PyString_AsString(%sobj);\n";
			else
				fmt = "%s%s = PyString_AsString(%sobj);\n";
			break;

		case float_type:
			fmt = "%s%s = (float)PyFloat_AsDouble(%sobj);\n";
			break;

		case double_type:
			fmt = "%s%s = PyFloat_AsDouble(%sobj);\n";
			break;

		case bool_type:
			fmt = "%s%s = (bool)PyInt_AsLong(%sobj);\n";
			break;

		case ushort_type:
			fmt = "%s%s = (unsigned short)PyInt_AsLong(%sobj);\n";
			break;

		case short_type:
			fmt = "%s%s = (short)PyInt_AsLong(%sobj);\n";
			break;

		case uint_type:
			fmt = "%s%s = (unsigned)PyInt_AsLong(%sobj);\n";
			break;

		case cint_type:
		case int_type:
			fmt = "%s%s = (int)PyInt_AsLong(%sobj);\n";
			break;

		case ulong_type:
			fmt = "%s%s = (unsigned long)PyInt_AsLong(%sobj);\n";
			break;

		case long_type:
			fmt = "%s%s = PyInt_AsLong(%sobj);\n";
			break;

		case struct_type:
			prcode(fp,
"%s%s = (struct %s *)sipConvertToVoidPtr(%sobj);\n"
				,indent,name,ad -> u.sname,name);
			break;

		case voidptr_type:
			fmt = "%s%s = sipConvertToVoidPtr(%sobj);\n";
			break;
		}

		if (fmt != NULL)
			prcode(fp,fmt,indent,name,name);
	}
}


/*
 * Generate the member functions for a class.
 */

static void generateClassFunctions(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	visibleList *vl;
	classList *cl;
	spmDef *sd;

	/* Any complex code. */

	if (isComplex(cvd))
		if (cannotCreate(cvd))
			generateVirtualHandlers(pt,cvd,fp);
		else
			generateComplexCode(pt,cvd,fp);

	/* The member functions. */

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		generateFunction(pt,vl -> m,vl -> cv -> overs,cvd,vl -> cv,TRUE,fp);
		generateFunction(pt,vl -> m,vl -> cv -> overs,cvd,vl -> cv,FALSE,fp);
	}

	/* The sub-class convertor if there is one. */

	if (cvd -> convtosubcode != NULL)
	{
		prcode(fp,
"\n"
"// Convert to a sub-class if possible.\n"
"\n"
"PyObject *sipSubClass_%C(const %C *sipCpp)\n"
"{\n"
"	PyObject *sipClass;\n"
			,classVersFQName(cvd),classVersFQName(cvd));

		generateCppCodeBlock(cvd -> convtosubcode,cvd,fp);

		prcode(fp,
"\n"
"	return sipClass;\n"
"}\n"
			);
	}

	/* The cast function. */

	prcode(fp,
"\n"
"// Cast a pointer to a type somewhere in its superclass hierachy.\n"
"\n"
"const void *sipCast_%C(const void *ptr,PyObject *targetClass)\n"
"{\n"
		,classVersFQName(cvd));

	if (cvd -> supers != NULL)
		prcode(fp,
"	const void *res;\n"
"\n"
			);

	prcode(fp,
"	if (targetClass == sipClass_%C)\n"
"		return ptr;\n"
		,classVersFQName(cvd));

	for (cl = cvd -> supers; cl != NULL; cl = cl -> next)
	{
		scopedNameDef *sname = cl -> c -> iff -> fqname;

		prcode(fp,
"\n"
"	if ((res = sipCast_%C((%S *)(%S *)ptr,targetClass)) != NULL)\n"
"		return res;\n"
			,sname,sname,classVersFQName(cvd));
	}

	prcode(fp,
"\n"
"	return NULL;\n"
"}\n"
		);

	/* The dealloc function. */

	prcode(fp,
"\n"
"static void sipDealloc_%C(sipThisType *sipThis)\n"
"{\n"
		,classVersFQName(cvd));

	if (cvd -> dtorcode != NULL)
		generateCppCodeBlock(cvd -> dtorcode,cvd,fp);

	/* Generate the right dtor call - if there is one. */

	if (!cannotCreate(cvd) && (isComplex(cvd) || !noPublicDtor(cvd)))
	{
		prcode(fp,
"	if (sipThis -> u.cppPtr != NULL)\n"
"	{\n"
			);

		/* Disable the virtual handlers. */

		if (isComplex(cvd))
			prcode(fp,
"		if (!sipIsSimple(sipThis))\n"
"			((sip%C *)sipThis -> u.cppPtr) -> sipPyThis = NULL;\n"
"\n"
				,classVersFQName(cvd));

		prcode(fp,
"		if (sipIsPyOwned(sipThis))\n"
			);

		if (isComplex(cvd))
		{
			if (noPublicDtor(cvd))
				prcode(fp,
"			if (!sipIsSimple(sipThis))\n"
"				delete (sip%C *)sipThis -> u.cppPtr;\n"
					,classVersFQName(cvd));
			else
				prcode(fp,
"			if (sipIsSimple(sipThis))\n"
"				delete (%S *)sipThis -> u.cppPtr;\n"
"			else\n"
"				delete (sip%C *)sipThis -> u.cppPtr;\n"
					,classVersFQName(cvd)
					,classVersFQName(cvd));
		}
		else
			prcode(fp,
"			delete (%S *)sipThis -> u.cppPtr;\n"
				,classVersFQName(cvd));

		prcode(fp,
"	}\n"
"\n"
			);
	}

	prcode(fp,
"	sipDeleteThis(sipThis);\n"
"}\n"
		);

	/* The special method implementations. */

	for (sd = cvd -> spms; sd != NULL; sd = sd -> next)
	{
		generateSpecialMethodStart(cvd,sd -> spm,fp);

		prcode(fp,"\n"
"{\n"
			);

		generateCppCodeBlock(sd -> code,cvd,fp);

		prcode(fp,
"}\n"
			);

		generateSpecialMethodEnd(sd -> spm,fp);
	}

	/* The Python constructor. */

	generatePythonConstructor(pt,cvd,fp);
}


/*
 * Generate the code specific to complex classes.
 */

static void generateComplexCode(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro, nrVirts;
	visibleList *vl;
	virtVersDef *vvd;
	ctorDef *ct;

	nrVirts = countVirtuals(cvd);

	/* Generate the wrapper class constructors. */

	for (ct = cvd -> ctors; ct != NULL; ct = ct -> next)
	{
		char *prefix;
		int a;

		if (isPrivateCtor(ct))
			continue;

		prcode(fp,
"\n"
			);

		generateVersionStart(&ct -> version,fp);

		prcode(fp,
"sip%C::sip%C(",classVersFQName(cvd),classVersFQName(cvd));

		generateArgs(&ct -> args,Definition,fp);

		prcode(fp,"): %S(",classVersFQName(cvd));

		prefix = "";

		for (a = 0; a < ct -> args.nrArgs; ++a)
		{
			prcode(fp,"%sa%d",prefix,a);
			prefix = ",";
		}

		prcode(fp,")\n"
"{\n"
"	sipCommonCtor(%s,%d);\n"
"}\n"
			,(nrVirts > 0 ? "sipPyMethods" : "NULL"),nrVirts);

		generateVersionEnd(&ct -> version,fp);
	}

	/* The destructor. */

	prcode(fp,
"\n"
"sip%C::~sip%C()\n"
"{\n"
"	sipCommonDtor(sipPyThis);\n"
"}\n"
		,classVersFQName(cvd),classVersFQName(cvd));

	/* Generate the virtual catchers. */
 
	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		int virtNr;
		virtOverDef *vod;
 
		virtNr = 0;
 
		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
			if (!isPrivate(vod -> o))
				generateVirtualCatcher(cvd,virtNr++,vod,fp);
	}

	/* Generate the virtual handlers. */

	generateVirtualHandlers(pt,cvd,fp);

	/* Generate the wrapper around each protected member function. */

	generateProtectedDefinitions(pt,cvd,fp);

	/* Generate the emitter functions. */

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			if (od -> common != vl -> m ||
			    !versionsOverlap(&od -> version,&cvd -> version))
				continue;

			if (isSignal(od))
			{
				generateEmitter(pt,cvd,vl,fp);
				break;
			}
		}
	}

	/* Generate the table of signals to support fan-outs. */

	noIntro = TRUE;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			if (od -> common != vl -> m ||
			    !versionsOverlap(&od -> version,&cvd -> version))
				continue;

			if (isSignal(od))
			{
				if (noIntro)
				{
					setHasSigSlots(cvd);

					prcode(fp,
"\n"
"static sipQtSignal sipSignals_%C[] = {\n"
						,classVersFQName(cvd));

					noIntro = FALSE;
				}

				prcode(fp,
"	{%N, %C_emit_%s},\n"
					,vl -> m -> pyname,classVersFQName(cvd),vl -> m -> pyname -> text);

				break;
			}
		}
	}

	if (!noIntro)
		prcode(fp,
"	{NULL, NULL}\n"
"};\n"
			);
}


/*
 * Generate the protected enums for a class.
 */

static void generateProtectedEnums(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	enumDef *ed;

	for (ed = pt -> enums; ed != NULL; ed = ed -> next)
	{
		char *eol;
		enumValueDef *evd;

		if (ed -> ecvd != cvd || !isProtectedEnum(ed))
			continue;

		prcode(fp,
"\n"
			);

		generateVersionStart(&ed -> version,fp);

		prcode(fp,
"	// Expose this protected enum.\n"
"\n"
"	enum");

		if (ed -> fqname != NULL)
			prcode(fp," sip%s",scopedNameTail(ed -> fqname));

		prcode(fp," {");

		eol = "\n";

		for (evd = ed -> values; evd != NULL; evd = evd -> next)
		{
			prcode(fp,"%s"
"		%s = %S::%s",eol,evd -> name -> text,classVersFQName(ed -> ecvd),evd -> name -> text);

			eol = ",\n";
		}

		prcode(fp,"\n"
"	};\n"
			);

		generateVersionEnd(&ed -> version,fp);
	}
}


/*
 * Generate the catcher for a virtual function.
 */

static void generateVirtualCatcher(classVersDef *cvd,int virtNr,virtOverDef *vod,FILE *fp)
{
	scopedNameDef *ncname = classVersFQName(vod -> nearc);
	scopedNameDef *fcname = classVersFQName(vod -> farc);
	char *pname = vod -> o -> common -> pyname -> text;
	char *mname = vod -> o -> cppname;
	char *cast;
	nameDef *pmname = vod -> o -> common -> pyname;
	overDef *od = vod -> o;

	prcode(fp,
"\n");

	generateVersionStart(&od -> version,fp);

	generateResultType(od -> result,fp);

	prcode(fp," sip%C::%s(",classVersFQName(cvd),mname);
	generateArgs(&od -> args,Definition,fp);
	prcode(fp,")%s\n"
"{\n"
"	int relLock;\n"
"\n"
		,(isConst(od) ? " const" : ""));

	cast = (isConst(od) ? "(sipMethodCache *)" : "");

	if (isAbstract(od))
	{
		prcode(fp,
"	if (sipIsPyMethod(%s&sipPyMethods[%d],sipPyThis,%N,%N,&relLock))\n"
"		%ssip%C::sipVH_%s(&sipPyMethods[%d],sipPyThis,relLock",cast,virtNr,cvd -> common -> iff -> name,pmname,(od -> result != NULL ? "return " : ""),fcname,mname,virtNr);
 
		if (od -> args.nrArgs > 0)
			prcode(fp,",");
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,");\n"
			);
	}
	else if (od -> result == NULL)
	{
		prcode(fp,
"	if (sipIsPyMethod(%s&sipPyMethods[%d],sipPyThis,NULL,%N,&relLock))\n"
"		sip%C::sipVH_%s(&sipPyMethods[%d],sipPyThis,relLock",cast,virtNr,pmname,fcname,mname,virtNr);
 
		if (od -> args.nrArgs > 0)
			prcode(fp,",");
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,");\n"
"	else\n"
"		%S::%s(",ncname,mname);
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,");\n"
			);
	}
	else
	{
		prcode(fp,
"	return sipIsPyMethod(%s&sipPyMethods[%d],sipPyThis,NULL,%N,&relLock) ?\n"
"		sip%C::sipVH_%s(&sipPyMethods[%d],sipPyThis,relLock",cast,virtNr,pmname,fcname,mname,virtNr);

		if (od -> args.nrArgs > 0)
			prcode(fp,",");
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,") :\n"
"		%S::%s(",ncname,mname);
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,");\n"
			);
	}
 
	prcode(fp,
"}\n"
		);

	generateVersionEnd(&od -> version,fp);
}


/*
 * Generate the emitter function for a signal.
 */

static void generateEmitter(sipSpec *pt,classVersDef *cvd,visibleList *vl,FILE *fp)
{
	char *pname = vl -> m -> pyname -> text;
	overDef *od;

	prcode(fp,
"\n"
"int sip%C::sipEmit_%s(PyObject *sipArgs)\n"
"{\n"
"	int sipArgsParsed = 0;\n"
		,classVersFQName(cvd),pname);

	for (od = vl -> cv -> overs; od != NULL; od = od -> next)
	{
		if (od -> common != vl -> m || !isSignal(od))
			continue;

		/*
		 * Generate the code that parses the args and emits the
		 * appropriate overloaded signal.
		 */

		prcode(fp,
"\n"
			);

		generateVersionStart(&od -> version,fp);

		prcode(fp,
"	{\n"
			);

		generateArgParser(&od -> args,TRUE,FALSE,fp);

		prcode(fp,
"		{\n"
			);

		generateArgConvertors(pt,&od -> args,FALSE,FALSE,"-1",fp);

		prcode(fp,
"			emit %s("
			,od -> cppname);

		generateArgs(&od -> args,Call,fp);

		prcode(fp,");\n"
"\n"
			);

		generateCallTidyUp(&od -> args,fp);

		prcode(fp,
"			return 0;\n"
"		}\n"
"	}\n"
			);

		generateVersionEnd(&od -> version,fp);
	}

	prcode(fp,
"\n"
"	sipNoMethod(sipArgsParsed,%N,%N);\n"
"\n"
"	return -1;\n"
"}\n"
"\n"
"static int %C_emit_%s(sipThisType *w,PyObject *sipArgs)\n"
"{\n"
"	sip%C *ptr;\n"
"\n"
"	if ((ptr = (sip%C *)sipGetComplexCppPtr(w)) == NULL)\n"
"		return -1;\n"
"\n"
"	return ptr -> sipEmit_%s(sipArgs);\n"
"}\n"
		,cvd -> common -> iff -> name,vl -> m -> pyname
		,classVersFQName(cvd),pname
		,classVersFQName(cvd)
		,classVersFQName(cvd)
		,pname);
}


/*
 * Generate the declarations of the protected wrapper functions for a class.
 */

static void generateProtectedDeclarations(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	visibleList *vl;

	noIntro = TRUE;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			if (od -> common != vl -> m || !isProtected(od) ||
			    !versionsOverlap(&od -> version,&cvd -> version) ||
			    isAbstract(od))
				continue;

			if (noIntro)
			{
				prcode(fp,
"\n"
"	// There is a public member function for every protected member\n"
"	// function visible from this class.\n"
"\n"
					);

				noIntro = FALSE;
			}

			generateVersionStart(&od -> version,fp);

			prcode(fp,
"	");

			generateResultType(od -> result,fp);

			prcode(fp," sipProtect_%s(",od -> cppname);
			generateArgs(&od -> args,Declaration,fp);
			prcode(fp,");\n"
				);

			generateVersionEnd(&od -> version,fp);
		}
	}
}


/*
 * Generate the definitions of the protected wrapper functions for a class.
 */

static void generateProtectedDefinitions(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	visibleList *vl;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			char *mname = od -> cppname;
			int a;

			if (od -> common != vl -> m || !isProtected(od) ||
			    !versionsOverlap(&od -> version,&cvd -> version) ||
			    isAbstract(od))
				continue;

			prcode(fp,
"\n"
				);

			generateVersionStart(&od -> version,fp);

			generateResultType(od -> result,fp);

			prcode(fp,
" sip%C::sipProtect_%s(",classVersFQName(cvd),mname);
			generateArgs(&od -> args,Definition,fp);
			prcode(fp,")\n"
"{\n"
				);

			if (od -> result != NULL)
				prcode(fp,
"	return ");
			else
				prcode(fp,
"	");

			prcode(fp,"%S::%s(",classVersFQName(vl -> cv),mname);

			for (a = 0; a < od -> args.nrArgs; ++a)
				prcode(fp,"%sa%d",(a == 0 ? "" : ","),a);

			prcode(fp,");\n"
"}\n"
				);

			generateVersionEnd(&od -> version,fp);
		}
	}
}


/*
 * Generate the argument list of a simple call.
 */

static void generateArgCallList(funcArgs *fa,FILE *fp)
{
	int a;

	for (a = 0; a < fa -> nrArgs; ++a)
		prcode(fp,"%sa%d",(a == 0 ? "" : ","),a);
}


/*
 * Generate the virtual handlers for a class version.
 */

static void generateVirtualHandlers(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	virtVersDef *vvd;

	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		virtOverDef *vod;
 
		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
			if (!isPrivate(vod -> o) && vod -> farc == cvd)
				generateVirtualHandler(pt,vod,fp);
	}
}


/*
 * Generate the function that does most of the work to handle a particular
 * virtual function.
 */

static void generateVirtualHandler(sipSpec *pt,virtOverDef *vod,FILE *fp)
{
	char *indir = "";
	scopedNameDef *cname = classVersFQName(vod -> farc);
	argDef *ad = vod -> o -> result;
	overDef *od = vod -> o;

	prcode(fp,
"\n"
"// The common handler for all classes that inherit this virtual member\n"
"// function.\n"
"\n"
		);

	generateVersionStart(&od -> version,fp);

	generateResultType(ad,fp);

	prcode(fp," sip%C::sipVH_%s(const sipMethodCache *pymc,sipThisType *sipThis,int sipRelLock",cname,od -> cppname);

	if (od -> args.nrArgs > 0)
		prcode(fp,",");

	generateArgs(&od -> args,Definition,fp);
	prcode(fp,")\n"
"{\n"
		);

	if (od -> virtcode != NULL)
	{
		generateCppCodeBlock(od -> virtcode,vod -> farc,fp);
		prcode(fp,
"}\n"
			);

		generateVersionEnd(&od -> version,fp);

		return;
	}

	if (ad != NULL)
	{
		prcode(fp,
"	");

		generateValueType(ad,fp);

		prcode(fp,"res;\n"
			);
	}

	prcode(fp,
"	PyObject *resobj;\n"
		);

	generateTupleBuilder(pt,&od -> args,TRUE,fp);

	/* Generate the call the the Python member function. */

	prcode(fp,
"\n"
"	if (sipArgs == NULL)\n"
"		goto reportError;\n"
"\n"
"	resobj = sipEvalMethod(&pymc -> pyMethod,sipArgs);\n"
"\n"
"	Py_DECREF(sipArgs);\n"
"\n"
"	if (resobj != NULL)\n"
"	{\n"
		);

	/* Handle void functions. */

	if (ad == NULL)
		prcode(fp,
"		Py_DECREF(resobj);\n"
"\n"
"		if (resobj == Py_None)\n"
"			goto releaseLock;\n"
			);
	else if (ad -> atype == class_type || ad -> atype == mapped_type)
	{
		if (ad -> nrderefs == 0)
			indir = "*";

		prcode(fp,
"		int iserr = 0;\n"
"\n"
			);

		/* Possible memory leak - who deletes res? */

		if (ad -> atype == class_type)
			prcode(fp,
"		res = sipForceConvertTo_%C(resobj,&iserr);\n"
"		sipTransferSelf(resobj,1);\n"
				,classVersFQName(ad -> u.cvd));
		else
			prcode(fp,
"		res = sipForceConvertTo_%C(resobj,&iserr);\n"
				,ad -> u.mtd -> iff -> fqname);


		prcode(fp,
"\n"
"		Py_DECREF(resobj);\n"
"\n"
"		if (!iserr)\n"
"			goto releaseLock;\n"
			);
	}
	else
	{
		/* Convert and check the result. */

		switch (ad -> atype)
		{
		case ustring_type:
			if (ad -> nrderefs == 0)
				prcode(fp,
"		unsigned char *resstr = (unsigned char *)PyString_AsString(resobj);\n"
					);
			else
				prcode(fp,
"		if (resobj == Py_None)\n"
"			res = NULL;\n"
"		else\n"
"			res = (unsigned char *)PyString_AsString(resobj);\n"
					);
			break;

		case string_type:
			if (ad -> nrderefs == 0)
				prcode(fp,
"		char *resstr = PyString_AsString(resobj);\n"
					);
			else
				prcode(fp,
"		if (resobj == Py_None)\n"
"			res = NULL;\n"
"		else\n"
"			res = PyString_AsString(resobj);\n"
					);
			break;

		case ushort_type:
			prcode(fp,
"		res = (unsigned short)PyInt_AsLong(resobj);\n"
				);
			break;

		case short_type:
			prcode(fp,
"		res = (short)PyInt_AsLong(resobj);\n"
				);
			break;

		case uint_type:
			prcode(fp,
"		res = (unsigned)PyInt_AsLong(resobj);\n"
				);
			break;

		case cint_type:
		case int_type:
			prcode(fp,
"		res = (int)PyInt_AsLong(resobj);\n"
				);
			break;

		case ulong_type:
			prcode(fp,
"		res = (unsigned long)PyInt_AsLong(resobj);\n"
				);
			break;

		case long_type:
			prcode(fp,
"		res = PyInt_AsLong(resobj);\n"
				);
			break;

		case struct_type:
			prcode(fp,
"		res = (struct %s *)sipConvertToVoidPtr(resobj);\n"
				,ad -> u.sname);
			break;

		case voidptr_type:
			prcode(fp,
"		res = sipConvertToVoidPtr(resobj);\n"
				);
			break;

		case bool_type:
			prcode(fp,
"		res = (bool)PyInt_AsLong(resobj);\n"
				);
			break;

		case float_type:
			prcode(fp,
"		res = (float)PyFloat_AsDouble(resobj);\n"
				);
			break;

		case double_type:
			prcode(fp,
"		res = PyFloat_AsDouble(resobj);\n"
				);
			break;

		case enum_type:
			prcode(fp,
"		res = (%S)PyInt_AsLong(resobj);\n"
				,ad -> u.ed -> fqname);
			break;
		}

		prcode(fp,
"\n"
"		Py_DECREF(resobj);\n"
"\n"
"		if (PyErr_Occurred() == NULL)\n"
"		{\n"
			);

		if ((ad -> atype == ustring_type || ad -> atype == string_type) && ad -> nrderefs == 0)
			prcode(fp,
"			res = *resstr;\n"
				);

		prcode(fp,
"			goto releaseLock;\n"
"		}\n"
			);
	}

	prcode(fp,
"\n"
"		sipBadVirtualResultType(%N,%N);\n"
"	}\n"
"\n"
"reportError:\n"
"	PyErr_Print();\n"
"\n"
"releaseLock:\n"
"	sipCondReleaseLock(sipRelLock);\n"
		,vod -> farc -> common -> iff -> name,vod -> o -> common -> pyname);

	if (ad != NULL)
		prcode(fp,
"\n"
"	return %sres;\n"
			,indir);

	prcode(fp,
"}\n"
		);

	generateVersionEnd(&od -> version,fp);
}


/*
 * Generate the code to build a tuple of Python arguments.
 */

static void generateTupleBuilder(sipSpec *pt,funcArgs *fa,int isMethod,FILE *fp)
{
	int a, arraylenarg;

	/* Generate the variables. */

	prcode(fp,
"	PyObject *sipArgs;\n"
		);

	for (a = 0; a < fa -> nrArgs; ++a)
	{
		argType atype = fa -> args[a].atype;

		if (atype == class_type || atype == mapped_type ||
		    atype == rxcon_type || atype == rxdis_type)
			prcode(fp,
"	PyObject *a%dobj;\n"
				,a);
	}

	/* Convert each class to a Python object. */

	for (a = 0; a < fa -> nrArgs; ++a)
	{
		argDef *ad = &fa -> args[a];

		if (ad -> atype == class_type || ad -> atype == mapped_type)
		{
			char *ptr = "";

			if (isReference(ad) || ad -> nrderefs == 0)
				ptr = "&";

			if (ad -> atype == mapped_type)
			{
				prcode(fp,
"\n"
"	a%dobj = sipConvertFrom_%T(%sa%d);\n"
					,a,ad,ptr,a);
			}
			else
			{
				classVersDef *cvd = ad -> u.cvd;

				prcode(fp,
"\n"
"	a%dobj = sipMapCppToSelf(%sa%d,",a,ptr,a);

				if (cvd -> convtosubcode == NULL)
					prcode(fp,"sipClass_%C",classVersFQName(cvd));
				else
					prcode(fp,"sipSubClass_%C(%sa%d)",classVersFQName(cvd),ptr,a);

				prcode(fp,");\n"
					);
			}
		}
		else if (ad -> atype == rxcon_type || ad -> atype == rxdis_type)
			prcode(fp,
"\n"
"	a%dobj = sipMapCppToSelf(a%d,sipClass_QObject);\n",
				a,a);
	}

	prcode(fp,
"\n"
"	sipArgs = Py_BuildValue(\"(");

	if (isMethod)
		prcode(fp,"O");

	for (a = 0; a < fa -> nrArgs; ++a)
	{
		char *fmt;

		switch (fa -> args[a].atype)
		{
		case ustring_type:
		case string_type:
			if (fa -> args[a].nrderefs == 0)
				fmt = "c";
			else
				fmt = "s";

			break;

		case enum_type:
		case bool_type:
		case uint_type:
		case cint_type:
		case int_type:
			fmt = "i";
			break;

		case ushort_type:
		case short_type:
			fmt = "h";
			break;

		case ulong_type:
		case long_type:
			fmt = "l";
			break;

		case struct_type:
		case voidptr_type:
			fmt = "v";
			break;

		case float_type:
			fmt = "f";
			break;

		case double_type:
			fmt = "d";
			break;

		case signal_type:
		case slotcon_type:
		case slotdis_type:
			fmt = "s";
			break;

		case rxcon_type:
		case rxdis_type:
		case mapped_type:
		case class_type:
			fmt = "N";
			break;

		case array_type:
		case uarray_type:
			fmt = "s#";
			break;

		case arraysize_type:
		case arrayusize_type:
			fmt = "";
			arraylenarg = a;
			break;
		}

		prcode(fp,fmt);
	}

	prcode(fp,")\"");

	if (isMethod)
		prcode(fp,",sipThis -> sipSelf");

	for (a = 0; a < fa -> nrArgs; ++a)
	{
		int atype = fa -> args[a].atype;

		if (atype != arraysize_type && atype != arrayusize_type)
			prcode(fp,",a%d",a);

		if (atype == array_type || atype == uarray_type)
			prcode(fp,",a%d",arraylenarg);

		if (atype == class_type || atype == mapped_type ||
		    atype == rxcon_type || atype == rxdis_type)
			prcode(fp,"obj");
	}

	prcode(fp,");\n");
}


/*
 * Generate the class interface #include directives required by either a class
 * or a module.
 */

static void generateUsedIncludes(ifaceFileList *iffl,FILE *fp)
{
	prcode(fp,
"\n"
		);

	while (iffl != NULL)
	{
		prcode(fp,
"#include \"sip%s%F.h\"\n"
			,iffl -> iff -> module -> name -> text,iffl -> iff -> fqname);

		iffl = iffl -> next;
	}

	prcode(fp,
"\n"
		);
}


/*
 * Generate the header file for the C++ interface.
 */

static void generateIfaceHeader(sipSpec *pt,ifaceFileDef *iff,char *codeDir)
{
	char *impexp, *wfile;
	char *cmname = iff -> module -> name -> module -> name -> text;
	classVersDef *cvd;
	mappedTypeDef *mtd;
	FILE *fp;

	/* See if we are importing or exporting. */

	impexp = (iff -> module == pt -> module) ? "EXPORT" : "IMPORT";

	/* Create the header file. */

	wfile = createIfaceFileName(codeDir,iff,".h");
	fp = createFile(pt,wfile,"//","C++ interface header file.");

	prcode(fp,
"\n"
"#ifndef _%s%F_h\n"
"#define	_%s%F_h\n"
"\n"
		,cmname,iff -> fqname,cmname,iff -> fqname);

	for (cvd = pt -> classversions; cvd != NULL; cvd = cvd -> next)
	{
		if (cvd -> common -> iff != iff)
			continue;

		generateClassVersHeader(cvd,impexp,pt,fp);
	}

	for (mtd = pt -> mappedtypes; mtd != NULL; mtd = mtd -> next)
	{
		if (mtd -> iff != iff)
			continue;

		generateMappedTypeHeader(mtd,impexp,pt,fp);
	}

	prcode(fp,
"\n"
"#endif\n"
		);

	closeFile(fp);
	free(wfile);
}


/*
 * Generate the C++ header code for a mapped type.
 */

static void generateMappedTypeHeader(mappedTypeDef *mtd,char *impexp,
				     sipSpec *pt,FILE *fp)
{
	generateVersionStart(&mtd -> version,fp);

	generateCppCodeBlock(mtd -> hdrcode,NULL,fp);

	generateConvertToDeclarations(mtd,NULL,impexp,fp);

	prcode(fp,
"extern SIP_%s PyObject *sipConvertFrom_%T Py_PROTO((const %B *));\n"
		,impexp,&mtd -> type,&mtd -> type);

	generateVersionEnd(&mtd -> version,fp);
}


/*
 * Generate the C++ header code for a class.
 */

static void generateClassVersHeader(classVersDef *cvd,char *impexp,sipSpec *pt,
				    FILE *fp)
{
	varDef *vd;

	generateVersionStart(&cvd -> version,fp);

	generateUsedIncludes(cvd -> used,fp);

	generateCppCodeBlock(cvd -> hdrcode,cvd,fp);

	if (isOpaque(cvd))
		prcode(fp,
"\n"
"class %S;\n"
			,classVersFQName(cvd));

	prcode(fp,
"\n"
"extern SIP_%s PyObject *sipClass_%C;\n"
"extern SIP_%s PyMethodDef sipClassAttrTab_%C[];\n"
		,impexp,classVersFQName(cvd)
		,impexp,classVersFQName(cvd));

	for (vd = pt -> vars; vd != NULL; vd = vd -> next)
		if (vd -> ecvd == cvd)
		{
			prcode(fp,
"extern SIP_%s PyMethodDef sipClassVarTab_%C[];\n"
"extern SIP_%s PyMethodDef *sipClassVarHierTab_%C[];\n"
				,impexp,classVersFQName(cvd),impexp,classVersFQName(cvd));

			break;
		}

	/* The cast and constructor functions. */

	prcode(fp,
"\n"
"extern SIP_%s const void *sipCast_%C Py_PROTO((const void *,PyObject *));\n"
"extern SIP_%s PyObject *sipNew_%C Py_PROTO((PyObject *,PyObject *));\n"
			,impexp,classVersFQName(cvd),impexp,classVersFQName(cvd));

	/* The sub-class convertor. */

	if (cvd -> convtosubcode != NULL)
		prcode(fp,
"extern SIP_%s PyObject *sipSubClass_%C Py_PROTO((const %C *));\n"
			,impexp,classVersFQName(cvd),classVersFQName(cvd));

	if (isComplex(cvd))
		if (cannotCreate(cvd))
			generateLimitedWrapperClassDeclaration(pt,cvd,fp);
		else
			generateWrapperClassDeclaration(pt,cvd,fp);

	generateConvertToDeclarations(NULL,cvd,impexp,fp);

	generateVersionEnd(&cvd -> version,fp);
}


/*
 * Generate the "convert to" function declarations.
 */

static void generateConvertToDeclarations(mappedTypeDef *mtd,classVersDef *cvd,
					  char *impexp,FILE *fp)
{
	codeBlock *convtocode;
	argDef type;

	if (cvd != NULL)
	{
		convtocode = cvd -> convtocode;

		type.atype = class_type;
		type.u.cvd = cvd;
	}
	else
	{
		convtocode = mtd -> convtocode;

		type.atype = mapped_type;
		type.u.mtd = mtd;
	}

	type.argflags = 0;
	type.nrderefs = 0;
	type.defval = NULL;

	prcode(fp,
"\n"
"extern SIP_%s int sipCanConvertTo_%T Py_PROTO((PyObject *));\n"
"extern SIP_%s %s sipConvertTo_%T Py_PROTO((PyObject *,%B **,int,int *));\n"
"extern SIP_%s %B *sipForceConvertTo_%T Py_PROTO((PyObject *,int *));\n"
		,impexp,&type
		,impexp,(convtocode != NULL ? "int" : "void"),&type,&type
		,impexp,&type,&type);

}


/*
 * Generate the declaration of a virtual handler.
 */

static void generateVirtualHandlerDeclaration(overDef *od,FILE *fp)
{
	prcode(fp,
"	static ");

	generateResultType(od -> result,fp);

	prcode(fp," sipVH_%s(const sipMethodCache *,sipThisType *,int",od -> cppname);

	if (od -> args.nrArgs > 0)
		prcode(fp,",");

	generateArgs(&od -> args,Declaration,fp);

	prcode(fp,");\n"
		);
}


/*
 * Generate the limited wrapper class declaration that might be needed by a
 * complex class that cannot be created in Python.
 */

static void generateLimitedWrapperClassDeclaration(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	virtVersDef *vvd;

	noIntro = TRUE;

	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		virtOverDef *vod;

		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
			if (!isPrivate(vod -> o) && vod -> farc == cvd)
			{
				if (noIntro)
				{
					noIntro = FALSE;

					prcode(fp,
"\n"
"\n"
"class SIP_%s sip%C\n"
"{\n"
"public:\n"
						,(cvd -> common -> iff -> module == pt -> module) ? "EXPORT" : "IMPORT"
						,classVersFQName(cvd));
				}

				generateVersionStart(&vod -> o -> version,fp);
				generateVirtualHandlerDeclaration(vod -> o,fp);
				generateVersionEnd(&vod -> o -> version,fp);
			}
	}

	if (!noIntro)
		prcode(fp,
"};\n"
			);
}


/*
 * Generate the wrapper class declaration.
 */

static void generateWrapperClassDeclaration(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro, nrVirts;
	ctorDef *ct;
	visibleList *vl;
	virtVersDef *vvd;

	nrVirts = countVirtuals(cvd);

	prcode(fp,
"\n"
"\n"
"class SIP_%s sip%C : public %S\n"
"{\n"
"public:\n"
		,(cvd -> common -> iff -> module == pt -> module) ? "EXPORT" : "IMPORT"
		,classVersFQName(cvd),classVersFQName(cvd));

	/* The constructor declarations. */

	for (ct = cvd -> ctors; ct != NULL; ct = ct -> next)
	{
		if (isPrivateCtor(ct))
			continue;

		generateVersionStart(&ct -> version,fp);

		prcode(fp,
"	sip%C(",classVersFQName(cvd));

		generateArgs(&ct -> args,Declaration,fp);

		prcode(fp,");\n"
			);

		generateVersionEnd(&ct -> version,fp);
	}

	/* The destructor. */

	prcode(fp,
"	~sip%C();\n"
		,classVersFQName(cvd));

	/* The exposure of protected enums. */

	generateProtectedEnums(pt,cvd,fp);

	/* The catcher around each virtual function in the hierachy. */

	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		virtOverDef *vod;
 
		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
		{
			overDef *od = vod -> o;

			if (isPrivate(od))
				continue;
 
			generateVersionStart(&od -> version,fp);

			prcode(fp,
"	");
 
			generateResultType(od -> result,fp);
 
			prcode(fp," %s(",od -> cppname);
			generateArgs(&od -> args,Declaration,fp);
 
			prcode(fp,")%s;\n"
				,(isConst(od) ? " const" : ""));

			if (vod -> farc == cvd)
				generateVirtualHandlerDeclaration(od,fp);
 
			generateVersionEnd(&od -> version,fp);
		}
	}

	/* The wrapper around each protected member function. */

	generateProtectedDeclarations(pt,cvd,fp);

	/* The public wrapper around each signal emitter. */

	noIntro = TRUE;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			if (od -> common != vl -> m || !isSignal(od) ||
			    !versionsOverlap(&od -> version,&cvd -> version))
				continue;

			if (noIntro)
			{
				prcode(fp,
"\n"
"	// There is a public member function for every Qt signal that can be\n"
"	// emitted by this object.  This function is called by Python to emit\n"
"	// the signal.\n"
"\n"
					);

				noIntro = FALSE;
			}

			prcode(fp,
"	int sipEmit_%s(PyObject *sipArgs);\n"
				,vl -> m -> pyname -> text);

			break;
		}
	}

	/* The private declarations. */

	prcode(fp,
"\n"
"	sipThisType *sipPyThis;\n"
		);

	prcode(fp,
"\n"
"private:\n"
"	sip%C(const sip%C &);\n"
"	sip%C &operator = (const sip%C &);\n"
		,classVersFQName(cvd),classVersFQName(cvd)
		,classVersFQName(cvd),classVersFQName(cvd));

	if (nrVirts > 0)
		prcode(fp,
"\n"
"	sipMethodCache sipPyMethods[%d];\n"
			,nrVirts);

	prcode(fp,
"};\n"
		);
}


/*
 * Generate the return type of a member function.
 */

static void generateResultType(argDef *ad,FILE *fp)
{
	if (ad == NULL)
		prcode(fp,"void");
	else
		generateSingleArg(ad,-1,Declaration,fp);
}


/*
 * Generate typed arguments.
 */

static void generateArgs(funcArgs *args,funcArgType ftype,FILE *fp)
{
	char *prefix = NULL;
	int a;

	for (a = 0; a < args -> nrArgs; ++a)
	{
		if (prefix == NULL)
			prefix = ",";
		else
			prcode(fp,prefix);

		generateSingleArg(&args -> args[a],a,ftype,fp);
	}
}


/*
 * Generate the declaration of a variable (excluding the name) to hold a result
 * from a C++ function call.
 */

static void generateValueType(argDef *ad,FILE *fp)
{
	argDef orig;

	orig = *ad;

	if (ad -> nrderefs == 0 && ad -> atype != class_type && ad -> atype != mapped_type)
		resetIsConstArg(ad);

	resetIsReference(ad);
	generateBaseType(ad,fp);
	*ad = orig;

	if (ad -> nrderefs == 0)
	{
		prcode(fp," ");

		if (ad -> atype == class_type || ad -> atype == mapped_type)
			prcode(fp,"*");
	}
}


/*
 * Generate a single argument.
 */

static void generateSingleArg(argDef *ad,int argnr,funcArgType ftype,FILE *fp)
{
	int genType, genName, derefPtr, toType;

	/* Break the type down to individual modifications. */

	genType = FALSE;
	genName = FALSE;
	derefPtr = FALSE;
	toType = FALSE;

	switch (ftype)
	{
	case Call:
		genName = TRUE;
		derefPtr = TRUE;
		toType = TRUE;
		break;

	case Declaration:
		genType = TRUE;
		break;

	case Definition:
		genType = TRUE;
		genName = TRUE;
		break;
	}

	if (genType)
		generateBaseType(ad,fp);

	if (genName)
	{
		if (derefPtr && (isReference(ad) ||
		    ((ad -> atype == class_type || ad -> atype == mapped_type) && ad -> nrderefs == 0)))
			prcode(fp,"*");

		prcode(fp," ");

		if (toType)
			if (ad -> atype == bool_type)
				prcode(fp,"(bool)");
			else if (ad -> atype == enum_type)
				prcode(fp,"(%S)",ad -> u.ed -> fqname);

		prcode(fp,"a%d",argnr);
	}
}


/*
 * Generate a C++ type.
 */

static void generateBaseType(argDef *ad,FILE *fp)
{
	int nr_derefs;

	nr_derefs = ad -> nrderefs;

	if (isConstArg(ad) || ad -> atype == signal_type)
		prcode(fp,"const ");

	switch (ad -> atype)
	{
	case uarray_type:
		nr_derefs = 1;

		/* Drop through. */

	case ustring_type:
		prcode(fp,"unsigned char");
		break;

	case signal_type:
	case slotcon_type:
	case slotdis_type:
	case array_type:
		nr_derefs = 1;

		/* Drop through. */

	case string_type:
		prcode(fp,"char");
		break;

	case ushort_type:
		prcode(fp,"unsigned short");
		break;

	case short_type:
		prcode(fp,"short");
		break;

	case arrayusize_type:
	case uint_type:
		prcode(fp,"unsigned");
		break;

	case arraysize_type:
	case cint_type:
	case int_type:
		prcode(fp,"int");
		break;

	case ulong_type:
		prcode(fp,"unsigned long");
		break;

	case long_type:
		prcode(fp,"long");
		break;

	case struct_type:
		prcode(fp,"struct %S",ad -> u.sname);
		break;

	case voidptr_type:
		prcode(fp,"void");
		break;

	case bool_type:
		prcode(fp,"bool");
		break;

	case float_type:
		prcode(fp,"float");
		break;

	case double_type:
		prcode(fp,"double");
		break;

	case defined_type:
		/*
		 * The only defined types still remaining are arguments to
		 * templates.
		 */

		prcode(fp,"%S",ad -> u.snd);
		break;

	case enum_type:
		prcode(fp,"%S",ad -> u.ed -> fqname);
		break;

	case rxcon_type:
	case rxdis_type:
		nr_derefs = 1;
		prcode(fp,"QObject");
		break;

	case mapped_type:
		generateBaseType(&ad -> u.mtd -> type,fp);
		break;

	case class_type:
		prcode(fp,"%S",classVersFQName(ad -> u.cvd));
		break;

	case template_type:
		{
			int a;
			templateDef *td = ad -> u.td;

			prcode(fp,"%S<",td -> fqname);

			for (a = 0; a < td -> types.nrArgs; ++a)
			{
				if (a > 0)
					prcode(fp,",");

				generateBaseType(&td -> types.args[a],fp);
			}

			prcode(fp,">");
			break;
		}
	}

	if (nr_derefs > 0)
	{
		int i;

		prcode(fp," ");

		for (i = 0; i < nr_derefs; ++i)
			prcode(fp,"*");
	}

	if (isReference(ad))
		prcode(fp,"&");
}


/*
 * Generate the definition of an argument variable and any supporting
 * variables.
 */

static void generateVariable(argDef *ad,int argnr,int secCall,FILE *fp)
{
	argType atype = ad -> atype;
	argDef orig;

	if ((atype == class_type || atype == mapped_type) &&
	    ad -> defval != NULL && ad -> defval -> vtype == fcall_value &&
	    (ad -> nrderefs == 0 || isReference(ad)))
	{
		/*
		 * Generate something to hold the default value as it cannot be
		 * assigned straight away.
		 */

		prcode(fp,
"		%B a%ddef = ",ad,argnr);
		generateSimpleFunctionCall(ad -> defval -> u.fcd,fp);
		prcode(fp,";\n"
			);
	}

	/* Massage some types. */

	orig = *ad;

	if (atype == class_type || atype == mapped_type)
	{
		/*
		 * For classes and mapped types we have to use a pointer rather
		 * than a reference.
		 */

		ad -> nrderefs = 1;
		resetIsReference(ad);
	}
	else if (atype == bool_type || atype == enum_type)
		ad -> atype = long_type;

	prcode(fp,
"		%B a%d",ad,argnr);

	*ad = orig;

	/* Handle any default value. */

	if (ad -> defval != NULL)
	{
		prcode(fp," = ");

		if (atype == class_type || atype == mapped_type)
		{
			if (ad -> defval -> vtype == fcall_value)
			{
				if (ad -> nrderefs == 0 || isReference(ad))
					prcode(fp,"&a%ddef",argnr);
				else
					generateSimpleFunctionCall(ad -> defval -> u.fcd,fp);
			}
			else if (ad -> defval -> vtype == scoped_value)
			{
				/* Cast if needed. */

				if (!isConstArg(ad))
					prcode(fp,"(%C *)",classVersFQName(ad -> u.cvd));

				prcode(fp,"&%S",ad -> defval -> u.vscp);
			}
			else
				prcode(fp,"NULL");
		}
		else
			generateExpression(ad -> defval,fp);
	}

	prcode(fp,";\n"
		);

	/* Some types have supporting variables. */

	switch (atype)
	{
	case rxcon_type:
	case rxdis_type:
		prcode(fp,
"		PyObject *a%dobj;\n"
			,argnr);
		break;

	case mapped_type:
	case class_type:
		prcode(fp,
"		PyObject *a%dobj%s;\n"
			,argnr,(ad -> defval == NULL ? "" : " = NULL"));
		break;
	}
}


/*
 * Generate a simple function call.
 */

static void generateSimpleFunctionCall(fcallDef *fcd,FILE *fp)
{
	int i;

	prcode(fp,"%S(",fcd -> name);

	for (i = 0; i < fcd -> nrArgs; ++i)
	{
		if (i > 0)
			prcode(fp,",");

		generateExpression(fcd -> args[i],fp);
	}

	prcode(fp,")");
}


/*
 * Generate the Python constructor function for the class.
 */

static void generatePythonConstructor(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	char *mname = pt -> module -> name -> text;
	ctorDef *ct;

	prcode(fp,
"\n"
"PyObject *sipNew_%C(PyObject *sipSelf,PyObject *sipArgs)\n"
"{\n"
"	static sipExtraType et = {\n"
"		sipCast_%C",classVersFQName(cvd),classVersFQName(cvd));

	if (pt -> sigslots)
	{
		prcode(fp,",\n"
"		(void *)sipNewProxy_%s,\n"
			,mname);

		if (hasSigSlots(cvd))
			prcode(fp,
"		sipSignals_%C"
				,classVersFQName(cvd));
		else
			prcode(fp,
"		NULL");
	}

	prcode(fp,"\n"
"	};\n"
"\n"
"	sipThisType *sipThis = NULL;\n"
"	const void *sipNew = NULL;\n"
"	int sipFlags = SIP_PY_OWNED;\n"
"	int sipArgsParsed = 0;\n"
"\n"
"	// See if there is something pending.\n"
"\n"
"	sipNew = sipGetPending(&sipFlags);\n"
		);

	/*
	 * Generate the code that parses the Python arguments and calls the
	 * correct constructor.
	 */

	if (!cannotCreate(cvd))
		for (ct = cvd -> ctors; ct != NULL; ct = ct -> next)
		{
			if (isPrivateCtor(ct))
				continue;

			prcode(fp,
"\n"
				);

			generateVersionStart(&ct -> version,fp);

			prcode(fp,
"	if (sipNew == NULL)\n"
"	{\n"
				);

			/* Use any code supplied in the specification. */

			if (ct -> cppcode != NULL)
				generateCppCodeBlock(ct -> cppcode,cvd,fp);
			else
			{
				int needSecCall;

				needSecCall = generateArgParser(&ct -> args,TRUE,FALSE,fp);
				generateConstructorCall(pt,cvd,ct,FALSE,fp);

				if (needSecCall)
				{
					prcode(fp,
"	}\n"
"\n"
"	if (sipNew == NULL)\n"
"	{\n"
						);

					generateArgParser(&ct -> args,TRUE,TRUE,fp);
					generateConstructorCall(pt,cvd,ct,TRUE,fp);
				}
			}

			prcode(fp,
"	}\n"
				);

			generateVersionEnd(&ct -> version,fp);
		}

	prcode(fp,
"\n"
"	if (sipNew == NULL)\n"
"	{\n"
"		sipNoCtor(sipArgsParsed,%N);\n"
"		return NULL;\n"
"	}\n"
"\n"
"	// Wrap the object.\n"
"\n"
"	if ((sipThis = sipCreateThis(sipSelf,sipNew,&sipType_%C,sipFlags,&et)) == NULL)\n"
"	{\n"
		,cvd -> common -> iff -> name
		,classVersFQName(cvd));

	if (!cannotCreate(cvd) && (isComplex(cvd) || !noPublicDtor(cvd)))
	{
		prcode(fp,
"		if (sipFlags & SIP_PY_OWNED)\n"
			);

		if (isComplex(cvd))
		{
			if (noPublicDtor(cvd))
				prcode(fp,
"			if (!(sipFlags & SIP_SIMPLE))\n"
"				delete (sip%C *)sipNew;\n"
					,classVersFQName(cvd));
			else
				prcode(fp,
"			if (sipFlags & SIP_SIMPLE)\n"
"				delete (%S *)sipNew;\n"
"			else\n"
"				delete (sip%C *)sipNew;\n"
					,classVersFQName(cvd),classVersFQName(cvd));
		}
		else
			prcode(fp,
"			delete (%S *)sipNew;\n"
				,classVersFQName(cvd));
	}

	prcode(fp,
"\n"
"		return NULL;\n"
"	}\n"
		);

	if (isComplex(cvd) && !cannotCreate(cvd))
		prcode(fp,
"\n"
"	if (!(sipFlags & SIP_SIMPLE))\n"
"		((sip%C *)sipNew) -> sipPyThis = sipThis;\n"
			,classVersFQName(cvd));

	prcode(fp,
"\n"
"	Py_INCREF(Py_None);\n"
"	return Py_None;\n"
"}\n"
		);
}


/*
 * Count the number of virtual members in a class.
 */
 
static int countVirtuals(classVersDef *cvd)
{
	int nrvirts;
	virtVersDef *vvd;
 
	nrvirts = 0;
 
	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		int thisVers = 0;

		virtOverDef *vod;

		/* Only count members than have a non-private overload. */

		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
			if (!isPrivate(vod -> o))
				++thisVers;

		if (thisVers > nrvirts)
			nrvirts = thisVers;
	}
 
	return nrvirts;
}
 

/*
 * Generate a single constructor call.
 */

static void generateConstructorCall(sipSpec *pt,classVersDef *cvd,ctorDef *ct,int secCall,FILE *fp)
{
	prcode(fp,
"		{\n"
		);

	generateArgConvertors(pt,&ct -> args,secCall,FALSE,NULL,fp);

	if (isComplex(cvd))
		prcode(fp,
"			sipNew = new sip%C(",classVersFQName(cvd));
	else
		prcode(fp,
"			sipNew = new %S(",classVersFQName(cvd));

	generateArgs(&ct -> args,Call,fp);

	prcode(fp,");\n"
		);

	generateCallTidyUp(&ct -> args,fp);

	prcode(fp,
"		}\n"
		);
}


/*
 * See if a member overload should be skipped.
 */

static int skipOverload(overDef *od,memberDef *md,classVersDef *cvd,classVersDef *ccvd,int doStatics)
{
	/* Skip if the versions clash. */

	if (!versionsOverlap(&od -> version,&cvd -> version))
		return TRUE;

	/* Skip if it's a signal or abstract. */

	if (od -> common != md || isSignal(od) || isAbstract(od))
		return TRUE;

	/* Skip if its not in the current class unless it is protected. */

	if (!isProtected(od) && ccvd != cvd)
		return TRUE;

	/* Skip if it is static and we aren't doing statics, or vice versa. */

	if ((doStatics && !isStatic(od)) || (!doStatics && isStatic(od)))
		return TRUE;

	return FALSE;
}


/*
 * Generate a class member function.
 */

static void generateFunction(sipSpec *pt,memberDef *md,overDef *overs,classVersDef *cvd,classVersDef *ocvd,int doStatics,FILE *fp)
{
	overDef *od;
	versionOrList *vol;

	/*
	 * Go through the overloads and build and or'ed list of versions for
	 * those that need to be handled.
	 */

	vol = NULL;

	for (od = overs; od != NULL; od = od -> next)
	{
		if (skipOverload(od,md,cvd,ocvd,doStatics))
			continue;

		orVersionQual(pt,&vol,&od -> version);
	}

	if (vol != NULL)
	{
		char *pname = md -> pyname -> text;

		prcode(fp,
"\n"
			);

		generateOredVersionStart(vol,fp);

		/* See if the member function is static. */

		if (doStatics)
			prcode(fp,
"static PyObject *sipDo_%C_%s(PyObject *,PyObject *sipArgs)\n"
"{\n"
"	int sipArgsParsed = 0;\n"
				,classVersFQName(cvd),pname);
		else
			prcode(fp,
"static PyObject *sipDo_%C_%s(PyObject *sipThisObj,PyObject *sipArgs)\n"
"{\n"
"	sipThisType *sipThis;\n"
"	int sipArgsParsed = 0;\n"
"\n"
"	if ((sipThis = sipGetThis(sipThisObj,&sipArgs,sipClass_%C)) == NULL)\n"
"		return NULL;\n"
				,classVersFQName(cvd),pname
				,classVersFQName(cvd));

		for (od = overs; od != NULL; od = od -> next)
		{
			if (skipOverload(od,md,cvd,ocvd,doStatics))
				continue;

			generateFunctionBody(pt,od,cvd,fp);
		}

		prcode(fp,
"\n"
"	// Report an error if the arguments couldn't be parsed.\n"
"\n"
"	sipNoMethod(sipArgsParsed,%N,%N);\n"
"\n"
"	return NULL;\n"
"}\n"
			,cvd -> common -> iff -> name,md -> pyname);

		generateOredVersionEnd(vol,fp);
		freeVersionOrList(vol);
	}
}


/*
 * Free an or'ed version list - but not the versions themselves.
 */

static void freeVersionOrList(versionOrList *vol)
{
	while (vol != NULL)
	{
		versionOrList *next;

		next = vol -> next;
		free(vol);
		vol = next;
	}
}


/*
 * Generate the function calls for a particular overload.
 */

static void generateFunctionBody(sipSpec *pt,overDef *od,classVersDef *cvd,FILE *fp)
{
	int noDecr;

	if (cvd == NULL)
		noDecr = TRUE;
	else
		noDecr = isStatic(od);

	prcode(fp,
"\n"
		);

	generateSpecialHandlingStart(pt,cvd,od,fp);
	generateVersionStart(&od -> version,fp);

	prcode(fp,
"	{\n"
		);

	/* Use any code supplied in the specification. */

	if (od -> cppcode != NULL)
		generateCppCodeBlock(od -> cppcode,cvd,fp);
	else if (!isPrivate(od))
	{
		int needSecCall;

		needSecCall = generateArgParser(&od -> args,noDecr,FALSE,fp);
		generateFunctionCall(pt,cvd,od,FALSE,fp);

		if (needSecCall)
		{
			prcode(fp,
"	}\n"
"\n"
"	{\n"
				);

			generateArgParser(&od -> args,noDecr,TRUE,fp);
			generateFunctionCall(pt,cvd,od,TRUE,fp);
		}
	}

	prcode(fp,
"	}\n"
		);

	generateVersionEnd(&od -> version,fp);
	generateSpecialHandlingEnd(pt,cvd,od,fp);
}


/*
 * Generate the code to handle the result of a call to a member function.
 */

static void generateHandleResult(sipSpec *pt,overDef *od,int isNew,FILE *fp)
{
	if (od -> result == NULL)
	{
		prcode(fp,
"			Py_INCREF(Py_None);\n"
"			return Py_None;\n"
			);

		return;
	}

	switch (od -> result -> atype)
	{
	case mapped_type:
		prcode(fp,
"			PyObject *resobj = sipConvertFrom_%T(res);\n"
			,od -> result);

		if (isNew)
			prcode(fp,
"			delete res;\n"
				);

		prcode(fp,
"\n"
"			return resobj;\n"
			);

		break;

	case class_type:
		{
			classVersDef *cvd = od -> result -> u.cvd;

			if (isNew)
			{
				prcode(fp,
"			return sipNewCppToSelf(res,");

				if (cvd -> convtosubcode == NULL)
					prcode(fp,"sipClass_%C",classVersFQName(cvd));
				else
					prcode(fp,"sipSubClass_%C(res)",classVersFQName(cvd));

				prcode(fp,",SIP_SIMPLE | SIP_PY_OWNED);\n"
					);
			}
			else
			{
				prcode(fp,
"			return sipMapCppToSelf(res,");

				if (cvd -> convtosubcode == NULL)
					prcode(fp,"sipClass_%C",classVersFQName(cvd));
				else
					prcode(fp,"sipSubClass_%C(res)",classVersFQName(cvd));

				prcode(fp,");\n"
					);
			}
		}
	
		break;

	case bool_type:
		prcode(fp,
"			return sipConvertFromBool((int)res);\n"
			);

		break;

	case ustring_type:
	case string_type:
		if (od -> result -> nrderefs == 0)
			prcode(fp,
"			return PyString_FromStringAndSize(%s&res,1);\n"
				,(od -> result -> atype == ustring_type) ? "(char *)" : "");
		else
			prcode(fp,
"			if (res == NULL)\n"
"			{\n"
"				Py_INCREF(Py_None);\n"
"				return Py_None;\n"
"			}\n"
"\n"
"			return PyString_FromString(res);\n"
				);

		break;

	case enum_type:
	case ushort_type:
	case short_type:
	case uint_type:
	case cint_type:
	case int_type:
	case ulong_type:
		prcode(fp,
"			return PyInt_FromLong((long)res);\n"
			);

		break;

	case long_type:
		prcode(fp,
"			return PyInt_FromLong(res);\n"
			);

		break;

	case struct_type:
	case voidptr_type:
		prcode(fp,
"			return sipConvertFromVoidPtr(res);\n"
			);

		break;

	case float_type:
		prcode(fp,
"			return PyFloat_FromDouble((double)res);\n"
			);

		break;

	case double_type:
		prcode(fp,
"			return PyFloat_FromDouble(res);\n"
			);

		break;
	}
}


/*
 * Generate a function call.
 */

static void generateFunctionCall(sipSpec *pt,classVersDef *cvd,overDef *od,int secCall,FILE *fp)
{
	char *mname = od -> cppname;
	int inNew;

	prcode(fp,
"		{\n"
		);

	/*
	 * If we can never create this class from Python, then all protected
	 * methods can never be called.
	 */

	if (isProtected(od) && cannotCreate(cvd))
	{
		prcode(fp,
"			sipGetComplexCppPtr(sipThis);\n"
"			return NULL;\n"
"		}\n"
			);

		return;
	}

	if (od -> result != NULL)
	{
		prcode(fp,
"			");
		generateValueType(od -> result,fp);
		prcode(fp,"res;\n"
			);
	}

	if (cvd != NULL)
	{
		if (isProtected(od))
			prcode(fp,
"			sip%C *ptr;\n"
"\n"
"			if ((ptr = (sip%C *)sipGetComplexCppPtr(sipThis)) == NULL)\n"
"				return NULL;\n"
				,classVersFQName(cvd),classVersFQName(cvd));
		else if (!isStatic(od))
			prcode(fp,
"			%S *ptr;\n"
"\n"
"			if ((ptr = (%S *)sipGetCppPtr(sipThis,sipClass_%C)) == NULL)\n"
"				return NULL;\n"
				,classVersFQName(cvd)
				,classVersFQName(cvd),classVersFQName(cvd));
	}

	if (od -> result != NULL || cvd != NULL)
		prcode(fp,
"\n"
			);

	generateArgConvertors(pt,&od -> args,secCall,isStatic(od),NULL,fp);

	/* Call any pre-hook. */

	if (od -> prehook != NULL)
		prcode(fp,
"			sipCallHook(\"%s\");\n"
			,od -> prehook);

	/* Save the thread if required. */

	if (isBlocking(od))
		prcode(fp,
"			sipReleaseLock();\n"
			);

	prcode(fp,
"			");

	if (od -> result != NULL)
		prcode(fp,"res = ");

	/*
	 * If the result is a class instance, then construct a copy on the
	 * heap.
	 */

	inNew = FALSE;

	if (od -> result != NULL &&
	    (od -> result -> atype == class_type || od -> result -> atype == mapped_type))
		if (isReference(od -> result))
			prcode(fp,"&");
		else if (od -> result -> nrderefs == 0)
		{
			argDef orig;

			orig = *od -> result;

			resetIsConstArg(od -> result);
			prcode(fp,"new %B(",od -> result);

			*od -> result = orig;

			inNew = TRUE;
		}

	/*
	 * If the function is protected then call the public wrapper.
	 * Otherwise explicitly call the real function and not any version in
	 * the wrapper class in case it is virtual.  This will prevent virtual
	 * loops.  Don't need to worry about indirected objects for protected
	 * functions.
	 */

	if (cvd == NULL)
		prcode(fp,"%s(",mname);
	else if (isProtected(od))
		prcode(fp,"ptr -> sipProtect_%s(",mname);
	else if (isStatic(od))
		prcode(fp,"%S::%s(",classVersFQName(cvd),mname);
	else
		prcode(fp,"ptr -> %S::%s(",classVersFQName(cvd),mname);

	generateArgs(&od -> args,Call,fp);

	prcode(fp,")");

	if (inNew)
		prcode(fp,")");

	prcode(fp,";\n"
		);

	/* Restore the thread if required. */

	if (isBlocking(od))
		prcode(fp,
"			sipAcquireLock();\n"
			);

	/* Call any post-hook. */

	if (od -> posthook != NULL)
		prcode(fp,
"			sipCallHook(\"%s\");\n"
			,od -> posthook);

	generateCallTidyUp(&od -> args,fp);

	prcode(fp,
"\n"
		);

	generateHandleResult(pt,od,inNew,fp);

	prcode(fp,
"		}\n"
		);
}


/*
 * Generate the code to convert the remaining unconverted arguments.
 */

static void generateArgConvertors(sipSpec *pt,funcArgs *args,int secCall,int isstat,char *retval,FILE *fp)
{
	char *mname = pt -> module -> name -> text, *this;
	int a, sigarg, rxconarg, rxdisarg, slotconarg, slotdisarg, noIntro;

	this = (isstat ? "NULL" : "sipThis");

	/* Do class ones together. */

	noIntro = TRUE;
	sigarg = rxconarg = rxdisarg = -1;

	for (a = 0; a < args -> nrArgs; ++a)
	{
		argDef *ad = &args -> args[a];

		switch (ad -> atype)
		{
		case mapped_type:
		case class_type:
			{
				codeBlock *convtocode;

				if (ad -> atype == class_type)
					convtocode = ad -> u.cvd -> convtocode;
				else
					convtocode = ad -> u.mtd -> convtocode;

				if (noIntro)
				{
					prcode(fp,
"			int iserr = 0;\n"
"\n"
						);

					noIntro = FALSE;
				}

				prcode(fp,
"			");

				if (convtocode != NULL)
					prcode(fp,"int istemp%d = ",a);

				prcode(fp,"sipConvertTo_%T(a%dobj,",ad,a);

				/* See if it needs a cast. */

				if (isConstArg(ad))
				{
					argDef orig;

					orig = *ad;

					resetIsConstArg(ad);
					resetIsReference(ad);
					ad -> nrderefs = 0;

					prcode(fp,"(%B **)",ad);

					*ad = orig;
				}

				prcode(fp,"&a%d,%s,&iserr);\n"
					,a,(ad -> nrderefs == 0 ? "1" : "0"));

				break;
			}

		case signal_type:
			sigarg = a;
			break;

		case rxcon_type:
			rxconarg = a;
			break;

		case rxdis_type:
			rxdisarg = a;
			break;

		case slotcon_type:
			slotconarg = a;
			break;

		case slotdis_type:
			slotdisarg = a;
			break;
		}
	}

	if (rxconarg >= 0 || rxdisarg >= 0)
		if (noIntro)
		{
			prcode(fp,
"			int iserr = 0;\n"
"\n"
				);

			noIntro = FALSE;
		}

	if (secCall)
	{
		if (rxconarg >= 0)
		{
			prcode(fp,
"			a%d = sipConvertRx(sipNewProxy_%s,%s,",rxconarg,mname,this);

			if (sigarg > 0)
				prcode(fp,"a%d",sigarg);
			else
			{
				prcode(fp,"\"(");

				generateArgs(args -> args[slotconarg].u.sa,Declaration,fp);

				prcode(fp,")\"");
			}

			prcode(fp,",a%dobj,NULL,&a%d,&iserr);\n"
				,rxconarg,slotconarg);
		}

		if (rxdisarg >= 0)
		{
			prcode(fp,
"			a%d = sipGetRx(%s,\"(",rxdisarg,this);

			generateArgs(args -> args[slotdisarg].u.sa,Declaration,fp);

			prcode(fp,")\",a%dobj,NULL,&a%d,&iserr);\n"
				,rxdisarg,slotdisarg);
		}
	}
	else
	{
		if (rxconarg >= 0)
		{
			prcode(fp,
"			a%d = sipConvertRx(sipNewProxy_%s,%s,",rxconarg,mname,this);

			if (sigarg > 0)
				prcode(fp,"a%d",sigarg);
			else
			{
				prcode(fp,"\"(");

				generateArgs(args -> args[slotconarg].u.sa,Declaration,fp);

				prcode(fp,")\"");
			}

			prcode(fp,",a%dobj,a%d,&a%d,&iserr);\n"
				,rxconarg,slotconarg,slotconarg);
		}

		if (rxdisarg >= 0)
		{
			prcode(fp,
"			a%d = sipGetRx(%s,\"(",rxdisarg,this);

			generateArgs(args -> args[slotdisarg].u.sa,Declaration,fp);

			prcode(fp,")\",a%dobj,a%d,&a%d,&iserr);\n"
				,rxdisarg,slotdisarg,slotdisarg);
		}
	}

	if (!noIntro)
	{
		if (retval == NULL)
			retval = "NULL";

		prcode(fp,
"\n"
"			if (iserr)\n"
"				return %s;\n"
"\n"
			,retval);
	}
}


/*
 * Generate the argument variables for a member function/constructor.
 */

static int generateArgParser(funcArgs *args,int noDecr,int secCall,FILE *fp)
{
	int a, needSecCall, optargs, arraylenarg;

	/* Assume there isn't a slot. */

	needSecCall = FALSE;

	/* Generate the local variables that will hold the parsed arguments. */

	for (a = 0; a < args -> nrArgs; ++a)
	{
		argType atype = args -> args[a].atype;

		if (atype == slotcon_type || atype == slotdis_type ||
			 atype == rxcon_type || atype == rxdis_type)
			needSecCall = TRUE;

		if (atype == arraysize_type || atype == arrayusize_type)
			arraylenarg = a;

		generateVariable(&args -> args[a],a,secCall,fp);
	}

	if (args -> nrArgs != 0)
		prcode(fp,
"\n"
			);

	/* Generate the call to sipParseArgs(). */

	prcode(fp,
"		if (sipParseArgs(&sipArgsParsed,sipArgs,\"");

	/* Generate the format string. */

	optargs = FALSE;

	if (noDecr)
		prcode(fp,"-");

	for (a = 0; a < args -> nrArgs; ++a)
	{
		char *fmt;

		if (args -> args[a].defval != NULL && !optargs)
		{
			prcode(fp,"|");
			optargs = TRUE;
		}

		switch (args -> args[a].atype)
		{
		case ustring_type:
		case string_type:
			if (args -> args[a].nrderefs == 0)
				fmt = "c";
			else
				fmt = "s";
			break;

		case uint_type:
		case int_type:
			fmt = "i";
			break;

		case cint_type:
			fmt = "C";
			break;

		case ushort_type:
		case short_type:
			fmt = "h";
			break;

		case enum_type:
		case bool_type:
		case ulong_type:
		case long_type:
			fmt = "l";
			break;

		case struct_type:
		case voidptr_type:
			fmt = "v";
			break;

		case float_type:
			fmt = "f";
			break;

		case double_type:
			fmt = "d";
			break;

		case signal_type:
			fmt = "G";
			break;

		case slotcon_type:
		case slotdis_type:
			fmt = (secCall ? "" : "S");
			break;

		case rxcon_type:
		case rxdis_type:
			fmt = (secCall ? "F" : "R");
			break;

		case mapped_type:
		case class_type:
			fmt = "I";
			break;

		case array_type:
		case uarray_type:
			fmt = "a";
			break;

		case arrayusize_type:
		case arraysize_type:
			fmt = "";
			break;
		}

		prcode(fp,fmt);
	}

	prcode(fp,"\"");

	/* Generate the parameters corresponding to the format string. */

	for (a = 0; a < args -> nrArgs; ++a)
	{
		int atype = args -> args[a].atype;

		switch (atype)
		{
		case mapped_type:
		case class_type:
			prcode(fp,",sipCanConvertTo_%T,&a%dobj",&args -> args[a],a);
			break;

		case rxcon_type:
		case rxdis_type:
			prcode(fp,",&a%dobj",a);
			break;

		case slotcon_type:
		case slotdis_type:
			if (!secCall)
				prcode(fp,",&a%d",a);

			break;

		case arraysize_type:
		case arrayusize_type:
			break;

		default:
			prcode(fp,",&a%d",a);

			if (atype == array_type || atype == uarray_type)
				prcode(fp,",&a%d",arraylenarg);
		}
	}

	prcode(fp,"))\n");

	return needSecCall;
}


/*
 * Generate the code to tidy up after a call to C++.
 */

static void generateCallTidyUp(funcArgs *args,FILE *fp)
{
	int a;

	for (a = 0; a < args -> nrArgs; ++a)
	{
		codeBlock *convtocode;

		if (args -> args[a].atype == class_type)
			convtocode = args -> args[a].u.cvd -> convtocode;
		else if (args -> args[a].atype == mapped_type)
			convtocode = args -> args[a].u.mtd -> convtocode;
		else
			continue;

		if (convtocode == NULL &&
		    !isTransferred(&args -> args[a]) &&
		    !isThisTransferred(&args -> args[a]) &&
		    !isTransferredBack(&args -> args[a]))
			continue;

		/* Delete temporary instances created by class convertors. */

		if (convtocode != NULL)
			prcode(fp,
"\n"
"			if (istemp%d)\n"
"				delete a%d;\n"
				,a,a);

		if (isTransferred(&args -> args[a]))
			prcode(fp,
"\n"
"			sipTransferSelf(a%dobj,1);\n"
				,a);

		if (isThisTransferred(&args -> args[a]))
			prcode(fp,
"\n"
"			if (a%d)\n"
"				sipFlags &= ~SIP_PY_OWNED;\n"
				,a);

		if (isTransferredBack(&args -> args[a]))
			prcode(fp,
"\n"
"			sipTransferSelf(a%dobj,0);\n"
				,a);
	}
}


/*
 * Generate a C++ code block.
 */

static void generateCppCodeBlock(codeBlock *code,classVersDef *cvd,FILE *fp)
{
	codeBlock *cb;

	for (cb = code; cb != NULL; cb = cb -> next)
	{
		prcode(fp,
"#line %d \"%s\"\n"
			,cb -> linenr,cb -> filename);

		generateMacroCode(cb,cvd,NULL,NULL,NULL,fp);
	}

	if (code != NULL)
	{
		char *bn;

		/* Just use the base name. */

		if ((bn = strrchr(currentFileName,'/')) != NULL)
			++bn;
		else
			bn = currentFileName;

		prcode(fp,
"#line %d \"%s\"\n"
			,currentLineNr + 1,bn);
	}
}


/*
 * Generate a code fragment with macro expansion.
 */

static void generateMacroCode(codeBlock *cb,classVersDef *cvd,sipSpec *pt,char *cppSuffix,char *objSuffix,FILE *fp)
{
	char *cp;

	for (cp = cb -> frag; *cp != '\0'; ++cp)
	{
		char mc;
		int invalid;

		if (*cp != MACRO_ESCAPE)
		{
			prcode(fp,"%c",*cp);
			continue;
		}

		invalid = FALSE;

		switch (mc = *++cp)
		{
		case MACRO_ESCAPE:
			prcode(fp,"%c",*cp);
			break;

		case MACRO_CLASS:
			if (cvd != NULL)
				prcode(fp,"%C",classVersFQName(cvd));
			else
				invalid = TRUE;

			break;

		case MACRO_CPPSUFF:
			if (cppSuffix != NULL)
				prcode(fp,"%s",cppSuffix);
			else
				invalid = TRUE;

			break;

		case MACRO_OBJSUFF:
			if (objSuffix != NULL)
				prcode(fp,"%s",objSuffix);
			else
				invalid = TRUE;

			break;

		case MACRO_SOURCES:
			if (pt != NULL)
				generateMacroSource(pt,cppSuffix,fp);
			else
				invalid = TRUE;

			break;

		case MACRO_OBJECTS:
			if (pt != NULL)
				generateMacroObject(pt,objSuffix,fp);
			else
				invalid = TRUE;

			break;

		case MACRO_CPPMODULE:
			prcode(fp,"%s",pt -> cppmname);
			break;

		case MACRO_PERCENT:
			prcode(fp,"%c",'%');
			break;

		case '\0':
			--cp;
			break;

		default:
			fatal("Invalid macro: %c%c\n",MACRO_ESCAPE,mc);
		}

		if (invalid)
			fatal("%c%c macro invalid in this context\n",MACRO_ESCAPE,mc);
	}
}


/*
 * Generate the macro expansion for the list of source files.
 */

static void generateMacroSource(sipSpec *pt,char *cppSuffix,FILE *fp)
{
	char *mname = pt -> module -> name -> text;
	ifaceFileDef *iff;

	prcode(fp,"%scmodule%s sip%sDecl%s.h",mname,cppSuffix,mname,mname);

	if (pt -> sigargs != NULL)
		prcode(fp," \\\n"
"	sip%sProxy%s.h",mname,mname);
 
	/* Dependencies on the source files. */
 
	for (iff = pt -> ifacefiles; iff != NULL; iff = iff -> next)
		if (hasMappedType(iff) || hasRealClassVers(iff))
			if (iff -> module == pt -> module)
				prcode(fp," \\\n"
"	sip%s%F.h sip%s%F%s",mname,iff -> fqname,mname,iff -> fqname,cppSuffix);
			else if (isUsed(iff))
				prcode(fp," \\\n"
"	sip%s%F.h",iff -> module -> name -> text,iff -> fqname);
}


/*
 * Generate the macro expansion for the list of object files.
 */

static void generateMacroObject(sipSpec *pt,char *objSuffix,FILE *fp)
{
	char *mname = pt -> module -> name -> text;
	ifaceFileDef *iff;

	prcode(fp,"%scmodule%s",mname,objSuffix);

	/* Dependencies on the object files. */
 
	for (iff = pt -> ifacefiles; iff != NULL; iff = iff -> next)
		if (hasMappedType(iff) || hasRealClassVers(iff))
			if (iff -> module == pt -> module)
				prcode(fp," \\\n"
"	sip%s%F%s",mname,iff -> fqname,objSuffix);
}


/*
 * Generate the pre-processor code to check an ored set of secondary versions.
 */

static void generateOredVersionStart(versionOrList *vol,FILE *fp)
{
	int needSep;

	/*
	 * If there is currently a version active then, because secondarys are
	 * mutually exclusive, this new expression can't make a difference, so
	 * don't bother.
	 */

	if (vqstackptr > 0)
		return;

	/* Check for the true values. */

	if (vol == NULL || vol -> version.secondary == NULL)
		return;

	prcode(fp,
"#if ");

	needSep = FALSE;

	do
	{
		if (needSep)
			prcode(fp," || ");

		prcode(fp,"%V",vol -> version.secondary);

		needSep = TRUE;
	}
	while ((vol = vol -> next) != NULL);

	prcode(fp,"\n"
		);
}


/*
 * Generate the pre-processor code to complete a version check.
 */

static void generateOredVersionEnd(versionOrList *vol,FILE *fp)
{
	/*
	 * If there is currently a version active then, because secondarys are
	 * mutually exclusive, this new expression can't make a difference, so
	 * don't bother.
	 */

	if (vqstackptr > 0)
		return;

	/* Check for the true values. */

	if (vol == NULL || vol -> version.secondary == NULL)
		return;

	prcode(fp,
"#endif\n"
		);
}


/*
 * Generate the pre-processor code to check a version.
 */

static void generateVersionStart(versionDef *vd,FILE *fp)
{
	versionQual *vq;

	if ((vq = pushVersion(vd)) != NULL)
		prcode(fp,
"#if %V\n"
			,vq);
}


/*
 * Generate the pre-processor code to complete a version check.
 */

static void generateVersionEnd(versionDef *vd,FILE *fp)
{
	popVersion();

	/*
	 * Push it again so that we can work out if anything was generated at
	 * the start of the version.
	 */

	if (pushVersion(vd) != NULL)
		prcode(fp,
"#endif\n"
			);

	popVersion();
}


/*
 * Create a file with an optional standard header.
 */

static FILE *createFile(sipSpec *pt,char *fname,char *commentStr,char *description)
{
	FILE *fp;

	/* Create the file. */

	if ((fp = fopen(fname,"w")) == NULL)
		fatal("Unable to create file \"%s\"\n",fname);

	currentLineNr = 1;
	currentFileName = fname;

	if (commentStr != NULL)
	{
		int needComment;
		codeBlock *cb;
		time_t now;

		/* Write the header. */

		now = time(NULL);

		prcode(fp,"%s %s\n",commentStr,description);
		prcode(fp,"%s\n",commentStr);
		prcode(fp,"%s Generated by SIP v%s on %s",commentStr,sipVersion,ctime(&now));

		if (pt -> copying != NULL)
			prcode(fp,"%s\n",commentStr);

		needComment = TRUE;

		for (cb = pt -> copying; cb != NULL; cb = cb -> next)
		{
			char *cp;

			for (cp = cb -> frag; *cp != '\0'; ++cp)
			{
				if (needComment)
				{
					needComment = FALSE;
					prcode(fp,"%s ",commentStr);
				}

				prcode(fp,"%c",*cp);

				if (*cp == '\n')
					needComment = TRUE;
			}
		}
	}

	return fp;
}


/*
 * Close a file and report any errors.
 */

static void closeFile(FILE *fp)
{
	if (ferror(fp))
		fatal("Error writing to \"%s\"\n",currentFileName);

	if (fclose(fp))
		fatal("Error closing \"%s\"\n",currentFileName);
}


/*
 * Print formatted code.
 */

static void prcode(FILE *fp,char *fmt,...)
{
	char ch;
	va_list ap;

	va_start(ap,fmt);

	while ((ch = *fmt++) != '\0')
		if (ch == '%')
		{
			ch = *fmt++;

			switch (ch)
			{
			case 'c':
				{
					char c = va_arg(ap,int);

					if (c == '\n')
						++currentLineNr;

					fputc(c,fp);
					break;
				}

			case 's':
				{
					char *cp = va_arg(ap,char *);

					while (*cp != '\0')
					{
						if (*cp == '\n')
							++currentLineNr;

						fputc(*cp,fp);
						++cp;
					}

					break;
				}

			case 'l':
				fprintf(fp,"%ld",va_arg(ap,long));
				break;

			case 'd':
				fprintf(fp,"%d",va_arg(ap,int));
				break;

			case 'g':
				fprintf(fp,"%g",va_arg(ap,double));
				break;

			case '\0':
				fputc('%',fp);
				--fmt;
				break;

			case '\n':
				fputc('\n',fp);
				++currentLineNr;
				break;

			case 'B':
				generateBaseType(va_arg(ap,argDef *),fp);
				break;

			case 'T':
				prTypeName(fp,va_arg(ap,argDef *),FALSE);
				break;

			case 'I':
				{
					int indent = va_arg(ap,int);

					while (indent-- > 0)
						fputc('\t',fp);

					break;
				}

			case 'N':
				{
					nameDef *nd = va_arg(ap,nameDef *);

					fprintf(fp,"sipName_%s_%s",nd -> module -> name -> text,nd -> text);
					break;
				}

			case 'F':
				prScopedName(fp,va_arg(ap,scopedNameDef *),"");
				break;

			case 'C':
				prScopedName(fp,va_arg(ap,scopedNameDef *),"_");
				break;

			case 'S':
				prScopedName(fp,va_arg(ap,scopedNameDef *),"::");
				break;

			case 'V':
				{
					versionQual *vq = va_arg(ap,versionQual *);

					fprintf(fp,"defined(SIP_VERS_%s)",vq -> name);
					break;
				}

			default:
				fputc(ch,fp);
			}
		}
		else if (ch == '\n')
		{
			fputc('\n',fp);
			++currentLineNr;
		}
		else
			fputc(ch,fp);

	va_end(ap);
}


/*
 * Generate a scoped name with the given separator string.
 */

static void prScopedName(FILE *fp,scopedNameDef *snd,char *sep)
{
	while (snd != NULL)
	{
		fprintf(fp,"%s",snd -> name);

		if ((snd = snd -> next) != NULL)
			fprintf(fp,"%s",sep);
	}
}


/*
 * Generate a type name to be used as part of an identifier name.
 */

static void prTypeName(FILE *fp,argDef *ad,int intmpl)
{
	if (intmpl)
	{
		char buf[10];
		int flgs;

		/* We use numbers so they don't conflict with names. */

		sprintf(buf,"%02d",ad -> atype);

		flgs = 0;

		if (isConstArg(ad))
			flgs += 1;

		if (isReference(ad))
			flgs += 2;

		prcode(fp,"%s%d%d",buf,flgs,ad -> nrderefs);
	}

	/* Now add a name if there is one associated with the type. */

	switch (ad -> atype)
	{
	case struct_type:
		prcode(fp,"%C",ad -> u.sname);
		break;

	case defined_type:
		prcode(fp,"%C",ad -> u.snd);
		break;

	case enum_type:
		prcode(fp,"%C",ad -> u.ed -> fqname);
		break;

	case mapped_type:
		prTypeName(fp,&ad -> u.mtd -> type,intmpl);
		break;

	case class_type:
		prcode(fp,"%C",classVersFQName(ad -> u.cvd));
		break;

	case template_type:
		{
			int a;
			templateDef *td = ad -> u.td;

			prcode(fp,"%C",td -> fqname);

			for (a = 0; a < td -> types.nrArgs; ++a)
			{
				prcode(fp,"_");
				prTypeName(fp,&td -> types.args[a],TRUE);
			}

			break;
		}
	}
}


/*
 * See if an overload requires any special handling.  At the moment there is
 * only one case.  If we ever need any more then this function should return a
 * set of flags describing the relevant cases.
 */

static int needsSpecialHandling(sipSpec *pt,classVersDef *cvd,overDef *od)
{
	classVersList *cvl;

	/* We are only interested in class methods. */

	if (cvd == NULL)
		return FALSE;

	/* We are only interested in the tr() method. */

	if (strcmp(od -> cppname,"tr") != 0)
		return FALSE;

	/* QObject::tr() doesn't need special handling. */

	if (pt -> qobjclass == cvd -> common -> classnr)
		return FALSE;

	/* See if we have QObject in the hierachy. */

	for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
		if (pt -> qobjclass == cvl -> cv -> common -> classnr)
			break;

	if (cvl == NULL)
		return FALSE;

	/*
	 * For the moment this simple test will do.  Future versions of Qt may
	 * need something more sophisticated.
	 */

	return (od -> args.nrArgs == 2);
}


/*
 * Generate the code for any special handling for an overload.
 */

static void generateSpecialHandlingStart(sipSpec *pt,classVersDef *cvd,
					 overDef *od,FILE *fp)
{
	if (needsSpecialHandling(pt,cvd,od))
		prcode(fp,
"// Auto-generation of this method was introduced with moc revision 9 (which\n"
"// was Qt v2.3.1).\n"
"#if QT_VERSION >= 231\n"
			);
}


/*
 * Generate anything required to tidy up special handling.
 */

static void generateSpecialHandlingEnd(sipSpec *pt,classVersDef *cvd,
				       overDef *od,FILE *fp)
{
	if (needsSpecialHandling(pt,cvd,od))
		prcode(fp,
"#endif\n"
			);
}
