/*
 * ----------------------------------------------------------------------------
 *
 * Interface between emulator and gdb using the gdb 
 * remote protocol
 *
 * (C) 2005 Jochen Karrer
 *   Author: Jochen Karrer
 *
 * State:
 *	minimum required set of operations are working	
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/ip.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <unistd.h>
#include <fcntl.h>
#include <fio.h>
#include <configfile.h>
#include <gdebug.h>
#include <sys/time.h>


#if 0
#define dprintf(x...) { fprintf(stderr,x); }
#else
#define dprintf(x...)
#endif

#define CMDBUF_SIZE (512)

typedef struct GdbSession {
	int fd;
	int rfh_is_active;
	FIO_FileHandler rfh;
	GDebugOps *dbgops;
	GdbServer *gserv;

	char cmdbuf[CMDBUF_SIZE];
	int cmdstate;
	int cmdbuf_wp;
	uint8_t csum;
	void *backend;
	struct GdbSession *next;
} GdbSession;

struct GdbServer {
	int sockfd;
	FIO_TcpServer gserv;
	GDebugOps *dbgops;
	void *backend;
	GdbSession *first_gsess;
};

#define CMDSTATE_WAIT_START (0)
#define CMDSTATE_WAIT_DATA	(1)
#define CMDSTATE_WAIT_CSUM1	(2)
#define CMDSTATE_WAIT_CSUM2	(3)

__attribute__((__unused__)) static inline void 
print_timestamp(char *str) 
{
#if 0
	struct timeval tv;
	gettimeofday(&tv,NULL);
	fprintf(stderr,"%s: %d\n",str,tv.tv_usec);
#endif
}

static int gsess_reply(GdbSession *,const char *format,...); //  __attribute__ ((format (printf, 2, 3)));;
/*
 * ----------------------------------------------------
 * gsess_output
 * 	Write to socket is currently done in blocking
 * 	mode
 * ----------------------------------------------------
 */

static void
gsess_output(GdbSession *gsess,char *buf,int count) {
        int result;
        fcntl(gsess->fd,F_SETFL,0);
        while(count) {
                result = write(gsess->fd,buf,count);
		print_timestamp("Output done"); 
                if(result<=0) {
                        /* Can not delete the session now */
                        return;
                }
                count-=result; buf+=result;
        }
        fcntl(gsess->fd,F_SETFL,O_NONBLOCK);
}

/*
 * ----------------------------------------
 * Assemble a reply using varargs/vsnprintf
 * ----------------------------------------
 */

static int
gsess_reply(GdbSession *gsess,const char *format,...) {
	va_list ap;
	char reply[1024];
	uint8_t chksum=0;
	int count;
	int i;
	count = sprintf(reply,"$");
        va_start (ap, format);
        count+=vsnprintf(reply+count,1024-count-4,format,ap);
        va_end(ap);
	for(i=1;i<count;i++) {
		chksum += reply[i];
	}
	count += sprintf(reply+count,"#%02x",chksum);
	gsess_output(gsess,reply,count);	
	return 0;
}

void
gsess_del(GdbSession *gsess) {
        uint8_t c;
	GdbServer *gserv = gsess->gserv;
        if(gsess->rfh_is_active) {
                gsess->rfh_is_active=0;
                dprintf("remove the filehandler\n");
                FIO_RemoveFileHandler(&gsess->rfh);
        }
        shutdown(gsess->fd,SHUT_RDWR);
        while(read(gsess->fd,&c,1)>0) {
                dprintf("READEMPTY %02x\n",c);
        }
        if(gsess->fd>2) {
                dprintf("close %d\n",gsess->fd);
                if(close(gsess->fd)<0) {
                        perror("close fd failed");
                } else {
                        gsess->fd=-1;
                }
        } else {
                dprintf("Do not close %d\n",gsess->fd);
        }
	if(gserv->first_gsess == gsess) {
		gserv->first_gsess = NULL;
	}
        free(gsess);
}

#define MAXREGS (30)
static void
gsess_getregs(GdbSession *gsess) 
{
	char reply[MAXREGS*8+1];
	int i,j;
	GDebugOps *dbgops = gsess->dbgops;
	if(!dbgops->getreg) {
		gsess_reply(gsess,"00000000");
	} else {
		for(i=0;i<MAXREGS;i++) {
			uint8_t value[4];
			if(dbgops->getreg(gsess->backend,value,i,4)<=0) {
				break;	
			}
			for(j=0;j<4;j++) {
				sprintf(reply+8*i+j*2,"%02x",value[j]);
			}
		}
		*(reply+i*8) = 0;
		dprintf("Got %d registers, %s\n",i,reply);
		gsess_reply(gsess,reply);
	}
}
static void
gsess_getreg(GdbSession *gsess,int index) 
{
	char reply[10];
	int j;
	GDebugOps *dbgops = gsess->dbgops;
	if(!dbgops->getreg) {
		gsess_reply(gsess,"00000000");
	} else {
		uint8_t value[4];
		if(dbgops->getreg(gsess->backend,value,index,4)<=0) {
			
		}
		for(j=0;j<4;j++) {
			sprintf(reply+j*2,"%02x",value[j]);
		}
		*(reply+8) = 0;
		dprintf("registers %d, %s\n",index,reply);
		gsess_reply(gsess,reply);
	}
}
static void
gsess_getmem(GdbSession *gsess,uint64_t addr,uint32_t len) 
{
	char reply[1024];
	int writep=0;
	uint8_t buf[32];
	int i;
	int result;
	GDebugOps *dbgops = gsess->dbgops;
	if(len > 256) {
		len=256;
	}
	if(!dbgops->getmem) {
		gsess_reply(gsess,"00000000");
	} else {
		while(len>=4)  {
			result = dbgops->getmem(gsess->backend,buf,addr,4);
			if(result > 0) {
				len -= result;	
				for(i=0;i<result;i++) {
					writep+=sprintf(reply + writep,"%02x",buf[i]);
				}
			}
			if(result < 4) {
				break;
			}
			addr+=4;
		}
		while(len>0)  {
			result = dbgops->getmem(gsess->backend,buf,addr,1);
			if(result > 0) {
				len -= result;	
				for(i=0;i<result;i++) {
					writep+=sprintf(reply + writep,"%02x",buf[i]);
				}
			} else {
				break;
			}
			addr+=1;
		}
		reply[writep] = 0;
		dprintf("count %d gm reply \"%s\"\n",writep, reply);
		gsess_reply(gsess,reply);
	}
}

static void
gsess_getstatus(GdbSession *gsess) 
{
	int result;
	GDebugOps *dbgops = gsess->dbgops;
	if(!dbgops->get_status) {
		gsess_reply(gsess,"S00");
	} else {
		result = dbgops->get_status(gsess->backend);
		if(result>=0) {
			gsess_reply(gsess,"S%02x",result);
		}		
	}
}
static void
gsess_stop(GdbSession *gsess) 
{
	int result;
	GDebugOps *dbgops = gsess->dbgops;
	if(!dbgops->stop) {
		gsess_reply(gsess,"S00");
	} else {
		result = dbgops->stop(gsess->backend);
		if(result>=0) {
			gsess_reply(gsess,"S%02x",result);
		}		
	}
}

/*
 * ------------------------------------------
 * Returns the number of sessions
 * ------------------------------------------
 */
int
GdbServer_Signal(GdbServer *gserv,int sig) 
{
	GdbSession *gsess = gserv->first_gsess;
	if(!gsess) {
		return 0;	
	}
	gsess_reply(gsess,"S%02x",sig);
	return 1;
}

static void
gsess_cont(GdbSession *gsess) 
{
	int result;
	GDebugOps *dbgops = gsess->dbgops;
	if(!dbgops->cont) {

	} else {
		result = dbgops->cont(gsess->backend);
		if(result<0) {
			gsess_reply(gsess,"S%02x",-result);
		}		
	}
}

static void
gsess_step(GdbSession *gsess,uint32_t addr,int use_addr) 
{
	GDebugOps *dbgops = gsess->dbgops;
	int result;
	if(!dbgops->step) {
		gsess_reply(gsess,"S00");
	} else {
		result = dbgops->step(gsess->backend,addr,use_addr);
		if(result>=0) {
			gsess_reply(gsess,"S%02x",result);
		}		
	}
}

static int
hexparse(const char *str,uint8_t *retval,int maxbytes) {
        int i,nibble;
	uint8_t val;
        for(i=0;i<maxbytes;i++) {
		for(nibble=0;nibble<2;nibble++) {
                	char c = *str;
			if(c>='0' && c<='9') {
				val =(c - '0');
			} else if(c>='A' && c<='F') {
				val =(c - 'A'+10);
			} else if(c>='a' && c<='f') {
				val =(c - 'a'+10);
			} else {
				return i;
			}
			if(nibble) {
				retval[i] = (retval[i] & ~0xf) | val;
			} else {
				retval[i] = (retval[i] & ~0xf0) | (val<<4);
			}
			str++;
		}
        }
        return maxbytes;
}

static void
gsess_setmem(GdbSession *gsess,char *data,int maxlen)
{
	uint32_t addr;	
	uint32_t len;
	int readlen;
	uint32_t readp=0;
	GDebugOps *dbgops = gsess->dbgops;
	if(!dbgops->setmem) {
		gsess_reply(gsess,"E00");
	}
	if(sscanf(data,"%x,%x:",&addr,&len) != 2) {
		gsess_reply(gsess,"E00");
		return;
	}
	while(readp < maxlen) {
		if(data[readp] == ':') {
			break;
		}	
	}
	if((readp+2*len)>=maxlen) {
		gsess_reply(gsess,"E00");
		return;
	}
	while(len) {
		uint8_t value[4];
		readlen = (len >= 4) ? 4 : len;	
		if(hexparse(data+readp,value,readlen) <= 0) {
			gsess_reply(gsess,"E00");
			return;
		}
		dbgops->setmem(gsess->backend,value,addr,readlen);
		len-=readlen;
		readp+=readlen;
	}
	gsess_reply(gsess,"OK");

}

static void
gsess_setreg(GdbSession *gsess,char *data)
{
	GDebugOps *dbgops = gsess->dbgops;
	int reg; 
	uint8_t value[8];
	int count;
	if(!dbgops->setreg) {
		gsess_reply(gsess,"E00");
		return;
	}
	if(sscanf(data,"%d=",&reg)!=1) {
		gsess_reply(gsess,"E00"); // fix the error number 
		return;
	}
	while(*data != '=') {
		data++;
	}
	data++;
	if((count = hexparse(data,value,8)) < 1) {
		gsess_reply(gsess,"E00");
		return;
	}
#if 1
	{
		int i;
		fprintf(stderr,"hexparsed %d bytes: ",count);
		for(i=0;i<4;i++) {
			fprintf(stderr,"%02x ",value[i]);	
		}
		fprintf(stderr,"\n");	
	}
#endif
	dbgops->setreg(gsess->backend,value,reg,count);
	gsess_reply(gsess,"OK");
}

static int 
gsess_execute_cmd(GdbSession *gsess) 
{
	char *cmd = gsess->cmdbuf;
	int result=0;
	dprintf("Das Kommando wird hingerichted\n");	
	switch(cmd[0]) {
		/* Status query */
		case '?':
			gsess_getstatus(gsess);
			break;

		case 's':
			{
				uint32_t addr;
				if(sscanf(cmd+1,"%x",&addr) == 1) {
					gsess_step(gsess,addr,1);
				} else {
					gsess_step(gsess,addr,0);
				}
			}
			break;

		/* continue: no reply until stopped again */
		case 'c':
			gsess_cont(gsess);
			break;

		case 'D':
			gsess_cont(gsess);
			result=-1;
			break;

		case 'p':
			{
				unsigned int reg;
				if(sscanf(cmd+1,"%02x#",&reg) == 1) {
					gsess_getreg(gsess,reg);
				}
			}
			break;

		case 'P': 
			gsess_setreg(gsess,cmd+1);
			break;

		/* Get registers */
		case 'g':
			gsess_getregs(gsess);
			break;


		/* Set registers */
		case 'G': 
			gsess_reply(gsess,"OK");
			break;

		/* Read memory */
		case 'm':
			{
				uint32_t addr;
				uint32_t len;
				sscanf(cmd+1,"%x,%x",&addr,&len);
				gsess_getmem(gsess,addr,len);
			}
			break;

		/* Write memory */
		case 'M':
			gsess_setmem(gsess,cmd+1,gsess->cmdbuf_wp-1);
			break;

		default:
			dprintf("unknown cmd \'%c\'\n",cmd[0]);
			gsess_output(gsess,"$#00",4); 
			break;
	}
	return result;
}

static int
feed_state_machine (GdbSession *gsess,uint8_t c) {
	int result = 0;
	dprintf("%c",c);
	switch(gsess->cmdstate) 
	{
		case CMDSTATE_WAIT_START:
			if(c=='$') {
				gsess->cmdbuf_wp = 0;	
				gsess->cmdstate = CMDSTATE_WAIT_DATA;
				gsess->csum = 0;
			} else if(c == '+') {
				dprintf("Got acknowledge\n");
			} else if(c == '-') {
				dprintf("Got NACK\n");
			} else if(c == 0x03) {
				gsess_stop(gsess);
				dprintf("received abort\n");
			} else {
				dprintf("received unknown byte %02x in START state\n",c);
			}
			break;

		case CMDSTATE_WAIT_DATA:
			if(c=='#') {
				gsess->cmdstate = CMDSTATE_WAIT_CSUM1;
				/* Terminate the string with 0 */
				gsess->cmdbuf[gsess->cmdbuf_wp] = 0;
			} else {
				if(gsess->cmdbuf_wp >= CMDBUF_SIZE) {
					dprintf("Message from gdb to long, ignoring\n");
				} else {
					gsess->cmdbuf[gsess->cmdbuf_wp++] = c;
					gsess->csum += c;
				}
			}			
			break;
			
		case CMDSTATE_WAIT_CSUM1:
			if((c >= '0') && (c <= '9')) {
				gsess->csum ^= (c - '0') << 4;
				gsess->cmdstate = CMDSTATE_WAIT_CSUM2;
			} else if((c >= 'a') && (c <= 'f')) {
				gsess->csum ^= (c - 'a'+10) << 4;
				gsess->cmdstate = CMDSTATE_WAIT_CSUM2;
			} else if((c >= 'A') && (c <= 'F')) {
				gsess->csum ^= (c - 'A'+10) << 4;
				gsess->cmdstate = CMDSTATE_WAIT_CSUM2;
			} else {
				gsess->cmdstate = CMDSTATE_WAIT_START;
				fprintf(stderr,"illegal byte %02x in chksum\n",c);	
			}
			break;

		case CMDSTATE_WAIT_CSUM2:
			gsess->cmdstate = CMDSTATE_WAIT_START;
			if((c >= '0') && (c <= '9')) {
				gsess->csum ^= (c - '0');
			} else if((c >= 'a') && (c <= 'f')) {
				gsess->csum ^= (c - 'a'+10);
			} else if((c >= 'A') && (c <= 'F')) {
				gsess->csum ^= (c - 'A'+10);
			} else {
				fprintf(stderr,"illegal byte %02x in chksum\n",c);	
				break;
			}
			dprintf("received complete command, csum %02x\n",gsess->csum); 
			if(gsess->csum == 0) {
				print_timestamp("Input"); 
				gsess_output(gsess,"+",1);
				result = gsess_execute_cmd(gsess);
			} else {
				fprintf(stderr,"Checksum error in gdb packet\n");
				gsess_output(gsess,"-",1);
			}
			break;
		break;
	}
	return result;
}
/*
 * ---------------------------------------------------------
 * gsess_input
 * 	The data sink for commands from gdb. It is called
 *	when filedescriptor is readable
 * ---------------------------------------------------------
 */

static int
gsess_input(void *cd,int mask) {
        int result;
        GdbSession *gsess = cd;
        uint8_t c;
        while((result=read(gsess->fd,&c,1))==1) {
                if(feed_state_machine (gsess, c)<0) {
                	gsess_del(gsess);
                        return 0;
                }
        }
        if((result<0) && (errno==EAGAIN))  {
                return 0;
        } else {
                dprintf("Connection lost\n");
                gsess_del(gsess);
                return 0;
        }
}


static void
gserv_accept(int fd,char *host,unsigned short port,void *clientData) 
{
	GdbSession *gsess;	
	int on=1;
	GdbServer *gserv = (GdbServer *)clientData;
	if(gserv->first_gsess) {
		fprintf(stderr,"Only one gdb session allowed\n");
		close(fd);
		return;
	}
	gsess = malloc(sizeof(GdbSession));	
	if(!gsess) {
		fprintf(stderr,"Out of memory allocating GDB session\n");
		exit(344);
	}
	memset(gsess,0,sizeof(GdbSession));
	gserv->first_gsess = gsess;
	gsess->fd = fd;
	gsess->gserv = gserv;
	gsess->cmdstate = CMDSTATE_WAIT_START;
	gsess->dbgops = gserv->dbgops;
	gsess->backend = gserv->backend;
	fcntl(gsess->fd,F_SETFL,O_NONBLOCK);
	if(setsockopt(gsess->fd,IPPROTO_TCP,TCP_NODELAY,(void *)&on,sizeof(on)) < 0) {
                        perror("Error setting sockopts");
	}
	FIO_AddFileHandler(&gsess->rfh,gsess->fd,FIO_READABLE,gsess_input,gsess);
	gsess->rfh_is_active = 1;
}

GdbServer *
GdbServer_New(GDebugOps *dbgops,void *backend)
{
        int fd;
	int port;
        GdbServer *gserv;
	char *host=Config_ReadVar("gdebug","host");
	if(!host || (Config_ReadInt32(&port,"gdebug","port")<0)) {
		fprintf(stderr,"GDB server is not configured\n");	
		return NULL;
	}
        gserv=malloc(sizeof(GdbServer));
        if(!gserv) {
		fprintf(stderr,"Out of memory for GDB server\n");
                return NULL;
	}
	memset(gserv,0,sizeof(GdbServer));
	gserv->dbgops = dbgops;
	gserv->backend = backend;
        if((fd=FIO_InitTcpServer(&gserv->gserv,gserv_accept,gserv,host,port))<0) {
                free(gserv);
                return NULL;
        }
        fprintf(stderr,"GDB server listening on host \"%s\" port %d\n",host,port);
	return gserv;
}
