/*
 * -----------------------------------------------------------------------
 * Emulation of MMC/SD-Cards 
 *
 * (C) 2006 Jochen Karrer
 *   Author: Jochen Karrer
 *
 * state: working with u-boot and linux with i.MX21 SDHC emulator
 * 	Secure commands are missing because they are secret
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * -----------------------------------------------------------------------
 */

#include <errno.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <libgen.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <bus.h>
#include <fio.h>
#include <signode.h>
#include <configfile.h>
#include <cycletimer.h>
#include <mmcard.h>
#include <mmcproto.h>
#include <diskimage.h>


#if 0
#define dprintf(x...) { fprintf(stderr,x); }
#else
#define dprintf(x...)
#endif

#define STATE_IDLE 		(0)
#define STATE_READY		(1)
#define STATE_IDENT		(2)
#define STATE_STBY		(3)
#define	STATE_TRANSFER		(4)
#define STATE_DATA		(5)
#define STATE_RCV		(6)
#define STATE_PRG		(7)
#define STATE_DIS	 	(8)
#define STATE_INACTIVE  	(10)
#define GET_STATUS(card) ((card)->card_status & ~STATUS_STATE_MASK) | ((card)->state << STATUS_STATE_SHIFT)

#define RST_NOT_STARTED 	(7)
#define RST_STARTED 		(8)
#define RST_DONE 		(9)
typedef struct MMCardSpec {
	char *producttype;
	int type;
	int usec_reset;
	uint32_t ocr;
	uint8_t cid[16];
	uint8_t csd[16];
	uint8_t scr[8];
} MMCardSpec;

static MMCardSpec cardspec[] = {
	{
		/* This is also the auto_sd template */
		.producttype = "Toshiba32M",
		.type = CARD_TYPE_SD,
		.ocr = 0x80ff8000,
		.usec_reset = 5000,
		.cid = {0x02, /* Toshiba */
			0x54, /* 'T'	*/
			0x4d, /* 'M'	*/
			0x53, /* 'S'	*/
			0x44, /* 'D'	*/
			0x30, /* '0'	*/
			0x33, /* '3'	*/
			0x32, /* '2'	*/
			0, 	/* Product revision */	
			/* Serial number has to be inserted from configuration file */
			0,0,0,0,
	 		0,
	 		0x63,
	 		1, // crc has to be inserted by software 
		},
		.scr = { 0, 0xa5,0,0,0,0,0,0 },
		.csd = {0,    /* 			Bits 120-127 */
			0x26, /* TAAC  			Bits 112-119 */
			0,    /* NSAC  			Bits 104-111 */
			0x32, /* TRAN_SPEED 		Bits 96-103 */
			0x13, /* CCC High 		Bits 88-95 */
			0x59, /* CCC low, block len 512 	Bits 80-87 */
			0x81, /*				Bits 72-79 */ 
			0xd2, /*				Bits 64-71 */
			0xf6, /*				Bits 56-63 */
			0xd9, /*				Bits 48-55 */
	 		0xc0, /*			Bits 40-47 */
	 		0x1f,
	 		0x12,
	 		0x40,
	 		0x40,
	 		0x0,
		}
	},
	{
		.producttype = "Toshiba64M",
		.type = CARD_TYPE_SD,
		.ocr = 0x80ff8000,
		.usec_reset = 5000,
		.cid = {
			0x02, 0x54, 0x4d, 0x53, 0x44, 0x30, 0x36, 0x34,
			0x07, 0x51, 0x17, 0xd4, 0x11, 0x00, 0x49, 0x00,
		},
		.scr = {
			0x00, 0xa5, 0x00, 0x00, 0x09, 0x02, 0x02, 0x02,
		},
		.csd = {
			0x00, 0x2d, 0x00, 0x32, 0x13, 0x59, 0x83, 0xb1,
			0xf6, 0xd9, 0xcf, 0x80, 0x16, 0x40, 0x00, 0x00,
		},
	},
	{
		/* This is also the auto_mmc template */
		.producttype = "ExtremeMemory128M",
		.type = CARD_TYPE_MMC,
		.ocr = 0x80ff8000,
		.usec_reset = 5000,
		.cid = {
			0x01, 0x00, 0x00, 0x49, 0x46, 0x58, 0x31, 0x32,
			0x38, 0x20, 0x19, 0xf2, 0xee, 0x0d, 0x17, 0x00,
		},
		.scr = {},
		.csd = {
			0x48, 0x0e, 0x01, 0x2a, 0x0f, 0xf9, 0x81, 0xe9,
			0xf6, 0xda, 0x81, 0xe1, 0x8a, 0x40, 0x00, 0x00,
		},
	}
};

struct MMCard {
	int type; /* SD/MMC .. */
	int state;
	int usec_reset;
	int is_app_cmd; /* > 0 app cmd state */
	int bus_width;
	uint32_t card_status;
	uint32_t ocr;
	uint8_t cid[16];
	uint8_t csd[16];
	uint16_t rca;
	uint16_t dsr;
	uint8_t scr[8];
	uint32_t blocklen;
	uint32_t erase_group_start;
	uint32_t erase_group_end;
	DiskImage *disk_image;
	uint32_t capacity;
	uint8_t cmd;	  /* command attribute for data read/write operations */
	uint32_t address; /* address attribute for data read/write operations */
	uint32_t transfer_count;  /* transfer_count incremented during data read/write operations */ 
	/* Reset delay timer: real card needs some undocumented time for reset */
	int reset_state;
	CycleCounter_t reset_start_time;
};

static void
dump_cardtypes() {
	int nr_types = sizeof(cardspec)/sizeof(MMCardSpec);
	int i;
	fprintf(stderr,"Possible card types:\n");
	for(i=0;i<nr_types;i++) {
		fprintf(stderr,"\"%s\"\n",cardspec[i].producttype);
	}
	fprintf(stderr,"\"auto_sd\"\n");
	fprintf(stderr,"\"auto_mmc\"\n");
}

static int 
get_bit(uint8_t *array,int bit) {
	int ofs = bit >> 3;	
	bit = bit & 7;
	return array[ofs] & (1<<bit) ? 1 : 0;
}

/*
 * ----------------------------------------------
 * CMD0: go_idle
 * No response
 * ----------------------------------------------
 */
static int
mmc_go_idle(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	card->state = STATE_IDLE;
	card->reset_state = RST_NOT_STARTED;
	card->rca=0;
	card->card_status = STATUS_READY_FOR_DATA;
	resp->len = 0;
	return MMC_ERR_NONE;
}

/*
 * -------------------------------------------------------------------------
 * send op cond: CMD1 Response format R3
 * 
 * -------------------------------------------------------------------------
 */
static int
mmc_send_op_cond(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	if(card->state != STATE_IDLE) { 
		fprintf(stderr,"MMCard: got SEND_OP_COND cmd in non idle state (%d)\n",card->state);
		resp->len = 0;
		return MMC_ERR_FAILED;
	}
	fprintf(stderr,"mmc op cond state %d\n",card->reset_state); // jk
	if(card->reset_state == RST_NOT_STARTED) {
		card->reset_start_time = CycleCounter_Get();
		card->ocr &= ~OCR_NOTBUSY;
		card->reset_state = RST_STARTED;
	} else if(card->reset_state == RST_STARTED) {
		int64_t usec = CyclesToMicroseconds(CycleCounter_Get() - card->reset_start_time);
		if(usec > card->usec_reset) {
			card->state = STATE_READY;
			card->ocr |= OCR_NOTBUSY;
			card->reset_state = RST_DONE;
		}
	} else if(card->reset_state != RST_DONE) {
		fprintf(stderr,"Emulator bug: MMC-Card reset_state %d not valid\n",card->reset_state);
		exit(1);
	}
	resp->data[0] = 0x3f;	
	resp->data[1] = card->ocr >> 24;
	resp->data[2] = card->ocr >> 16;
	resp->data[3] = card->ocr >> 8;
	resp->data[4] = card->ocr;
	resp->data[5] = 0xff;	
	resp->len = 6;
	dprintf("Send op cond arg 0x%08x\n",arg);
	return MMC_ERR_NONE;
}

/*
 * ---------------------------------------------------------------------
 * CMD2, no argument
 * Response format R2 (136 Bits)
 * doesn't respond when  rca is not 0.  (Toshiba docu says this)
 * ---------------------------------------------------------------------
 */
static int
mmc_all_send_cid(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int i;
	if(card->rca != 0) {
		return MMC_ERR_TIMEOUT;
	}
	if(card->state != STATE_READY) {
		/* Real card seems to behave this way ? Or is it the controller ? */
		fprintf(stderr,"ALL_SEND_CID: card not in ready state\n");
		resp->len = 0;
		return MMC_ERR_TIMEOUT;
	}
	resp->len = 17;
	resp->data[0] = 0x3f;
	for(i=0;i<16;i++) {
		resp->data[1+i] = card->cid[i]; 
	}
	card->state = STATE_IDENT;
	return MMC_ERR_NONE;
}
/*
 * ---------------------------------------------------------------------------------
 * CMD3 for MMC cards: set relative address
 * for SD-Cards see sd_send relative address
 * Argument: Bits 16-31 RCA
 * Response format R1
 * --------------------------------------------------------------------------------
 */
static int
mmc_set_relative_addr(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint32_t card_status = GET_STATUS(card);
	if(card->state != STATE_IDENT) {
		fprintf(stderr,"Got SET_RCA but not in IDENT state (%d)\n",card->state);
		resp->len = 0;
		return MMC_ERR_TIMEOUT;
	}
	card->state = STATE_STBY;
	card->rca = (arg >> 16) & 0xffff;
	fprintf(stderr,"New rca is %d\n",card->rca);
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}
/*
 * ---------------------------------------------------------------------------------
 * CMD3 for SD-Cards, send relative address
 * Argument: stuff bits
 * Response format R6
 * --------------------------------------------------------------------------------
 */
static int
sd_send_relative_addr(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint32_t card_status = GET_STATUS(card);
	if(card->state != STATE_IDENT) {
		resp->len = 0;
		return MMC_ERR_TIMEOUT;
	}
	card->state = STATE_STBY;
	card->rca++;
	if(!card->rca)
		card->rca++;
	//card->card_status  &= ~(0xfd3fc020);
	card->card_status  &= ~(0xc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card->rca >> 8) & 0xff;
	resp->data[2] = card->rca & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

/*
 * -------------------------------------------------------------
 * CMD4 SET_DSR optional command
 * only present when bit 96 in CSD is set
 * STATE_STBY -> STATE_STBY (Sandisk state table)
 * -------------------------------------------------------------
 */
static int
mmc_set_dsr(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	if(!get_bit(card->csd,CSD_BIT_DSR_IMP)) {
		return MMC_ERR_TIMEOUT;	
	}	
	if(card->state != STATE_STBY) {
		fprintf(stderr,"MMCard: Got set dsr cmd in wrong state %d\n",card->state);
		return MMC_ERR_TIMEOUT;
	}
	card->dsr = arg>>16;	
	return MMC_ERR_NONE;
}


/*
 * ------------------------------------------------------------------------------
 * CMD6
 *	MMC_SWITCH
 * ------------------------------------------------------------------------------
 */ 
static int
mmc_switch(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	fprintf(stderr,"MMC switch command not implemented\n");
	return MMC_ERR_TIMEOUT;
}

/*
 * ------------------------------------------------------------------------------
 * CMD7 select Card
 * Response format R1 arg argument RCA
 * ------------------------------------------------------------------------------
 */
static int
mmc_select_card(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint16_t rca = arg >> 16;
	uint32_t card_status = GET_STATUS(card);
	if((card->rca != rca) || (rca == 0)) {
		if((card->state == STATE_STBY) || (card->state == STATE_TRANSFER) 
		   || (card->state == STATE_DATA)) {
			card->state = STATE_STBY;
		} else if(card->state == STATE_PRG) {
			card->state = STATE_DIS;
		} else {
			fprintf(stderr,"CMD7 unsel card not allowed in state %d\n",card->state);
			return MMC_ERR_TIMEOUT;
		}
	} else {
		if((card->state == STATE_DIS)) {
			card->state = STATE_PRG;
		} else if(card->state == STATE_STBY) {
			card->state = STATE_TRANSFER;
		} else {
			fprintf(stderr,"CMD7 sel card not allowed in state %d\n",card->state);
			return MMC_ERR_TIMEOUT;
		}
	}
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}
/*
 * ------------------------------------------------------------------------------
 * CMD8
 *	MMC_SEND_EXT_CSD
 * ------------------------------------------------------------------------------
 */ 
static int
mmc_send_ext_csd(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	fprintf(stderr,"MMC_SEND_EXT_CSD  command not implemented\n");
	return MMC_ERR_TIMEOUT;
}

/*
 * -----------------------------------------------------------------------
 * CMD9 MMC_SEND_CSD
 *	read card specific data 
 * arg: rca in high
 * response: 136 Bit R2
 * STATE_STBY -> STATE_STBY
 * -----------------------------------------------------------------------
 */
static int
mmc_send_csd(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int i;
	uint16_t rca = arg>>16;
	if(card->state != STATE_STBY) {
		fprintf(stderr,"MMCard: SEND_CSD but not in standby state\n");
		return MMC_ERR_TIMEOUT;
	}
	if(rca!=card->rca) {
		fprintf(stderr,"SEND CSD: card not selected, rca %d\n",rca);
		dprintf("SEND CSD: card not selected\n");
		return MMC_ERR_TIMEOUT;
	}
	resp->len = 17;
	resp->data[0] = 0x3f;
	for(i=0;i<16;i++) {
		resp->data[1+i] = card->csd[i]; 
	}
	return MMC_ERR_NONE;
}

/*
 * -----------------------------------------------------------------
 * CMD10 SEND_CID
 * arg: rca
 * return R2 (128 + 8 Bits)
 * STATE_STBY -> STATE_STBY
 * -----------------------------------------------------------------
 */

static int
mmc_send_cid(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int i;
	uint16_t rca = arg>>16;
	if(card->state != STATE_STBY) {
		resp->len = 0;
		return MMC_ERR_TIMEOUT;
	}
	if(rca!=card->rca) {
		dprintf("SEND CID: card not selected\n");
		return MMC_ERR_TIMEOUT;
	}
	resp->len = 17;
	resp->data[0] = 0x3f;
	for(i=0;i<16;i++) {
		resp->data[1+i] = card->cid[i]; 
	}
	return MMC_ERR_NONE;
}

/*
 * ---------------------------------------------------------
 * CMD11 READ_DATA_UNTIL_STOP
 * arg: none
 * resp R1 
 * STATE ??? not found in SanDisk docu
 * ---------------------------------------------------------
 */
static int
mmc_read_dat_until_stop(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint32_t card_status = GET_STATUS(card);
	if(card->state != STATE_TRANSFER) {
		fprintf(stderr,"MMCard: read data until stop in wrong state %d\n",card->state);
		return MMC_ERR_TIMEOUT;
	}
	/* Store the attributes for the recognized data operation */
	card->cmd = MMC_READ_DAT_UNTIL_STOP;
	card->address = arg;
	card->transfer_count = 0;

	card->state = STATE_DATA;
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

/*
 * ---------------------------------------------------------------------------------
 * CMD12 STOP_TRANSMISSION
 * arg: none
 * resp: R1b 
 * STATE: Read:  STATE_DATA -> STATE_TRANSFER
 *	  Write: STATE_RCV -> STATE_PRG ... (delay) -> STATE_TRANSFER 
 * ---------------------------------------------------------------------------------
 */
static int
mmc_stop_transmission(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	uint32_t card_status = GET_STATUS(card);
	if((card->state != STATE_DATA) || (card->state == STATE_RCV)) {
		fprintf(stderr,"MMCard: STOP_TRANSMISSION but in wrong state: %d\n",card->state);
		return MMC_ERR_TIMEOUT;
	}
	if(card->state == STATE_DATA) {
		card->state = STATE_TRANSFER;
	} else if(card->state == STATE_RCV) {
		/* This should go to STATE_PRG and after a delay go to STATE_TRANSFER ! */
		card->state = STATE_TRANSFER; 
	} else {
		fprintf(stderr,"MMCard Bug: stop tranmission in state %d\n",card->state);
	}
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}
/*
 * -------------------------------------------------------------------
 * CMD13 MMC_SEND_STATUS
 * arg rca
 * response format R1
 * Keep STATE 
 * -------------------------------------------------------------------
 */
static int
mmc_send_status(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int state = card->state;
	uint32_t card_status = GET_STATUS(card);
	uint16_t rca = (arg >> 16) & 0xffff; 
	if(rca != card->rca) {
		dprintf("MMC SEND STATUS: Card not selected\n");
		return MMC_ERR_TIMEOUT;
	}
	if((state != STATE_STBY) && (state != STATE_TRANSFER) && (state != STATE_DATA)
		&& (state != STATE_RCV) && (state != STATE_PRG) && (state != STATE_DIS)) {
		fprintf(stderr,"MMC SEND STATUS: in wrong state %d\n",state);
		return MMC_ERR_TIMEOUT;
	}
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	dprintf("MMCard CMD13: Card status %08x\n",card->card_status);
	return MMC_ERR_NONE;
}

/*
 * ------------------------------------------------------------------------------
 * CMD14
 *	MMC_BUSTEST_R
 * ------------------------------------------------------------------------------
 */ 
static int
mmc_bustest_r(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	fprintf(stderr,"MMC_BUSTEST_R  command not implemented\n");
	return MMC_ERR_TIMEOUT;
}
/*
 * ------------------------------------------------------------------
 * CMD15 GO_INACTIVE
 * arg RCA
 * response none
 * STATE_xxx -> STATE_INACTIVE (Sandisk manual)
 * ------------------------------------------------------------------
 */
static int
mmc_go_inactive_state(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint16_t rca = (arg >> 16) & 0xffff; 
	int state = card->state;
	resp->len = 0;
	if(rca != card->rca) {
		fprintf(stderr,"MMC SEND STATUS: Card not selected\n");
		return MMC_ERR_TIMEOUT;
	}
	if((state != STATE_STBY) && (state != STATE_TRANSFER) && (state != STATE_DATA)
		&& (state != STATE_RCV) && (state != STATE_PRG) && (state != STATE_DIS)) {
		fprintf(stderr,"MMC SEND STATUS: in wrong state %d\n",state);
		return MMC_ERR_TIMEOUT;
	}
	card->state = STATE_INACTIVE;
	return MMC_ERR_NONE;
}

/*
 * -----------------------------------------------------------------------------------------------
 * CMD16 MMC_SET_BLOCKLEN 
 * arg: Blocklen in bytes for all following block read write commands
 *  Response R1
 * STATE_TRANFER -> STATE_TRANSFER
 * -----------------------------------------------------------------------------------------------
 */
static int
mmc_set_blocklen(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int state = card->state;
	uint32_t card_status = GET_STATUS(card);
	if(state != STATE_TRANSFER) {
		fprintf(stderr,"MMC SET BLOCKLEN: in wrong state %d\n",state);
		return MMC_ERR_TIMEOUT;
	}
	card->blocklen = arg;
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

/*
 * ------------------------------------------------------------------------
 * CMD17 Read single block
 *	Argument: address
 *	Response Format R1
 * STATE_TRANSFER -> STATE_DATA
 * ------------------------------------------------------------------------
 */
static int
mmc_read_single_block(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int state = card->state;
	uint32_t card_status = GET_STATUS(card);
	if(state != STATE_TRANSFER) {
		fprintf(stderr,"MMC READ single block: in wrong state %d\n",state);
		return MMC_ERR_TIMEOUT;
	}
	/* Store the attributes for the recognized data operation */
	card->cmd = MMC_READ_SINGLE_BLOCK;
	card->address = arg;
	card->transfer_count=0;

	card->state = STATE_DATA;
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

/*
 * --------------------------------------------------------------------------
 * CMD18
 * 	MMC read multiple block
 *	Argument: address 
 *	Response format R1
 * --------------------------------------------------------------------------
 */
static int
mmc_read_multiple_block(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int state = card->state;
	uint32_t card_status = GET_STATUS(card);
	if(state != STATE_TRANSFER) {
		fprintf(stderr,"MMC READ multiple block: in wrong state %d\n",state);
		return MMC_ERR_TIMEOUT;
	}

	/* Store the attributes for the recognized data operation */
	card->cmd = MMC_READ_MULTIPLE_BLOCK;
	card->address = arg;
	card->transfer_count=0;

	card->state = STATE_DATA;
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

/*
 * ------------------------------------------------------------------------------
 * CMD19
 *	MMC_BUSTEST_W
 * ------------------------------------------------------------------------------
 */ 
static int
mmc_bustest_w(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	fprintf(stderr,"MMC_BUSTEST_W  command not implemented\n");
	return MMC_ERR_TIMEOUT;
}

/*
 * --------------------------------------------------------------------
 * CMD20 MMC_WRITE_DAT_UNTIL_STOP
 * arg: address
 * response R1
 * STATE: ???? not mentioned in SanDisk docu
 * --------------------------------------------------------------------
 */
static int
mmc_write_dat_until_stop(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint32_t card_status = GET_STATUS(card);
	if(card->state != STATE_TRANSFER) {
		fprintf(stderr,"MMCard: write dat until stop in wrong state %d\n",card->state);
		return MMC_ERR_TIMEOUT;
	}
	/* Store the attributes for the recognized data operation */
	card->cmd = MMC_WRITE_DAT_UNTIL_STOP;
	card->address = arg;
	card->transfer_count=0;

	card->state = STATE_DATA;
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

/* 
 * ---------------------------------------------------------------------------------
 * CMD23 MMC_SET_BLOCK_COUNT
 * arg: bits 0-15 block count
 *	16-31: 0
 * response: R1
 * STATE: ?????
 * Set block count for immediately following CMD 18 (read mult.) (write mult ????)
 * Found no vendor supporting this command
 * ---------------------------------------------------------------------------------
 */ 
static int
mmc_set_block_count(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint32_t card_status = GET_STATUS(card);
	if(card->state != STATE_TRANSFER) {
		fprintf(stderr,"MMCard: set_block_count in wrong state %d\n",card->state);
		return MMC_ERR_TIMEOUT;
	}
	fprintf(stderr,"MMCard: Warning: set block count %d is ignored\n",arg);
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

/*
 * ------------------------------------------------------------------------
 * CMD24 MMC_WRITE_SINGLE_BLOCK
 *	arg: address
 *	resp: R1
 *	STATE_TRANSFER -> STATE_RCV	
 * ------------------------------------------------------------------------
 */
static int
mmc_write_single_block(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int state = card->state;
	uint32_t card_status = GET_STATUS(card);
	if(state != STATE_TRANSFER) {
		fprintf(stderr,"MMC WRITE single block: in wrong state %d\n",state);
		return MMC_ERR_TIMEOUT;
	}
	/* Store the attributes for the recognized data operation */
	card->cmd = MMC_WRITE_SINGLE_BLOCK;
	card->address = arg;
	card->transfer_count=0;

	card->state = STATE_RCV;
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

/*
 * ------------------------------------------------------------------------
 * CMD25 MMC_WRITE_MULTIPLE_BLOCK
 *	arg: address
 *	resp: R1
 *	STATE_TRANSFER -> STATE_RCV	
 * ------------------------------------------------------------------------
 */
static int
mmc_write_multiple_block(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int state = card->state;
	uint32_t card_status = GET_STATUS(card);
	if(state != STATE_TRANSFER) {
		fprintf(stderr,"MMC write multiple block: in wrong state %d\n",state);
		return MMC_ERR_TIMEOUT;
	}
	/* Store the attributes for the recognized data operation */
	card->cmd = MMC_WRITE_MULTIPLE_BLOCK;
	card->address = arg;
	card->transfer_count=0;

	card->state = STATE_RCV;
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

/* 
 * ----------------------------------------------------------------------------
 * CMD26 PROGRAM_CID 
 * reserved for manufacturer
 * ----------------------------------------------------------------------------
 */
static int
mmc_program_cid(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	fprintf(stderr,"MMCard: Reserved command PROGRAM_CID\n");
	return MMC_ERR_TIMEOUT;
}

/* 
 * ----------------------------------------------------------------------------
 * CMD26 PROGRAM_CSD 
 * reserved for manufacturer
 * ----------------------------------------------------------------------------
 */
static int
mmc_program_csd(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	fprintf(stderr,"MMCard: Reserved command PROGRAMM_CSD\n");
	return MMC_ERR_TIMEOUT;
}

/*
 * -------------------------------------------------------------------------------
 * CMD28 SET_WRITE_PROT
 * arg: address of sector group
 * Response R1b
 * Set writeprotect for sector group. Group and sector size is stored in CSD
 * -------------------------------------------------------------------------------
 */
static int
mmc_set_write_prot(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	fprintf(stderr,"SET WRITE PROT not implemented\n");
	return MMC_ERR_TIMEOUT;
}

/*
 * ---------------------------------------------------------------------------
 * CMD29 CLR_WRITE_PROT
 * arg: address of sector group
 * Response R1b
 * ---------------------------------------------------------------------------
 */
static int
mmc_clr_write_prot(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	fprintf(stderr,"CLR_WRITE_PROT\n");
	return MMC_ERR_FAILED;
}

/*
 * ---------------------------------------------------------------------------
 * CMD30 SEND_WRITE_PROT
 * Response Format R1
 * Send the write protection of 32 groups on the data lines 
 * ---------------------------------------------------------------------------
 */
static int
mmc_send_write_prot(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	fprintf(stderr,"cmd 0x%02x not implemented\n",cmd);
	return MMC_ERR_FAILED;
}

/*
 * ------------------------------------------------------------------------
 * CMD35 erase group start
 * arg: address of first write block to be erased (erase group ???)
 * Response R1
 * State ?
 * ------------------------------------------------------------------------
 */
static int
mmc_erase_group_start(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint32_t card_status = GET_STATUS(card);
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	card->erase_group_start = arg;
	return MMC_ERR_NONE;
}

/*
 * ----------------------------------------------------------------------
 * CMD36 ERASE_GROUP_END
 * arg: address of last write block to be erased 
 * Response R1
 * ----------------------------------------------------------------------
 */
static int
mmc_erase_group_end(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint32_t card_status = GET_STATUS(card);
	card->card_status  &= ~(0xfd3fc020);
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	card->erase_group_end = arg;
	return MMC_ERR_NONE;
}

/*
 * ---------------------------------------------------------------------------
 * CMD38 ERASE
 * erase previously selected blocks
 * arg: none
 * Response R1b
 * ---------------------------------------------------------------------------
 */
static int
mmc_erase(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	fprintf(stderr,"cmd 0x%02x not implemented\n",cmd);
	return MMC_ERR_FAILED;
}

static int
mmc_fast_io(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	fprintf(stderr,"cmd 0x%02x not implemented\n",cmd);
	return MMC_ERR_FAILED;
}

static int
mmc_go_irq_state(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	fprintf(stderr,"cmd 0x%02x not implemented\n",cmd);
	return MMC_ERR_FAILED;
}

/*
 * ----------------------------------------------------------
 * CMD55 app_cmd
 * Response format R1
 * ----------------------------------------------------------
 */
static int
mmc_app_cmd(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint32_t card_status = GET_STATUS(card) | STATUS_APP_CMD;
	uint16_t rca = (arg >> 16) & 0xffff; 
	/* behaviour for rca == 0 ???? */
	if(/*(rca != 0) &&*/(rca != card->rca)) {
		dprintf("MMC APP CMD: card not selected\n");
		fprintf(stderr,"MMC APP CMD: card not selected, rca %d instead of %d\n",rca,card->rca);
		return MMC_ERR_TIMEOUT;
	}
	card->card_status  &= ~(0xfd3fc020);
	card->is_app_cmd = 1;
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}

static int
mmc_gen_cmd(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	fprintf(stderr,"cmd 0x%02x not implemented\n",cmd);
	return MMC_ERR_FAILED;
}

/*
 * ----------------------------------------------------------------------------
 * ACMD6 SD_APP_SET_BUS_WIDTH
 * ----------------------------------------------------------------------------
 */
static int
sd_app_set_bus_width(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{

	uint32_t card_status = GET_STATUS(card) | STATUS_APP_CMD;
	if(card->type != CARD_TYPE_SD) {
		fprintf(stderr,"SD_APP_SET_BUS_WIDTH: Not an SD-Card !\n");
		return MMC_ERR_TIMEOUT;
	} 
	if(card->state != STATE_TRANSFER) {
		fprintf(stderr,"MMCard: SD_APP_BUS_WIDTH in state %d\n",card->state);
		return MMC_ERR_TIMEOUT;
	}
	card->card_status  &= ~(0xfd3fc020);
	card->bus_width = arg;
	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	return MMC_ERR_NONE;
}


/*
 * ----------------------------------------------------------------------
 * SD app op condition CMD41 (0x29)
 * Response format R3
 * STATE_IDLE -> STATE_READY when voltage good and card not busy else
 * go to inactive
 * ----------------------------------------------------------------------
 */
static int
sd_app_op_cond(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	resp->len = 0;
	if(card->type != CARD_TYPE_SD) {
		fprintf(stderr,"SD_AP_OP_COND for non SD-Card\n");
		return MMC_ERR_FAILED;
	}
	if(card->state != STATE_IDLE) {
		resp->len = 0;
		fprintf(stderr,"SD-Card: SD_AP_OP_COND when in IDLE state (%d)\n",card->state);
		return MMC_ERR_TIMEOUT;	
	}
	if(card->reset_state == RST_NOT_STARTED) {
		card->reset_start_time = CycleCounter_Get();
		card->ocr &= ~OCR_NOTBUSY;
		card->reset_state = RST_STARTED;
	} else if(card->reset_state == RST_STARTED) {
		int64_t usec = CyclesToMicroseconds(CycleCounter_Get()-card->reset_start_time);
		if(usec > card->usec_reset) {
			card->state = STATE_READY;
			card->ocr |= OCR_NOTBUSY;
			card->reset_state = RST_DONE;
		}
	} else if(card->reset_state != RST_DONE) {
		fprintf(stderr,"Emulator bug: MMC-Card reset_state %d not valid\n",card->reset_state);
		exit(1);
	}
	// init card is missing here, need some time, return busy the first few times
	resp->data[0] = 0x3f;	
	resp->data[1] = card->ocr >> 24;
	resp->data[2] = card->ocr >> 16;
	resp->data[3] = card->ocr >> 8;
	resp->data[4] = card->ocr;
	resp->data[5] = 0xff;	
	resp->len = 6;
	dprintf("SD app op cond arg 0x%08x\n",arg);
	return MMC_ERR_NONE;
}

/*
 * ------------------------------------------------------------------
 * ACMD51: SEND SCR
 * Response format R1, dummy argument, goes to data state
 * ------------------------------------------------------------------
 */
static int
sd_app_send_scr(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	uint32_t card_status = GET_STATUS(card);
	if(card->type != CARD_TYPE_SD) {
		fprintf(stderr,"SD_APP_SEND_SCR: Not an SD-Card !\n");
		return MMC_ERR_TIMEOUT;
	} 
	if(card->state != STATE_TRANSFER) {
		fprintf(stderr,"Send scr in wrong state\n");
		return MMC_ERR_TIMEOUT;
	}
	card->transfer_count = 0;
	card->state = STATE_DATA;
	card->card_status  &= ~(0xfd3fc020);

	/* Store the attributes for the recognized data operation */
	card->cmd = cmd;

	resp->len = 6;
	resp->data[0] = cmd & 0x3f;
	resp->data[1] = (card_status >> 24) & 0xff;
	resp->data[2] = (card_status >> 16) & 0xff;
	resp->data[3] = (card_status >> 8)  & 0xff;
	resp->data[4] = card_status & 0xff;
	resp->data[5] = 1; /* CRC missing here */
	resp->len = 0;
	/* put data to somewhere */
	return MMC_ERR_NONE;
}

/*
 * ----------------------------------------------------------------
 * MMCard_Read:
 * retval: number of Bytes, -ERRCODE on error  0=eofdata
 * ----------------------------------------------------------------
 */
int 
MMCard_Read(MMCard *card,uint8_t *buf,int count) 
{
	int i;
	if(!card) {
		return 0;
	}
	if(card->state != STATE_DATA) {
		return 0;
	}
	if(card->cmd == SD_APP_SEND_SCR) {
		for(i=0;(i<count) && (card->transfer_count < 8);i++) {
			buf[i]= card->scr[card->transfer_count];
			card->transfer_count++;
		}
		if(card->transfer_count == 8) {
			card->state = STATE_TRANSFER; 
		}
		return i;
	} else if(card->cmd ==  MMC_READ_SINGLE_BLOCK) {
		uint32_t address = card->address + card->transfer_count;
		if(card->transfer_count + count > card->blocklen) {
			count = card->blocklen - card->transfer_count;
			if(count<0) {
				fprintf(stderr,"transfer count < 0 should never happen\n");
				return 0;
			}
		}
		if(address >= card->capacity) {
			return 0;
		}
		if((address+count) > card->capacity) { 
			count = card->capacity - address;
		}
		if(DiskImage_Read(card->disk_image,address,buf,count)<count) {
			fprintf(stderr,"MMCard: Error reading from diskimage\n");
		}
		card->transfer_count += count;
		if(card->transfer_count == card->blocklen) {
			card->state = STATE_TRANSFER;
		}
		return count;
	} else if(card->cmd == MMC_READ_MULTIPLE_BLOCK) {
		uint32_t address = card->address + card->transfer_count;
		if((address+count) > card->capacity) { 
			count = card->capacity - address;
		}
		if(DiskImage_Read(card->disk_image,address,buf,count)<count) {
			fprintf(stderr,"MMCard: Error reading from diskimage\n");
		}
		card->transfer_count += count;
		return count;
	} else if(card->cmd == MMC_READ_DAT_UNTIL_STOP) {
		uint32_t address = card->address + card->transfer_count;
		if((address+count) > card->capacity) { 
			count = card->capacity - address;
		}
		if(DiskImage_Read(card->disk_image,address,buf,count) < count) {
			fprintf(stderr,"MMCard: Error reading from diskimage\n");
		}
		card->transfer_count += count;
		return count;
	} else {
		fprintf(stderr,"MMC card read with unknown command %d\n",card->cmd);
		return 0;
	}
}

/*
 * ------------------------------------------------------------------
 * MMCard_Write
 * retval: number of Bytes, -errcode on error
 * Busy polling implenentation is missing
 * ------------------------------------------------------------------
 */
int 
MMCard_Write(MMCard *card,const uint8_t *buf,int count) 
{
	if(!card) {
		return 0;
	}
	if(card->state != STATE_RCV) {
		return 0;
	} 
	if(card->cmd ==  MMC_WRITE_SINGLE_BLOCK) {
		uint32_t address  = card->address + card->transfer_count;
		if(card->transfer_count + count > card->blocklen) {
			count = card->blocklen - card->transfer_count;
			if(count<0) {
				fprintf(stderr,"transfer count < 0 should never happen\n");
				return 0;
			}
		}
		if((address+count) > card->capacity) { 
			count = card->capacity - address;
		}
		if(DiskImage_Write(card->disk_image,address,buf,count)<count) {
			fprintf(stderr,"MMCard: Error writing to diskimage\n");
		}
		card->transfer_count += count;
		if(card->transfer_count == card->blocklen) {
			card->state = STATE_TRANSFER;
		}
		return count;
	} else if(card->cmd == MMC_WRITE_MULTIPLE_BLOCK) { 
		uint32_t address  = card->address + card->transfer_count;
		if((address+count) > card->capacity) { 
			count = card->capacity - address;
		}
		if(DiskImage_Write(card->disk_image,address,buf,count)<count) {
			fprintf(stderr,"MMCard: Error writing to diskimage\n");
		}
		card->transfer_count += count;
		return count;
	} else if(card->cmd == MMC_WRITE_DAT_UNTIL_STOP) { 
		uint32_t address  = card->address + card->transfer_count;
		if((address+count) > card->capacity) { 
			count = card->capacity - address;
		}
		if(DiskImage_Write(card->disk_image,address,buf,count)<count) {
			fprintf(stderr,"MMCard: Error reading from diskimage\n");
		}
		card->transfer_count += count;
		return count;
	} else {
		fprintf(stderr,"Write cmd %d not known\n",card->cmd);
		return 0;
	}
}

static int
DoAppCmd(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{

	int result = MMC_ERR_FAILED;
	switch(cmd) {
		case SD_APP_SET_BUS_WIDTH:
			result = sd_app_set_bus_width(card,cmd,arg,resp);
			break;

		case SD_APP_OP_COND:
			result = sd_app_op_cond(card,cmd,arg,resp);
			break;

		case SD_APP_SEND_SCR:
			result = sd_app_send_scr(card,cmd,arg,resp);	
			break;
		default:
			fprintf(stderr,"Unrecognized app cmd %d\n",cmd);
	}
	card->is_app_cmd=0;
	return result;
}

int
MMCard_DoCmd(MMCard *card,uint32_t cmd,uint32_t arg,MMCResponse *resp) 
{
	int result = MMC_ERR_FAILED;
	if(!card) {
		return MMC_ERR_TIMEOUT;
	}
	dprintf("MMCard CMD%d ,arg 0x%08x\n",cmd,arg);	
	resp->len = 0;
	memset(resp->data,0xff,MMCARD_MAX_RESPLEN);
	if(card->is_app_cmd)  {
		return DoAppCmd(card,cmd,arg,resp);
	}
	switch(cmd) {
		case MMC_GO_IDLE_STATE:
			result = mmc_go_idle(card,cmd,arg,resp);
			break;

		case MMC_SEND_OP_COND:
			result = mmc_send_op_cond(card,cmd,arg,resp);
			break;

		case MMC_ALL_SEND_CID:
			result = mmc_all_send_cid(card,cmd,arg,resp);
			break;

		case MMC_SET_RELATIVE_ADDR:
			if(card->type == CARD_TYPE_SD) {
				result = sd_send_relative_addr(card,cmd,arg,resp);
			} else if(card->type == CARD_TYPE_MMC) {
				result = mmc_set_relative_addr(card,cmd,arg,resp);
			} else {
				fprintf(stderr,"Uninitalized card type %d\n",card->type);
			}
			break;

		case MMC_SET_DSR:
			result = mmc_set_dsr(card,cmd,arg,resp);
			break;

		case MMC_SWITCH:
			result = mmc_switch(card,cmd,arg,resp);
			break;

		case MMC_SELECT_CARD:
			result = mmc_select_card(card,cmd,arg,resp);
			break;

		case MMC_SEND_EXT_CSD:
			result = mmc_send_ext_csd(card,cmd,arg,resp);
			break;

		case MMC_SEND_CSD: 
			result = mmc_send_csd(card,cmd,arg,resp);
			break;

		case MMC_SEND_CID: 
			result = mmc_send_cid(card,cmd,arg,resp);
			break;

		case MMC_READ_DAT_UNTIL_STOP:
			result = mmc_read_dat_until_stop(card,cmd,arg,resp);
			break;

		case MMC_STOP_TRANSMISSION:
			result = mmc_stop_transmission(card,cmd,arg,resp);
			break;

		case MMC_SEND_STATUS: 
			result = mmc_send_status(card,cmd,arg,resp);
			break;

		case MMC_BUSTEST_R:
			result = mmc_bustest_r(card,cmd,arg,resp);
			break;

		case MMC_GO_INACTIVE_STATE:
			result = mmc_go_inactive_state(card,cmd,arg,resp);	
			break;

		case MMC_BUSTEST_W:
			result = mmc_bustest_w(card,cmd,arg,resp);
			break;

		case MMC_SET_BLOCKLEN: 
			result = mmc_set_blocklen(card,cmd,arg,resp);
			break;

		case MMC_READ_SINGLE_BLOCK:
			result = mmc_read_single_block(card,cmd,arg,resp);
			break;

		case MMC_READ_MULTIPLE_BLOCK:
			result = mmc_read_multiple_block(card,cmd,arg,resp);
			break;

		case MMC_WRITE_DAT_UNTIL_STOP:
			result = mmc_write_dat_until_stop(card,cmd,arg,resp);
			break;

		case MMC_SET_BLOCK_COUNT: 
			result = mmc_set_block_count(card,cmd,arg,resp);
			break;

		case MMC_WRITE_SINGLE_BLOCK: 
			result = mmc_write_single_block(card,cmd,arg,resp);
			break;

		case MMC_WRITE_MULTIPLE_BLOCK:
			result = mmc_write_multiple_block(card,cmd,arg,resp);
			break;

		case MMC_PROGRAM_CID:
			result = mmc_program_cid(card,cmd,arg,resp);
			break;

		case MMC_PROGRAM_CSD:
			result = mmc_program_csd(card,cmd,arg,resp);
			break;

		case MMC_SET_WRITE_PROT:
			result = mmc_set_write_prot(card,cmd,arg,resp);
			break;

		case MMC_CLR_WRITE_PROT:
			result = mmc_clr_write_prot(card,cmd,arg,resp);
			break;

		case MMC_SEND_WRITE_PROT:
			result = mmc_send_write_prot(card,cmd,arg,resp);
			break;

		case MMC_ERASE_GROUP_START:
			result = mmc_erase_group_start(card,cmd,arg,resp);
			break;

		case MMC_ERASE_GROUP_END:
			result = mmc_erase_group_end(card,cmd,arg,resp);
			break;

		case MMC_ERASE:
			result = mmc_erase(card,cmd,arg,resp);
			break;

		case MMC_FAST_IO:
			result = mmc_fast_io(card,cmd,arg,resp);
			break;

		case MMC_GO_IRQ_STATE:
			result = mmc_go_irq_state(card,cmd,arg,resp);
			break;

		case MMC_APP_CMD:
			result = mmc_app_cmd(card,cmd,arg,resp);
			break;

		case MMC_GEN_CMD:
			result = mmc_gen_cmd(card,cmd,arg,resp);
			break;

		default:
			fprintf(stderr,"MMCard CMD%d not implemented\n",cmd);
			break;
	}
	if(resp->len > 17)  {
		fprintf(stderr,"Emulator: invalid response len %d\n",resp->len);
	}
	return result;
}


static uint32_t
hash_string(const char *s) {
        uint32_t hash=0;
        while(*s) {
                hash = *s + (hash << 6) + (hash << 16) - hash;
                s++;
        }
        return hash;
}

/*
 * -----------------------------------------------------------------
 * Fill the card registers with values from a template
 * identified by cardtype
 * -----------------------------------------------------------------
 */
static int 
init_from_template(MMCard *card,const char *cardtype) 
{
	int nr_types = sizeof(cardspec)/sizeof(MMCardSpec);
	int i;
	MMCardSpec *spec;
	for(i=0;i<nr_types;i++) {
		spec = &cardspec[i];
		if(strcmp(cardtype,spec->producttype) ==  0) {
			break;
		}
	}	
	if(i==nr_types) {
		return -1;
	}
	card->type = spec->type;
	card->usec_reset = spec->usec_reset;
	card->ocr = spec->ocr &= ~OCR_NOTBUSY;
	for(i=0;i<8;i++) {
		card->scr[i] = spec->scr[i];
	}
	for(i=0;i<16;i++) {
		card->cid[i] = spec->cid[i];
		card->csd[i] = spec->csd[i];
	}
	return 0;
}

/*
 * -------------------------------------------------------------------------
 * For emulation of card where you have only a diskimage but no CID  
 * and CSD register values the values are taken from a template
 * and the size bits in CSD is calculated from filesize
 * -------------------------------------------------------------------------
 */
static void
init_auto_card_from_filesize(MMCard *card,char *filename) {
	int nr_sectors;
	int shift=0;
	int c_size_mult, c_size;
	int fd;
	int blkbits = 9;
	struct stat stat_buf;
	fd = open(filename,O_RDONLY);
	if(fd<0) {
		fprintf(stderr,"MMC/SD card auto type requires an existing diskimage \"%s\"\n",filename);
		perror("");
		exit(1);
	}
	fstat(fd,&stat_buf);
	close(fd);
	fprintf(stderr,"MM/SD Card: Using filesize of \"%s\"\n",basename(filename));
	nr_sectors = stat_buf.st_size >> blkbits; 
	if(nr_sectors > 2097152) {
		nr_sectors >>=1;
		blkbits++;	
	}
	while((nr_sectors == ((nr_sectors>>1)<<1)) && (shift != 9)) {
		shift++;
		nr_sectors >>= 1;
	}
	if(shift < 2) {
		fprintf(stderr,"\"%s\" is not a valid MM/SD card image: impossible sector count\n",filename);
		exit(1);
	}
	if(nr_sectors > 4096) {
		fprintf(stderr,"MMCard to big\n");
		exit(1);
	}
	c_size=nr_sectors - 1;
	c_size_mult = shift - 2;
	card->csd[8] = (card->csd[8] & 0x3f) | ((c_size & 3)<<6);
	card->csd[7] = (c_size >> 2) & 0xff;
	card->csd[6] = (card->csd[6] & 0xfc) | ((c_size >> 10) & 3);
	card->csd[5] = (blkbits & 0xf) | (card->csd[5] & 0xf0);

	card->blocklen = 1<<blkbits;	
	card->csd[10] = (card->csd[10] & 0x7f) | ((c_size_mult & 1) << 7);
	card->csd[9] = (card->csd[9] & 0xfc) | ((c_size_mult >> 1) & 3); 
	card->capacity = card->blocklen * ((c_size+1)<<(c_size_mult + 2)); 
	card->cid[5]='x';	
	card->cid[6]='y';	
	card->cid[7]='z';	
}

int MMCard_GetType(MMCard *card) 
{
	if(card) {
		return card->type;
	} else {
		return 0;
	}
}

void
MMCard_Delete(MMCard *card) 
{
	DiskImage_Close(card->disk_image);
	card->disk_image = NULL;
	free(card);
}
/*
 * ---------------------------------------------------------------------
 * MMCard_New
 * Constructor for MMC/SD Cards
 * ---------------------------------------------------------------------
 */
MMCard *
MMCard_New(const char *name)
{
	MMCard *card = malloc(sizeof(MMCard));
	char *imgdirname;
	char *producttype;
	uint32_t c_size,c_size_mult;
	uint32_t psn = hash_string(name);
	int autotype = 0;
	if(!card) {
		fprintf(stderr,"Out of memory\n");
		exit(1);
	}
	memset(card,0,sizeof(*card));
	producttype = Config_ReadVar(name,"type");
	if(!producttype) {
		fprintf(stderr,"No product type found for MMC card \"%s\". Create nothing\n",name);
		dump_cardtypes();
		free(card);
		return NULL;
	}
	if(strcmp(producttype,"auto_sd") == 0) {
		if(init_from_template(card,"Toshiba32M") < 0) {
			fprintf(stderr,"Emulator bug, builtin SD card not found\n");
			exit(1);
		}
		autotype = 1;
	} else if(strcmp(producttype,"auto_mmc") == 0) {
		if(init_from_template(card,"ExtremeMemory128M") < 0) {
			fprintf(stderr,"Emulator bug, builtin SD card not found\n");
			exit(1);
		}
		autotype = 1;
		card->type = CARD_TYPE_MMC;
	} else {
		if(init_from_template(card,producttype) < 0) {
			fprintf(stderr,"MMCard Product \"%s\" not found. Please fix configfile\n",producttype);
			dump_cardtypes();
			free(card);
			exit(1);
		}
		c_size = (card->csd[8] >> 6) | (card->csd[7] << 2) | ((card->csd[6]&3)<<10);
		c_size_mult = ((card->csd[10]>>7) & 1) | ((card->csd[9] & 3) << 1);
		card->blocklen = 1<<(card->csd[5] & 0xf);	
		card->capacity = card->blocklen * ((c_size+1)<<(c_size_mult + 2)); 
	}
	card->cid[9] = psn >> 24;
	card->cid[10] = psn >> 16;
	card->cid[11] = psn >> 8;
	card->cid[12] = psn >> 0;
	card->card_status = STATUS_READY_FOR_DATA;
	card->reset_state = RST_NOT_STARTED;
	imgdirname=Config_ReadVar("global","imagedir");
        if(imgdirname) {
                char *imagename = alloca(strlen(imgdirname) + strlen(name) + 20);
                sprintf(imagename,"%s/%s.img",imgdirname,name);
		if(autotype) {
			init_auto_card_from_filesize(card,imagename);
		}
                card->disk_image = DiskImage_Open(imagename,card->capacity,DI_RDWR | DI_CREAT_FF);
                if(!card->disk_image) {
                        fprintf(stderr,"Failed to open disk_image \"%s\"\n",imagename);
			perror("msg");
                        exit(1);
                }
        } else {
		fprintf(stderr,"No diskimage given for SD/MMC Card \"%s\"\n",name);
		exit(1);
        }
	fprintf(stderr,"MM/SD-card \"%s\" of type \"%s\" cap. %db blksize %d\n",name,producttype,card->capacity,card->blocklen);
	return card;	
}
