#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "ladspa-util.h"
#define COMB_SIZE 12000

#define COMB_FREQ                      0
#define COMB_FB                        1
#define COMB_INPUT                     2
#define COMB_OUTPUT                    3

LADSPA_Descriptor *combDescriptor = NULL;

typedef struct {
	LADSPA_Data *freq;
	LADSPA_Data *fb;
	LADSPA_Data *input;
	LADSPA_Data *output;
	long         comb_pos;
	LADSPA_Data *comb_tbl;
	long         last_offset;
	long         sample_rate;
	LADSPA_Data run_adding_gain;
} Comb;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return combDescriptor;
	default:
		return NULL;
	}
}

void activateComb(LADSPA_Handle instance) {
	Comb *plugin_data = (Comb *)instance;
	long comb_pos = plugin_data->comb_pos;
	LADSPA_Data *comb_tbl = plugin_data->comb_tbl;
	long last_offset = plugin_data->last_offset;
	long sample_rate = plugin_data->sample_rate;
	int i;

	for (i = 0; i < COMB_SIZE; i++) {
	        comb_tbl[i] = 0;
	}
	comb_pos = 0;
	last_offset = 1000;
	sample_rate = sample_rate;
	plugin_data->comb_pos = comb_pos;
	plugin_data->comb_tbl = comb_tbl;
	plugin_data->last_offset = last_offset;
	plugin_data->sample_rate = sample_rate;

}

void cleanupComb(LADSPA_Handle instance) {
	free(instance);
}

void connectPortComb(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Comb *plugin;

	plugin = (Comb *)instance;
	switch (port) {
	case COMB_FREQ:
		plugin->freq = data;
		break;
	case COMB_FB:
		plugin->fb = data;
		break;
	case COMB_INPUT:
		plugin->input = data;
		break;
	case COMB_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateComb(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Comb *plugin_data = (Comb *)malloc(sizeof(Comb));
	long comb_pos;
	LADSPA_Data *comb_tbl = NULL;
	long last_offset;
	long sample_rate;

	sample_rate = s_rate;
	comb_tbl = malloc(sizeof(LADSPA_Data) * COMB_SIZE);
	comb_pos = 0;
	last_offset = 1000;

	plugin_data->comb_pos = comb_pos;
	plugin_data->comb_tbl = comb_tbl;
	plugin_data->last_offset = last_offset;
	plugin_data->sample_rate = sample_rate;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runComb(LADSPA_Handle instance, unsigned long sample_count) {
	Comb *plugin_data = (Comb *)instance;

	/* Band separation (Hz) (float value) */
	LADSPA_Data freq = *(plugin_data->freq);

	/* Feedback (float value) */
	LADSPA_Data fb = *(plugin_data->fb);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	long comb_pos = plugin_data->comb_pos;
	LADSPA_Data * comb_tbl = plugin_data->comb_tbl;
	long last_offset = plugin_data->last_offset;
	long sample_rate = plugin_data->sample_rate;

	int offset;
	long data_pos;
	unsigned long pos;
	float xf, xf_step;

	offset = sample_rate / freq;
	offset = LIMIT(offset, 0, COMB_SIZE - 1);
	xf_step = 1.0f / (float)sample_count;
	xf = 0.0f;

	for (pos = 0; pos < sample_count; pos++) {
	        xf += xf_step;
	        data_pos = MOD(comb_pos - LIN_INTERP(xf, last_offset, offset), COMB_SIZE);
	        comb_tbl[comb_pos] = *input + fb*comb_tbl[data_pos];
	        buffer_write(*(output++), (*(input++) + comb_tbl[data_pos]) * 0.5);
	        comb_pos = (comb_pos + 1) % COMB_SIZE;
	}

	plugin_data->comb_pos = comb_pos;
	plugin_data->last_offset = offset;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainComb(LADSPA_Handle instance, LADSPA_Data gain) {
	((Comb *)instance)->run_adding_gain = gain;
}

void runAddingComb(LADSPA_Handle instance, unsigned long sample_count) {
	Comb *plugin_data = (Comb *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Band separation (Hz) (float value) */
	LADSPA_Data freq = *(plugin_data->freq);

	/* Feedback (float value) */
	LADSPA_Data fb = *(plugin_data->fb);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	long comb_pos = plugin_data->comb_pos;
	LADSPA_Data * comb_tbl = plugin_data->comb_tbl;
	long last_offset = plugin_data->last_offset;
	long sample_rate = plugin_data->sample_rate;

	int offset;
	long data_pos;
	unsigned long pos;
	float xf, xf_step;

	offset = sample_rate / freq;
	offset = LIMIT(offset, 0, COMB_SIZE - 1);
	xf_step = 1.0f / (float)sample_count;
	xf = 0.0f;

	for (pos = 0; pos < sample_count; pos++) {
	        xf += xf_step;
	        data_pos = MOD(comb_pos - LIN_INTERP(xf, last_offset, offset), COMB_SIZE);
	        comb_tbl[comb_pos] = *input + fb*comb_tbl[data_pos];
	        buffer_write(*(output++), (*(input++) + comb_tbl[data_pos]) * 0.5);
	        comb_pos = (comb_pos + 1) % COMB_SIZE;
	}

	plugin_data->comb_pos = comb_pos;
	plugin_data->last_offset = offset;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	combDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (combDescriptor) {
		combDescriptor->UniqueID = 1190;
		combDescriptor->Label = strdup("comb");
		combDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		combDescriptor->Name =
		 strdup("Comb Filter");
		combDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		combDescriptor->Copyright =
		 strdup("GPL");
		combDescriptor->PortCount = 4;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(4,
		 sizeof(LADSPA_PortDescriptor));
		combDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(4,
		 sizeof(LADSPA_PortRangeHint));
		combDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(4, sizeof(char*));
		combDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Band separation (Hz) */
		port_descriptors[COMB_FREQ] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[COMB_FREQ] =
		 strdup("Band separation (Hz)");
		port_range_hints[COMB_FREQ].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[COMB_FREQ].LowerBound = 16;
		port_range_hints[COMB_FREQ].UpperBound = 640;

		/* Parameters for Feedback */
		port_descriptors[COMB_FB] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[COMB_FB] =
		 strdup("Feedback");
		port_range_hints[COMB_FB].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[COMB_FB].LowerBound = -0.99;
		port_range_hints[COMB_FB].UpperBound = 0.99;

		/* Parameters for Input */
		port_descriptors[COMB_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[COMB_INPUT] =
		 strdup("Input");
		port_range_hints[COMB_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[COMB_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[COMB_OUTPUT] =
		 strdup("Output");
		port_range_hints[COMB_OUTPUT].HintDescriptor = 0;

		combDescriptor->activate = activateComb;
		combDescriptor->cleanup = cleanupComb;
		combDescriptor->connect_port = connectPortComb;
		combDescriptor->deactivate = NULL;
		combDescriptor->instantiate = instantiateComb;
		combDescriptor->run = runComb;
		combDescriptor->run_adding = runAddingComb;
		combDescriptor->set_run_adding_gain = setRunAddingGainComb;
	}
}

void _fini() {
	int i;

	if (combDescriptor) {
		free((char *)combDescriptor->Label);
		free((char *)combDescriptor->Name);
		free((char *)combDescriptor->Maker);
		free((char *)combDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)combDescriptor->PortDescriptors);
		for (i = 0; i < combDescriptor->PortCount; i++)
			free((char *)(combDescriptor->PortNames[i]));
		free((char **)combDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)combDescriptor->PortRangeHints);
		free(combDescriptor);
	}

}
