#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#define DIVIDER_DENOMINATOR            0
#define DIVIDER_INPUT                  1
#define DIVIDER_OUTPUT                 2

LADSPA_Descriptor *dividerDescriptor = NULL;

typedef struct {
	LADSPA_Data *denominator;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data  amp;
	float        count;
	LADSPA_Data  lamp;
	LADSPA_Data  last;
	int          zeroxs;
	LADSPA_Data run_adding_gain;
} Divider;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return dividerDescriptor;
	default:
		return NULL;
	}
}

void cleanupDivider(LADSPA_Handle instance) {
	free(instance);
}

void connectPortDivider(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Divider *plugin;

	plugin = (Divider *)instance;
	switch (port) {
	case DIVIDER_DENOMINATOR:
		plugin->denominator = data;
		break;
	case DIVIDER_INPUT:
		plugin->input = data;
		break;
	case DIVIDER_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateDivider(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Divider *plugin_data = (Divider *)malloc(sizeof(Divider));
	plugin_data->run_adding_gain = 1.0f;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runDivider(LADSPA_Handle instance, unsigned long sample_count) {
	Divider *plugin_data = (Divider *)instance;

	/* Denominator (float value) */
	LADSPA_Data denominator = *(plugin_data->denominator);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data amp = plugin_data->amp;
	float count = plugin_data->count;
	LADSPA_Data lamp = plugin_data->lamp;
	LADSPA_Data last = plugin_data->last;
	int zeroxs = plugin_data->zeroxs;

	/* Current output signal */
	static LADSPA_Data out = 1.0f;
	/* Integer version of denominator */
	int den = (int)denominator;
	
	unsigned long pos;
	
	for (pos = 0; pos < sample_count; pos++) {
	        count += 1.0f;
	        if ((input[pos] > 0.0f && last <= 0.0f) ||
	         (input[pos] < 0.0f && last >= 0.0)) {
	                zeroxs++;
	                if (den == 1) {
	                        out = out > 0 ? -1 : +1;
	                        lamp = amp / count;
	                        zeroxs = 0;
	                        count = 0;
	                        amp = 0;
	                }
	        }
	        amp += fabs(input[pos]);
	        if (den > 1 && (zeroxs % den) == den-1) {
	                out = out > 0 ? -1 : +1;
	                lamp = amp / count;
	                zeroxs = 0;
	                count = 0;
	                amp = 0;
	        }
	        last = input[pos];
	        buffer_write(*(output++), out * lamp);
	}
	
	plugin_data->last = last;
	plugin_data->amp = amp;
	plugin_data->lamp = lamp;
	plugin_data->zeroxs = zeroxs;
	plugin_data->count = count;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainDivider(LADSPA_Handle instance, LADSPA_Data gain) {
	((Divider *)instance)->run_adding_gain = gain;
}

void runAddingDivider(LADSPA_Handle instance, unsigned long sample_count) {
	Divider *plugin_data = (Divider *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Denominator (float value) */
	LADSPA_Data denominator = *(plugin_data->denominator);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data amp = plugin_data->amp;
	float count = plugin_data->count;
	LADSPA_Data lamp = plugin_data->lamp;
	LADSPA_Data last = plugin_data->last;
	int zeroxs = plugin_data->zeroxs;

	/* Current output signal */
	static LADSPA_Data out = 1.0f;
	/* Integer version of denominator */
	int den = (int)denominator;
	
	unsigned long pos;
	
	for (pos = 0; pos < sample_count; pos++) {
	        count += 1.0f;
	        if ((input[pos] > 0.0f && last <= 0.0f) ||
	         (input[pos] < 0.0f && last >= 0.0)) {
	                zeroxs++;
	                if (den == 1) {
	                        out = out > 0 ? -1 : +1;
	                        lamp = amp / count;
	                        zeroxs = 0;
	                        count = 0;
	                        amp = 0;
	                }
	        }
	        amp += fabs(input[pos]);
	        if (den > 1 && (zeroxs % den) == den-1) {
	                out = out > 0 ? -1 : +1;
	                lamp = amp / count;
	                zeroxs = 0;
	                count = 0;
	                amp = 0;
	        }
	        last = input[pos];
	        buffer_write(*(output++), out * lamp);
	}
	
	plugin_data->last = last;
	plugin_data->amp = amp;
	plugin_data->lamp = lamp;
	plugin_data->zeroxs = zeroxs;
	plugin_data->count = count;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	dividerDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (dividerDescriptor) {
		dividerDescriptor->UniqueID = 1186;
		dividerDescriptor->Label = strdup("divider");
		dividerDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		dividerDescriptor->Name =
		 strdup("Audio Divider (Suboctave Generator)");
		dividerDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		dividerDescriptor->Copyright =
		 strdup("GPL");
		dividerDescriptor->PortCount = 3;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(3,
		 sizeof(LADSPA_PortDescriptor));
		dividerDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(3,
		 sizeof(LADSPA_PortRangeHint));
		dividerDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(3, sizeof(char*));
		dividerDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Denominator */
		port_descriptors[DIVIDER_DENOMINATOR] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[DIVIDER_DENOMINATOR] =
		 strdup("Denominator");
		port_range_hints[DIVIDER_DENOMINATOR].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_INTEGER;
		port_range_hints[DIVIDER_DENOMINATOR].LowerBound = 1;
		port_range_hints[DIVIDER_DENOMINATOR].UpperBound = 8;

		/* Parameters for Input */
		port_descriptors[DIVIDER_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[DIVIDER_INPUT] =
		 strdup("Input");
		port_range_hints[DIVIDER_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[DIVIDER_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[DIVIDER_OUTPUT] =
		 strdup("Output");
		port_range_hints[DIVIDER_OUTPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[DIVIDER_OUTPUT].LowerBound = -1;
		port_range_hints[DIVIDER_OUTPUT].UpperBound = +1;

		dividerDescriptor->activate = NULL;
		dividerDescriptor->cleanup = cleanupDivider;
		dividerDescriptor->connect_port = connectPortDivider;
		dividerDescriptor->deactivate = NULL;
		dividerDescriptor->instantiate = instantiateDivider;
		dividerDescriptor->run = runDivider;
		dividerDescriptor->run_adding = runAddingDivider;
		dividerDescriptor->set_run_adding_gain = setRunAddingGainDivider;
	}
}

void _fini() {
	int i;

	if (dividerDescriptor) {
		free((char *)dividerDescriptor->Label);
		free((char *)dividerDescriptor->Name);
		free((char *)dividerDescriptor->Maker);
		free((char *)dividerDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)dividerDescriptor->PortDescriptors);
		for (i = 0; i < dividerDescriptor->PortCount; i++)
			free((char *)(dividerDescriptor->PortNames[i]));
		free((char **)dividerDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)dividerDescriptor->PortRangeHints);
		free(dividerDescriptor);
	}

}
