#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#define LN_2_2 0.34657359 // ln(2)/2
#define LIMIT(v,l,u) (v<l?l:(v>u?u:v))

typedef struct {
        float a1;
        float a2;
        float b0;
        float b1;
        float b2;
        float z0;
        float z1;
        float z2;
} biquad;

inline void biquad_init(biquad *f) {
        f->z0 = 0.0f;
        f->z1 = 0.0f;
        f->z2 = 0.0f;
}

inline void ls_set_params(biquad *f, float fc, float gain, float slope,
                          float fs) {
        float w = 2.0f * M_PI * LIMIT(fc, 1.0, fs/2.0) / fs;
        float cw = cos(w);
        float sw = sin(w);
        float A = pow(10.0f, gain / 40.0f);
        float b = sqrt(((1 + A * A) / LIMIT(slope, 0.0001, 1)) - ((A - 1.0) * (A - 1.0)));
        float apc = cw * (A + 1.0);
        float amc = cw * (A - 1.0);
        float bs = b * sw;
        float a0r = 1.0 / (A + 1.0 + amc + bs);

        f->b0 = a0r * A * (A + 1.0 - amc + bs);
        f->b1 = a0r * 2.0 * A * (A - 1.0 - apc);
        f->b2 = a0r * A * (A + 1.0 - amc - bs);
        f->a1 = a0r * 2.0 * (A - 1.0 + apc);
        f->a2 = a0r * (-A - 1.0 - amc + bs);
}

inline void hs_set_params(biquad *f, float fc, float gain, float slope,
                          float fs) {
        float w = 2.0f * M_PI * LIMIT(fc, 1.0, fs/2.0) / fs;
        float cw = cos(w);
        float sw = sin(w);
        float A = pow(10.0f, gain / 40.0f);
        float b = sqrt(((1 + A * A) / LIMIT(slope, 0.0001, 1)) - ((A - 1.0) * (A - 1.0)));
        float apc = cw * (A + 1.0);
        float amc = cw * (A - 1.0);
        float bs = b * sw;
        float a0r = 1.0 / (A + 1.0 - amc + bs);

        f->b0 = a0r * A * (A + 1.0 + amc + bs);
        f->b1 = a0r * -2.0 * A * (A - 1.0 + apc);
        f->b2 = a0r * (A + 1.0 + amc - bs);
        f->a1 = a0r * -2.0 * (A - 1.0 - apc);
        f->a2 = a0r * (-A - 1.0 + amc + bs);
}

inline float biquad_run(biquad *f, float in) {
        float out;

        f->z0 = in + (f->a1)*(f->z1) + (f->a2)*(f->z2);
        out = (f->z0)*(f->b0) + (f->z1)*(f->b1) + (f->z2)*(f->b2);
        f->z2 = f->z1;
        f->z1 = f->z0;

        return out;
}

#define SURROUNDENCODER_L              0
#define SURROUNDENCODER_R              1
#define SURROUNDENCODER_C              2
#define SURROUNDENCODER_S              3
#define SURROUNDENCODER_LT             4
#define SURROUNDENCODER_RT             5

LADSPA_Descriptor *surroundEncoderDescriptor = NULL;

typedef struct {
	LADSPA_Data *l;
	LADSPA_Data *r;
	LADSPA_Data *c;
	LADSPA_Data *s;
	LADSPA_Data *lt;
	LADSPA_Data *rt;
	LADSPA_Data *buffer;
	unsigned int buffer_pos;
	unsigned int buffer_size;
	biquad *     hc;
	biquad *     lc;
	LADSPA_Data run_adding_gain;
} SurroundEncoder;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return surroundEncoderDescriptor;
	default:
		return NULL;
	}
}

void cleanupSurroundEncoder(LADSPA_Handle instance) {
	free(instance);
}

void connectPortSurroundEncoder(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	SurroundEncoder *plugin;

	plugin = (SurroundEncoder *)instance;
	switch (port) {
	case SURROUNDENCODER_L:
		plugin->l = data;
		break;
	case SURROUNDENCODER_R:
		plugin->r = data;
		break;
	case SURROUNDENCODER_C:
		plugin->c = data;
		break;
	case SURROUNDENCODER_S:
		plugin->s = data;
		break;
	case SURROUNDENCODER_LT:
		plugin->lt = data;
		break;
	case SURROUNDENCODER_RT:
		plugin->rt = data;
		break;
	}
}

LADSPA_Handle instantiateSurroundEncoder(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	SurroundEncoder *plugin_data = (SurroundEncoder *)malloc(sizeof(SurroundEncoder));
	LADSPA_Data *buffer = NULL;
	unsigned int buffer_pos;
	unsigned int buffer_size;
	biquad *hc = NULL;
	biquad *lc = NULL;

	buffer_size = (int)(0.0072f * s_rate);
	buffer_pos = 0;
	buffer = calloc(buffer_size, sizeof(LADSPA_Data));
	lc = malloc(sizeof(biquad));
	hc = malloc(sizeof(biquad));
	biquad_init(lc);
	biquad_init(hc);
	ls_set_params(lc, 100.0f, -70.0f, 1.0f, s_rate);
	hs_set_params(hc, 7000.0f, -70.0f, 1.0f, s_rate);

	plugin_data->buffer = buffer;
	plugin_data->buffer_pos = buffer_pos;
	plugin_data->buffer_size = buffer_size;
	plugin_data->hc = hc;
	plugin_data->lc = lc;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runSurroundEncoder(LADSPA_Handle instance, unsigned long sample_count) {
	SurroundEncoder *plugin_data = (SurroundEncoder *)instance;

	/* L (array of floats of length sample_count) */
	LADSPA_Data *l = plugin_data->l;

	/* R (array of floats of length sample_count) */
	LADSPA_Data *r = plugin_data->r;

	/* C (array of floats of length sample_count) */
	LADSPA_Data *c = plugin_data->c;

	/* S (array of floats of length sample_count) */
	LADSPA_Data *s = plugin_data->s;

	/* Lt (array of floats of length sample_count) */
	LADSPA_Data *lt = plugin_data->lt;

	/* Rt (array of floats of length sample_count) */
	LADSPA_Data *rt = plugin_data->rt;
	LADSPA_Data * buffer = plugin_data->buffer;
	unsigned int buffer_pos = plugin_data->buffer_pos;
	unsigned int buffer_size = plugin_data->buffer_size;
	biquad * hc = plugin_data->hc;
	biquad * lc = plugin_data->lc;

	unsigned long pos;
	LADSPA_Data s_delayed;

	for (pos = 0; pos < sample_count; pos++) {
	  s_delayed = buffer[buffer_pos];
	  buffer[buffer_pos++] = s[pos];
	  buffer_pos %= buffer_size;

	  s_delayed = biquad_run(lc, s_delayed);
	  s_delayed = biquad_run(hc, s_delayed);

	  buffer_write(lt[pos], l[pos] + c[pos] * 0.707946f +
	               s_delayed * 0.707946f);
	  buffer_write(rt[pos], r[pos] + c[pos] * 0.707946f -
	               s_delayed * 0.707946f);
	}

	plugin_data->buffer_pos = buffer_pos;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainSurroundEncoder(LADSPA_Handle instance, LADSPA_Data gain) {
	((SurroundEncoder *)instance)->run_adding_gain = gain;
}

void runAddingSurroundEncoder(LADSPA_Handle instance, unsigned long sample_count) {
	SurroundEncoder *plugin_data = (SurroundEncoder *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* L (array of floats of length sample_count) */
	LADSPA_Data *l = plugin_data->l;

	/* R (array of floats of length sample_count) */
	LADSPA_Data *r = plugin_data->r;

	/* C (array of floats of length sample_count) */
	LADSPA_Data *c = plugin_data->c;

	/* S (array of floats of length sample_count) */
	LADSPA_Data *s = plugin_data->s;

	/* Lt (array of floats of length sample_count) */
	LADSPA_Data *lt = plugin_data->lt;

	/* Rt (array of floats of length sample_count) */
	LADSPA_Data *rt = plugin_data->rt;
	LADSPA_Data * buffer = plugin_data->buffer;
	unsigned int buffer_pos = plugin_data->buffer_pos;
	unsigned int buffer_size = plugin_data->buffer_size;
	biquad * hc = plugin_data->hc;
	biquad * lc = plugin_data->lc;

	unsigned long pos;
	LADSPA_Data s_delayed;

	for (pos = 0; pos < sample_count; pos++) {
	  s_delayed = buffer[buffer_pos];
	  buffer[buffer_pos++] = s[pos];
	  buffer_pos %= buffer_size;

	  s_delayed = biquad_run(lc, s_delayed);
	  s_delayed = biquad_run(hc, s_delayed);

	  buffer_write(lt[pos], l[pos] + c[pos] * 0.707946f +
	               s_delayed * 0.707946f);
	  buffer_write(rt[pos], r[pos] + c[pos] * 0.707946f -
	               s_delayed * 0.707946f);
	}

	plugin_data->buffer_pos = buffer_pos;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	surroundEncoderDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (surroundEncoderDescriptor) {
		surroundEncoderDescriptor->UniqueID = 1401;
		surroundEncoderDescriptor->Label = strdup("surroundEncoder");
		surroundEncoderDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		surroundEncoderDescriptor->Name =
		 strdup("Surround matrix encoder");
		surroundEncoderDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		surroundEncoderDescriptor->Copyright =
		 strdup("GPL");
		surroundEncoderDescriptor->PortCount = 6;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(6,
		 sizeof(LADSPA_PortDescriptor));
		surroundEncoderDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(6,
		 sizeof(LADSPA_PortRangeHint));
		surroundEncoderDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(6, sizeof(char*));
		surroundEncoderDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for L */
		port_descriptors[SURROUNDENCODER_L] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[SURROUNDENCODER_L] =
		 strdup("L");
		port_range_hints[SURROUNDENCODER_L].HintDescriptor = 0;

		/* Parameters for R */
		port_descriptors[SURROUNDENCODER_R] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[SURROUNDENCODER_R] =
		 strdup("R");
		port_range_hints[SURROUNDENCODER_R].HintDescriptor = 0;

		/* Parameters for C */
		port_descriptors[SURROUNDENCODER_C] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[SURROUNDENCODER_C] =
		 strdup("C");
		port_range_hints[SURROUNDENCODER_C].HintDescriptor = 0;

		/* Parameters for S */
		port_descriptors[SURROUNDENCODER_S] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[SURROUNDENCODER_S] =
		 strdup("S");
		port_range_hints[SURROUNDENCODER_S].HintDescriptor = 0;

		/* Parameters for Lt */
		port_descriptors[SURROUNDENCODER_LT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[SURROUNDENCODER_LT] =
		 strdup("Lt");
		port_range_hints[SURROUNDENCODER_LT].HintDescriptor = 0;

		/* Parameters for Rt */
		port_descriptors[SURROUNDENCODER_RT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[SURROUNDENCODER_RT] =
		 strdup("Rt");
		port_range_hints[SURROUNDENCODER_RT].HintDescriptor = 0;

		surroundEncoderDescriptor->activate = NULL;
		surroundEncoderDescriptor->cleanup = cleanupSurroundEncoder;
		surroundEncoderDescriptor->connect_port = connectPortSurroundEncoder;
		surroundEncoderDescriptor->deactivate = NULL;
		surroundEncoderDescriptor->instantiate = instantiateSurroundEncoder;
		surroundEncoderDescriptor->run = runSurroundEncoder;
		surroundEncoderDescriptor->run_adding = runAddingSurroundEncoder;
		surroundEncoderDescriptor->set_run_adding_gain = setRunAddingGainSurroundEncoder;
	}
}

void _fini() {
	int i;

	if (surroundEncoderDescriptor) {
		free((char *)surroundEncoderDescriptor->Label);
		free((char *)surroundEncoderDescriptor->Name);
		free((char *)surroundEncoderDescriptor->Maker);
		free((char *)surroundEncoderDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)surroundEncoderDescriptor->PortDescriptors);
		for (i = 0; i < surroundEncoderDescriptor->PortCount; i++)
			free((char *)(surroundEncoderDescriptor->PortNames[i]));
		free((char **)surroundEncoderDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)surroundEncoderDescriptor->PortRangeHints);
		free(surroundEncoderDescriptor);
	}

}
