#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "ladspa-util.h"

#define BASE_BUFFER 4 // Tape length (inches)

#define TAPEDELAY_SPEED                0
#define TAPEDELAY_DA                   1
#define TAPEDELAY_T1D                  2
#define TAPEDELAY_T1A                  3
#define TAPEDELAY_T2D                  4
#define TAPEDELAY_T2A                  5
#define TAPEDELAY_T3D                  6
#define TAPEDELAY_T3A                  7
#define TAPEDELAY_T4D                  8
#define TAPEDELAY_T4A                  9
#define TAPEDELAY_INPUT                10
#define TAPEDELAY_OUTPUT               11

LADSPA_Descriptor *tapeDelayDescriptor = NULL;

typedef struct {
	LADSPA_Data *speed;
	LADSPA_Data *da;
	LADSPA_Data *t1d;
	LADSPA_Data *t1a;
	LADSPA_Data *t2d;
	LADSPA_Data *t2a;
	LADSPA_Data *t3d;
	LADSPA_Data *t3a;
	LADSPA_Data *t4d;
	LADSPA_Data *t4a;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data *buffer;
	long         buffer_size;
	LADSPA_Data  last2_in;
	LADSPA_Data  last3_in;
	LADSPA_Data  last_in;
	int          last_phase;
	float        phase;
	long         sample_rate;
	LADSPA_Data  z0;
	LADSPA_Data  z1;
	LADSPA_Data  z2;
	LADSPA_Data run_adding_gain;
} TapeDelay;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return tapeDelayDescriptor;
	default:
		return NULL;
	}
}

void activateTapeDelay(LADSPA_Handle instance) {
	TapeDelay *plugin_data = (TapeDelay *)instance;
	LADSPA_Data *buffer = plugin_data->buffer;
	long buffer_size = plugin_data->buffer_size;
	LADSPA_Data last2_in = plugin_data->last2_in;
	LADSPA_Data last3_in = plugin_data->last3_in;
	LADSPA_Data last_in = plugin_data->last_in;
	int last_phase = plugin_data->last_phase;
	float phase = plugin_data->phase;
	long sample_rate = plugin_data->sample_rate;
	LADSPA_Data z0 = plugin_data->z0;
	LADSPA_Data z1 = plugin_data->z1;
	LADSPA_Data z2 = plugin_data->z2;
	int i;

	for (i = 0; i < buffer_size; i++) {
	        buffer[i] = 0;
	}
	phase = 0;
	last_phase = 0;
	last_in = 0.0f;
	last2_in = 0.0f;
	last3_in = 0.0f;
	sample_rate = sample_rate;
	z0 = 0.0f;
	z1 = 0.0f;
	z2 = 0.0f;
	plugin_data->buffer = buffer;
	plugin_data->buffer_size = buffer_size;
	plugin_data->last2_in = last2_in;
	plugin_data->last3_in = last3_in;
	plugin_data->last_in = last_in;
	plugin_data->last_phase = last_phase;
	plugin_data->phase = phase;
	plugin_data->sample_rate = sample_rate;
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;

}

void cleanupTapeDelay(LADSPA_Handle instance) {
	free(instance);
}

void connectPortTapeDelay(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	TapeDelay *plugin;

	plugin = (TapeDelay *)instance;
	switch (port) {
	case TAPEDELAY_SPEED:
		plugin->speed = data;
		break;
	case TAPEDELAY_DA:
		plugin->da = data;
		break;
	case TAPEDELAY_T1D:
		plugin->t1d = data;
		break;
	case TAPEDELAY_T1A:
		plugin->t1a = data;
		break;
	case TAPEDELAY_T2D:
		plugin->t2d = data;
		break;
	case TAPEDELAY_T2A:
		plugin->t2a = data;
		break;
	case TAPEDELAY_T3D:
		plugin->t3d = data;
		break;
	case TAPEDELAY_T3A:
		plugin->t3a = data;
		break;
	case TAPEDELAY_T4D:
		plugin->t4d = data;
		break;
	case TAPEDELAY_T4A:
		plugin->t4a = data;
		break;
	case TAPEDELAY_INPUT:
		plugin->input = data;
		break;
	case TAPEDELAY_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateTapeDelay(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	TapeDelay *plugin_data = (TapeDelay *)malloc(sizeof(TapeDelay));
	LADSPA_Data *buffer = NULL;
	long buffer_size;
	LADSPA_Data last2_in;
	LADSPA_Data last3_in;
	LADSPA_Data last_in;
	int last_phase;
	float phase;
	long sample_rate;
	LADSPA_Data z0;
	LADSPA_Data z1;
	LADSPA_Data z2;

	sample_rate = s_rate;
	buffer_size = BASE_BUFFER * s_rate;
	buffer = calloc(buffer_size, sizeof(LADSPA_Data));
	phase = 0;
	last_phase = 0;
	last_in = 0.0f;
	last2_in = 0.0f;
	last3_in = 0.0f;
	z0 = 0.0f;
	z1 = 0.0f;
	z2 = 0.0f;

	plugin_data->buffer = buffer;
	plugin_data->buffer_size = buffer_size;
	plugin_data->last2_in = last2_in;
	plugin_data->last3_in = last3_in;
	plugin_data->last_in = last_in;
	plugin_data->last_phase = last_phase;
	plugin_data->phase = phase;
	plugin_data->sample_rate = sample_rate;
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runTapeDelay(LADSPA_Handle instance, unsigned long sample_count) {
	TapeDelay *plugin_data = (TapeDelay *)instance;

	/* Tape speed (inches/sec, 1=normal) (float value) */
	LADSPA_Data speed = *(plugin_data->speed);

	/* Dry level (dB) (float value) */
	LADSPA_Data da = *(plugin_data->da);

	/* Tap 1 distance (inches) (float value) */
	LADSPA_Data t1d = *(plugin_data->t1d);

	/* Tap 1 level (dB) (float value) */
	LADSPA_Data t1a = *(plugin_data->t1a);

	/* Tap 2 distance (inches) (float value) */
	LADSPA_Data t2d = *(plugin_data->t2d);

	/* Tap 2 level (dB) (float value) */
	LADSPA_Data t2a = *(plugin_data->t2a);

	/* Tap 3 distance (inches) (float value) */
	LADSPA_Data t3d = *(plugin_data->t3d);

	/* Tap 3 level (dB) (float value) */
	LADSPA_Data t3a = *(plugin_data->t3a);

	/* Tap 4 distance (inches) (float value) */
	LADSPA_Data t4d = *(plugin_data->t4d);

	/* Tap 4 level (dB) (float value) */
	LADSPA_Data t4a = *(plugin_data->t4a);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data * buffer = plugin_data->buffer;
	long buffer_size = plugin_data->buffer_size;
	LADSPA_Data last2_in = plugin_data->last2_in;
	LADSPA_Data last3_in = plugin_data->last3_in;
	LADSPA_Data last_in = plugin_data->last_in;
	int last_phase = plugin_data->last_phase;
	float phase = plugin_data->phase;
	long sample_rate = plugin_data->sample_rate;
	LADSPA_Data z0 = plugin_data->z0;
	LADSPA_Data z1 = plugin_data->z1;
	LADSPA_Data z2 = plugin_data->z2;

	long int pos;
	float increment = LIMIT(speed, 0, 40);
	float lin_int, lin_inc;
	int track;
	int fph;
	int t1d_s, t2d_s, t3d_s, t4d_s; // Delay in samples
	LADSPA_Data out;
	
	// Biquad filter params, an R.B-J 2-pole lp filter
	/* Filer seems to have too may side effects, wrong values?
	float w = M_PI * LIMIT(increment, 0.1f, 1.0f);
	float cw = cos(w);
	float sw = sin(w);
	float alpha = sw * sinh(0.5f / 0.2);
	float a0r = 1.0f / (1.0f + alpha);
	float a1 = -2.0f * cw * a0r;
	float a2 = (1.0f - alpha) * a0r;
	float b1 = (1.0f - cw) * a0r;
	float b0 = 0.5f * b1;
	float b2 = b0;
	*/
	
	da = DB_CO(da);
	t1a = DB_CO(t1a);
	t2a = DB_CO(t2a);
	t3a = DB_CO(t3a);
	t4a = DB_CO(t4a);
	t1d_s = LIMIT(f_round(t1d * sample_rate), 0, buffer_size - 1);
	t2d_s = LIMIT(f_round(t2d * sample_rate), 0, buffer_size - 1);
	t3d_s = LIMIT(f_round(t3d * sample_rate), 0, buffer_size - 1);
	t4d_s = LIMIT(f_round(t4d * sample_rate), 0, buffer_size - 1);
	
	for (pos = 0; pos < sample_count; pos++) {
	        fph = f_trunc(phase);
	        last_phase = fph;
	        lin_int = phase - (float)fph;
	        out = buffer[NEG_MOD(fph - t1d_s, buffer_size)] * t1a;
	        out += buffer[NEG_MOD(fph - t2d_s, buffer_size)] * t2a;
	        out += buffer[NEG_MOD(fph - t3d_s, buffer_size)] * t3a;
	        out += buffer[NEG_MOD(fph - t4d_s, buffer_size)] * t4a;
	        //z0 = out + a1*z1 + a2*z2;
	        //out = z0*b0 + z1*b1 + z2*b2;
	        //z2 = z1; z1 = z0;
	        phase += increment;
	        lin_inc = 1.0f / (floor(phase) - last_phase + 1);
	        lin_inc = lin_inc > 1.0f ? 1.0f : lin_inc;
	        lin_int = 0.0f;
	        for (track = last_phase; track < phase; track++) {
	                lin_int += lin_inc;
	                buffer[track % buffer_size] =
	                 cube_interp(lin_int, last3_in, last2_in, last_in, input[pos]);
	        }
	        last3_in = last2_in;
	        last2_in = last_in;
	        last_in = input[pos];
	        out += input[pos] * da;
	        buffer_write(output[pos], out);
	        if (phase >= buffer_size) {
	                phase -= buffer_size;
	        }
	}
	
	// Store current phase in instance
	plugin_data->phase = phase;
	plugin_data->last_phase = last_phase;
	plugin_data->last_in = last_in;
	plugin_data->last2_in = last2_in;
	plugin_data->last3_in = last3_in;
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainTapeDelay(LADSPA_Handle instance, LADSPA_Data gain) {
	((TapeDelay *)instance)->run_adding_gain = gain;
}

void runAddingTapeDelay(LADSPA_Handle instance, unsigned long sample_count) {
	TapeDelay *plugin_data = (TapeDelay *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Tape speed (inches/sec, 1=normal) (float value) */
	LADSPA_Data speed = *(plugin_data->speed);

	/* Dry level (dB) (float value) */
	LADSPA_Data da = *(plugin_data->da);

	/* Tap 1 distance (inches) (float value) */
	LADSPA_Data t1d = *(plugin_data->t1d);

	/* Tap 1 level (dB) (float value) */
	LADSPA_Data t1a = *(plugin_data->t1a);

	/* Tap 2 distance (inches) (float value) */
	LADSPA_Data t2d = *(plugin_data->t2d);

	/* Tap 2 level (dB) (float value) */
	LADSPA_Data t2a = *(plugin_data->t2a);

	/* Tap 3 distance (inches) (float value) */
	LADSPA_Data t3d = *(plugin_data->t3d);

	/* Tap 3 level (dB) (float value) */
	LADSPA_Data t3a = *(plugin_data->t3a);

	/* Tap 4 distance (inches) (float value) */
	LADSPA_Data t4d = *(plugin_data->t4d);

	/* Tap 4 level (dB) (float value) */
	LADSPA_Data t4a = *(plugin_data->t4a);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data * buffer = plugin_data->buffer;
	long buffer_size = plugin_data->buffer_size;
	LADSPA_Data last2_in = plugin_data->last2_in;
	LADSPA_Data last3_in = plugin_data->last3_in;
	LADSPA_Data last_in = plugin_data->last_in;
	int last_phase = plugin_data->last_phase;
	float phase = plugin_data->phase;
	long sample_rate = plugin_data->sample_rate;
	LADSPA_Data z0 = plugin_data->z0;
	LADSPA_Data z1 = plugin_data->z1;
	LADSPA_Data z2 = plugin_data->z2;

	long int pos;
	float increment = LIMIT(speed, 0, 40);
	float lin_int, lin_inc;
	int track;
	int fph;
	int t1d_s, t2d_s, t3d_s, t4d_s; // Delay in samples
	LADSPA_Data out;
	
	// Biquad filter params, an R.B-J 2-pole lp filter
	/* Filer seems to have too may side effects, wrong values?
	float w = M_PI * LIMIT(increment, 0.1f, 1.0f);
	float cw = cos(w);
	float sw = sin(w);
	float alpha = sw * sinh(0.5f / 0.2);
	float a0r = 1.0f / (1.0f + alpha);
	float a1 = -2.0f * cw * a0r;
	float a2 = (1.0f - alpha) * a0r;
	float b1 = (1.0f - cw) * a0r;
	float b0 = 0.5f * b1;
	float b2 = b0;
	*/
	
	da = DB_CO(da);
	t1a = DB_CO(t1a);
	t2a = DB_CO(t2a);
	t3a = DB_CO(t3a);
	t4a = DB_CO(t4a);
	t1d_s = LIMIT(f_round(t1d * sample_rate), 0, buffer_size - 1);
	t2d_s = LIMIT(f_round(t2d * sample_rate), 0, buffer_size - 1);
	t3d_s = LIMIT(f_round(t3d * sample_rate), 0, buffer_size - 1);
	t4d_s = LIMIT(f_round(t4d * sample_rate), 0, buffer_size - 1);
	
	for (pos = 0; pos < sample_count; pos++) {
	        fph = f_trunc(phase);
	        last_phase = fph;
	        lin_int = phase - (float)fph;
	        out = buffer[NEG_MOD(fph - t1d_s, buffer_size)] * t1a;
	        out += buffer[NEG_MOD(fph - t2d_s, buffer_size)] * t2a;
	        out += buffer[NEG_MOD(fph - t3d_s, buffer_size)] * t3a;
	        out += buffer[NEG_MOD(fph - t4d_s, buffer_size)] * t4a;
	        //z0 = out + a1*z1 + a2*z2;
	        //out = z0*b0 + z1*b1 + z2*b2;
	        //z2 = z1; z1 = z0;
	        phase += increment;
	        lin_inc = 1.0f / (floor(phase) - last_phase + 1);
	        lin_inc = lin_inc > 1.0f ? 1.0f : lin_inc;
	        lin_int = 0.0f;
	        for (track = last_phase; track < phase; track++) {
	                lin_int += lin_inc;
	                buffer[track % buffer_size] =
	                 cube_interp(lin_int, last3_in, last2_in, last_in, input[pos]);
	        }
	        last3_in = last2_in;
	        last2_in = last_in;
	        last_in = input[pos];
	        out += input[pos] * da;
	        buffer_write(output[pos], out);
	        if (phase >= buffer_size) {
	                phase -= buffer_size;
	        }
	}
	
	// Store current phase in instance
	plugin_data->phase = phase;
	plugin_data->last_phase = last_phase;
	plugin_data->last_in = last_in;
	plugin_data->last2_in = last2_in;
	plugin_data->last3_in = last3_in;
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	tapeDelayDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (tapeDelayDescriptor) {
		tapeDelayDescriptor->UniqueID = 1211;
		tapeDelayDescriptor->Label = strdup("tapeDelay");
		tapeDelayDescriptor->Properties =
		 0;
		tapeDelayDescriptor->Name =
		 strdup("Tape Delay Simulation");
		tapeDelayDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		tapeDelayDescriptor->Copyright =
		 strdup("GPL");
		tapeDelayDescriptor->PortCount = 12;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(12,
		 sizeof(LADSPA_PortDescriptor));
		tapeDelayDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(12,
		 sizeof(LADSPA_PortRangeHint));
		tapeDelayDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(12, sizeof(char*));
		tapeDelayDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Tape speed (inches/sec, 1=normal) */
		port_descriptors[TAPEDELAY_SPEED] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_SPEED] =
		 strdup("Tape speed (inches/sec, 1=normal)");
		port_range_hints[TAPEDELAY_SPEED].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_SPEED].LowerBound = 0;
		port_range_hints[TAPEDELAY_SPEED].UpperBound = 10;

		/* Parameters for Dry level (dB) */
		port_descriptors[TAPEDELAY_DA] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_DA] =
		 strdup("Dry level (dB)");
		port_range_hints[TAPEDELAY_DA].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_DA].LowerBound = -70;
		port_range_hints[TAPEDELAY_DA].UpperBound = 0;

		/* Parameters for Tap 1 distance (inches) */
		port_descriptors[TAPEDELAY_T1D] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_T1D] =
		 strdup("Tap 1 distance (inches)");
		port_range_hints[TAPEDELAY_T1D].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_T1D].LowerBound = 0;
		port_range_hints[TAPEDELAY_T1D].UpperBound = 4;

		/* Parameters for Tap 1 level (dB) */
		port_descriptors[TAPEDELAY_T1A] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_T1A] =
		 strdup("Tap 1 level (dB)");
		port_range_hints[TAPEDELAY_T1A].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_T1A].LowerBound = -70;
		port_range_hints[TAPEDELAY_T1A].UpperBound = 0;

		/* Parameters for Tap 2 distance (inches) */
		port_descriptors[TAPEDELAY_T2D] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_T2D] =
		 strdup("Tap 2 distance (inches)");
		port_range_hints[TAPEDELAY_T2D].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_T2D].LowerBound = 0;
		port_range_hints[TAPEDELAY_T2D].UpperBound = 4;

		/* Parameters for Tap 2 level (dB) */
		port_descriptors[TAPEDELAY_T2A] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_T2A] =
		 strdup("Tap 2 level (dB)");
		port_range_hints[TAPEDELAY_T2A].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_T2A].LowerBound = -70;
		port_range_hints[TAPEDELAY_T2A].UpperBound = 0;

		/* Parameters for Tap 3 distance (inches) */
		port_descriptors[TAPEDELAY_T3D] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_T3D] =
		 strdup("Tap 3 distance (inches)");
		port_range_hints[TAPEDELAY_T3D].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_T3D].LowerBound = 0;
		port_range_hints[TAPEDELAY_T3D].UpperBound = 4;

		/* Parameters for Tap 3 level (dB) */
		port_descriptors[TAPEDELAY_T3A] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_T3A] =
		 strdup("Tap 3 level (dB)");
		port_range_hints[TAPEDELAY_T3A].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_T3A].LowerBound = -70;
		port_range_hints[TAPEDELAY_T3A].UpperBound = 0;

		/* Parameters for Tap 4 distance (inches) */
		port_descriptors[TAPEDELAY_T4D] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_T4D] =
		 strdup("Tap 4 distance (inches)");
		port_range_hints[TAPEDELAY_T4D].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_T4D].LowerBound = 0;
		port_range_hints[TAPEDELAY_T4D].UpperBound = 4;

		/* Parameters for Tap 4 level (dB) */
		port_descriptors[TAPEDELAY_T4A] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[TAPEDELAY_T4A] =
		 strdup("Tap 4 level (dB)");
		port_range_hints[TAPEDELAY_T4A].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[TAPEDELAY_T4A].LowerBound = -70;
		port_range_hints[TAPEDELAY_T4A].UpperBound = 0;

		/* Parameters for Input */
		port_descriptors[TAPEDELAY_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[TAPEDELAY_INPUT] =
		 strdup("Input");
		port_range_hints[TAPEDELAY_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[TAPEDELAY_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[TAPEDELAY_OUTPUT] =
		 strdup("Output");
		port_range_hints[TAPEDELAY_OUTPUT].HintDescriptor = 0;

		tapeDelayDescriptor->activate = activateTapeDelay;
		tapeDelayDescriptor->cleanup = cleanupTapeDelay;
		tapeDelayDescriptor->connect_port = connectPortTapeDelay;
		tapeDelayDescriptor->deactivate = NULL;
		tapeDelayDescriptor->instantiate = instantiateTapeDelay;
		tapeDelayDescriptor->run = runTapeDelay;
		tapeDelayDescriptor->run_adding = runAddingTapeDelay;
		tapeDelayDescriptor->set_run_adding_gain = setRunAddingGainTapeDelay;
	}
}

void _fini() {
	int i;

	if (tapeDelayDescriptor) {
		free((char *)tapeDelayDescriptor->Label);
		free((char *)tapeDelayDescriptor->Name);
		free((char *)tapeDelayDescriptor->Maker);
		free((char *)tapeDelayDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)tapeDelayDescriptor->PortDescriptors);
		for (i = 0; i < tapeDelayDescriptor->PortCount; i++)
			free((char *)(tapeDelayDescriptor->PortNames[i]));
		free((char **)tapeDelayDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)tapeDelayDescriptor->PortRangeHints);
		free(tapeDelayDescriptor);
	}

}
