/* rpackagecache.cc - package cache wrapper
 * 
 * Copyright (c) 2000, 2001 Conectiva S/A 
 * 
 * Author: Alfredo K. Kojima <kojima@conectiva.com.br>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include "config.h"

#include "rpackagecache.h"

#include "i18n.h"

#include <apt-pkg/error.h>
#include <apt-pkg/sourcelist.h>
#include <apt-pkg/pkgcachegen.h>
#include <apt-pkg/configuration.h>
#ifdef HAVE_RPM
# include <apt-pkg/systemfactory.h>
#else
# include <apt-pkg/pkgsystem.h>
# include <apt-pkg/policy.h>
#endif



bool RPackageCache::open(OpProgress &progress)
{
#ifdef HAVE_RPM
    _lock = new pkgRpmLock(true);
#else
    _system->Lock();
#endif

    if (_error->PendingError())
	return false;
    
    // Read the source list
    pkgSourceList list;
    if (!list.ReadMainList())
       return _error->Error(_("The list of sources could not be read."));
    
#ifdef HAVE_RPM
    _system->makeStatusCache(list, progress);
    progress.Done();
#else
    pkgMakeStatusCache(list, progress);
#endif

    if (_error->PendingError())
	return _error->Error(_("The package lists or status file could not be parsed or opened."));

    // Open the cache file
    FileFd File(_config->FindFile("Dir::Cache::pkgcache"),FileFd::ReadOnly);
    if (_error->PendingError())
	return false;

    _map = new MMap(File,MMap::Public | MMap::ReadOnly);
    if (_error->PendingError())
	return false;
    
    // Create the dependency cache
#ifdef HAVE_RPM
    _dcache = new pkgDepCache(*_map, progress);
#else
    _cache = new pkgCache(_map);
    if (_error->PendingError())
       return false;
    
    // The policy engine
    _policy = new pkgPolicy(_cache);
    if (_error->PendingError() == true)
       return false;
    if (ReadPinFile(*_policy) == false)
       return false;
    
    _dcache = new pkgDepCache(_cache, _policy);
    
    _dcache->Init(&progress);
#endif

    progress.Done();
    if (_error->PendingError())
	return false;

    // Check that the system is OK
    if (_dcache->DelCount() != 0 || _dcache->InstCount() != 0)
	return _error->Error("Internal Error, non-zero counts");

    _locked = true;

    return true;
}



bool RPackageCache::reset(OpProgress &progress)
{
    delete _dcache;
    delete _map;

    releaseLock();

#ifdef HAVE_RPM
    if (_lock) {
	delete _lock;
	_lock = 0;
    }
#endif
    return open(progress);
}



bool RPackageCache::lock()
{
    if (_locked)
       return true;

#ifdef HAVE_RPM
    if (_lock) {
	_lock->GetLock(true);
    }
#else    
    _system->Lock();
#endif
    _locked = true;
}


void RPackageCache::releaseLock()
{
    if (!_locked)
	return;

#ifdef HAVE_RPM
    if (_lock) {
	_lock->Close();
    }
#else    
    _system->UnLock();
#endif
    _locked = false;
}
